/**
 * Copyright 2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.micrometer.tracing;

import io.micrometer.common.lang.Nullable;

import java.util.Collections;
import java.util.Map;

/**
 * Manages {@link Baggage} entries. Upon retrieval / creation of a baggage entry puts it
 * in scope. Scope must be closed.
 *
 * @author OpenTelemetry Authors
 * @author Marcin Grzejszczak
 * @since 1.0.0
 */
public interface BaggageManager {

    /**
     * A noop implementation.
     */
    BaggageManager NOOP = new BaggageManager() {
        @Override
        public Map<String, String> getAllBaggage() {
            return Collections.emptyMap();
        }

        @Override
        public Baggage getBaggage(String name) {
            return Baggage.NOOP;
        }

        @Override
        public Baggage getBaggage(TraceContext traceContext, String name) {
            return Baggage.NOOP;
        }

        @Override
        public Baggage createBaggage(String name) {
            return Baggage.NOOP;
        }

        @Override
        public Baggage createBaggage(String name, String value) {
            return Baggage.NOOP;
        }
    };

    /**
     * @return mapping of all baggage entries from the given scope
     */
    Map<String, String> getAllBaggage();

    /**
     * Retrieves {@link Baggage} for the given name.
     * @param name baggage name
     * @return baggage if present or creates a new one if missing with {@code null} value
     */
    Baggage getBaggage(String name);

    /**
     * Retrieves {@link Baggage} for the given name.
     * @param traceContext trace context with baggage attached to it
     * @param name baggage name
     * @return baggage or {@code null} if not present on the {@link TraceContext}
     */
    @Nullable
    Baggage getBaggage(TraceContext traceContext, String name);

    /**
     * Creates a new {@link Baggage} entry for the given name or returns an existing one
     * if it's already present.
     * @param name baggage name
     * @return new or already created baggage
     */
    Baggage createBaggage(String name);

    /**
     * Creates a new {@link Baggage} entry for the given name or returns an existing one
     * if it's already present.
     * @param name baggage name
     * @param value baggage value
     * @return new or already created baggage
     */
    Baggage createBaggage(String name, String value);

}
