/*
 * Copyright 2017-2020 original authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.micronaut.configuration.kafka.serde;

import io.micronaut.context.BeanContext;
import io.micronaut.core.reflect.ClassUtils;
import jakarta.inject.Singleton;
import org.apache.kafka.common.serialization.Serde;
import org.apache.kafka.common.serialization.Serdes;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * A {@link SerdeRegistry} that computes {@link Serde} instances that use Jackson to JSON serialization.
 *
 * @author Graeme Rocher
 * @since 1.0
 */
@Singleton
public class JsonSerdeRegistry implements SerdeRegistry {

    private final BeanContext beanContext;
    private final Map<Class, JsonObjectSerde> serdes = new ConcurrentHashMap<>();

    /**
     * Constructs a new instance.
     *
     * @param beanContext The bean context
     */
    protected JsonSerdeRegistry(BeanContext beanContext) {
        this.beanContext = beanContext;
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> Serde<T> getSerde(Class<T> type) {
        if (ClassUtils.isJavaBasicType(type)) {
            return (Serde<T>) Serdes.serdeFrom(String.class);
        } else {
            JsonObjectSerde jsonSerde = serdes.get(type);
            if (jsonSerde == null) {
                jsonSerde = beanContext.createBean(JsonObjectSerde.class, type);
                serdes.put(type, jsonSerde);
            }
            return jsonSerde;
        }
    }
}
