/*
 * Copyright 2017-2020 original authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.micronaut.core.beans;

import io.micronaut.core.annotation.NonNull;
import io.micronaut.core.annotation.Nullable;
import io.micronaut.core.annotation.AnnotationMetadata;
import io.micronaut.core.annotation.Internal;
import io.micronaut.core.annotation.UsedByGeneratedCode;
import io.micronaut.core.type.Argument;
import io.micronaut.core.type.ReturnType;

import java.util.Map;

/**
 * Abstract implementation of the {@link BeanMethod} interface.
 * @param <B> The bean type
 * @param <T> The return type
 *
 * @author graemerocher
 * @since 2.3.0
 */
@Internal
@UsedByGeneratedCode
public abstract class AbstractBeanMethod<B, T> implements BeanMethod<B, T> {
    private final String name;
    private final AnnotationMetadata annotationMetadata;
    private final Argument<?>[] arguments;
    private final Argument<T> returnType;
    private final BeanIntrospection<B> introspection;

    /**
     * Default constructor.
     * @param introspection The associated introspection
     * @param returnType The return type
     * @param name The name of the method
     * @param annotationMetadata The annotation metadata
     * @param arguments The argument types
     */
    @UsedByGeneratedCode
    protected AbstractBeanMethod(
            @NonNull BeanIntrospection<B> introspection,
            @NonNull Argument<T> returnType,
            @NonNull String name,
            @Nullable AnnotationMetadata annotationMetadata,
            @Nullable Argument<?>... arguments) {
        this.introspection = introspection;
        this.name = name;
        this.annotationMetadata = annotationMetadata == null ? AnnotationMetadata.EMPTY_METADATA : annotationMetadata;
        this.arguments = arguments == null ? Argument.ZERO_ARGUMENTS : arguments;
        this.returnType = returnType;
    }

    @NonNull
    @Override
    public BeanIntrospection<B> getDeclaringBean() {
        return introspection;
    }

    @Override
    public final @NonNull ReturnType<T> getReturnType() {
        return new ReturnType<>() {
            @Override
            public Class<T> getType() {
                return returnType.getType();
            }

            @Override
            @NonNull
            public Argument<T> asArgument() {
                return returnType;
            }

            @Override
            public Map<String, Argument<?>> getTypeVariables() {
                return returnType.getTypeVariables();
            }

            @NonNull
            @Override
            public AnnotationMetadata getAnnotationMetadata() {
                return returnType.getAnnotationMetadata();
            }
        };
    }

    @NonNull
    @Override
    public final AnnotationMetadata getAnnotationMetadata() {
        return annotationMetadata;
    }

    @NonNull
    @Override
    public final String getName() {
        return name;
    }

    @Override
    public final Argument<?>[] getArguments() {
        return arguments;
    }

    @SuppressWarnings("java:S2638")
    @Override
    public T invoke(@NonNull B instance, Object... arguments) {
        return invokeInternal(instance, arguments);
    }

    /**
     * Abstract implementation implemented by generated byte code.
     * @param instance The instance
     * @param arguments The arguments
     * @return The result
     */
    @UsedByGeneratedCode
    @Internal
    protected abstract T invokeInternal(B instance, Object... arguments);
}
