/*
 * Decompiled with CFR 0.152.
 */
package io.micronaut.http.client.netty;

import io.micronaut.core.annotation.Internal;
import io.micronaut.core.annotation.NonNull;
import io.micronaut.core.annotation.Nullable;
import io.micronaut.core.execution.DelayedExecutionFlow;
import io.micronaut.core.execution.ExecutionFlow;
import io.micronaut.http.client.HttpClientConfiguration;
import io.micronaut.http.client.exceptions.HttpClientException;
import io.micronaut.http.client.netty.BlockHint;
import io.micronaut.http.client.netty.ConnectionManager;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Deque;
import java.util.Iterator;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import org.slf4j.Logger;

@Internal
abstract class PoolResizer {
    private final Logger log;
    private final HttpClientConfiguration.ConnectionPoolConfiguration connectionPoolConfiguration;
    private final AtomicReference<WorkState> state = new AtomicReference<WorkState>(WorkState.IDLE);
    private final AtomicInteger pendingConnectionCount = new AtomicInteger(0);
    private final Deque<PendingRequest> pendingRequests = new ConcurrentLinkedDeque<PendingRequest>();
    private final ConnectionList http1Connections = new ConnectionList();
    private final ConnectionList http2Connections = new ConnectionList();

    PoolResizer(Logger log, HttpClientConfiguration.ConnectionPoolConfiguration connectionPoolConfiguration) {
        this.log = log;
        this.connectionPoolConfiguration = connectionPoolConfiguration;
    }

    private void dirty() {
        WorkState endState;
        WorkState before = this.state.getAndUpdate(ws -> {
            if (ws == WorkState.IDLE) {
                return WorkState.ACTIVE_WITHOUT_PENDING_WORK;
            }
            return WorkState.ACTIVE_WITH_PENDING_WORK;
        });
        if (before != WorkState.IDLE) {
            return;
        }
        do {
            try {
                this.doSomeWork();
            }
            catch (Throwable t) {
                this.state.set(WorkState.IDLE);
                throw t;
            }
        } while ((endState = this.state.updateAndGet(ws -> {
            if (ws == WorkState.ACTIVE_WITH_PENDING_WORK) {
                return WorkState.ACTIVE_WITHOUT_PENDING_WORK;
            }
            return WorkState.IDLE;
        })) != WorkState.IDLE);
    }

    private ResizerConnection[] sort(PendingRequest request, ConnectionList connections) {
        ResizerConnection[] items = connections.unsafeItems;
        if (items.length == 0) {
            return items;
        }
        HttpClientConfiguration.ConnectionPoolConfiguration.ConnectionLocality locality = this.connectionPoolConfiguration.getConnectionLocality();
        if (locality == HttpClientConfiguration.ConnectionPoolConfiguration.ConnectionLocality.PREFERRED) {
            int copies = 0;
            for (int i = 1; i < items.length; ++i) {
                ResizerConnection connection = items[i];
                if (!connection.inEventLoop(request.requestingThread)) continue;
                System.arraycopy(items, 0, items, 1, i);
                items[0] = connection;
                if (copies++ <= 4) {
                    continue;
                }
                break;
            }
        } else if (locality == HttpClientConfiguration.ConnectionPoolConfiguration.ConnectionLocality.ENFORCED_IF_SAME_GROUP || locality == HttpClientConfiguration.ConnectionPoolConfiguration.ConnectionLocality.ENFORCED_ALWAYS) {
            ArrayList<ResizerConnection> options = new ArrayList<ResizerConnection>();
            for (ResizerConnection item : items) {
                if (!item.inEventLoop(request.requestingThread)) continue;
                options.add(item);
            }
            if (!options.isEmpty() || locality == HttpClientConfiguration.ConnectionPoolConfiguration.ConnectionLocality.ENFORCED_ALWAYS || this.containsThread(request.requestingThread)) {
                return options.toArray(new ResizerConnection[0]);
            }
        }
        return items;
    }

    private void doSomeWork() {
        PendingRequest toDispatch;
        BlockHint blockedPendingRequests = null;
        while ((toDispatch = this.pendingRequests.pollFirst()) != null) {
            boolean dispatched = false;
            for (ResizerConnection c : this.sort(toDispatch, this.http2Connections)) {
                if (!this.dispatchSafe(c, toDispatch)) continue;
                dispatched = true;
                break;
            }
            if (!dispatched) {
                for (ResizerConnection c : this.sort(toDispatch, this.http1Connections)) {
                    if (!this.dispatchSafe(c, toDispatch)) continue;
                    dispatched = true;
                    break;
                }
            }
            if (dispatched) continue;
            this.pendingRequests.addFirst(toDispatch);
            blockedPendingRequests = BlockHint.combine(blockedPendingRequests, toDispatch.blockHint);
            break;
        }
        int pendingRequestCount = this.pendingRequests.size();
        int pendingConnectionCount = this.pendingConnectionCount.get();
        int http1ConnectionCount = this.http1Connections.unsafeItems.length;
        int http2ConnectionCount = this.http2Connections.unsafeItems.length;
        if (pendingRequestCount == 0) {
            return;
        }
        int connectionsToOpen = pendingRequestCount - pendingConnectionCount;
        connectionsToOpen = Math.min(connectionsToOpen, this.connectionPoolConfiguration.getMaxPendingConnections() - pendingConnectionCount);
        if (http1ConnectionCount > 0 || http2ConnectionCount == 0) {
            connectionsToOpen = Math.min(connectionsToOpen, this.connectionPoolConfiguration.getMaxConcurrentHttp1Connections() - http1ConnectionCount);
        }
        if (http2ConnectionCount > 0 || http1ConnectionCount == 0) {
            connectionsToOpen = Math.min(connectionsToOpen, this.connectionPoolConfiguration.getMaxConcurrentHttp2Connections() - http2ConnectionCount);
        }
        if (connectionsToOpen > 0) {
            Iterator<PendingRequest> pendingRequestIterator = this.pendingRequests.iterator();
            if (!pendingRequestIterator.hasNext()) {
                return;
            }
            Thread preferredThread = pendingRequestIterator.next().requestingThread;
            this.pendingConnectionCount.addAndGet(connectionsToOpen);
            for (int i = 0; i < connectionsToOpen; ++i) {
                try {
                    this.openNewConnection(blockedPendingRequests, preferredThread);
                }
                catch (Exception e) {
                    try {
                        this.onNewConnectionFailure(e);
                    }
                    catch (Exception f) {
                        this.log.error("Internal error", (Throwable)f);
                    }
                }
                if (!pendingRequestIterator.hasNext()) continue;
                preferredThread = pendingRequestIterator.next().requestingThread;
            }
            this.dirty();
        }
    }

    private boolean dispatchSafe(ResizerConnection connection, PendingRequest toDispatch) {
        try {
            return connection.dispatch(toDispatch);
        }
        catch (Exception e) {
            try {
                if (!toDispatch.tryCompleteExceptionally(e)) {
                    this.log.debug("Failure during connection dispatch operation, but dispatch request was already complete.", (Throwable)e);
                }
            }
            catch (Exception f) {
                this.log.error("Internal error", (Throwable)f);
            }
            return true;
        }
    }

    abstract void openNewConnection(@Nullable BlockHint var1, @NonNull Thread var2) throws Exception;

    abstract boolean containsThread(@NonNull Thread var1);

    static boolean incrementWithLimit(AtomicInteger variable, int limit) {
        int old;
        do {
            if ((old = variable.get()) < limit) continue;
            return false;
        } while (!variable.compareAndSet(old, old + 1));
        return true;
    }

    void onNewConnectionFailure(@Nullable Throwable error) throws Exception {
        this.pendingConnectionCount.decrementAndGet();
        this.dirty();
    }

    final void onNewConnectionEstablished1(ResizerConnection connection) {
        this.http1Connections.add(connection);
        this.pendingConnectionCount.decrementAndGet();
        this.dirty();
    }

    final void onNewConnectionEstablished2(ResizerConnection connection) {
        this.http2Connections.add(connection);
        this.pendingConnectionCount.decrementAndGet();
        this.dirty();
    }

    final void onConnectionInactive1(ResizerConnection connection) {
        this.http1Connections.remove(connection);
        this.dirty();
    }

    final void onConnectionInactive2(ResizerConnection connection) {
        this.http2Connections.remove(connection);
        this.dirty();
    }

    final void addPendingRequest(PendingRequest sink) {
        int maxPendingAcquires = this.connectionPoolConfiguration.getMaxPendingAcquires();
        if (maxPendingAcquires != Integer.MAX_VALUE && this.pendingRequests.size() >= maxPendingAcquires) {
            sink.tryCompleteExceptionally(new HttpClientException("Cannot acquire connection, exceeded max pending acquires configuration"));
            return;
        }
        this.pendingRequests.addLast(sink);
        this.dirty();
    }

    final PendingRequest pollPendingRequest() {
        PendingRequest req = this.pendingRequests.pollFirst();
        if (req != null) {
            this.dirty();
        }
        return req;
    }

    final void markConnectionAvailable() {
        this.dirty();
    }

    final void forEachConnection(Consumer<ResizerConnection> c) {
        this.http1Connections.forEach(c);
        this.http2Connections.forEach(c);
    }

    private static enum WorkState {
        IDLE,
        ACTIVE_WITH_PENDING_WORK,
        ACTIVE_WITHOUT_PENDING_WORK;

    }

    private static final class ConnectionList {
        private static final ResizerConnection[] EMPTY = new ResizerConnection[0];
        private final Lock lock = new ReentrantLock();
        private volatile ResizerConnection[] unsafeItems = EMPTY;
        private ResizerConnection[] safeItems = EMPTY;

        private ConnectionList() {
        }

        void forEach(Consumer<ResizerConnection> c) {
            ResizerConnection[] items;
            this.lock.lock();
            try {
                items = this.safeItems;
            }
            finally {
                this.lock.unlock();
            }
            for (ResizerConnection item : items) {
                c.accept(item);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void add(ResizerConnection connection) {
            this.lock.lock();
            try {
                ResizerConnection[] prev = this.safeItems;
                ResizerConnection[] next = Arrays.copyOf(prev, prev.length + 1);
                next[prev.length] = connection;
                this.safeItems = next;
                this.unsafeItems = (ResizerConnection[])next.clone();
            }
            finally {
                this.lock.unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void remove(ResizerConnection connection) {
            this.lock.lock();
            try {
                ResizerConnection[] prev = this.safeItems;
                int index = Arrays.asList(prev).indexOf(connection);
                if (index == -1) {
                    return;
                }
                ResizerConnection[] next = Arrays.copyOf(prev, prev.length - 1);
                System.arraycopy(prev, index + 1, next, index, prev.length - index - 1);
                this.safeItems = next;
                this.unsafeItems = (ResizerConnection[])next.clone();
            }
            finally {
                this.lock.unlock();
            }
        }
    }

    static abstract class ResizerConnection {
        ResizerConnection() {
        }

        abstract boolean inEventLoop(Thread var1);

        abstract boolean dispatch(PendingRequest var1) throws Exception;
    }

    static final class PendingRequest
    extends AtomicBoolean {
        final Thread requestingThread = Thread.currentThread();
        @Nullable
        final BlockHint blockHint;
        private final DelayedExecutionFlow<ConnectionManager.PoolHandle> sink = DelayedExecutionFlow.create();

        PendingRequest(@Nullable BlockHint blockHint) {
            this.blockHint = blockHint;
        }

        ExecutionFlow<ConnectionManager.PoolHandle> flow() {
            return this.sink;
        }

        boolean tryCompleteExceptionally(Throwable t) {
            if (this.compareAndSet(false, true)) {
                this.sink.completeExceptionally(t);
                return true;
            }
            return false;
        }

        boolean tryComplete(ConnectionManager.PoolHandle value) {
            if (this.compareAndSet(false, true)) {
                if (this.sink.isCancelled()) {
                    return false;
                }
                this.sink.complete((Object)value);
                return true;
            }
            return false;
        }
    }
}

