/*
 * Decompiled with CFR 0.152.
 */
package io.micronaut.security.token.jwt.validator;

import com.nimbusds.jose.EncryptionMethod;
import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWEAlgorithm;
import com.nimbusds.jose.JWEHeader;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jwt.EncryptedJWT;
import com.nimbusds.jwt.JWT;
import com.nimbusds.jwt.JWTParser;
import com.nimbusds.jwt.PlainJWT;
import com.nimbusds.jwt.SignedJWT;
import io.micronaut.security.token.jwt.encryption.EncryptionConfiguration;
import io.micronaut.security.token.jwt.generator.claims.JwtClaims;
import io.micronaut.security.token.jwt.generator.claims.JwtClaimsSetAdapter;
import io.micronaut.security.token.jwt.signature.SignatureConfiguration;
import io.micronaut.security.token.jwt.validator.JwtClaimsValidator;
import java.text.ParseException;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import javax.annotation.Nonnull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class JwtTokenValidatorUtils {
    private static final Logger LOG = LoggerFactory.getLogger(JwtTokenValidatorUtils.class);

    private JwtTokenValidatorUtils() {
    }

    public static Optional<JWT> validatePlainJWTSignature(JWT jwt, List<SignatureConfiguration> signatureConfigurations) {
        if (signatureConfigurations.isEmpty()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("JWT is not signed and no signature configurations -> verified");
            }
            return Optional.of(jwt);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("A non-signed JWT cannot be accepted as signature configurations have been defined");
        }
        return Optional.empty();
    }

    public static Optional<JWT> validateSignedJWTSignature(SignedJWT signedJWT, List<SignatureConfiguration> signatureConfigurations) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("JWT is signed");
        }
        JWSAlgorithm algorithm = signedJWT.getHeader().getAlgorithm();
        for (SignatureConfiguration config : signatureConfigurations) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Using signature configuration: {}", (Object)config.toString());
            }
            try {
                if (config.verify(signedJWT)) {
                    return Optional.of(signedJWT);
                }
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("JWT Signature verification failed: {}", (Object)signedJWT.getParsedString());
            }
            catch (JOSEException e) {
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("Verification fails with signature configuration: {}, passing to the next one", (Object)config);
            }
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("No signature algorithm found for JWT: {}", (Object)signedJWT.getParsedString());
        }
        return Optional.empty();
    }

    public static boolean verifyClaims(JwtClaims claims, Collection<? extends JwtClaimsValidator> claimsValidators) {
        return claimsValidators.stream().allMatch(jwtClaimsValidator -> jwtClaimsValidator.validate(claims));
    }

    public static Optional<JWT> validateEncryptedJWTSignature(@Nonnull EncryptedJWT encryptedJWT, @Nonnull String token, @Nonnull List<SignatureConfiguration> signatureConfigurations, @Nonnull List<EncryptionConfiguration> encryptionConfigurations) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("JWT is encrypted");
        }
        JWEHeader header = encryptedJWT.getHeader();
        JWEAlgorithm algorithm = header.getAlgorithm();
        EncryptionMethod method = header.getEncryptionMethod();
        for (EncryptionConfiguration config : encryptionConfigurations) {
            if (!config.supports(algorithm, method)) continue;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Using encryption configuration: {}", (Object)config.toString());
            }
            try {
                config.decrypt(encryptedJWT);
                SignedJWT signedJWT = encryptedJWT.getPayload().toSignedJWT();
                if (signedJWT == null) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("encrypted JWT could couldn't be converted to a signed JWT.");
                    }
                    return Optional.empty();
                }
                return JwtTokenValidatorUtils.validateSignedJWTSignature(signedJWT, signatureConfigurations);
            }
            catch (JOSEException e) {
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("Decryption fails with encryption configuration: {}, passing to the next one", (Object)config.toString());
            }
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("No encryption algorithm found for JWT: {}", (Object)token);
        }
        return Optional.empty();
    }

    public static Optional<JWT> validateJwtSignatureAndClaims(String token, Collection<? extends JwtClaimsValidator> claimsValidators, List<SignatureConfiguration> signatureConfigurations, List<EncryptionConfiguration> encryptionConfigurations) {
        block4: {
            Optional<JWT> jwt = JwtTokenValidatorUtils.parseJwtIfValidSignature(token, signatureConfigurations, encryptionConfigurations);
            if (jwt.isPresent()) {
                try {
                    if (JwtTokenValidatorUtils.verifyClaims(new JwtClaimsSetAdapter(jwt.get().getJWTClaimsSet()), claimsValidators)) {
                        return jwt;
                    }
                }
                catch (ParseException e) {
                    if (!LOG.isErrorEnabled()) break block4;
                    LOG.error("ParseException creating authentication", (Object)e.getMessage());
                }
            }
        }
        return Optional.empty();
    }

    public static Optional<JWT> parseJwtIfValidSignature(String token, List<SignatureConfiguration> signatureConfigurations, List<EncryptionConfiguration> encryptionConfigurations) {
        block5: {
            try {
                JWT jwt = JWTParser.parse((String)token);
                if (jwt instanceof PlainJWT) {
                    return JwtTokenValidatorUtils.validatePlainJWTSignature(jwt, signatureConfigurations);
                }
                if (jwt instanceof EncryptedJWT) {
                    EncryptedJWT encryptedJWT = (EncryptedJWT)jwt;
                    return JwtTokenValidatorUtils.validateEncryptedJWTSignature(encryptedJWT, token, signatureConfigurations, encryptionConfigurations);
                }
                if (jwt instanceof SignedJWT) {
                    SignedJWT signedJWT = (SignedJWT)jwt;
                    return JwtTokenValidatorUtils.validateSignedJWTSignature(signedJWT, signatureConfigurations);
                }
            }
            catch (ParseException e) {
                if (!LOG.isDebugEnabled()) break block5;
                LOG.debug("Cannot decrypt / verify JWT: {}", (Object)e.getMessage());
            }
        }
        return Optional.empty();
    }
}

