/*
 * Copyright 2017-2023 original authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.micronaut.security.rules;

import io.micronaut.context.annotation.Requires;
import io.micronaut.core.annotation.Nullable;
import io.micronaut.http.HttpRequest;
import io.micronaut.security.authentication.Authentication;
import io.micronaut.security.config.SecurityConfiguration;
import io.micronaut.security.config.SecurityConfigurationProperties;
import io.micronaut.security.token.RolesFinder;
import jakarta.inject.Singleton;
import java.net.InetSocketAddress;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Mono;

import static io.micronaut.security.utils.LoggingUtils.debug;

/**
 * A security rule implementation backed by the {@link SecurityConfigurationProperties#getIpPatterns()} ()}.
 *
 * @author Sergio del Amo
 * @since 1.0
 */
@Requires(classes = HttpRequest.class)
@Singleton
public class IpPatternsRule extends AbstractSecurityRule<HttpRequest<?>> {

    /**
     * The order of the rule.
     */
    public static final Integer ORDER = SecuredAnnotationRule.ORDER - 100;

    private static final Logger LOG = LoggerFactory.getLogger(IpPatternsRule.class);

    private final List<Pattern> patternList;

    /**
     * @param rolesFinder Roles Parser
     * @param securityConfiguration Security Configuration
     */
    public IpPatternsRule(RolesFinder rolesFinder,
                          SecurityConfiguration securityConfiguration) {
        super(rolesFinder);
        this.patternList = securityConfiguration.getIpPatterns()
                .stream()
                .map(Pattern::compile)
                .collect(Collectors.toList());
    }

    @Override
    public int getOrder() {
        return ORDER;
    }

    @Override
    public Publisher<SecurityRuleResult> check(HttpRequest<?> request, @Nullable Authentication authentication) {

        if (patternList.isEmpty()) {
            debug(LOG, "No IP patterns provided. Skipping host address check.");
            return Mono.just(SecurityRuleResult.UNKNOWN);
        } else {
            try {
                InetSocketAddress socketAddress = request.getRemoteAddress();
                //noinspection ConstantConditions https://github.com/micronaut-projects/micronaut-security/issues/186
                if (socketAddress == null) {
                    debug(LOG, "Request remote address was not found. Continuing request processing.");
                    return Mono.just(SecurityRuleResult.UNKNOWN);
                }
                if (socketAddress.getAddress() == null) {
                    debug(LOG, "Could not resolve the InetAddress. Continuing request processing.");
                    return Mono.just(SecurityRuleResult.UNKNOWN);
                }
                String hostAddress = socketAddress.getAddress().getHostAddress();
                if (patternList.stream().anyMatch(pattern ->
                        pattern.pattern().equals(SecurityConfigurationProperties.ANYWHERE) ||
                                pattern.matcher(hostAddress).matches())) {
                            debug(LOG, "One or more of the IP patterns matched the host address [{}]. Continuing request processing.", hostAddress);
                            return Mono.just(SecurityRuleResult.UNKNOWN);
                } else {
                    debug(LOG, "None of the IP patterns [{}] matched the host address [{}]. Rejecting the request.", patternList.stream().map(Pattern::pattern).collect(Collectors.toList()), hostAddress);
                    return Mono.just(SecurityRuleResult.REJECTED);
                }
            } catch (IllegalArgumentException e) {
                debug(LOG, "IllegalArgumentException thrown while getting the request remote address. Continuing request processing.");
                return Mono.just(SecurityRuleResult.UNKNOWN);
            }
        }
    }
}
