/*
 * Decompiled with CFR 0.152.
 */
package io.micronaut.serde.jackson;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import io.micronaut.core.annotation.Internal;
import io.micronaut.core.annotation.NonNull;
import io.micronaut.core.annotation.Nullable;
import io.micronaut.core.type.Argument;
import io.micronaut.json.tree.JsonNode;
import io.micronaut.serde.Decoder;
import io.micronaut.serde.exceptions.InvalidFormatException;
import io.micronaut.serde.exceptions.SerdeException;
import io.micronaut.serde.support.util.JsonNodeDecoder;
import java.io.EOFException;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

@Internal
public final class JacksonDecoder
implements Decoder {
    @Internal
    private final JsonParser parser;
    private JsonToken currentToken;
    private boolean currentlyUnwrappingArray;
    private int depth = 0;

    private JacksonDecoder(JsonParser parser) throws IOException {
        this.parser = parser;
        if (!parser.hasCurrentToken()) {
            this.currentToken = parser.nextToken();
            if (!parser.hasCurrentToken()) {
                throw new EOFException("No JSON input to parse");
            }
        } else {
            this.currentToken = parser.currentToken();
        }
    }

    public static Decoder create(JsonParser parser) throws IOException {
        return new JacksonDecoder(parser);
    }

    public IOException createDeserializationException(String message, Object invalidValue) {
        if (invalidValue != null) {
            return new InvalidFormatException(message + " \n at " + this.parser.getCurrentLocation(), null, invalidValue);
        }
        return new SerdeException(message + " \n at " + this.parser.getCurrentLocation());
    }

    private IOException unexpectedToken(JsonToken expected) {
        return this.createDeserializationException("Unexpected token " + this.currentToken + ", expected " + expected, null);
    }

    public void finishStructure(boolean consumeLeftElements) throws IOException {
        if (consumeLeftElements) {
            while (this.currentToken != null && this.currentToken != JsonToken.END_ARRAY && this.currentToken != JsonToken.END_OBJECT) {
                if (this.currentToken == JsonToken.START_ARRAY || this.currentToken == JsonToken.START_OBJECT) {
                    this.parser.skipChildren();
                }
                this.currentToken = this.parser.nextToken();
            }
        } else if (this.currentToken != JsonToken.END_ARRAY && this.currentToken != JsonToken.END_OBJECT) {
            throw new IllegalStateException("Not all elements have been consumed yet");
        }
        if (--this.depth != 0) {
            this.nextToken();
        }
    }

    public void finishStructure() throws IOException {
        if (this.currentToken != JsonToken.END_ARRAY && this.currentToken != JsonToken.END_OBJECT) {
            throw new IllegalStateException("Not all elements have been consumed yet");
        }
        if (--this.depth != 0) {
            this.nextToken();
        }
    }

    public boolean hasNextArrayValue() {
        return this.currentToken != JsonToken.END_ARRAY;
    }

    @Nullable
    public String decodeKey() throws IOException {
        if (this.currentToken == JsonToken.END_OBJECT) {
            return null;
        }
        if (this.currentToken != JsonToken.FIELD_NAME) {
            throw new IllegalStateException("Not at a field name");
        }
        String fieldName = this.parser.getCurrentName();
        this.nextToken();
        return fieldName;
    }

    @NonNull
    public JacksonDecoder decodeArray(Argument<?> type) throws IOException {
        if (this.currentToken != JsonToken.START_ARRAY) {
            throw this.unexpectedToken(JsonToken.START_ARRAY);
        }
        ++this.depth;
        this.nextToken();
        return this;
    }

    public JacksonDecoder decodeArray() throws IOException {
        if (this.currentToken != JsonToken.START_ARRAY) {
            throw this.unexpectedToken(JsonToken.START_ARRAY);
        }
        ++this.depth;
        this.nextToken();
        return this;
    }

    @NonNull
    public JacksonDecoder decodeObject(Argument<?> type) throws IOException {
        if (this.currentToken != JsonToken.START_OBJECT) {
            throw this.unexpectedToken(JsonToken.START_OBJECT);
        }
        ++this.depth;
        this.nextToken();
        return this;
    }

    public JacksonDecoder decodeObject() throws IOException {
        if (this.currentToken != JsonToken.START_OBJECT) {
            throw this.unexpectedToken(JsonToken.START_OBJECT);
        }
        ++this.depth;
        this.nextToken();
        return this;
    }

    @NonNull
    public String decodeString() throws IOException {
        switch (this.currentToken) {
            case VALUE_STRING: {
                String value = this.parser.getText();
                this.nextToken();
                return value;
            }
            case START_ARRAY: {
                if (this.beginUnwrapArray(this.currentToken)) {
                    String unwrapped = this.decodeString();
                    if (this.endUnwrapArray()) {
                        return unwrapped;
                    }
                    throw this.createDeserializationException("Expected one string, but got array of multiple values", null);
                }
                throw this.unexpectedToken(JsonToken.VALUE_STRING);
            }
        }
        String def = this.parser.getValueAsString();
        this.nextToken();
        return def;
    }

    public boolean decodeBoolean() throws IOException {
        switch (this.currentToken) {
            case VALUE_TRUE: {
                this.nextToken();
                return true;
            }
            case VALUE_FALSE: {
                this.nextToken();
                return false;
            }
            case VALUE_NUMBER_FLOAT: {
                float floatValue = this.parser.getFloatValue();
                this.nextToken();
                return (double)floatValue != 0.0;
            }
            case START_ARRAY: {
                if (this.beginUnwrapArray(this.currentToken)) {
                    boolean unwrapped = this.decodeBoolean();
                    if (this.endUnwrapArray()) {
                        return unwrapped;
                    }
                    throw this.createDeserializationException("Expected one string, but got array of multiple values", null);
                }
                throw this.unexpectedToken(JsonToken.VALUE_TRUE);
            }
        }
        boolean value = this.parser.getValueAsBoolean();
        this.nextToken();
        return value;
    }

    public byte decodeByte() throws IOException {
        switch (this.currentToken) {
            case VALUE_TRUE: {
                this.nextToken();
                return 1;
            }
            case VALUE_FALSE: {
                this.nextToken();
                return 0;
            }
            case START_ARRAY: {
                if (this.beginUnwrapArray(this.currentToken)) {
                    byte unwrapped = this.decodeByte();
                    if (this.endUnwrapArray()) {
                        return unwrapped;
                    }
                    throw this.createDeserializationException("Expected one string, but got array of multiple values", null);
                }
                throw this.unexpectedToken(JsonToken.VALUE_NUMBER_INT);
            }
        }
        byte value = this.parser.getByteValue();
        this.nextToken();
        return value;
    }

    public short decodeShort() throws IOException {
        switch (this.currentToken) {
            case VALUE_TRUE: {
                this.nextToken();
                return 1;
            }
            case VALUE_FALSE: {
                this.nextToken();
                return 0;
            }
            case START_ARRAY: {
                if (this.beginUnwrapArray(this.currentToken)) {
                    byte unwrapped = this.decodeByte();
                    if (this.endUnwrapArray()) {
                        return unwrapped;
                    }
                    throw this.createDeserializationException("Expected one string, but got array of multiple values", null);
                }
                throw this.unexpectedToken(JsonToken.VALUE_NUMBER_INT);
            }
        }
        short value = this.parser.getShortValue();
        this.nextToken();
        return value;
    }

    public char decodeChar() throws IOException {
        switch (this.currentToken) {
            case START_ARRAY: {
                if (this.beginUnwrapArray(this.currentToken)) {
                    char unwrapped = this.decodeChar();
                    if (this.endUnwrapArray()) {
                        return unwrapped;
                    }
                    throw this.createDeserializationException("Expected one string, but got array of multiple values", null);
                }
                throw this.unexpectedToken(JsonToken.VALUE_NUMBER_INT);
            }
            case VALUE_STRING: {
                String string = this.parser.getText();
                if (string.length() != 1) {
                    throw this.createDeserializationException("When decoding char value, must give a single character", string);
                }
                char c = string.charAt(0);
                this.nextToken();
                return c;
            }
            case VALUE_NUMBER_INT: {
                char value = (char)this.parser.getIntValue();
                this.nextToken();
                return value;
            }
        }
        String text = this.parser.getText();
        if (text.length() == 0) {
            throw this.createDeserializationException("No characters found", text);
        }
        this.nextToken();
        return text.charAt(0);
    }

    public int decodeInt() throws IOException {
        switch (this.currentToken) {
            case VALUE_NUMBER_INT: {
                int value = this.parser.getIntValue();
                this.nextToken();
                return value;
            }
            case VALUE_STRING: {
                int value;
                String string = this.parser.getText();
                try {
                    value = Integer.parseInt(string);
                }
                catch (NumberFormatException e) {
                    throw this.createDeserializationException("Unable to coerce string to integer", string);
                }
                this.nextToken();
                return value;
            }
            case START_ARRAY: {
                if (this.beginUnwrapArray(this.currentToken)) {
                    int unwrapped = this.decodeInt();
                    if (this.endUnwrapArray()) {
                        return unwrapped;
                    }
                    throw this.createDeserializationException("Expected one string, but got array of multiple values", null);
                }
                throw this.unexpectedToken(JsonToken.VALUE_NUMBER_INT);
            }
            case VALUE_FALSE: {
                this.nextToken();
                return 0;
            }
            case VALUE_TRUE: {
                this.nextToken();
                return 1;
            }
        }
        int value = this.parser.getValueAsInt();
        this.nextToken();
        return value;
    }

    public long decodeLong() throws IOException {
        switch (this.currentToken) {
            case VALUE_NUMBER_INT: {
                long value = this.parser.getLongValue();
                this.nextToken();
                return value;
            }
            case VALUE_STRING: {
                long value;
                String string = this.parser.getText();
                try {
                    value = Long.parseLong(string);
                }
                catch (NumberFormatException e) {
                    throw this.createDeserializationException("Unable to coerce string to integer", string);
                }
                this.nextToken();
                return value;
            }
            case VALUE_FALSE: {
                this.nextToken();
                return 0L;
            }
            case VALUE_TRUE: {
                this.nextToken();
                return 1L;
            }
            case START_ARRAY: {
                if (this.beginUnwrapArray(this.currentToken)) {
                    long unwrapped = this.decodeLong();
                    if (this.endUnwrapArray()) {
                        return unwrapped;
                    }
                    throw this.createDeserializationException("Expected one string, but got array of multiple values", null);
                }
                throw this.unexpectedToken(JsonToken.VALUE_NUMBER_INT);
            }
        }
        long value = this.parser.getValueAsLong();
        this.nextToken();
        return value;
    }

    public float decodeFloat() throws IOException {
        switch (this.currentToken) {
            case VALUE_STRING: {
                float value;
                String string = this.parser.getText();
                try {
                    value = Float.parseFloat(string);
                }
                catch (NumberFormatException e) {
                    throw this.createDeserializationException("Unable to coerce string to float", string);
                }
                this.nextToken();
                return value;
            }
            case START_ARRAY: {
                if (this.beginUnwrapArray(this.currentToken)) {
                    float unwrapped = this.decodeFloat();
                    if (this.endUnwrapArray()) {
                        return unwrapped;
                    }
                    throw this.createDeserializationException("Expected one string, but got array of multiple values", null);
                }
                throw this.unexpectedToken(JsonToken.VALUE_NUMBER_FLOAT);
            }
            case VALUE_FALSE: {
                this.nextToken();
                return 0.0f;
            }
            case VALUE_TRUE: {
                this.nextToken();
                return 1.0f;
            }
        }
        float value = this.parser.getFloatValue();
        this.nextToken();
        return value;
    }

    public double decodeDouble() throws IOException {
        switch (this.currentToken) {
            case VALUE_NUMBER_FLOAT: 
            case VALUE_NUMBER_INT: {
                double value = this.parser.getDoubleValue();
                this.nextToken();
                return value;
            }
            case VALUE_STRING: {
                double value;
                String string = this.parser.getText();
                try {
                    value = Double.parseDouble(string);
                }
                catch (NumberFormatException e) {
                    throw this.createDeserializationException("Unable to coerce string to double", string);
                }
                this.nextToken();
                return value;
            }
            case START_ARRAY: {
                if (this.beginUnwrapArray(this.currentToken)) {
                    double unwrapped = this.decodeDouble();
                    if (this.endUnwrapArray()) {
                        return unwrapped;
                    }
                    throw this.createDeserializationException("Expected one string, but got array of multiple values", null);
                }
                throw this.unexpectedToken(JsonToken.VALUE_NUMBER_INT);
            }
            case VALUE_FALSE: {
                this.nextToken();
                return 0.0;
            }
            case VALUE_TRUE: {
                this.nextToken();
                return 1.0;
            }
        }
        double value = this.parser.getValueAsDouble();
        this.nextToken();
        return value;
    }

    @NonNull
    public BigInteger decodeBigInteger() throws IOException {
        switch (this.currentToken) {
            case VALUE_STRING: {
                BigInteger value;
                String string = this.parser.getText();
                try {
                    value = new BigInteger(string);
                }
                catch (NumberFormatException e) {
                    throw this.createDeserializationException("Unable to coerce string to integer", string);
                }
                this.nextToken();
                return value;
            }
            case START_ARRAY: {
                if (this.beginUnwrapArray(this.currentToken)) {
                    BigInteger unwrapped = this.decodeBigInteger();
                    if (this.endUnwrapArray()) {
                        return unwrapped;
                    }
                    throw this.createDeserializationException("Expected one string, but got array of multiple values", null);
                }
                throw this.unexpectedToken(JsonToken.VALUE_NUMBER_INT);
            }
            case VALUE_FALSE: {
                this.nextToken();
                return BigInteger.ZERO;
            }
            case VALUE_TRUE: {
                this.nextToken();
                return BigInteger.ONE;
            }
        }
        BigInteger value = this.parser.getBigIntegerValue();
        this.nextToken();
        return value;
    }

    @NonNull
    public BigDecimal decodeBigDecimal() throws IOException {
        switch (this.currentToken) {
            case VALUE_STRING: {
                BigDecimal value;
                String string = this.parser.getText();
                try {
                    value = new BigDecimal(string);
                }
                catch (NumberFormatException e) {
                    throw this.createDeserializationException("Unable to coerce string to BigDecimal", string);
                }
                this.nextToken();
                return value;
            }
            case START_ARRAY: {
                if (this.beginUnwrapArray(this.currentToken)) {
                    BigDecimal unwrapped = this.decodeBigDecimal();
                    if (this.endUnwrapArray()) {
                        return unwrapped;
                    }
                    throw this.createDeserializationException("Expected one string, but got array of multiple values", null);
                }
                throw this.unexpectedToken(JsonToken.VALUE_NUMBER_INT);
            }
            case VALUE_FALSE: {
                this.nextToken();
                return BigDecimal.ZERO;
            }
            case VALUE_TRUE: {
                this.nextToken();
                return BigDecimal.ONE;
            }
        }
        BigDecimal value = this.parser.getDecimalValue();
        this.nextToken();
        return value;
    }

    private Number decodeNumber() throws IOException {
        Number number = this.parser.getNumberValue();
        this.nextToken();
        return number;
    }

    public boolean decodeNull() throws IOException {
        if (this.currentToken == JsonToken.VALUE_NULL) {
            this.nextToken();
            return true;
        }
        return false;
    }

    private boolean beginUnwrapArray(JsonToken currentToken) throws IOException {
        if (this.currentlyUnwrappingArray) {
            return false;
        }
        if (currentToken != JsonToken.START_ARRAY) {
            throw new IllegalStateException("Not an array");
        }
        this.currentlyUnwrappingArray = true;
        this.nextToken();
        return true;
    }

    private void nextToken() throws IOException {
        this.currentToken = this.parser.nextToken();
    }

    private boolean endUnwrapArray() throws IOException {
        this.currentlyUnwrappingArray = false;
        if (this.currentToken == JsonToken.END_ARRAY) {
            this.nextToken();
            return true;
        }
        return false;
    }

    public Decoder decodeBuffer() throws IOException {
        JsonNode node = this.decodeNode();
        return JsonNodeDecoder.create((JsonNode)node);
    }

    @NonNull
    public JsonNode decodeNode() throws IOException {
        return switch (this.currentToken) {
            case JsonToken.START_OBJECT -> JacksonDecoder.decodeObjectNode(this.decodeObject());
            case JsonToken.START_ARRAY -> JacksonDecoder.decodeArrayNode(this.decodeArray());
            case JsonToken.VALUE_STRING -> JsonNode.createStringNode((String)this.decodeString());
            case JsonToken.VALUE_NUMBER_FLOAT, JsonToken.VALUE_NUMBER_INT -> {
                JsonNode bestNumberNode = this.getBestNumberNode();
                this.nextToken();
                yield bestNumberNode;
            }
            case JsonToken.VALUE_TRUE, JsonToken.VALUE_FALSE -> JsonNode.createBooleanNode((boolean)this.decodeBoolean());
            case JsonToken.VALUE_NULL -> {
                this.decodeNull();
                yield JsonNode.nullNode();
            }
            default -> throw this.createDeserializationException("Unexpected token " + this.currentToken + ", expected value", null);
        };
    }

    private JsonNode getBestNumberNode() throws IOException {
        Number number = this.parser.getNumberValue();
        if (number instanceof Byte || number instanceof Short || number instanceof Integer) {
            return JsonNode.createNumberNode((int)number.intValue());
        }
        if (number instanceof Long) {
            return JsonNode.createNumberNode((long)number.longValue());
        }
        if (number instanceof Float) {
            return JsonNode.createNumberNode((float)number.floatValue());
        }
        if (number instanceof Double) {
            return JsonNode.createNumberNode((double)number.doubleValue());
        }
        if (number instanceof BigInteger) {
            return JsonNode.createNumberNode((BigInteger)((BigInteger)number));
        }
        if (number instanceof BigDecimal) {
            return JsonNode.createNumberNode((BigDecimal)((BigDecimal)number));
        }
        return JsonNode.createNumberNode((BigDecimal)this.parser.getDecimalValue());
    }

    private static JsonNode decodeObjectNode(JacksonDecoder elementDecoder) throws IOException {
        String key;
        LinkedHashMap<String, JsonNode> result = new LinkedHashMap<String, JsonNode>();
        while ((key = elementDecoder.decodeKey()) != null) {
            result.put(key, elementDecoder.decodeNode());
        }
        elementDecoder.finishStructure();
        return JsonNode.createObjectNode(result);
    }

    private static JsonNode decodeArrayNode(JacksonDecoder elementDecoder) throws IOException {
        ArrayList<JsonNode> result = new ArrayList<JsonNode>();
        while (elementDecoder.hasNextArrayValue()) {
            result.add(elementDecoder.decodeNode());
        }
        elementDecoder.finishStructure();
        return JsonNode.createArrayNode(result);
    }

    @Nullable
    public Object decodeArbitrary() throws IOException {
        RootBuilder root;
        for (ArbitraryBuilder currentStructure = root = new RootBuilder(this); currentStructure != null; currentStructure = currentStructure.proceed()) {
        }
        return root.result;
    }

    public void skipValue() throws IOException {
        this.parser.skipChildren();
        this.nextToken();
    }

    private static final class RootBuilder
    extends ArbitraryBuilder {
        boolean done = false;
        Object result;

        RootBuilder(JacksonDecoder decoder) {
            super(null, decoder);
        }

        @Override
        void put(String key, Object value) {
            this.result = value;
            this.done = true;
        }

        @Override
        String decodeKey() {
            return !this.done ? "" : null;
        }
    }

    private static abstract class ArbitraryBuilder {
        final ArbitraryBuilder parent;
        final JacksonDecoder elementDecoder;

        ArbitraryBuilder(ArbitraryBuilder parent, JacksonDecoder elementDecoder) {
            this.parent = parent;
            this.elementDecoder = elementDecoder;
        }

        abstract String decodeKey() throws IOException;

        abstract void put(String var1, Object var2);

        ArbitraryBuilder proceed() throws IOException {
            String key = this.decodeKey();
            if (key != null) {
                JsonParser jsonParser = this.elementDecoder.parser;
                switch (this.elementDecoder.currentToken) {
                    case START_OBJECT: {
                        MapBuilder map = new MapBuilder(this, this.elementDecoder.decodeObject());
                        this.put(key, map.items);
                        return map;
                    }
                    case START_ARRAY: {
                        ListBuilder list = new ListBuilder(this, this.elementDecoder.decodeArray());
                        this.put(key, list.items);
                        return list;
                    }
                    case VALUE_STRING: {
                        this.put(key, this.elementDecoder.decodeString());
                        return this;
                    }
                    case VALUE_NUMBER_FLOAT: 
                    case VALUE_NUMBER_INT: {
                        this.put(key, this.elementDecoder.decodeNumber());
                        return this;
                    }
                    case VALUE_TRUE: 
                    case VALUE_FALSE: {
                        this.put(key, this.elementDecoder.decodeBoolean());
                        return this;
                    }
                    case VALUE_NULL: {
                        this.elementDecoder.decodeNull();
                        this.put(key, null);
                        return this;
                    }
                }
                throw this.elementDecoder.createDeserializationException("Unexpected token " + this.elementDecoder.currentToken + ", expected value", null);
            }
            return this.parent;
        }
    }

    private static final class MapBuilder
    extends ArbitraryBuilder {
        private final Map<String, Object> items = new LinkedHashMap<String, Object>();

        MapBuilder(ArbitraryBuilder parent, JacksonDecoder elementDecoder) {
            super(parent, elementDecoder);
        }

        @Override
        void put(String key, Object value) {
            this.items.put(key, value);
        }

        @Override
        String decodeKey() throws IOException {
            String key = this.elementDecoder.decodeKey();
            if (key == null) {
                this.elementDecoder.finishStructure();
            }
            return key;
        }
    }

    private static final class ListBuilder
    extends ArbitraryBuilder {
        private final List<Object> items = new ArrayList<Object>();

        ListBuilder(ArbitraryBuilder parent, JacksonDecoder decoder) {
            super(parent, decoder);
        }

        @Override
        void put(String key, Object value) {
            this.items.add(value);
        }

        @Override
        String decodeKey() throws IOException {
            if (this.elementDecoder.hasNextArrayValue()) {
                return "";
            }
            this.elementDecoder.finishStructure();
            return null;
        }
    }
}

