/*
 * Copyright 2017-2020 original authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.micronaut.spring.boot.annotation;

import io.micronaut.context.annotation.Requires;
import io.micronaut.core.annotation.AnnotationClassValue;
import io.micronaut.core.annotation.AnnotationMetadata;
import io.micronaut.core.annotation.AnnotationValue;
import io.micronaut.core.annotation.NonNull;
import io.micronaut.core.util.ArrayUtils;
import io.micronaut.inject.visitor.VisitorContext;
import io.micronaut.spring.annotation.AbstractSpringAnnotationMapper;

import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

/**
 * Maps ConditionalOnBean to Micronaut Requires.
 *
 * @author graemerocher
 * @since 1.0
 */
public class ConditionalOnBeanAnnotationMapper extends AbstractSpringAnnotationMapper {

    @Override
    public String getName() {
        return "org.springframework.boot.autoconfigure.condition.ConditionalOnBean";
    }

    @Override
    protected List<AnnotationValue<?>> mapInternal(AnnotationValue<Annotation> annotation, VisitorContext visitorContext) {
        final AnnotationClassValue<?>[] classValues = annotation.annotationClassValues(AnnotationMetadata.VALUE_MEMBER);
        if (ArrayUtils.isNotEmpty(classValues)) {
            return Collections.singletonList(
                    AnnotationValue.builder(Requires.class)
                            .member(requiresMethodName(), classValues).build()
            );
        } else {
            final String[] types = annotation.stringValues(typesMemberName());
            if (ArrayUtils.isNotEmpty(types)) {
                final AnnotationClassValue[] classesValues = Arrays.stream(types).map(AnnotationClassValue::new).toArray(AnnotationClassValue[]::new);
                return Collections.singletonList(
                        AnnotationValue.builder(Requires.class)
                                .member(requiresMethodName(), classesValues).build()
                );
            }
        }

        return Collections.emptyList();
    }

    /**
     * The annotation member name that specifies the types.
     * @return The name
     */
    protected @NonNull String typesMemberName() {
        return "types";
    }

    /**
     * The annotation member name for requires.
     * @return The member name
     */
    protected @NonNull String requiresMethodName() {
        return "beans";
    }
}
