/*
 * Decompiled with CFR 0.152.
 */
package org.testcontainers.utility;

import com.github.dockerjava.api.exception.NotFoundException;
import com.github.dockerjava.api.model.AuthConfig;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.util.Base64;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.testcontainers.DockerClientFactory;
import org.testcontainers.shaded.com.fasterxml.jackson.databind.JsonNode;
import org.testcontainers.shaded.com.fasterxml.jackson.databind.ObjectMapper;
import org.testcontainers.shaded.com.google.common.annotations.VisibleForTesting;
import org.testcontainers.shaded.org.apache.commons.lang3.StringUtils;
import org.testcontainers.shaded.org.apache.commons.lang3.SystemUtils;
import org.testcontainers.shaded.org.zeroturnaround.exec.InvalidResultException;
import org.testcontainers.shaded.org.zeroturnaround.exec.ProcessExecutor;
import org.testcontainers.shaded.org.zeroturnaround.exec.ProcessResult;
import org.testcontainers.shaded.org.zeroturnaround.exec.stream.LogOutputStream;
import org.testcontainers.utility.AuthConfigUtil;
import org.testcontainers.utility.DockerImageName;

public class RegistryAuthLocator {
    private static final Logger log = LoggerFactory.getLogger(RegistryAuthLocator.class);
    private static final String DEFAULT_REGISTRY_NAME = "https://index.docker.io/v1/";
    private static final String DOCKER_AUTH_ENV_VAR = "DOCKER_AUTH_CONFIG";
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private static RegistryAuthLocator instance;
    private final String commandPathPrefix;
    private final String commandExtension;
    private final File configFile;
    private final String configEnv;
    private final Map<String, Optional<AuthConfig>> cache = new ConcurrentHashMap<String, Optional<AuthConfig>>();
    private final Map<String, String> CREDENTIALS_HELPERS_NOT_FOUND_MESSAGE_CACHE;

    @VisibleForTesting
    RegistryAuthLocator(File configFile, String configEnv, String commandPathPrefix, String commandExtension, Map<String, String> notFoundMessageHolderReference) {
        this.configFile = configFile;
        this.configEnv = configEnv;
        this.commandPathPrefix = commandPathPrefix;
        this.commandExtension = commandExtension;
        this.CREDENTIALS_HELPERS_NOT_FOUND_MESSAGE_CACHE = notFoundMessageHolderReference;
    }

    protected RegistryAuthLocator() {
        String dockerConfigLocation = System.getenv().getOrDefault("DOCKER_CONFIG", System.getProperty("user.home") + "/.docker");
        this.configFile = new File(dockerConfigLocation + "/config.json");
        this.configEnv = System.getenv(DOCKER_AUTH_ENV_VAR);
        this.commandPathPrefix = "";
        this.commandExtension = "";
        this.CREDENTIALS_HELPERS_NOT_FOUND_MESSAGE_CACHE = new HashMap<String, String>();
    }

    public static synchronized RegistryAuthLocator instance() {
        if (instance == null) {
            instance = new RegistryAuthLocator();
        }
        return instance;
    }

    @VisibleForTesting
    static void setInstance(RegistryAuthLocator overrideInstance) {
        instance = overrideInstance;
    }

    public AuthConfig lookupAuthConfig(DockerImageName dockerImageName, AuthConfig defaultAuthConfig) {
        String registryName = this.effectiveRegistryName(dockerImageName);
        log.debug("Looking up auth config for image: {} at registry: {}", (Object)dockerImageName, (Object)registryName);
        Optional cachedAuth = this.cache.computeIfAbsent(registryName, __ -> this.lookupUncachedAuthConfig(registryName, dockerImageName));
        if (cachedAuth.isPresent()) {
            log.debug("Cached auth found: [{}]", (Object)AuthConfigUtil.toSafeString((AuthConfig)cachedAuth.get()));
            return (AuthConfig)cachedAuth.get();
        }
        log.debug("No matching Auth Configs - falling back to defaultAuthConfig [{}]", (Object)AuthConfigUtil.toSafeString(defaultAuthConfig));
        return defaultAuthConfig;
    }

    private Optional<AuthConfig> lookupUncachedAuthConfig(String registryName, DockerImageName dockerImageName) {
        try {
            JsonNode config = this.getDockerAuthConfig();
            log.debug("registryName [{}] for dockerImageName [{}]", (Object)registryName, (Object)dockerImageName);
            AuthConfig helperAuthConfig = this.authConfigUsingHelper(config, registryName);
            if (helperAuthConfig != null) {
                log.debug("found helper auth config [{}]", (Object)AuthConfigUtil.toSafeString(helperAuthConfig));
                return Optional.of(helperAuthConfig);
            }
            AuthConfig storeAuthConfig = this.authConfigUsingStore(config, registryName);
            if (storeAuthConfig != null) {
                log.debug("found creds store auth config [{}]", (Object)AuthConfigUtil.toSafeString(storeAuthConfig));
                return Optional.of(storeAuthConfig);
            }
            AuthConfig existingAuthConfig = this.findExistingAuthConfig(config, registryName);
            if (existingAuthConfig != null) {
                log.debug("found existing auth config [{}]", (Object)AuthConfigUtil.toSafeString(existingAuthConfig));
                return Optional.of(existingAuthConfig);
            }
        }
        catch (Exception e) {
            log.info("Failure when attempting to lookup auth config. Please ignore if you don't have images in an authenticated registry. Details: (dockerImageName: {}, configFile: {}, configEnv: {}). Falling back to docker-java default behaviour. Exception message: {}", dockerImageName, this.configFile, DOCKER_AUTH_ENV_VAR, e.getMessage());
        }
        return Optional.empty();
    }

    private JsonNode getDockerAuthConfig() throws Exception {
        log.debug("RegistryAuthLocator has configFile: {} ({}) configEnv: {} ({}) and commandPathPrefix: {}", this.configFile, this.configFile.exists() ? "exists" : "does not exist", DOCKER_AUTH_ENV_VAR, this.configEnv != null ? "exists" : "does not exist", this.commandPathPrefix);
        if (this.configEnv != null) {
            log.debug("RegistryAuthLocator reading from environment variable: {}", (Object)DOCKER_AUTH_ENV_VAR);
            return OBJECT_MAPPER.readTree(this.configEnv);
        }
        if (this.configFile.exists()) {
            log.debug("RegistryAuthLocator reading from configFile: {}", (Object)this.configFile);
            return OBJECT_MAPPER.readTree(this.configFile);
        }
        throw new NotFoundException("No config supplied. Checked in order: " + this.configFile + " (file not found), " + DOCKER_AUTH_ENV_VAR + " (not set)");
    }

    private AuthConfig findExistingAuthConfig(JsonNode config, String reposName) throws Exception {
        Map.Entry<String, JsonNode> entry = this.findAuthNode(config, reposName);
        if (entry != null && entry.getValue() != null && entry.getValue().size() > 0) {
            String rawAuth;
            String[] splitRawAuth;
            AuthConfig deserializedAuth = OBJECT_MAPPER.treeToValue(entry.getValue(), AuthConfig.class).withRegistryAddress(entry.getKey());
            if (StringUtils.isBlank(deserializedAuth.getUsername()) && StringUtils.isBlank(deserializedAuth.getPassword()) && !StringUtils.isBlank(deserializedAuth.getAuth()) && (splitRawAuth = (rawAuth = new String(Base64.getDecoder().decode(deserializedAuth.getAuth()))).split(":", 2)).length == 2) {
                deserializedAuth.withUsername(splitRawAuth[0]);
                deserializedAuth.withPassword(splitRawAuth[1]);
            }
            return deserializedAuth;
        }
        return null;
    }

    private AuthConfig authConfigUsingHelper(JsonNode config, String reposName) throws Exception {
        JsonNode helperNode;
        JsonNode credHelpers = config.get("credHelpers");
        if (credHelpers != null && credHelpers.size() > 0 && (helperNode = credHelpers.get(reposName)) != null && helperNode.isTextual()) {
            String helper = helperNode.asText();
            return this.runCredentialProvider(reposName, helper);
        }
        return null;
    }

    private AuthConfig authConfigUsingStore(JsonNode config, String reposName) throws Exception {
        JsonNode credsStoreNode = config.get("credsStore");
        if (credsStoreNode != null && !credsStoreNode.isMissingNode() && credsStoreNode.isTextual()) {
            String credsStore = credsStoreNode.asText();
            if (StringUtils.isBlank(credsStore)) {
                log.warn("Docker auth config credsStore field will be ignored, because value is blank");
                return null;
            }
            return this.runCredentialProvider(reposName, credsStore);
        }
        return null;
    }

    private Map.Entry<String, JsonNode> findAuthNode(JsonNode config, String reposName) {
        JsonNode auths = config.get("auths");
        if (auths != null && auths.size() > 0) {
            Iterator<Map.Entry<String, JsonNode>> fields = auths.fields();
            while (fields.hasNext()) {
                Map.Entry<String, JsonNode> entry = fields.next();
                if (!entry.getKey().endsWith("://" + reposName) && !entry.getKey().equals(reposName)) continue;
                return entry;
            }
        }
        return null;
    }

    private AuthConfig runCredentialProvider(String hostName, String helperOrStoreName) throws Exception {
        CredentialOutput data;
        if (StringUtils.isBlank(hostName)) {
            log.debug("There is no point in locating AuthConfig for blank hostName. Returning NULL to allow fallback");
            return null;
        }
        String credentialProgramName = this.getCredentialProgramName(helperOrStoreName);
        log.debug("Executing docker credential provider: {} to locate auth config for: {}", (Object)credentialProgramName, (Object)hostName);
        try {
            data = this.runCredentialProgram(hostName, credentialProgramName);
            if (data.getExitValue() == 1) {
                String responseErrorMsg = data.getStdout();
                if (!StringUtils.isBlank(responseErrorMsg)) {
                    String credentialsNotFoundMsg = this.getGenericCredentialsNotFoundMsg(responseErrorMsg, credentialProgramName);
                    if (credentialsNotFoundMsg != null && credentialsNotFoundMsg.equals(responseErrorMsg)) {
                        log.info("Credential helper/store ({}) does not have credentials for {}", (Object)credentialProgramName, (Object)hostName);
                        return null;
                    }
                    log.debug("Failure running docker credential helper/store ({}) with output '{}' and error '{}'", credentialProgramName, responseErrorMsg, data.getStderr());
                } else {
                    log.debug("Failure running docker credential helper/store ({})", (Object)credentialProgramName);
                }
                throw new InvalidResultException(data.getStdout(), null);
            }
        }
        catch (Exception e) {
            log.debug("Failure running docker credential helper/store ({})", (Object)credentialProgramName);
            throw e;
        }
        JsonNode helperResponse = OBJECT_MAPPER.readTree(data.getStdout());
        log.debug("Credential helper/store provided auth config for: {}", (Object)hostName);
        String username = helperResponse.at("/Username").asText();
        String password = helperResponse.at("/Secret").asText();
        String serverUrl = helperResponse.at("/ServerURL").asText();
        AuthConfig authConfig = new AuthConfig().withRegistryAddress(serverUrl.isEmpty() ? hostName : serverUrl);
        if ("<token>".equals(username)) {
            return authConfig.withIdentityToken(password);
        }
        return authConfig.withUsername(username).withPassword(password);
    }

    private String getCredentialProgramName(String credHelper) {
        return this.commandPathPrefix + "docker-credential-" + credHelper + this.commandExtension;
    }

    private String effectiveRegistryName(DockerImageName dockerImageName) {
        String registry = dockerImageName.getRegistry();
        if (!StringUtils.isEmpty(registry)) {
            return registry;
        }
        return StringUtils.defaultString(DockerClientFactory.instance().getInfo().getIndexServerAddress(), DEFAULT_REGISTRY_NAME);
    }

    private String getGenericCredentialsNotFoundMsg(String credentialsNotFoundMsg, String credentialHelperName) {
        if (!this.CREDENTIALS_HELPERS_NOT_FOUND_MESSAGE_CACHE.containsKey(credentialHelperName)) {
            this.CREDENTIALS_HELPERS_NOT_FOUND_MESSAGE_CACHE.put(credentialHelperName, credentialsNotFoundMsg);
        }
        return this.CREDENTIALS_HELPERS_NOT_FOUND_MESSAGE_CACHE.get(credentialHelperName);
    }

    private CredentialOutput runCredentialProgram(String hostName, String credentialHelperName) throws InterruptedException, TimeoutException, IOException {
        String[] stringArray;
        if (SystemUtils.IS_OS_WINDOWS) {
            String[] stringArray2 = new String[4];
            stringArray2[0] = "cmd";
            stringArray2[1] = "/c";
            stringArray2[2] = credentialHelperName;
            stringArray = stringArray2;
            stringArray2[3] = "get";
        } else {
            String[] stringArray3 = new String[2];
            stringArray3[0] = credentialHelperName;
            stringArray = stringArray3;
            stringArray3[1] = "get";
        }
        String[] command = stringArray;
        final StringBuffer stdout = new StringBuffer();
        final StringBuffer stderr = new StringBuffer();
        ProcessResult processResult = new ProcessExecutor().command(command).redirectInput(new ByteArrayInputStream(hostName.getBytes())).redirectOutput(new LogOutputStream(){

            @Override
            protected void processLine(String line) {
                stdout.append(line).append(System.lineSeparator());
            }
        }).redirectError(new LogOutputStream(){

            @Override
            protected void processLine(String line) {
                stderr.append(line).append(System.lineSeparator());
            }
        }).timeout(30L, TimeUnit.SECONDS).execute();
        int exitValue = processResult.getExitValue();
        return new CredentialOutput(exitValue, stdout.toString(), stderr.toString());
    }

    static class CredentialOutput {
        private final int exitValue;
        private final String stdout;
        private final String stderr;

        public CredentialOutput(int exitValue, String stdout, String stderr) {
            this.exitValue = exitValue;
            this.stdout = stdout.trim();
            this.stderr = stderr.trim();
        }

        int getExitValue() {
            return this.exitValue;
        }

        String getStdout() {
            return this.stdout;
        }

        String getStderr() {
            return this.stderr;
        }
    }
}

