/*
 * Decompiled with CFR 0.152.
 */
package io.milvus.bulkwriter;

import com.google.common.collect.Lists;
import com.google.gson.JsonElement;
import com.google.gson.JsonNull;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import io.milvus.bulkwriter.common.clientenum.BulkFileType;
import io.milvus.bulkwriter.common.clientenum.TypeSize;
import io.milvus.bulkwriter.common.utils.V2AdapterUtils;
import io.milvus.bulkwriter.writer.CSVFileWriter;
import io.milvus.bulkwriter.writer.FormatFileWriter;
import io.milvus.bulkwriter.writer.JSONFileWriter;
import io.milvus.bulkwriter.writer.ParquetFileWriter;
import io.milvus.common.utils.ExceptionUtils;
import io.milvus.common.utils.Float16Utils;
import io.milvus.grpc.FieldSchema;
import io.milvus.param.ParamUtils;
import io.milvus.param.collection.FieldType;
import io.milvus.v2.common.DataType;
import io.milvus.v2.service.collection.request.CreateCollectionReq;
import io.milvus.v2.utils.SchemaUtils;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.SortedMap;
import java.util.UUID;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class BulkWriter
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(BulkWriter.class);
    protected CreateCollectionReq.CollectionSchema collectionSchema;
    protected long chunkSize;
    protected BulkFileType fileType;
    protected String localPath;
    protected String uuid;
    protected int flushCount;
    protected FormatFileWriter fileWriter;
    protected final Map<String, Object> config;
    protected long totalSize;
    protected long totalRowCount;
    protected ReentrantLock appendLock;
    protected ReentrantLock fileWriteLock;
    protected boolean firstWrite;

    protected BulkWriter(CreateCollectionReq.CollectionSchema collectionSchema, long chunkSize, BulkFileType fileType, String localPath, Map<String, Object> config) throws IOException {
        this.collectionSchema = collectionSchema;
        this.chunkSize = chunkSize;
        this.fileType = fileType;
        this.localPath = localPath;
        this.uuid = UUID.randomUUID().toString();
        this.config = config;
        if (CollectionUtils.isEmpty((Collection)collectionSchema.getFieldSchemaList())) {
            ExceptionUtils.throwUnExpectedException((String)"collection schema fields list is empty");
        }
        if (!this.hasPrimaryField(collectionSchema.getFieldSchemaList())) {
            ExceptionUtils.throwUnExpectedException((String)"primary field is null");
        }
        this.appendLock = new ReentrantLock();
        this.makeDir();
        this.fileWriteLock = new ReentrantLock();
        this.fileWriter = null;
        this.newFileWriter();
        this.firstWrite = true;
    }

    protected Long getTotalSize() {
        return this.totalSize;
    }

    public Long getTotalRowCount() {
        return this.totalRowCount;
    }

    protected Long getChunkSize() {
        return this.chunkSize;
    }

    protected FormatFileWriter getFileWriter() {
        return this.fileWriter;
    }

    protected FormatFileWriter newFileWriter() throws IOException {
        FormatFileWriter oldFileWriter = this.fileWriter;
        this.fileWriteLock.lock();
        this.createWriterByType();
        this.fileWriteLock.unlock();
        return oldFileWriter;
    }

    private void createWriterByType() throws IOException {
        ++this.flushCount;
        Path path = Paths.get(this.localPath, new String[0]);
        Path filePathPrefix = path.resolve(String.valueOf(this.flushCount));
        switch (this.fileType) {
            case PARQUET: {
                this.fileWriter = new ParquetFileWriter(this.collectionSchema, filePathPrefix.toString());
                break;
            }
            case JSON: {
                this.fileWriter = new JSONFileWriter(this.collectionSchema, filePathPrefix.toString());
                break;
            }
            case CSV: {
                this.fileWriter = new CSVFileWriter(this.collectionSchema, filePathPrefix.toString(), this.config);
                break;
            }
            default: {
                ExceptionUtils.throwUnExpectedException((String)("Unsupported file type: " + (Object)((Object)this.fileType)));
            }
        }
    }

    private void makeDir() throws IOException {
        Path path = Paths.get(this.localPath, new String[0]);
        this.createDirIfNotExist(path);
        Path fullPath = path.resolve(this.uuid);
        this.createDirIfNotExist(fullPath);
        this.localPath = fullPath.toString();
    }

    private void createDirIfNotExist(Path path) throws IOException {
        try {
            Files.createDirectories(path, new FileAttribute[0]);
            logger.info("Data path created: {}", (Object)path);
        }
        catch (IOException e) {
            logger.error("Data Path create failed: {}", (Object)path);
            throw e;
        }
    }

    public void appendRow(JsonObject row) throws IOException, InterruptedException {
        Map<String, Object> rowValues = this.verifyRow(row);
        List<Object> filePaths = Lists.newArrayList();
        this.appendLock.lock();
        this.fileWriter.appendRow(rowValues, this.firstWrite);
        this.firstWrite = false;
        if (this.getTotalSize() > this.getChunkSize()) {
            filePaths = this.commitIfFileReady(true);
        }
        this.appendLock.unlock();
        if (CollectionUtils.isNotEmpty((Collection)filePaths)) {
            this.callBackIfCommitReady(filePaths);
        }
    }

    protected abstract List<String> commitIfFileReady(boolean var1) throws IOException;

    protected abstract void callBackIfCommitReady(List<String> var1) throws IOException, InterruptedException;

    protected void commit() {
        this.appendLock.lock();
        this.totalSize = 0L;
        this.totalRowCount = 0L;
        this.appendLock.unlock();
    }

    protected String getDataPath() {
        return "";
    }

    private JsonElement setDefaultValue(Object defaultValue, JsonObject row, String fieldName) {
        if (defaultValue instanceof Boolean) {
            row.addProperty(fieldName, (Boolean)defaultValue);
            return new JsonPrimitive((Boolean)defaultValue);
        }
        if (defaultValue instanceof String) {
            row.addProperty(fieldName, (String)defaultValue);
            return new JsonPrimitive((String)defaultValue);
        }
        row.addProperty(fieldName, (Number)defaultValue);
        return new JsonPrimitive((Number)defaultValue);
    }

    protected Map<String, Object> verifyRow(JsonObject row) {
        int rowSize = 0;
        HashMap<String, Object> rowValues = new HashMap<String, Object>();
        List<String> outputFieldNames = V2AdapterUtils.getOutputFieldNames(this.collectionSchema);
        block7: for (CreateCollectionReq.FieldSchema field : this.collectionSchema.getFieldSchemaList()) {
            JsonElement obj;
            String fieldName = field.getName();
            if (field.getIsPrimaryKey().booleanValue() && field.getAutoID().booleanValue()) {
                if (!row.has(fieldName)) continue;
                String msg = String.format("The primary key field '%s' is auto-id, no need to provide", fieldName);
                ExceptionUtils.throwUnExpectedException((String)msg);
            }
            if ((obj = row.get(fieldName)) == null) {
                obj = JsonNull.INSTANCE;
            }
            if (outputFieldNames.contains(fieldName)) {
                if (obj instanceof JsonNull) continue;
                String msg = String.format("The field '%s'  is function output, no need to provide", fieldName);
                ExceptionUtils.throwUnExpectedException((String)msg);
            }
            Object defaultValue = field.getDefaultValue();
            if (field.getIsNullable().booleanValue()) {
                if (defaultValue != null) {
                    if (obj instanceof JsonNull) {
                        obj = this.setDefaultValue(defaultValue, row, fieldName);
                    }
                } else if (obj instanceof JsonNull) {
                    row.add(fieldName, (JsonElement)JsonNull.INSTANCE);
                }
            } else if (defaultValue != null) {
                if (obj instanceof JsonNull) {
                    obj = this.setDefaultValue(defaultValue, row, fieldName);
                }
            } else if (obj instanceof JsonNull) {
                String msg = String.format("The field '%s' is not nullable, not allow null value", fieldName);
                ExceptionUtils.throwUnExpectedException((String)msg);
            }
            DataType dataType = field.getDataType();
            switch (dataType) {
                case BinaryVector: 
                case FloatVector: 
                case Float16Vector: 
                case BFloat16Vector: 
                case SparseFloatVector: {
                    Pair<Object, Integer> objectAndSize = this.verifyVector(obj, field);
                    rowValues.put(fieldName, objectAndSize.getLeft());
                    rowSize += ((Integer)objectAndSize.getRight()).intValue();
                    continue block7;
                }
                case VarChar: {
                    Pair<Object, Integer> objectAndSize = this.verifyVarchar(obj, field);
                    rowValues.put(fieldName, objectAndSize.getLeft());
                    rowSize += ((Integer)objectAndSize.getRight()).intValue();
                    continue block7;
                }
                case JSON: {
                    Pair<Object, Integer> objectAndSize = this.verifyJSON(obj, field);
                    rowValues.put(fieldName, objectAndSize.getLeft());
                    rowSize += ((Integer)objectAndSize.getRight()).intValue();
                    continue block7;
                }
                case Array: {
                    Pair<Object, Integer> objectAndSize = this.verifyArray(obj, field);
                    rowValues.put(fieldName, objectAndSize.getLeft());
                    rowSize += ((Integer)objectAndSize.getRight()).intValue();
                    continue block7;
                }
                case Bool: 
                case Int8: 
                case Int16: 
                case Int32: 
                case Int64: 
                case Float: 
                case Double: {
                    Pair<Object, Integer> objectAndSize = this.verifyScalar(obj, field);
                    rowValues.put(fieldName, objectAndSize.getLeft());
                    rowSize += ((Integer)objectAndSize.getRight()).intValue();
                    continue block7;
                }
            }
            String msg = String.format("Unsupported data type of field '%s', not implemented in BulkWriter.", fieldName);
            ExceptionUtils.throwUnExpectedException((String)msg);
        }
        if (this.collectionSchema.isEnableDynamicField()) {
            JsonObject dynamicValues = new JsonObject();
            if (row.has("$meta")) {
                JsonElement value = row.get("$meta");
                if (!(value instanceof JsonObject)) {
                    String msg = String.format("Dynamic field '%s' value should be JSON dict format", "$meta");
                    ExceptionUtils.throwUnExpectedException((String)msg);
                }
                dynamicValues = (JsonObject)value;
            }
            for (String key : row.keySet()) {
                if (key.equals("$meta") || rowValues.containsKey(key)) continue;
                dynamicValues.add(key, row.get(key));
            }
            String strValues = dynamicValues.toString();
            rowValues.put("$meta", strValues);
            rowSize += strValues.length();
        }
        this.appendLock.lock();
        this.totalSize += (long)rowSize;
        ++this.totalRowCount;
        this.appendLock.unlock();
        return rowValues;
    }

    private Pair<Object, Integer> verifyVector(JsonElement object, CreateCollectionReq.FieldSchema field) {
        FieldSchema grpcField = SchemaUtils.convertToGrpcFieldSchema((CreateCollectionReq.FieldSchema)field);
        Object vector = ParamUtils.checkFieldValue((FieldType)ParamUtils.ConvertField((FieldSchema)grpcField), (JsonElement)object);
        DataType dataType = field.getDataType();
        switch (dataType) {
            case FloatVector: {
                return Pair.of((Object)vector, (Object)(((List)vector).size() * 4));
            }
            case BinaryVector: {
                return Pair.of((Object)vector, (Object)((ByteBuffer)vector).limit());
            }
            case Float16Vector: 
            case BFloat16Vector: {
                if (this.fileType == BulkFileType.CSV || this.fileType == BulkFileType.JSON) {
                    ByteBuffer bv = (ByteBuffer)vector;
                    bv.order(ByteOrder.LITTLE_ENDIAN);
                    List v = dataType == DataType.Float16Vector ? Float16Utils.fp16BufferToVector((ByteBuffer)bv) : Float16Utils.bf16BufferToVector((ByteBuffer)bv);
                    return Pair.of((Object)v, (Object)(v.size() * 4));
                }
                return Pair.of((Object)vector, (Object)(((ByteBuffer)vector).limit() * 2));
            }
            case SparseFloatVector: {
                return Pair.of((Object)vector, (Object)(((SortedMap)vector).size() * 12));
            }
        }
        ExceptionUtils.throwUnExpectedException((String)"Unknown vector type");
        return null;
    }

    private Pair<Object, Integer> verifyVarchar(JsonElement object, CreateCollectionReq.FieldSchema field) {
        if (object.isJsonNull()) {
            return Pair.of(null, (Object)0);
        }
        FieldSchema grpcField = SchemaUtils.convertToGrpcFieldSchema((CreateCollectionReq.FieldSchema)field);
        Object varchar = ParamUtils.checkFieldValue((FieldType)ParamUtils.ConvertField((FieldSchema)grpcField), (JsonElement)object);
        return Pair.of((Object)varchar, (Object)String.valueOf(varchar).length());
    }

    private Pair<Object, Integer> verifyJSON(JsonElement object, CreateCollectionReq.FieldSchema field) {
        if (object.isJsonNull()) {
            return Pair.of(null, (Object)0);
        }
        String str = object.toString();
        return Pair.of((Object)str, (Object)str.length());
    }

    private Pair<Object, Integer> verifyArray(JsonElement object, CreateCollectionReq.FieldSchema field) {
        FieldSchema grpcField = SchemaUtils.convertToGrpcFieldSchema((CreateCollectionReq.FieldSchema)field);
        Object array = ParamUtils.checkFieldValue((FieldType)ParamUtils.ConvertField((FieldSchema)grpcField), (JsonElement)object);
        if (array == null) {
            return Pair.of(null, (Object)0);
        }
        int rowSize = 0;
        DataType elementType = field.getElementType();
        if (TypeSize.contains(elementType)) {
            rowSize = TypeSize.getSize(elementType) * ((List)array).size();
        } else if (elementType == DataType.VarChar) {
            for (String str : (List)array) {
                rowSize += str.length();
            }
        } else {
            String msg = String.format("Unsupported element type for array field '%s'", field.getName());
            ExceptionUtils.throwUnExpectedException((String)msg);
        }
        return Pair.of((Object)array, (Object)rowSize);
    }

    private Pair<Object, Integer> verifyScalar(JsonElement object, CreateCollectionReq.FieldSchema field) {
        String msg;
        if (object.isJsonNull()) {
            return Pair.of(null, (Object)0);
        }
        if (!object.isJsonPrimitive()) {
            String msg2 = String.format("Unsupported value type for field '%s'", field.getName());
            ExceptionUtils.throwUnExpectedException((String)msg2);
        }
        JsonPrimitive value = object.getAsJsonPrimitive();
        DataType dataType = field.getDataType();
        String fieldName = field.getName();
        if (dataType == DataType.Bool) {
            if (!value.isBoolean()) {
                String msg3 = String.format("Unsupported value type for field '%s', value is not boolean", fieldName);
                ExceptionUtils.throwUnExpectedException((String)msg3);
            }
            return Pair.of((Object)value.getAsBoolean(), (Object)TypeSize.getSize(dataType));
        }
        if (!value.isNumber()) {
            msg = String.format("Unsupported value type for field '%s', value is not a number", fieldName);
            ExceptionUtils.throwUnExpectedException((String)msg);
        }
        switch (dataType) {
            case Int8: 
            case Int16: {
                return Pair.of((Object)value.getAsShort(), (Object)TypeSize.getSize(dataType));
            }
            case Int32: {
                return Pair.of((Object)value.getAsInt(), (Object)TypeSize.getSize(dataType));
            }
            case Int64: {
                return Pair.of((Object)value.getAsLong(), (Object)TypeSize.getSize(dataType));
            }
            case Float: {
                return Pair.of((Object)Float.valueOf(value.getAsFloat()), (Object)TypeSize.getSize(dataType));
            }
            case Double: {
                return Pair.of((Object)value.getAsDouble(), (Object)TypeSize.getSize(dataType));
            }
        }
        msg = String.format("Field '%s' is not a scalar field", fieldName);
        ExceptionUtils.throwUnExpectedException((String)msg);
        return Pair.of(null, null);
    }

    private boolean hasPrimaryField(List<CreateCollectionReq.FieldSchema> fields) {
        Optional<CreateCollectionReq.FieldSchema> primaryKeyField = fields.stream().filter(CreateCollectionReq.FieldSchema::getIsPrimaryKey).findFirst();
        return primaryKeyField.isPresent();
    }
}

