package io.mongock.jwt.parser

import io.jsonwebtoken.ExpiredJwtException
import io.jsonwebtoken.Jwts
import io.mongock.jwt.api.*
import java.security.KeyFactory
import java.security.PublicKey
import java.security.spec.X509EncodedKeySpec
import java.time.ZoneOffset
import java.util.*

class JwtParser(rsaPublicKey: String) {
    private val publicKey: PublicKey = rsaPublicKey
        .replace("-----BEGIN PUBLIC KEY-----", "")
        .replace("-----END PUBLIC KEY-----", "")
        .let { X509EncodedKeySpec(Base64.getDecoder().decode(it)) }
        .let { KeyFactory.getInstance("RSA").generatePublic(it) }


    fun parse(jwtToken: String): JwtMongock = try {
        Jwts.parserBuilder()
            .setSigningKey(publicKey)
            .build()
            .parseClaimsJws(jwtToken)
            .let {
                JwtMongock(
                    signature = it.signature,
                    alg = it.header.algorithm,
                    //body
                    id = it.body.id,
                    audience = it.body.audience,
                    issuer = it.body.issuer,
                    subject = it.body.subject,
                    issuedAt = it.body.issuedAt.toInstant().atOffset(ZoneOffset.UTC).toLocalDateTime(),
                    expiresAt = it.body.expiration.toInstant().atOffset(ZoneOffset.UTC).toLocalDateTime(),
                    // custom
                    licenseId = it.body[LICENSE_ID_FIELD, String::class.java],
                    licenseType = LicenseType.valueOf(it.body[LICENSE_TYPE_FIELD, String::class.java]),
                    email = it.body[EMAIL_FIELD, String::class.java],
                    companyName = it.body[COMPANY_NAME_FIELD, String::class.java],
                    schemas = it.body[SCHEMAS_FIELD] as Int
                )
            }
    } catch (ex: ExpiredJwtException) {
        throw JwtExpiredException(ex)
    }





}