/*
 * Copyright (c) 2014-2017 by The Monix Project Developers.
 * See the project homepage at: https://monix.io
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package monix.reactive.internal.builders

import monix.execution.Cancelable
import monix.execution.cancelables.{MultiAssignmentCancelable, SingleAssignmentCancelable}
import monix.reactive.Observable
import monix.reactive.observables.ChainedObservable
import monix.reactive.observables.ChainedObservable.{subscribe => chain}
import monix.reactive.observers.Subscriber

private[reactive] final class ConsObservable[+A](head: A, tail: Observable[A])
  extends ChainedObservable[A] {

  def unsafeSubscribeFn(conn: MultiAssignmentCancelable, out: Subscriber[A]): Unit = {
    import out.{scheduler => s}
    out.onNext(head).syncOnContinue {
      // Should provide light async boundary
      chain(tail, conn, out)
    }(s)
  }

  private def simpleSubscribe(conn: SingleAssignmentCancelable, out: Subscriber[A]): Unit = {
    import out.{scheduler => s}
    out.onNext(head).syncOnContinue {
      conn := tail.unsafeSubscribeFn(out)
    }(s)
  }

  def unsafeSubscribeFn(out: Subscriber[A]): Cancelable = {
    if (!tail.isInstanceOf[ChainedObservable[_]]) {
      val conn = SingleAssignmentCancelable()
      simpleSubscribe(conn, out)
      conn
    } else {
      val conn = MultiAssignmentCancelable()
      unsafeSubscribeFn(conn, out)
      conn
    }
  }
}
