/*
 * Copyright (c) 2014-2018 by The Monix Project Developers.
 * See the project homepage at: https://monix.io
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package monix.reactive

import java.io.{BufferedReader, InputStream, PrintStream, Reader}

import cats.effect.IO
import cats.{Apply, Applicative, CoflatMap, Eq, Eval, FlatMap, MonadError, Monoid, MonoidK, NonEmptyParallel, Order, ~>}
import monix.eval.Coeval.Eager
import monix.eval.{Callback, Coeval, Task, TaskLift, TaskLike}
import monix.execution.Ack.{Continue, Stop}
import monix.execution._
import monix.execution.cancelables.{BooleanCancelable, SingleAssignCancelable}
import monix.execution.exceptions.UpstreamTimeoutException

import scala.util.control.NonFatal
import monix.reactive.Observable.Operator
import monix.reactive.OverflowStrategy.Synchronous
import monix.reactive.internal.builders
import monix.reactive.internal.builders._
import monix.reactive.internal.operators._
import monix.reactive.internal.subscribers.ForeachSubscriber
import monix.reactive.observables._
import monix.reactive.observers._
import monix.reactive.subjects._
import org.reactivestreams.{Publisher => RPublisher, Subscriber => RSubscriber}

import scala.collection.mutable
import scala.concurrent.duration.{Duration, FiniteDuration}
import scala.concurrent.{Future, Promise}
import scala.util.{Failure, Success, Try}

/** The `Observable` type that implements the Reactive Pattern.
  *
  * Provides methods of subscribing to the Observable and operators
  * for combining observable sources, filtering, modifying,
  * throttling, buffering, error handling and others.
  *
  * See the available documentation at: [[https://monix.io]]
  *
  * @define concatMergeDifference The difference between the `concat` operation
  *         and `merge`is that `concat` cares about the ordering of sequences
  *         (e.g. all items emitted by the first observable in the sequence
  *         will come before the elements emitted by the second observable),
  *         whereas `merge` doesn't care about that (elements get
  *         emitted as they come). Because of back-pressure applied to
  *         observables, `concat` is safe to use in all contexts, whereas
  *         `merge` requires buffering.
  *
  * @define concatDescription Concatenates the sequence
  *         of observables emitted by the source into one observable,
  *         without any transformation.
  *
  *         You can combine the items emitted by multiple observables
  *         so that they act like a single sequence by using this
  *         operator.
  *
  *         $concatMergeDifference
  *
  * @define delayErrorsDescription This version is reserving onError
  *         notifications until all of the observables complete and only
  *         then passing the issued errors(s) downstream. Note that
  *         the streamed error is a
  *         [[monix.execution.exceptions.CompositeException CompositeException]],
  *         since multiple errors from multiple streams can happen.
  *
  * @define concatReturn an observable that emits items that are the result of
  *         flattening the items emitted by the observables emitted by the source
  *
  * @define switchDescription Convert an observable that emits observables
  *         into a single observable that emits the items emitted by
  *         the most-recently-emitted of those observables.
  *
  * @define switchMapDescription Returns a new observable that emits the items
  *         emitted by the observable most recently generated by the
  *         mapping function.
  *
  * @define overflowStrategyParam the [[OverflowStrategy overflow strategy]]
  *         used for buffering, which specifies what to do in case
  *         we're dealing with a slow consumer - should an unbounded
  *         buffer be used, should back-pressure be applied, should
  *         the pipeline drop newer or older events, should it drop
  *         the whole buffer? See [[OverflowStrategy]] for more
  *         details.
  *
  * @define defaultOverflowStrategy this operation needs to do buffering
  *         and by not specifying an [[OverflowStrategy]], the
  *         [[OverflowStrategy.Default default strategy]] is being
  *         used.
  *
  * @define mergeMapDescription Creates a new observable by applying a
  *         function that you supply to each item emitted by the
  *         source observable, where that function returns an
  *         observable, and then merging those resulting observable
  *         and emitting the results of this merger.
  *
  *         The difference between this and `concatMap` is that
  *         `concatMap` cares about ordering of emitted
  *         items (e.g. all items emitted by the first observable in
  *         the sequence will come before the elements emitted by the
  *         second observable), whereas `merge` doesn't care about
  *         that (elements get emitted as they come). Because of
  *         back-pressure applied to observables, the `concat` operation
  *         is safe to use in all contexts, whereas `merge` requires
  *         buffering.
  *
  * @define mergeMapReturn an observable that emits the result of applying the
  *         transformation function to each item emitted by the source
  *         observable and merging the results of the observables
  *         obtained from this transformation.
  *
  * @define mergeDescription Merges the sequence of Observables emitted by
  *         the source into one Observable, without any transformation.
  *
  *         You can combine the items emitted by multiple Observables
  *         so that they act like a single Observable by using this
  *         operator.
  *
  * @define mergeReturn an Observable that emits items that are the
  *         result of flattening the items emitted by the Observables
  *         emitted by `this`.
  *
  * @define asyncBoundaryDescription Forces a buffered asynchronous boundary.
  *
  *         Internally it wraps the observer implementation given to
  *         `onSubscribe` into a
  *         [[monix.reactive.observers.BufferedSubscriber BufferedSubscriber]].
  *
  *         Normally Monix's implementation guarantees that events are
  *         not emitted concurrently, and that the publisher MUST NOT
  *         emit the next event without acknowledgement from the
  *         consumer that it may proceed, however for badly behaved
  *         publishers, this wrapper provides the guarantee that the
  *         downstream [[monix.reactive.Observer Observer]] given in
  *         `subscribe` will not receive concurrent events.
  *
  *         WARNING: if the buffer created by this operator is
  *         unbounded, it can blow up the process if the data source
  *         is pushing events faster than what the observer can
  *         consume, as it introduces an asynchronous boundary that
  *         eliminates the back-pressure requirements of the data
  *         source. Unbounded is the default
  *         [[monix.reactive.OverflowStrategy overflowStrategy]], see
  *         [[monix.reactive.OverflowStrategy OverflowStrategy]] for
  *         options.
  *
  * @define onOverflowParam a function that is used for signaling a special
  *         event used to inform the consumers that an overflow event
  *         happened, function that receives the number of dropped
  *         events as a parameter (see [[OverflowStrategy.Evicted]])
  *
  * @define bufferWithSelectorDesc Periodically gather items emitted by
  *         an observable into bundles and emit these bundles rather than
  *         emitting the items one at a time, whenever the `selector`
  *         observable signals an event.
  *
  *         The resulting observable collects the elements of the source
  *         in a buffer and emits that buffer whenever the given `selector`
  *         observable emits an `onNext` event, when the buffer is emitted
  *         as a sequence downstream and then reset. Thus the resulting
  *         observable emits connected, non-overlapping bundles triggered
  *         by the given `selector`.
  *
  *         If `selector` terminates with an `onComplete`, then the resulting
  *         observable also terminates normally. If `selector` terminates with
  *         an `onError`, then the resulting observable also terminates with an
  *         error.
  *
  *         If the source observable completes, then the current buffer gets
  *         signaled downstream. If the source triggers an error then the
  *         current buffer is being dropped and the error gets propagated
  *         immediately.
  *
  * @define catsOrderDesc In case type `A` is a primitive type and an
  *         `Order[A]` instance is not in scope, then you probably
  *         need this import:
  *
  *         {{{
  *           import cats.instances.all._
  *         }}}
  *
  *         Or in case your type `A` does not have an `Order[A]`
  *         instance defined for it, but it does have a Scala
  *         `Ordering` defined, then you can define one like this:
  *
  *         {{{
  *           import cats.Order
  *
  *           implicit val orderA = Order.fromOrdering[A]
  *         }}}
  */
abstract class Observable[+A] extends Serializable { self =>
  /** Characteristic function for an `Observable` instance, that creates
    * the subscription and that eventually starts the streaming of
    * events to the given [[Observer]], to be provided by observable
    * implementations.
    *
    * WARNING: This function is "unsafe" to call because it does not
    * protect the calls to the given [[Observer]] implementation in
    * regards to unexpected exceptions that violate the contract,
    * therefore the given instance must respect its contract and not
    * throw any exceptions when the observable calls `onNext`,
    * `onComplete` and `onError`. If it does, then the behavior is
    * undefined.
    *
    * Prefer normal [[Observable.subscribe(subscriber* subscribe]]
    * when consuming a stream, these unsafe subscription methods
    * being useful when building operators and for testing purposes.
    */
  def unsafeSubscribeFn(subscriber: Subscriber[A]): Cancelable

  /** Given an [[monix.reactive.Observer observer]] and a
    * [[monix.execution.Scheduler scheduler]] for managing async
    * boundaries, subscribes to this observable for events.
    *
    * Helper for calling the
    * [[Observable.unsafeSubscribeFn(subscriber* abstract method]].
    *
    * WARNING: this method is unsafe for calling directly because
    * subscriber instances aren't automatically wrapped in
    * [[monix.reactive.observers.SafeSubscriber SafeSubscriber]].
    *
    * Prefer normal [[Observable.subscribe(observer* subscribe]]
    * when consuming a stream, these unsafe subscription methods
    * being useful when building operators and for testing purposes.
    */
  final def unsafeSubscribeFn(observer: Observer[A])(implicit s: Scheduler): Cancelable =
    unsafeSubscribeFn(Subscriber(observer, s))

  /** Subscribes to the stream.
    *
    * @return a subscription that can be used to cancel the streaming.
    * @see [[consumeWith]] for another way of consuming observables
    */
  final def subscribe(observer: Observer[A])(implicit s: Scheduler): Cancelable =
    subscribe(Subscriber(observer, s))

  /** Subscribes to the stream.
    *
    * @return a subscription that can be used to cancel the streaming.
    * @see [[consumeWith]] for another way of consuming observables
    */
  final def subscribe(subscriber: Subscriber[A]): Cancelable =
    unsafeSubscribeFn(SafeSubscriber[A](subscriber))

  /** Subscribes to the stream.
    *
    * @return a subscription that can be used to cancel the streaming.
    * @see [[consumeWith]] for another way of consuming observables
    */
  final def subscribe(nextFn: A => Future[Ack], errorFn: Throwable => Unit)(implicit s: Scheduler): Cancelable =
    subscribe(nextFn, errorFn, () => ())

  /** Subscribes to the stream.
    *
    * @return a subscription that can be used to cancel the streaming.
    * @see [[consumeWith]] for another way of consuming observables
    */
  final def subscribe()(implicit s: Scheduler): Cancelable =
    subscribe(_ => Continue)

  /** Subscribes to the stream.
    *
    * @return a subscription that can be used to cancel the streaming.
    * @see [[consumeWith]] for another way of consuming observables
    */
  final def subscribe(nextFn: A => Future[Ack])(implicit s: Scheduler): Cancelable =
    subscribe(nextFn, error => s.reportFailure(error), () => ())

  /** Subscribes to the stream.
    *
    * @return a subscription that can be used to cancel the streaming.
    * @see [[consumeWith]] for another way of consuming observables
    */
  final def subscribe(nextFn: A => Future[Ack], errorFn: Throwable => Unit, completedFn: () => Unit)
    (implicit s: Scheduler): Cancelable = {

    subscribe(new Subscriber[A] {
      implicit val scheduler = s

      def onNext(elem: A) = nextFn(elem)
      def onComplete() = completedFn()
      def onError(ex: Throwable) = errorFn(ex)
    })
  }

  /** Transforms the source using the given operator. */
  final def liftByOperator[B](operator: Operator[A, B]): Observable[B] =
    new LiftByOperatorObservable(self, operator)

  /** On execution, consumes the source observable
    * with the given [[Consumer]], effectively transforming the
    * source observable into a [[monix.eval.Task Task]].
    */
  final def consumeWith[R](f: Consumer[A, R]): Task[R] =
    f(self)

  /** Creates a new Observable that emits the given element and then it
    * also emits the events of the source (prepend operation).
    */
  final def +:[B >: A](elem: B): Observable[B] =
    Observable.cons(elem, self)

  /** Creates a new Observable that emits the events of the source and
    * then it also emits the given element (appended to the stream).
    */
  final def :+[B >: A](elem: B): Observable[B] =
    self ++ Observable.now(elem)

  /** Given the source observable and another `Observable`, emits all of
    * the items from the first of these Observables to emit an item
    * and cancel the other.
    */
  final def ambWith[B >: A](other: Observable[B]): Observable[B] =
    Observable.firstStartedOf(self, other)

  /** Periodically gather items emitted by an observable into bundles
    * and emit these bundles rather than emitting the items one at a
    * time. This version of `buffer` is emitting items once the
    * internal buffer has reached the given count.
    *
    * If the source observable completes, then the current buffer gets
    * signaled downstream. If the source triggers an error then the
    * current buffer is being dropped and the error gets propagated
    * immediately.
    *
    * @param count the maximum size of each buffer before it should
    *        be emitted
    */
  final def bufferTumbling(count: Int): Observable[Seq[A]] =
    bufferSliding(count, count)

  /** Returns an observable that emits buffers of items it collects from
    * the source observable. The resulting observable emits buffers
    * every `skip` items, each containing `count` items.
    *
    * If the source observable completes, then the current buffer gets
    * signaled downstream. If the source triggers an error then the
    * current buffer is being dropped and the error gets propagated
    * immediately.
    *
    * For `count` and `skip` there are 3 possibilities:
    *
    *  1. in case `skip == count`, then there are no items dropped and
    *     no overlap, the call being equivalent to `bufferTumbling(count)`
    *  1. in case `skip < count`, then overlap between buffers
    *     happens, with the number of elements being repeated being
    *     `count - skip`
    *  1. in case `skip > count`, then `skip - count` elements start
    *     getting dropped between windows
    *
    * @param count the maximum size of each buffer before it should
    *        be emitted
    * @param skip how many items emitted by the source observable should
    *        be skipped before starting a new buffer. Note that when
    *        skip and count are equal, this is the same operation as
    *        `bufferTumbling(count)`
    */
  final def bufferSliding(count: Int, skip: Int): Observable[Seq[A]] =
    liftByOperator(new BufferSlidingOperator(count, skip))

  /** Periodically gather items emitted by an observable into bundles
    * and emit these bundles rather than emitting the items one at a
    * time.
    *
    * This version of `buffer` emits a new bundle of items
    * periodically, every timespan amount of time, containing all
    * items emitted by the source Observable since the previous bundle
    * emission.
    *
    * If the source observable completes, then the current buffer gets
    * signaled downstream. If the source triggers an error then the
    * current buffer is being dropped and the error gets propagated
    * immediately.
    *
    * @param timespan the interval of time at which it should emit
    *        the buffered bundle
    */
  final def bufferTimed(timespan: FiniteDuration): Observable[Seq[A]] =
    bufferTimedAndCounted(timespan, 0)

  /** Periodically gather items emitted by an observable into bundles
    * and emit these bundles rather than emitting the items one at a
    * time.
    *
    * The resulting observable emits connected, non-overlapping
    * buffers, each of a fixed duration specified by the `timespan`
    * argument or a maximum size specified by the `maxCount` argument
    * (whichever is reached first).
    *
    * If the source observable completes, then the current buffer gets
    * signaled downstream. If the source triggers an error then the
    * current buffer is being dropped and the error gets propagated
    * immediately.
    *
    * @param timespan the interval of time at which it should emit
    *        the buffered bundle
    * @param maxCount is the maximum bundle size, after which the
    *        buffered bundle gets forcefully emitted
    */
  final def bufferTimedAndCounted(timespan: FiniteDuration, maxCount: Int): Observable[Seq[A]] =
    new BufferTimedObservable[A](self, timespan, maxCount)

  /** Periodically gather items emitted by an observable into bundles
    * and emit these bundles rather than emitting the items one at a
    * time. Back-pressure the source when the buffer is full.
    *
    * The resulting observable emits connected, non-overlapping
    * buffers, each of a fixed duration specified by the `period`
    * argument.
    *
    * The bundles are emitted at a fixed rate. If the source is
    * silent, then the resulting observable will start emitting empty
    * sequences.
    *
    * If the source observable completes, then the current buffer gets
    * signaled downstream. If the source triggers an error then the
    * current buffer is being dropped and the error gets propagated
    * immediately.
    *
    * A `maxSize` argument is specified as the capacity of the
    * bundle. In case the source is too fast and `maxSize` is reached,
    * then the source will be back-pressured.
    *
    * The difference with [[bufferTimedAndCounted]] is that
    * [[bufferTimedWithPressure]] applies back-pressure from the time
    * when the buffer is full until the buffer is emitted, whereas
    * [[bufferTimedAndCounted]] will forcefully emit the buffer when
    * it's full.
    *
    * @param period the interval of time at which it should emit
    *        the buffered bundle
    * @param maxSize is the maximum buffer size, after which the
    *        source starts being back-pressured
    */
  final def bufferTimedWithPressure(period: FiniteDuration, maxSize: Int): Observable[Seq[A]] = {
    val sampler = Observable.intervalAtFixedRate(period, period)
    new BufferWithSelectorObservable(self, sampler, maxSize)
  }

  /** $bufferWithSelectorDesc
    *
    * @param selector is the observable that triggers the
    *        signaling of the current buffer
    */
  final def bufferWithSelector[S](selector: Observable[S]): Observable[Seq[A]] =
    new BufferWithSelectorObservable[A, S](self, selector, 0)

  /** $bufferWithSelectorDesc
    *
    * A `maxSize` argument is specified as the capacity of the
    * bundle. In case the source is too fast and `maxSize` is reached,
    * then the source will be back-pressured.
    *
    * @param selector is the observable that triggers the signaling of the
    *        current buffer
    * @param maxSize is the maximum bundle size, after which the
    *        source starts being back-pressured
    */
  final def bufferWithSelector[S](selector: Observable[S], maxSize: Int): Observable[Seq[A]] =
    new BufferWithSelectorObservable(self, selector, maxSize)

  /** Buffers signals while busy, after which it emits the
    * buffered events as a single bundle.
    *
    * This operator starts applying back-pressure when the
    * underlying buffer's size is exceeded.
    */
  final def bufferIntrospective(maxSize: Int): Observable[List[A]] =
    new BufferIntrospectiveObservable[A](self, maxSize)

  /** Applies the given partial function to the source
    * for each element for which the given partial function is defined.
    *
    * @param pf the function that filters and maps the source
    * @return an observable that emits the transformed items by the
    *         given partial function
    */
  final def collect[B](pf: PartialFunction[A, B]): Observable[B] =
    self.liftByOperator(new CollectOperator(pf))

  /** Creates a new observable from the source and another given
    * observable, by emitting elements combined in pairs. If one of
    * the observables emits fewer events than the other, then the rest
    * of the unpaired events are ignored.
    *
    * See [[zip]] for an alternative that pairs the items in strict sequence.
    *
    * @param other is an observable that gets paired with the source
    */
  final def combineLatest[B](other: Observable[B]): Observable[(A, B)] =
    new CombineLatest2Observable[A, B, (A, B)](self, other)((a, b) => (a, b))

  /** Creates a new observable from the source and another given
    * observable, by emitting elements combined in pairs. If one of
    * the observables emits fewer events than the other, then the rest
    * of the unpaired events are ignored.
    *
    * See [[zipMap]] for an alternative that pairs the items
    * in strict sequence.
    *
    * @param other is an observable that gets paired with the source
    * @param f is a mapping function over the generated pairs
    */
  final def combineLatestMap[B, R](other: Observable[B])(f: (A, B) => R): Observable[R] =
    new CombineLatest2Observable[A, B, R](self, other)(f)

  /** Ignores all items emitted by the source Observable and only calls
    * onCompleted or onError.
    *
    * @return an empty Observable that only calls onCompleted or onError,
    *         based on which one is called by the source Observable
    */
  final def completed: Observable[Nothing] =
    self.liftByOperator(CompletedOperator)

  /** Doesn't emit anything until a `timeout` period passes without the
    * source emitting anything. When that timeout happens, we
    * subscribe to the observable generated by the given function, an
    * observable that will keep emitting until the source will break
    * the silence by emitting another event.
    *
    * Note: If the source observable keeps emitting items more
    * frequently than the length of the time window, then no items
    * will be emitted by the resulting Observable.
    *
    * @param f is a function that receives the last element generated
    *        by the source, generating an observable to be subscribed
    *        when the source is timing out
    * @param timeout the length of the window of time that must pass after
    *        the emission of an item from the source Observable in
    *        which that Observable emits no items in order for the
    *        item to be emitted by the resulting Observable
    */
  final def debounceTo[B](timeout: FiniteDuration, f: A => Observable[B]): Observable[B] =
    self.switchMap(a => f(a).delayExecution(timeout))

  /** Hold an Observer's subscription request for a specified amount of
    * time before passing it on to the source Observable.
    *
    * @param timespan is the time to wait before the subscription
    *        is being initiated.
    */
  final def delayExecution(timespan: FiniteDuration): Observable[A] =
    new DelayExecutionByTimespanObservable(self, timespan)

  /** $switchMapDescription */
  final def switchMap[B](f: A => Observable[B]): Observable[B] =
    new SwitchMapObservable[A, B](self, f)

  /** Emits the last item from the source Observable if a particular
    * timespan has passed without it emitting another item, and keeps
    * emitting that item at regular intervals until the source breaks
    * the silence.
    *
    * So compared to regular [[debounceTo]] this version
    * keeps emitting the last item of the source.
    *
    * Note: If the source Observable keeps emitting items more
    * frequently than the length of the time window then no items will
    * be emitted by the resulting Observable.
    *
    * @param period the length of the window of time that must pass after
    *        the emission of an item from the source Observable in
    *        which that Observable emits no items in order for the
    *        item to be emitted by the resulting Observable at regular
    *        intervals, also determined by period
    * @see [[echoRepeated]] for a similar operator that also mirrors
    *     the source observable
    */
  final def debounceRepeated(period: FiniteDuration): Observable[A] =
    new DebounceObservable(self, period, repeat = true)

  /** Emit items from the source, or emit a default item if
    * the source completes after emitting no items.
    */
  final def defaultIfEmpty[B >: A](default: => B): Observable[B] =
    self.liftByOperator(new DefaultIfEmptyOperator[B](default _))

  /** Delays emitting the final `onComplete` event by the specified amount. */
  final def delayOnComplete(delay: FiniteDuration): Observable[A] =
    new DelayOnCompleteObservable(self, delay)

  /** Returns an Observable that emits the items emitted by the source
    * Observable shifted forward in time by a specified delay.
    *
    * Each time the source Observable emits an item, delay starts a
    * timer, and when that timer reaches the given duration, the
    * Observable returned from delay emits the same item.
    *
    * NOTE: this delay refers strictly to the time between the
    * `onNext` event coming from our source and the time it takes the
    * downstream observer to get this event. On the other hand the
    * operator is also applying back-pressure, so on slow observers
    * the actual time passing between two successive events may be
    * higher than the specified `duration`.
    *
    * @param duration - the delay to shift the source by
    * @return the source Observable shifted in time by the specified delay
    */
  final def delayOnNext(duration: FiniteDuration): Observable[A] =
    new DelayByTimespanObservable[A](self, duration)

  /** Returns an Observable that emits the items emitted by the source
    * Observable shifted forward in time.
    *
    * This variant of `delay` sets its delay duration on a per-item
    * basis by passing each item from the source Observable into a
    * function that returns an Observable and then monitoring those
    * Observables. When any such Observable emits an item or
    * completes, the Observable returned by delay emits the associated
    * item.
    *
    * @param selector is a function that returns an Observable for
    *        each item emitted by the source Observable, which is then
    *        used to delay the emission of that item by the resulting
    *        Observable until the Observable returned from `selector`
    *        emits an item
    * @return the source Observable shifted in time by
    *         the specified delay
    */
  final def delayOnNextBySelector[B](selector: A => Observable[B]): Observable[A] =
    new DelayBySelectorObservable[A, B](self, selector)

  /** Hold an Observer's subscription request until the given `trigger`
    * observable either emits an item or completes, before passing it
    * on to the source Observable.
    *
    * If the given `trigger` completes in error, then the subscription is
    * terminated with `onError`.
    *
    * @param trigger the observable that must either emit an item or
    *        complete in order for the source to be subscribed.
    */
  final def delayExecutionWith(trigger: Observable[Any]): Observable[A] =
    new DelayExecutionWithTriggerObservable(self, trigger)

  /** Converts the source Observable that emits `Notification[A]` (the
    * result of [[materialize]]) back to an Observable that emits `A`.
    */
  final def dematerialize[B](implicit ev: A <:< Notification[B]): Observable[B] =
    self.asInstanceOf[Observable[Notification[B]]].liftByOperator(new DematerializeOperator[B])

  /** Suppress duplicate consecutive items emitted by the source.
    *
    * Example:
    * {{{
    *   // Yields 1, 2, 1, 3, 2, 4
    *   Observable(1, 1, 1, 2, 2, 1, 1, 3, 3, 3, 2, 2, 4, 4, 4)
    *     .distinctUntilChanged
    * }}}
    *
    * Duplication is detected by using the equality relationship
    * provided by the `cats.Eq` type class. This allows one to
    * override the equality operation being used (e.g. maybe the
    * default `.equals` is badly defined, or maybe you want reference
    * equality, so depending on use case).
    *
    * In case type `A` is a primitive type and an `Eq[A]` instance
    * is not in scope, then you probably need this import:
    *
    * {{{
    *   import cats.instances.all._
    * }}}
    *
    * Or in case your type `A` does not have an `Eq[A]` instance
    * defined for it, then you can quickly define one like this:
    * {{{
    *   import cats.Eq
    *
    *   implicit val eqA = Eq.fromUniversalEquals[A]
    * }}}
    *
    * @param A is the `cats.Eq` instance that defines equality
    *        for the elements emitted by the source
    */
  final def distinctUntilChanged[AA >: A](implicit A: Eq[AA]): Observable[AA] =
    self.liftByOperator(new DistinctUntilChangedOperator()(A))

  /** Given a function that returns a key for each element emitted by
    * the source, suppress consecutive duplicate items.
    *
    * Example:
    *
    * {{{
    *   // Yields 1, 2, 3, 4
    *   Observable(1, 3, 2, 4, 2, 3, 5, 7, 4)
    *     .distinctUntilChangedBy(_ % 2)
    * }}}
    *
    * Duplication is detected by using the equality relationship
    * provided by the `cats.Eq` type class. This allows one to
    * override the equality operation being used (e.g. maybe the
    * default `.equals` is badly defined, or maybe you want reference
    * equality, so depending on use case).
    *
    * In case type `K` is a primitive type and an `Eq[K]` instance
    * is not in scope, then you probably need this import:
    *
    * {{{
    *   import cats.instances.all._
    * }}}
    *
    * Or in case your type `K` does not have an `Eq[K]` instance
    * defined for it, then you can quickly define one like this:
    *
    * {{{
    *   import cats.Eq
    *
    *   implicit val eqK = Eq.fromUniversalEquals[K]
    * }}}
    *
    * @param key is a function that returns a `K` key for each element,
    *        a value that's then used to do the deduplication
    *
    * @param K is the `cats.Eq` instance that defines equality for
    *        the key type `K`
    */
  final def distinctUntilChangedByKey[K](key: A => K)(implicit K: Eq[K]): Observable[A] =
    self.liftByOperator(new DistinctUntilChangedByKeyOperator(key)(K))

  /** Executes the given callback when the streaming is stopped
    * due to a downstream [[monix.execution.Ack.Stop Stop]] signal
    * returned by [[monix.reactive.Observer.onNext onNext]].
    *
    * @see [[doOnEarlyStopTask]] for a version that allows for
    *      asynchronous evaluation by means of [[monix.eval.Task Task]],
    *      or [[doOnEarlyStopEval]] that allows using other data types
    *      for dealing with evaluation and side effects.
    */
  final def doOnEarlyStop(cb: () => Unit): Observable[A] =
    self.liftByOperator(new DoOnEarlyStopOperator[A](cb))

  /** Executes the given task when the streaming is stopped
    * due to a downstream [[monix.execution.Ack.Stop Stop]] signal
    * returned by [[monix.reactive.Observer.onNext onNext]].
    *
    * The given `effect` gets evaluated *before* the upstream
    * receives the `Stop` event (is back-pressured). This
    * version of [[doOnEarlyStop]] is using any `F[_]` data type
    * that implements `cats.effect.Effect` (e.g. `Task`, `IO`, etc).
    *
    * @see [[doOnEarlyStop]] for a simpler version, or
    *      [[doOnEarlyStopTask]] for a version that's specialized for
    *      [[monix.eval.Task Task]].
    */
  final def doOnEarlyStopEval[F[_]](effect: F[Unit])(implicit F: TaskLike[F]): Observable[A] =
    doOnEarlyStopTask(F.toTask(effect))

  /** Executes the given task when the streaming is stopped
    * due to a downstream [[monix.execution.Ack.Stop Stop]] signal
    * returned by [[monix.reactive.Observer.onNext onNext]].
    *
    * The given `task` gets evaluated *before* the upstream
    * receives the `Stop` event (is back-pressured).
    *
    * @see [[doOnEarlyStop]] for a simpler version, or
    *      [[doOnEarlyStopEval]] for a version that can use any
    *      data type that implements `cats.effect.Effect`.
    */
  final def doOnEarlyStopTask(task: Task[Unit]): Observable[A] =
    self.liftByOperator(new EvalOnEarlyStopOperator[A](task))

  /** Executes the given callback when the connection is being
    * [[monix.execution.Cancelable.cancel cancelled]].
    */
  final def doOnSubscriptionCancel(cb: () => Unit): Observable[A] =
    new DoOnSubscriptionCancelObservable[A](self, cb)

  /** Executes the given callback when the stream has ended with an
    * `onComplete` event, but before the complete event is emitted.
    *
    * Unless you know what you're doing, you probably want to use
    * [[doOnTerminate]] and [[doOnSubscriptionCancel]] for proper
    * disposal of resources on completion.
    *
    * @param cb the callback to execute when the `onComplete`
    *        event gets emitted
    * @see [[doOnCompleteTask]] for a version that allows for asynchronous
    *      evaluation by means of [[monix.eval.Task Task]].
    */
  final def doOnComplete(cb: () => Unit): Observable[A] =
    self.liftByOperator(new DoOnCompleteOperator[A](cb))

  /** Evaluates the given `effect` when the stream has ended with an
    * `onComplete` event, but before the complete event is emitted.
    *
    * The `effect` gets evaluated and is finished *before* the
    * `onComplete` signal gets sent downstream.
    *
    * This version of [[doOnComplete]] evaluates the given side
    * effects using the specified `F[_]` data type, which should
    * implement `cats.effect.Effect`.
    *
    * Unless you know what you're doing, you probably want to use
    * [[doOnTerminateTask]] and [[doOnSubscriptionCancel]] for proper
    * disposal of resources on completion.
    *
    * @param effect the action to execute when the `onComplete`
    *        event gets emitted, its type being one that implements
    *        `cats.effect.Effect` and thus capable of delayed and
    *        asynchronous evaluation
    * @see [[doOnComplete]] for a simpler version that doesn't
    *     do asynchronous execution, or [[doOnCompleteTask]] for
    *     a version specialized for [[monix.eval.Task Task]]
    */
  final def doOnCompleteEval[F[_]](effect: F[Unit])(implicit F: TaskLike[F]): Observable[A] =
    doOnCompleteTask(F.toTask(effect))

  /** Evaluates the given task when the stream has ended with an
    * `onComplete` event, but before the complete event is emitted.
    *
    * The task gets evaluated and is finished *before* the `onComplete`
    * signal gets sent downstream.
    *
    * Unless you know what you're doing, you probably want to use
    * [[doOnTerminateTask]] and [[doOnSubscriptionCancel]] for proper
    * disposal of resources on completion.
    *
    * @see [[doOnComplete]] for a simpler version that doesn't
    *      do asynchronous execution, or [[doOnCompleteEval]] for
    *      a version that can work with any data type implementing
    *      `cats.effect.Effect`.
    * @param task the task to execute when the `onComplete`
    *        event gets emitted
    */
  final def doOnCompleteTask(task: Task[Unit]): Observable[A] =
    self.liftByOperator(new EvalOnCompleteOperator[A](task))

  /** Executes the given callback when the stream is interrupted with an
    * error, before the `onError` event is emitted downstream.
    *
    * NOTE: should protect the code in this callback, because if it
    * throws an exception the `onError` event will prefer signaling
    * the original exception and otherwise the behavior is undefined.
    *
    * @see [[doOnTerminate]] and [[doOnSubscriptionCancel]] for
    *      handling resource disposal, also see [[doOnErrorTask]] for
    *      a version that does asynchronous evaluation by means of
    *      [[monix.eval.Task Task]].
    */
  final def doOnError(cb: Throwable => Unit): Observable[A] =
    self.liftByOperator(new DoOnErrorOperator[A](cb))

  /** Executes the given `cb` when the stream is interrupted with an
    * error, before the `onError` event is emitted downstream.
    *
    * This version of [[doOnError]] evaluates the given side
    * effects using the specified `F[_]` data type, which should
    * implement `cats.effect.Effect`, thus being able of asynchronous
    * execution.
    *
    * NOTE: should protect the code in this callback, because if it
    * throws an exception the `onError` event will prefer signaling
    * the original exception and otherwise the behavior is undefined.
    *
    * @see [[doOnTerminateTask]] and [[doOnSubscriptionCancel]] for
    *      handling resource disposal, also see [[doOnError]] for a
    *      simpler version that doesn't do asynchronous execution,
    *      or [[doOnErrorTask]] for a version specialized on
    *      [[monix.eval.Task Task]].
    */
  final def doOnErrorEval[F[_]](cb: Throwable => F[Unit])(implicit F: TaskLike[F]): Observable[A] =
    doOnErrorTask(e => F.toTask(cb(e)))

  /** Executes the given task when the stream is interrupted with an
    * error, before the `onError` event is emitted downstream.
    *
    * NOTE: should protect the code in this callback, because if it
    * throws an exception the `onError` event will prefer signaling
    * the original exception and otherwise the behavior is undefined.
    *
    * @see [[doOnTerminateTask]] and [[doOnSubscriptionCancel]] for
    *      handling resource disposal, also see [[doOnError]] for a
    *      simpler version that doesn't do asynchronous execution,
    *      or [[doOnErrorEval]] for a version that can use any
    *      `Effect` data type.
    */
  final def doOnErrorTask(cb: Throwable => Task[Unit]): Observable[A] =
    self.liftByOperator(new EvalOnErrorOperator[A](cb))

  /** Executes the given callback right before the streaming is ended either
    * with an `onComplete` or `onError` event, or when the streaming stops by
    * a downstream `Stop` being signaled.
    *
    * It is the equivalent of calling:
    *
    *   - [[doOnComplete]]
    *   - [[doOnError]]
    *   - [[doOnEarlyStop]]
    *
    * This differs from [[doAfterTerminate]] in that this happens *before*
    * the `onComplete` or `onError` notification.
    *
    * @see [[doOnTerminateTask]] for a version that allows for asynchronous
    *      evaluation by means of [[monix.eval.Task Task]].
    */
  final def doOnTerminate(cb: Option[Throwable] => Unit): Observable[A] =
    self.liftByOperator(new DoOnTerminateOperator[A](cb, happensBefore = true))

  /** Evaluates the callback right before the streaming is ended
    * either with an `onComplete` or `onError` event, or when the
    * streaming stops by a downstream `Stop` being signaled.
    *
    * The callback-generated `F[_]` is a data type that should
    * implement `cats.effect.Effect` and is thus capable of
    * asynchronous evaluation, back-pressuring the source
    * when applied for `Stop` events returned by `onNext` and thus
    * the upstream source will receive the `Stop` result only after
    * the task has finished executing.
    *
    * It is the equivalent of calling:
    *
    *   - [[doOnCompleteTask]]
    *   - [[doOnErrorTask]]
    *   - [[doOnEarlyStopTask]]
    *
    * This differs from [[doAfterTerminateTask]] in that this happens
    * *before* the `onComplete` or `onError` notification.
    *
    * @see [[doOnTerminate]] for a simpler version that doesn't allow
    *     asynchronous execution, or [[doOnTerminateTask]] for a
    *     version that's specialized on [[monix.eval.Task Task]].
    */
  final def doOnTerminateEval[F[_]](cb: Option[Throwable] => F[Unit])(implicit F: TaskLike[F]): Observable[A] =
    doOnTerminateTask(opt => F.toTask(cb(opt)))

  /** Evaluates the task generated by the given callback right before
    * the streaming is ended either with an `onComplete` or `onError`
    * event, or when the streaming stops by a downstream `Stop` being
    * signaled.
    *
    * The callback-generated [[monix.eval.Task Task]] will
    * back-pressure the source when applied for `Stop` events returned
    * by `onNext` and thus the upstream source will receive the `Stop`
    * result only after the task has finished executing.
    *
    * It is the equivalent of calling:
    *
    *   - [[doOnCompleteTask]]
    *   - [[doOnErrorTask]]
    *   - [[doOnEarlyStopTask]]
    *
    * This differs from [[doAfterTerminateTask]] in that this happens
    * *before* the `onComplete` or `onError` notification.
    *
    * @see [[doOnTerminate]] for a simpler version that doesn't allow
    *     asynchronous execution, or [[doOnTerminateEval]] for a
    *     version that can work with any `cats.effect.Effect` type.
    */
  final def doOnTerminateTask(cb: Option[Throwable] => Task[Unit]): Observable[A] =
    self.liftByOperator(new EvalOnTerminateOperator[A](cb, happensBefore = true))

  /** Executes the given callback after the stream has ended either with
    * an `onComplete` or `onError` event, or when the streaming stops by
    * a downstream `Stop` being signaled.
    *
    * This differs from [[doOnTerminate]] in that this happens *after*
    * the `onComplete` or `onError` notification.
    *
    * @see [[doAfterTerminateTask]] for a version that allows for asynchronous
    *      evaluation by means of [[monix.eval.Task Task]].
    */
  final def doAfterTerminate(cb: Option[Throwable] => Unit): Observable[A] =
    self.liftByOperator(new DoOnTerminateOperator[A](cb, happensBefore = false))

  /** Evaluates the effect generated by the given callback after the
    * stream has ended either with an `onComplete` or `onError` event,
    * or when the streaming stops by a downstream `Stop` being signaled.
    *
    * This operation subsumes [[doOnEarlyStopEval]] and the
    * callback-generated value will back-pressure the source when
    * applied for `Stop` events returned by `onNext` and thus the
    * upstream source will receive the `Stop` result only after the
    * task has finished executing.
    *
    * The callback-generated `F[_]` is a data type that should
    * implement `cats.effect.Effect` and is thus capable of
    * asynchronous evaluation (e.g. `Task`, `IO`, etc).
    *
    * This differs from [[doOnTerminateEval]] in that this happens
    * *after* the `onComplete` or `onError` notification.
    *
    * @see [[doAfterTerminate]] for a simpler version that doesn't
    *     allow asynchronous execution, or [[doAfterTerminateTask]]
    *     for a version that's specialized for [[monix.eval.Task Task]].
    */
  final def doAfterTerminateEval[F[_]](cb: Option[Throwable] => F[Unit])(implicit F: TaskLike[F]): Observable[A] =
    doAfterTerminateTask(opt => F.toTask(cb(opt)))

  /** Evaluates the task generated by the given callback after the
    * stream has ended either with an `onComplete` or `onError` event,
    * or when the streaming stops by a downstream `Stop` being signaled.
    *
    * This operation subsumes [[doOnEarlyStopTask]] and the
    * callback-generated [[monix.eval.Task Task]] will back-pressure
    * the source when applied for `Stop` events returned by `onNext`
    * and thus the upstream source will receive the `Stop` result only
    * after the task has finished executing.
    *
    * This differs from [[doOnTerminateTask]] in that this happens
    * *after* the `onComplete` or `onError` notification.
    *
    * @see [[doAfterTerminate]] for a simpler version that doesn't allow
    *     asynchronous execution.
    */
  final def doAfterTerminateTask(cb: Option[Throwable] => Task[Unit]): Observable[A] =
    self.liftByOperator(new EvalOnTerminateOperator[A](cb, happensBefore = false))

  /** Executes the given callback for each element generated by the
    * source Observable, useful for doing side-effects.
    *
    * @return a new Observable that executes the specified
    *         callback for each element
    * @see [[doOnNextTask]] for a version that allows for asynchronous
    *      evaluation by means of [[monix.eval.Task Task]].
    */
  final def doOnNext(cb: A => Unit): Observable[A] =
    self.map { a => cb(a); a }

  /** Evaluates the given callback for each element generated by the
    * source Observable, useful for triggering async side-effects.
    *
    * This version of [[doOnNext]] evaluates side effects with any
    * data type that implements `cats.effect.Effect` (e.g. `Task`, `IO`).
    *
    * @return a new Observable that executes the specified
    *         callback for each element
    *
    * @see [[doOnNext]] for a simpler version that doesn't allow
    *     asynchronous execution, or [[doOnNextTask]] for a version
    *     that's specialized on [[monix.eval.Task Task]].
    */
  final def doOnNextEval[F[_]](cb: A => F[Unit])(implicit F: TaskLike[F]): Observable[A] =
    doOnNextTask(a => F.toTask(cb(a)))

  /** Evaluates the given callback for each element generated by the
    * source Observable, useful for triggering async side-effects.
    *
    * @return a new Observable that executes the specified
    *         callback for each element
    *
    * @see [[doOnNext]] for a simpler version that doesn't allow
    *     asynchronous execution.
    */
  final def doOnNextTask(cb: A => Task[Unit]): Observable[A] =
    self.mapTask(a => cb(a).map(_ => a))

  /** Maps elements from the source using a function that can do
    * asynchronous processing by means of [[monix.eval.Task Task]].
    *
    * Given a source observable, this function is basically the
    * equivalent of doing:
    *
    * {{{
    *   observable.concatMap(a => Observable.fromTask(f(a)))
    * }}}
    *
    * However prefer this operator to `concatMap` because it
    * is more clear and has better performance.
    *
    * @see [[mapFuture]] for the version that can work with
    *      `scala.concurrent.Future`
    */
  final def mapTask[B](f: A => Task[B]): Observable[B] =
    new MapTaskObservable[A, B](self, f)

  /** Executes the given callback on each acknowledgement
    * received from the downstream subscriber.
    *
    * This method helps in executing logic after messages get
    * processed, for example when messages are polled from
    * some distributed message queue and an acknowledgement
    * needs to be sent after each message in order to mark it
    * as processed.
    *
    * @see [[doOnNextAckTask]] for a version that allows for asynchronous
    *      evaluation by means of [[monix.eval.Task Task]], or
    *      [[doOnNextEval]] for a version that can do evaluation with
    *      any data type implementing `cats.effect.Effect`.
    */
  final def doOnNextAck(cb: (A, Ack) => Unit): Observable[A] =
    self.liftByOperator(new DoOnNextAckOperator[A](cb))

  /** Executes the given callback on each acknowledgement received
    * from the downstream subscriber, executing a generated
    * effect and back-pressuring until it is done executing.
    *
    * This method helps in executing logic after messages get
    * processed, for example when messages are polled from
    * some distributed message queue and an acknowledgement
    * needs to be sent after each message in order to mark it
    * as processed.
    *
    * This version of [[doOnNext]] evaluates side effects with any
    * data type that implements `cats.effect.Effect` (e.g. `Task`, `IO`).
    *
    * @see [[doOnNextAck]] for a simpler version that doesn't allow
    *     asynchronous execution, or [[doOnNextAckTask]] for a version
    *     that's specialized on [[monix.eval.Task Task]].
    */
  final def doOnNextAckEval[F[_]](cb: (A, Ack) => F[Unit])(implicit F: TaskLike[F]): Observable[A] =
    doOnNextAckTask((a, ack) => Task.from(cb(a, ack))(F))

  /** Executes the given callback on each acknowledgement received from
    * the downstream subscriber, executing a generated
    * [[monix.eval.Task Task]] and back-pressuring until the task
    * is done.
    *
    * This method helps in executing logic after messages get
    * processed, for example when messages are polled from
    * some distributed message queue and an acknowledgement
    * needs to be sent after each message in order to mark it
    * as processed.
    *
    * @see [[doOnNextAck]] for a simpler version that doesn't allow
    *     asynchronous execution, or [[doOnNextAckTask]] for a version
    *     that's specialized on [[monix.eval.Task Task]].
    */
  final def doOnNextAckTask(cb: (A, Ack) => Task[Unit]): Observable[A] =
    self.liftByOperator(new EvalOnNextAckOperator[A](cb))

  /** Executes the given callback only for the first element generated
    * by the source Observable, useful for doing a piece of
    * computation only when the stream starts.
    *
    * @return a new Observable that executes the specified callback
    *         only for the first element
    */
  final def doOnStart(cb: A => Unit): Observable[A] =
    self.liftByOperator(new DoOnStartOperator[A](cb))

  /** Executes the given callback just before the subscription happens.
    *
    * @see [[doAfterSubscribe]] for executing a callback just after
    *     a subscription happens.
    */
  final def doOnSubscribe(cb: () => Unit): Observable[A] =
    new DoOnSubscribeObservable.Before[A](self, cb)

  /** Executes the given callback just after the subscription happens.
    *
    * @see [[doOnSubscribe]] for executing a callback just before
    *     a subscription happens.
    */
  final def doAfterSubscribe(cb: () => Unit): Observable[A] =
    new DoOnSubscribeObservable.After[A](self, cb)

  /** Creates a new observable that drops the events of the source, only
    * for the specified `timestamp` window.
    *
    * @param timespan the window of time during which the new observable
    *        must drop events emitted by the source
    */
  final def dropByTimespan(timespan: FiniteDuration): Observable[A] =
    new DropByTimespanObservable(self, timespan)

  /** Drops the last `n` elements (from the end).
    *
    * @param n the number of elements to drop
    * @return a new Observable that drops the first ''n'' elements
    *         emitted by the source
    */
  final def dropLast(n: Int): Observable[A] =
    self.liftByOperator(new DropLastOperator[A](n))

  /** Discard items emitted by the source until a second
    * observable emits an item or completes.
    *
    * If the `trigger` observable completes in error, then the
    * resulting observable will also end in error when it notices
    * it (next time an element is emitted by the source).
    *
    * @param trigger the observable that has to emit an item before the
    *        source begin to be mirrored by the resulting observable
    */
  final def dropUntil(trigger: Observable[Any]): Observable[A] =
    new DropUntilObservable(self, trigger)

  /** Drops the longest prefix of elements that satisfy the given
    * predicate and returns a new observable that emits the rest.
    */
  final def dropWhile(p: A => Boolean): Observable[A] =
    self.liftByOperator(new DropByPredicateOperator(p))

  /** Drops the longest prefix of elements that satisfy the given
    * function and returns a new observable that emits the rest. In
    * comparison with [[dropWhile]], this version accepts a function
    * that takes an additional parameter: the zero-based index of the
    * element.
    */
  final def dropWhileWithIndex(p: (A, Int) => Boolean): Observable[A] =
    self.liftByOperator(new DropByPredicateWithIndexOperator(p))

  /** Utility that can be used for debugging purposes.
    */
  final def dump(prefix: String, out: PrintStream = System.out): Observable[A] =
    new DumpObservable[A](self, prefix, out)

  /** Mirror the source observable as long as the source keeps emitting
    * items, otherwise if `timeout` passes without the source emitting
    * anything new then the observable will emit the last item.
    *
    * This is the rough equivalent of:
    * {{{
    *   Observable.merge(source, source.debounce(period))
    * }}}
    *
    * Note: If the source Observable keeps emitting items more
    * frequently than the length of the time window then the resulting
    * observable will mirror the source exactly.
    *
    * @param timeout the window of silence that must pass in order for the
    *        observable to echo the last item
    */
  final def echoOnce(timeout: FiniteDuration): Observable[A] =
    new EchoObservable(self, timeout, onlyOnce = true)

  /** Mirror the source observable as long as the source keeps emitting
    * items, otherwise if `timeout` passes without the source emitting
    * anything new then the observable will start emitting the last
    * item repeatedly.
    *
    * Note: If the source Observable keeps emitting items more
    * frequently than the length of the time window then the resulting
    * observable will mirror the source exactly.
    *
    * @param timeout the window of silence that must pass in order for the
    *        observable to start echoing the last item
    */
  final def echoRepeated(timeout: FiniteDuration): Observable[A] =
    new EchoObservable(self, timeout, onlyOnce = false)

  /** Creates a new Observable that emits the events of the source and
    * then it also emits the given elements (appended to the stream).
    */
  final def endWith[B >: A](elems: Seq[B]): Observable[B] =
    self ++ Observable.fromIterable(elems)

  /** Concatenates the source with another observable.
    *
    * Ordering of subscription is preserved, so the second observable
    * starts only after the source observable is completed
    * successfully with an `onComplete`. On the other hand, the second
    * observable is never subscribed if the source completes with an
    * error.
    */
  final def ++[B >: A](other: Observable[B]): Observable[B] =
    new ConcatObservable[B](self, other)

  /** Emits the given exception instead of `onComplete`.
    *
    * @param error the exception to emit onComplete
    * @return a new Observable that emits an exception onComplete
    */
  final def endWithError(error: Throwable): Observable[A] =
    self.liftByOperator(new EndWithErrorOperator[A](error))

  /** Returns an observable that emits a single Throwable, in case an
    * error was thrown by the source, otherwise it isn't going to emit
    * anything.
    */
  final def failed: Observable[Throwable] =
    self.liftByOperator(FailedOperator)

  /** Emits the first element emitted by the source, or otherwise if the
    * source is completed without emitting anything, then the
    * `default` is emitted.
    *
    * Alias for `headOrElse`.
    */
  final def firstOrElseF[B >: A](default: => B): Observable[B] =
    headOrElseF(default)

  /** Emits the first element emitted by the source, or otherwise if the
    * source is completed without emitting anything, then the
    * `default` is emitted.
    */
  final def headOrElseF[B >: A](default: => B): Observable[B] =
    headF.foldLeftF(Option.empty[B])((_, elem) => Some(elem)).map {
      case Some(elem) => elem
      case None => default
    }

  /** Returns a new observable that applies the given function
    * to each item emitted by the source and emits the result.
    */
  final def map[B](f: A => B): Observable[B] =
    self.liftByOperator(new MapOperator(f))

  /** Applies a function that you supply to each item emitted by the
    * source observable, where that function returns sequences that
    * [[Observable can be observed]], and then concatenating those
    * resulting sequences and emitting the results of this concatenation.
    *
    * Alias for [[concatMap]].
    *
    * $concatMergeDifference
    */
  final def flatMap[B](f: A => Observable[B]): Observable[B] =
    self.concatMap(f)

  /** Applies a function that you supply to each item emitted by the
    * source observable, where that function returns observables,
    * and then concatenating those resulting sequences and
    * emitting the results of this concatenation.
    *
    * $concatMergeDifference
    */
  final def concatMap[B](f: A => Observable[B]): Observable[B] =
    new ConcatMapObservable[A, B](self, f, delayErrors = false)

  /** Applies a function that you supply to each item emitted by the
    * source observable, where that function returns sequences
    * and then concatenating those resulting sequences and emitting the
    * results of this concatenation.
    *
    * It's an alias for [[concatMapDelayErrors]].
    *
    * @param f a function that, when applied to an item emitted by
    *        the source Observable, returns an Observable
    * @return an Observable that emits the result of applying the
    *         transformation function to each item emitted by the
    *         source Observable and concatenating the results of the
    *         Observables obtained from this transformation.
    */
  final def flatMapDelayErrors[B](f: A => Observable[B]): Observable[B] =
    concatMapDelayErrors(f)

  /** An alias of [[switchMap]].
    *
    * $switchMapDescription
    */
  final def flatMapLatest[B](f: A => Observable[B]): Observable[B] =
    self.switchMap(f)

  /** Applies a binary operator to a start value and to elements
    * produced by the source observable, going from left to right,
    * producing and concatenating observables along the way.
    *
    * It's the combination between [[scan]] and [[flatMap]].
    *
    * @see [[flatScan0]] for the version that emits seed element at the beginning
    */
  final def flatScan[R](seed: => R)(op: (R, A) => Observable[R]): Observable[R] =
    new FlatScanObservable[A, R](self, seed _, op, delayErrors = false)

  /** Applies a binary operator to a start value and to elements
    * produced by the source observable, going from left to right,
    * producing and concatenating observables along the way.
    *
    * It's the combination between [[scan0]] and [[flatMap]].
    */
  final def flatScan0[R](seed: => R)(op: (R, A) => Observable[R]): Observable[R] =
    Observable.eval(seed).flatMap(s => s +: flatScan(s)(op))

  /** Applies a binary operator to a start value and to elements
    * produced by the source observable, going from left to right,
    * producing and concatenating observables along the way.
    *
    * This version of [[flatScan]] delays all errors until `onComplete`,
    * when it will finally emit a
    * [[monix.execution.exceptions.CompositeException CompositeException]].
    * It's the combination between [[scan]] and [[flatMapDelayErrors]].
    */
  final def flatScanDelayErrors[R](seed: => R)(op: (R, A) => Observable[R]): Observable[R] =
    new FlatScanObservable[A, R](self, seed _, op, delayErrors = true)

  /** Applies a binary operator to a start value and to elements
    * produced by the source observable, going from left to right,
    * producing and concatenating observables along the way.
    *
    * This version of [[flatScan0]] delays all errors until `onComplete`,
    * when it will finally emit a
    * [[monix.execution.exceptions.CompositeException CompositeException]].
    * It's the combination between [[scan0]] and [[flatMapDelayErrors]].
    */
  final def flatScan0DelayErrors[R](seed: => R)(op: (R, A) => Observable[R]): Observable[R] =
    Observable.eval(seed).flatMap(s => s +: flatScanDelayErrors(s)(op))

  /** $concatDescription
    *
    * Alias for [[concat]].
    *
    * @return $concatReturn
    */
  final def flatten[B](implicit ev: A <:< Observable[B]): Observable[B] =
    concat

  /** $concatDescription
    *
    * @return $concatReturn
    */
  final def concat[B](implicit ev: A <:< Observable[B]): Observable[B] =
    concatMap[B](x => x)

  /** Alias for [[concatDelayErrors]].
    *
    * $concatDescription
    * $delayErrorsDescription
    *
    * @return $concatReturn
    */
  final def flattenDelayErrors[B](implicit ev: A <:< Observable[B]): Observable[B] =
    concatDelayErrors

  /** $concatDescription
    *
    * $delayErrorsDescription
    *
    * @return $concatReturn
    */
  final def concatDelayErrors[B](implicit ev: A <:< Observable[B]): Observable[B] =
    concatMapDelayErrors(x => x)

  /** Applies a function that you supply to each item emitted by the
    * source observable, where that function returns sequences
    * and then concatenating those resulting sequences and emitting the
    * results of this concatenation.
    *
    * $delayErrorsDescription
    *
    * @param f a function that, when applied to an item emitted by
    *        the source, returns an observable
    * @return $concatReturn
    */
  final def concatMapDelayErrors[B](f: A => Observable[B]): Observable[B] =
    new ConcatMapObservable[A, B](self, f, delayErrors = true)

  /** Alias for [[switch]]
    *
    * $switchDescription
    */
  final def flattenLatest[B](implicit ev: A <:< Observable[B]): Observable[B] =
    self.switch

  /** $switchDescription */
  final def switch[B](implicit ev: A <:< Observable[B]): Observable[B] =
    self.switchMap(x => x)

  /** Returns an Observable that emits a single boolean, either true, in
    * case the given predicate holds for all the items emitted by the
    * source, or false in case at least one item is not verifying the
    * given predicate.
    *
    * @param p is a function that evaluates the items emitted by the source
    *        Observable, returning `true` if they pass the filter
    * @return an Observable that emits only true or false in case the given
    *         predicate holds or not for all the items
    */
  final def forAllF(p: A => Boolean): Observable[Boolean] =
    existsF(e => !p(e)).map(r => !r)

  /** Returns an Observable which emits a single value, either true, in
    * case the given predicate holds for at least one item, or false
    * otherwise.
    *
    * @param p is a function that evaluates the items emitted by the
    *        source Observable, returning `true` if they pass the
    *        filter
    * @return an Observable that emits only true or false in case
    *         the given predicate holds or not for at least one item
    */
  final def existsF(p: A => Boolean): Observable[Boolean] =
    findF(p).foldLeftF(false)((_, _) => true)

  /** Groups the items emitted by an Observable according to a specified
    * criterion, and emits these grouped items as GroupedObservables,
    * one GroupedObservable per group.
    *
    * Note: A [[monix.reactive.observables.GroupedObservable GroupedObservable]]
    * will cache the items it is to emit until such time as it is
    * subscribed to. For this reason, in order to avoid memory leaks,
    * you should not simply ignore those GroupedObservables that do
    * not concern you. Instead, you can signal to them that they may
    * discard their buffers by doing something like `source.take(0)`.
    *
    * @param keySelector  a function that extracts the key for each item
    */
  final def groupBy[K](keySelector: A => K)
    (implicit keysBuffer: Synchronous[Nothing] = OverflowStrategy.Unbounded): Observable[GroupedObservable[K, A]] =
    self.liftByOperator(new GroupByOperator[A, K](keysBuffer, keySelector))

  /** Alias for [[completed]]. Ignores all items emitted by
    * the source and only calls onCompleted or onError.
    *
    * @return an empty sequence that only calls onCompleted or onError,
    *         based on which one is called by the source Observable
    */
  final def ignoreElements: Observable[Nothing] =
    self.liftByOperator(CompletedOperator)

  /** Creates a new observable from this observable and another given
    * observable by interleaving their items into a strictly
    * alternating sequence.
    *
    * So the first item emitted by the new observable will be the item
    * emitted by `self`, the second item will be emitted by the other
    * observable, and so forth; when either `self` or `other` calls
    * `onCompletes`, the items will then be directly coming from the
    * observable that has not completed; when `onError` is called by
    * either `self` or `other`, the new observable will call `onError`
    * and halt.
    *
    * See [[merge]] for a more relaxed alternative that doesn't emit
    * items in strict alternating sequence.
    *
    * @param other is an observable that interleaves with the source
    * @return a new observable sequence that alternates emission of
    *         the items from both child streams
    */
  final def interleave[B >: A](other: Observable[B]): Observable[B] =
    new Interleave2Observable(self, other)

  /** Only emits the last element emitted by the source observable,
    * after which it's completed immediately.
    */
  final def lastF: Observable[A] = takeLast(1)

  /** Creates a new observable that only emits the last `n` elements
    * emitted by the source.
    *
    * In case the source triggers an error, then the underlying
    * buffer gets dropped and the error gets emitted immediately.
    */
  final def takeLast(n: Int): Observable[A] =
    if (n <= 0) Observable.empty else self.liftByOperator(new TakeLastOperator(n))

  /** Maps elements from the source using a function that can do
    * lazy or asynchronous processing by means of any `F[_]` data
    * type that implements `cats.effect.Effect` (e.g. `Task`, `IO`).
    *
    * Given a source observable, this function is basically the
    * equivalent of doing:
    *
    * {{{
    *   observable.mapTask(a => Task.fromEffect(f(a)))
    * }}}
    *
    * @see [[mapTask]] for a version specialized for
    *      [[monix.eval.Task Task]] and [[mapFuture]] for the version
    *      that can work with `scala.concurrent.Future`
    */
  final def mapEval[F[_], B](f: A => F[B])(implicit F: TaskLike[F]): Observable[B] =
    mapTask(a => Task.from(f(a))(F))

  /** Maps elements from the source using a function that can do
    * asynchronous processing by means of `scala.concurrent.Future`.
    *
    * Given a source observable, this function is basically the
    * equivalent of doing:
    *
    * {{{
    *   observable.concatMap(a => Observable.fromFuture(f(a)))
    * }}}
    *
    * However prefer this operator to `concatMap` because it
    * is more clear and has better performance.
    *
    * @see [[mapTask]] for the version that can work with
    *      [[monix.eval.Task Task]]
    */
  final def mapFuture[B](f: A => Future[B]): Observable[B] =
    mapTask { elemA =>
      try Task.fromFuture(f(elemA))
      catch {
        case ex if NonFatal(ex) => Task.raiseError(ex)
      }
    }

  /** Given a mapping function that maps events to [[monix.eval.Task tasks]],
    * applies it in parallel on the source, but with a specified
    * `parallelism`, which indicates the maximum number of tasks that
    * can be executed in parallel.
    *
    * Similar in spirit with
    * [[monix.reactive.Consumer.loadBalance[A,R](parallelism* Consumer.loadBalance]],
    * but expressed as an operator that executes [[monix.eval.Task Task]]
    * instances in parallel.
    *
    * Note that when the specified `parallelism` is 1, it has the same
    * behavior as [[mapTask]].
    *
    * @param parallelism is the maximum number of tasks that can be executed
    *        in parallel, over which the source starts being
    *        back-pressured
    *
    * @param f is the mapping function that produces tasks to execute
    *        in parallel, which will eventually produce events for the
    *        resulting observable stream
    *
    * @see [[mapTask]] for serial execution
    */
  final def mapParallelUnordered[B](parallelism: Int)(f: A => Task[B])
    (implicit os: OverflowStrategy[B] = OverflowStrategy.Default): Observable[B] =
    new MapParallelUnorderedObservable[A, B](self, parallelism, f, os)

  /** Converts the source Observable that emits `A` into an Observable
    * that emits `Notification[A]`.
    */
  final def materialize: Observable[Notification[A]] =
    self.liftByOperator(new MaterializeOperator[A])

  /** $mergeDescription
    *
    * @note $defaultOverflowStrategy
    * @return $mergeReturn
    */
  final def merge[B](implicit ev: A <:< Observable[B],
    os: OverflowStrategy[B] = OverflowStrategy.Default): Observable[B] =
    self.mergeMap(x => x)(os)

  /** $mergeMapDescription
    *
    * @param f - the transformation function
    * @return $mergeMapReturn
    */
  final def mergeMap[B](f: A => Observable[B])
    (implicit os: OverflowStrategy[B] = OverflowStrategy.Default): Observable[B] =
    new MergeMapObservable[A, B](self, f, os, delayErrors = false)

  /** $mergeDescription
    *
    * $delayErrorsDescription
    *
    * @note $defaultOverflowStrategy
    * @return $mergeReturn
    */
  final def mergeDelayErrors[B](implicit ev: A <:< Observable[B],
    os: OverflowStrategy[B] = OverflowStrategy.Default): Observable[B] =
    self.mergeMap(x => x)(os)

  /** $mergeMapDescription
    *
    * $delayErrorsDescription
    *
    * @param f - the transformation function
    * @return $mergeMapReturn
    */
  final def mergeMapDelayErrors[B](f: A => Observable[B])
    (implicit os: OverflowStrategy[B] = OverflowStrategy.Default): Observable[B] =
    new MergeMapObservable[A, B](self, f, os, delayErrors = true)

  /** Overrides the default [[monix.execution.Scheduler Scheduler]],
    * possibly forcing an asynchronous boundary on subscription
    * (if `forceAsync` is set to `true`, the default).
    *
    * When an `Observable` is subscribed with
    * [[Observable.subscribe(subscriber* subscribe]],
    * it needs a `Scheduler`, which is going to be injected in the
    * processing pipeline, to be used for managing asynchronous
    * boundaries, scheduling execution with delay, etc.
    *
    * Normally the [[monix.execution.Scheduler Scheduler]] gets injected
    * implicitly when doing `subscribe`, but this operator overrides
    * the injected subscriber for the given source. And if the source is
    * normally using that injected scheduler (given by `subscribe`),
    * then the effect will be that all processing will now happen
    * on the override.
    *
    * To put it in other words, in Monix it's usually the consumer and
    * not the producer that specifies the scheduler and this operator
    * allows for a different behavior.
    *
    * This operator also subsumes the effects of [[subscribeOn]],
    * meaning that the subscription logic itself will start on
    * the provided scheduler if `forceAsync = true` (the default).
    *
    * @see [[observeOn(s:monix\.execution\.Scheduler)* observeOn]]
    *      and [[subscribeOn]].
    *
    * @param s is the [[monix.execution.Scheduler Scheduler]] to use
    *        for overriding the default scheduler and for forcing
    *        an asynchronous boundary if `forceAsync` is `true`
    *
    * @param forceAsync indicates whether an asynchronous boundary
    *        should be forced right before the subscription of the
    *        source `Observable`, managed by the provided `Scheduler`
    *
    * @return a new `Observable` that mirrors the source on subscription,
    *         but that uses the provided scheduler for overriding
    *         the default and possibly force an extra asynchronous
    *         boundary on execution
    */
  final def executeOn(s: Scheduler, forceAsync: Boolean = true): Observable[A] =
    new ExecuteOnObservable[A](self, s, forceAsync)

  /** Mirrors the source observable, but upon subscription ensure
    * that the evaluation forks into a separate (logical) thread.
    *
    * The execution is managed by the injected
    * [[monix.execution.Scheduler scheduler]] in `subscribe()`.
    */
  final def executeAsync: Observable[A] =
    new ExecuteAsyncObservable(self)

  /** Returns a new observable that will execute the source with a different
    * [[monix.execution.ExecutionModel ExecutionModel]].
    *
    * This allows fine-tuning the options injected by the scheduler
    * locally. Example:
    *
    * {{{
    *   observable.executeWithModel(AlwaysAsyncExecution)
    * }}}
    *
    * @param em is the
    *        [[monix.execution.ExecutionModel ExecutionModel]]
    *        that will be used when evaluating the source.
    */
  final def executeWithModel(em: ExecutionModel): Observable[A] =
    new ExecuteWithModelObservable[A](self, em)

  /** Operator that specifies a different
    * [[monix.execution.Scheduler Scheduler]], on which subscribers
    * will observe events, instead of the default one.
    *
    * An `Observable` with an applied `observeOn` call will forward
    * events into a buffer that uses the specified `Scheduler`
    * reference to cycle through events and to make `onNext` calls to
    * downstream listeners.
    *
    * Example:
    * {{{
    *   import monix.execution.Scheduler
    *   val io = Scheduler.io("my-io")
    *
    *   source.map(_ + 1)
    *     .observeOn(io)
    *     .foreach(x => println(x))
    * }}}
    *
    * In the above example the first `map` (whatever comes before the
    * `observeOn` call) gets executed using the default `Scheduler`
    * (might execute on the current thread even), however the
    * `foreach` that's specified after `observeOn` will get executed
    * on the indicated `Scheduler`.
    *
    * NOTE: this operator does not guarantee that downstream listeners
    * will actually use the specified `Scheduler` to process events,
    * because this depends on the rest of the pipeline. E.g. this will
    * not work OK:
    *
    * {{{
    *   source.observeOn(io).asyncBoundary(Unbounded)
    * }}}
    *
    * This sample might not do what a user of `observeOn` would
    * want. Indeed the implementation will use the provided `io`
    * reference for calling `onNext` / `onComplete` / `onError`
    * events, however because of the following asynchronous boundary
    * created the actual listeners will probably end up being execute
    * on a different `Scheduler`.
    *
    * The underlying implementation uses
    * [[monix.reactive.observers.BufferedSubscriber a buffer]]
    * to forward events. The
    * [[monix.reactive.OverflowStrategy OverflowStrategy]]
    * being applied is the
    * [[monix.reactive.OverflowStrategy.Default default one]].
    *
    * @see [[observeOn[B>:A](s:monix\.execution\.Scheduler,os:monix\.reactive\.OverflowStrategy[B]* observeOn(Scheduler, OverflowStrategy)]]
    *      for the version that allows customizing the
    *      [[monix.reactive.OverflowStrategy OverflowStrategy]]
    *      being used by the underlying buffer.
    *
    * @param s is the alternative `Scheduler` reference to use
    *        for observing events
    */
  final def observeOn(s: Scheduler): Observable[A] =
    observeOn(s, OverflowStrategy.Default)

  /** Operator that specifies a different
    * [[monix.execution.Scheduler Scheduler]], on which subscribers
    * will observe events, instead of the default one.
    *
    * This overloaded version of `observeOn` takes an extra
    * [[monix.reactive.OverflowStrategy OverflowStrategy]]
    * parameter specifying the behavior of the underlying buffer.
    *
    * @see [[observeOn(s:monix\.execution\.Scheduler)* observeOn(Scheduler)]] for
    *      the version that does not take an `OverflowStrategy` parameter.
    *
    * @param s is the alternative `Scheduler` reference to use
    *        for observing events
    * @param os is the [[monix.reactive.OverflowStrategy OverflowStrategy]]
    *        to apply to the underlying buffer
    */
  final def observeOn[B >: A](s: Scheduler, os: OverflowStrategy[B]): Observable[B] =
    new ObserveOnObservable[B](self, s, os)

  /** If the connection is [[monix.execution.Cancelable.cancel cancelled]]
    * then trigger a `CancellationException`.
    *
    * A connection can be cancelled with the help of the
    * [[monix.execution.Cancelable Cancelable]]
    * returned on [[Observable.subscribe(subscriber* subscribe]].
    *
    * Because the cancellation is effectively concurrent with the
    * signals the [[monix.reactive.Observer Observer]] receives and because
    * we need to uphold the contract, this operator will effectively
    * synchronize access to [[monix.reactive.Observer.onNext onNext]],
    * [[monix.reactive.Observer.onComplete onComplete]] and
    * [[monix.reactive.Observer.onError onError]]. It will also watch
    * out for asynchronous [[monix.execution.Ack.Stop Stop]] events.
    *
    * In other words, this operator does heavy synchronization, can
    * prove to be inefficient and you should avoid using it because
    * the signaled error can interfere with functionality from other
    * operators that use cancellation internally and cancellation in
    * general is a side-effecting operation that should be avoided,
    * unless it's necessary.
    */
  final def onCancelTriggerError: Observable[A] =
    new OnCancelTriggerErrorObservable[A](self)

  /** Returns an Observable that mirrors the behavior of the source,
    * unless the source is terminated with an `onError`, in which case
    * the streaming of events continues with the specified backup
    * sequence.
    *
    * The created Observable mirrors the behavior of the source in
    * case the source does not end with an error.
    *
    * NOTE that compared with `onErrorResumeNext` from Rx.NET, the
    * streaming is not resumed in case the source is terminated
    * normally with an `onComplete`.
    *
    * @param that is a backup sequence that's being subscribed
    *        in case the source terminates with an error.
    */
  final def onErrorFallbackTo[B >: A](that: Observable[B]): Observable[B] =
    self.onErrorHandleWith(_ => that)

  /** Returns an observable that mirrors the behavior of the source,
    * unless the source is terminated with an `onError`, in which
    * case the streaming of events fallbacks to an observable
    * emitting a single element generated by the backup function.
    *
    * See [[onErrorRecover]] for the version that takes a
    * partial function as a parameter.
    *
    * @param f - a function that matches errors with a
    *        backup element that is emitted when the source
    *        throws an error.
    */
  final def onErrorHandle[B >: A](f: Throwable => B): Observable[B] =
    onErrorHandleWith { elem => Observable.now(f(elem)) }

  /** Returns an observable that mirrors the behavior of the source,
    * unless the source is terminated with an `onError`, in which
    * case the streaming of events fallbacks to an observable
    * emitting a single element generated by the backup function.
    *
    * The created Observable mirrors the behavior of the source
    * in case the source does not end with an error or if the
    * thrown `Throwable` is not matched.
    *
    * See [[onErrorHandle]] for the version that takes a
    * total function as a parameter.
    *
    * @param pf - a function that matches errors with a
    *        backup element that is emitted when the source
    *        throws an error.
    */
  final def onErrorRecover[B >: A](pf: PartialFunction[Throwable, B]): Observable[B] =
    onErrorHandleWith(ex => (pf andThen Observable.now).applyOrElse(ex, Observable.raiseError))

  /** Returns an Observable that mirrors the behavior of the source,
    * unless the source is terminated with an `onError`, in which case
    * the streaming of events continues with the specified backup
    * sequence generated by the given function.
    *
    * The created Observable mirrors the behavior of the source in
    * case the source does not end with an error or if the thrown
    * `Throwable` is not matched.
    *
    * See [[onErrorHandleWith]] for the version that takes a
    * total function as a parameter.
    *
    * @param pf is a function that matches errors with a
    *        backup throwable that is subscribed when the source
    *        throws an error.
    */
  final def onErrorRecoverWith[B >: A](pf: PartialFunction[Throwable, Observable[B]]): Observable[B] =
    onErrorHandleWith(ex => pf.applyOrElse(ex, Observable.raiseError))

  /** Returns an Observable that mirrors the behavior of the source,
    * unless the source is terminated with an `onError`, in which case
    * the streaming of events continues with the specified backup
    * sequence generated by the given function.
    *
    * See [[onErrorRecoverWith]] for the version that takes a
    * partial function as a parameter.
    *
    * @param f is a function that matches errors with a
    *        backup throwable that is subscribed when the source
    *        throws an error.
    */
  final def onErrorHandleWith[B >: A](f: Throwable => Observable[B]): Observable[B] =
    new OnErrorRecoverWithObservable(self, f)

  /** Returns an Observable that mirrors the behavior of the source,
    * unless the source is terminated with an `onError`, in which case
    * it tries subscribing to the source again in the hope that it
    * will complete without an error.
    *
    * The number of retries is limited by the specified `maxRetries`
    * parameter, so for an Observable that always ends in error the
    * total number of subscriptions that will eventually happen is
    * `maxRetries + 1`.
    */
  final def onErrorRestart(maxRetries: Long): Observable[A] = {
    require(maxRetries >= 0, "maxRetries should be positive")
    new OnErrorRetryCountedObservable(self, maxRetries)
  }

  /** Returns an Observable that mirrors the behavior of the source,
    * unless the source is terminated with an `onError`, in which case
    * it tries subscribing to the source again in the hope that it
    * will complete without an error.
    *
    * The given predicate establishes if the subscription should be
    * retried or not.
    */
  final def onErrorRestartIf(p: Throwable => Boolean): Observable[A] =
    new OnErrorRetryIfObservable[A](self, p)

  /** Returns an Observable that mirrors the behavior of the source,
    * unless the source is terminated with an `onError`, in which case
    * it tries subscribing to the source again in the hope that it
    * will complete without an error.
    *
    * NOTE: The number of retries is unlimited, so something like
    * `Observable.error(new RuntimeException).onErrorRestartUnlimited`
    * will loop forever.
    */
  final def onErrorRestartUnlimited: Observable[A] =
    new OnErrorRetryCountedObservable(self, -1)

  /** Given a [[monix.reactive.Pipe Pipe]], transform
    * the source observable with it.
    */
  final def pipeThrough[I >: A, B](pipe: Pipe[I, B]): Observable[B] =
    new PipeThroughObservable(self, pipe)

  /** Returns an observable that emits the results of invoking a
    * specified selector on items emitted by a
    * [[monix.reactive.observables.ConnectableObservable ConnectableObservable]],
    * which shares a single subscription to the underlying sequence.
    *
    * @param f is a selector function that can use the multicasted source sequence
    *        as many times as needed, without causing multiple subscriptions
    *        to the source sequence. Observers to the given source will
    *        receive all notifications of the source from the time of the
    *        subscription forward.
    */
  final def publishSelector[R](f: Observable[A] => Observable[R]): Observable[R] =
    pipeThroughSelector(Pipe.publish[A], f)

  /** Returns an observable that emits the results of invoking a
    * specified selector on items emitted by a
    * [[monix.reactive.observables.ConnectableObservable ConnectableObservable]],
    * which shares a single subscription to the underlying sequence.
    *
    * @param pipe is the [[Pipe]] used to transform the source into a multicast
    *        (hot) observable that can be shared in the selector function
    *
    * @param f is a selector function that can use the multicasted source sequence
    *        as many times as needed, without causing multiple subscriptions
    *        to the source sequence. Observers to the given source will
    *        receive all notifications of the source from the time of the
    *        subscription forward.
    */
  final def pipeThroughSelector[S >: A, B, R](pipe: Pipe[S, B], f: Observable[B] => Observable[R]): Observable[R] =
    new PipeThroughSelectorObservable[S, B, R](self, pipe, f)

  /** Applies a binary operator to a start value and all elements of
    * this Observable, going left to right and returns a new
    * Observable that emits only one item before `onComplete`.
    */
  final def reduce[B >: A](op: (B, B) => B): Observable[B] =
    self.liftByOperator(new ReduceOperator[B](op))

  /** Repeats the items emitted by the source continuously. It
    * caches the generated items until `onComplete` and repeats them
    * forever.
    *
    * It terminates either on error or if the source is empty.
    */
  final def repeat: Observable[A] =
    new RepeatSourceObservable[A](self)

  /** Keeps restarting / resubscribing the source until the predicate
    * returns `true` for the the first emitted element, after which
    * it starts mirroring the source.
    */
  final def restartUntil(p: A => Boolean): Observable[A] =
    new RestartUntilObservable[A](self, p)

  /** Emit the most recent items emitted by an observable within
    * periodic time intervals. If no new value has been emitted since
    * the last time it was sampled, it signals the last emitted value
    * anyway.
    *
    * @see [[sample]] for a variant that doesn't repeat the last value on silence
    * @see [[sampleRepeatedBy]] for fine control
    * @param period the timespan at which sampling occurs
    */
  final def sampleRepeated(period: FiniteDuration): Observable[A] =
    self.sampleRepeatedBy(Observable.intervalAtFixedRate(period, period))

  /** Returns an observable that, when the specified sampler observable
    * emits an item or completes, emits the most recently emitted item
    * (if any) emitted by the source Observable since the previous
    * emission from the sampler observable. If no new value has been
    * emitted since the last time it was sampled, it signals the last
    * emitted value anyway.
    *
    * @see [[sampleBy]] for a variant that doesn't repeat the last value on silence
    * @see [[sampleRepeated]] for a periodic sampling
    * @param sampler - the Observable to use for sampling the source Observable
    */
  final def sampleRepeatedBy[B](sampler: Observable[B]): Observable[A] =
    new ThrottleLastObservable[A, B](self, sampler, shouldRepeatOnSilence = true)

  /** Applies a binary operator to a start value and all elements of
    * this Observable, going left to right and returns a new
    * Observable that emits on each step the result of the applied
    * function.
    *
    * Similar to [[foldLeftF]], but emits the state on each
    * step. Useful for modeling finite state machines.
    *
    * @see [[scan0]] for the version that emits seed element at the beginning
    */
  final def scan[S](seed: => S)(op: (S, A) => S): Observable[S] =
    new ScanObservable[A, S](self, seed _, op)

  /** Applies a binary operator to a start value and all elements of
    * this Observable, going left to right and returns a new
    * Observable that emits on each step the result of the applied
    * function.
    *
    * This is a version of [[scan]] that emits seed element at the beginning,
    * similar to `scanLeft` on Scala collections
    */
  final def scan0[S](seed: => S)(op: (S, A) => S): Observable[S] =
    Observable.eval(seed).flatMap(s => s +: scan(s)(op))

  /** Applies a binary operator to a start value and all elements of
    * this stream, going left to right and returns a new stream that
    * emits on each step the result of the applied function.
    *
    * Similar with [[scan]], but this can suspend and evaluate
    * side effects with an `F[_]` data type that implements the
    * `cats.effect.Effect` type class, thus allowing for lazy or
    * asynchronous data processing.
    *
    * Similar to [[foldLeftF]] and [[foldWhileLeftF]], but emits the
    * state on each step. Useful for modeling finite state machines.
    *
    * Example showing how state can be evolved and acted upon:
    *
    * {{{
    *   // Using cats.effect.IO for evaluating our side effects
    *   import cats.effect.IO
    *
    *   sealed trait State[+A] { def count: Int }
    *   case object Init extends State[Nothing] { def count = 0 }
    *   case class Current[A](current: Option[A], count: Int)
    *     extends State[A]
    *
    *   case class Person(id: Int, name: String)
    *
    *   // Initial state
    *   val seed = IO.pure(Init : State[Person])
    *
    *   val scanned = source.scanEval(seed) { (state, id) =>
    *     requestPersonDetails(id).map { person =>
    *       state match {
    *         case Init =>
    *           Current(person, 1)
    *         case Current(_, count) =>
    *           Current(person, count + 1)
    *       }
    *     }
    *   }
    *
    *   scanned
    *     .takeWhile(_.count < 10)
    *     .collect { case Current(a, _) => a }
    * }}}
    *
    * @see [[scanEval0]] for the version that emits seed element at the beginning
    *
    * @see [[scan]] for the synchronous, non-lazy version, or
    *      [[scanTask]] for the [[monix.eval.Task Task]]-specialized
    *      version.
    *
    * @param seed is the initial state
    * @param op is the function that evolves the current state
    *
    * @param F is the `cats.effect.Effect` type class implementation
    *        for type `F`, which controls the evaluation. `F` can be
    *        a data type such as [[monix.eval.Task]] or `cats.effect.IO`,
    *        which implement `Effect`.
    *
    * @return a new observable that emits all intermediate states being
    *         resulted from applying the given function
    */
  final def scanEval[F[_], S](seed: F[S])(op: (S, A) => F[S])
    (implicit F: TaskLike[F]): Observable[S] =
    scanTask(Task.from(seed)(F))((s, a) => Task.from(op(s, a))(F))

/** Applies a binary operator to a start value and all elements of
  * this stream, going left to right and returns a new stream that
  * emits on each step the result of the applied function.
  *
  * This is a version of [[scanEval]] that emits seed element at the beginning,
  * similar to `scanLeft` on Scala collections
  */
  final def scanEval0[F[_], S](seed: F[S])(op: (S, A) => F[S])
    (implicit F: TaskLike[F], A: Applicative[F]): Observable[S] =
    Observable.fromTaskLike(seed).flatMap(s => s +: scanEval(A.pure(s))(op))

  /** Given a mapping function that returns a `B` type for which we have
    * a [[cats.Monoid]] instance, returns a new stream that folds the incoming
    * elements of the sources using the provided `Monoid[B].combine`, with the
    * initial seed being the `Monoid[B].empty` value, emitting the generated values
    * at each step.
    *
    * Equivalent with [[scan]] applied with the given [[cats.Monoid]], so given
    * our `f` mapping function returns a `B`, this law holds:
    * {{{
    * val B = implicitly[Monoid[B]]
    *
    * stream.scanMap(f) <-> stream.scan(B.empty)(B.combine)
    * }}}
    *
    * Example:
    * {{{
    * // Yields 2, 6, 12, 20, 30, 42
    * Observable(1, 2, 3, 4, 5, 6).scanMap(x => x * 2)
    * }}}
    *
    * @param f is the mapping function applied to every incoming element of this `Observable`
    *          before folding using `Monoid[B].combine`
    *
    * @return a new `Observable` that emits all intermediate states being
    *         resulted from applying `Monoid[B].combine` function
    */
  final def scanMap[B](f: A => B)(implicit B: Monoid[B]): Observable[B] =
    self.scan(B.empty)((acc, a) => B.combine(acc, f(a)))

  /** Given a mapping function that returns a `B` type for which we have
    * a [[cats.Monoid]] instance, returns a new stream that folds the incoming
    * elements of the sources using the provided `Monoid[B].combine`, with the
    * initial seed being the `Monoid[B].empty` value, emitting the generated values
    * at each step.
    *
    * This is a version of [[scanMap]] that emits seed element at the beginning.
    */
  final def scanMap0[B](f: A => B)(implicit B: Monoid[B]): Observable[B] =
    B.empty +: scanMap(f)

  /** Applies a binary operator to a start value and all elements of
    * this stream, going left to right and returns a new stream that
    * emits on each step the result of the applied function.
    *
    * Similar with [[scan]], but this can suspend and evaluate
    * side effects with [[monix.eval.Task Task]], thus allowing for
    * asynchronous data processing.
    *
    * Similar to [[foldLeftF]] and [[foldWhileLeftF]], but emits the
    * state on each step. Useful for modeling finite state machines.
    *
    * Example showing how state can be evolved and acted upon:
    *
    * {{{
    *   sealed trait State[+A] { def count: Int }
    *   case object Init extends State[Nothing] { def count = 0 }
    *   case class Current[A](current: Option[A], count: Int)
    *     extends State[A]
    *
    *   case class Person(id: Int, name: String)
    *
    *   // Initial state
    *   val seed = Task.now(Init : State[Person])
    *
    *   val scanned = source.scanTask(seed) { (state, id) =>
    *     requestPersonDetails(id).map { person =>
    *       state match {
    *         case Init =>
    *           Current(person, 1)
    *         case Current(_, count) =>
    *           Current(person, count + 1)
    *       }
    *     }
    *   }
    *
    *   scanned
    *     .takeWhile(_.count < 10)
    *     .collect { case Current(a, _) => a }
    * }}}
    *
    * @see [[scanTask0]] for the version that emits seed element at the beginning
    * @see [[scan]] for the version that does not require using `Task`
    *      in the provided operator
    *
    * @param seed is the initial state
    * @param op is the function that evolves the current state
    *
    * @return a new observable that emits all intermediate states being
    *         resulted from applying the given function
    */
  final def scanTask[S](seed: Task[S])(op: (S, A) => Task[S]): Observable[S] =
    new ScanTaskObservable(self, seed, op)

  /** Applies a binary operator to a start value and all elements of
    * this stream, going left to right and returns a new stream that
    * emits on each step the result of the applied function.
    *
    * This is a version of [[scanTask]] that emits seed element at the beginning.
    */
  final def scanTask0[S](seed: Task[S])(op: (S, A) => Task[S]): Observable[S] =
    Observable.fromTask(seed).flatMap(s => s +: scanTask(Task.pure(s))(op))

  /** Creates a new Observable that emits the given elements and then
    * it also emits the events of the source (prepend operation).
    */
  final def startWith[B >: A](elems: Seq[B]): Observable[B] =
    Observable.fromIterable(elems) ++ self

  /** Returns a new Observable that uses the specified `Scheduler` for
    * initiating the subscription.
    */
  final def subscribeOn(scheduler: Scheduler): Observable[A] =
    new SubscribeOnObservable[A](self, scheduler)

  /** In case the source is empty, switch to the given backup. */
  final def switchIfEmpty[B >: A](backup: Observable[B]): Observable[B] =
    new SwitchIfEmptyObservable[B](self, backup)

  /** Drops the first element of the source observable,
    * emitting the rest.
    */
  final def tail: Observable[A] = drop(1)

  /** Drops the first `n` elements (from the start).
    *
    * @param n the number of elements to drop
    * @return a new Observable that drops the first ''n'' elements
    *         emitted by the source
    */
  final def drop(n: Int): Observable[A] =
    self.liftByOperator(new DropFirstOperator(n))

  /** Creates a new Observable that emits the events of the source, only
    * for the specified `timestamp`, after which it completes.
    *
    * @param timespan the window of time during which the new Observable
    *        is allowed to emit the events of the source
    */
  final def takeByTimespan(timespan: FiniteDuration): Observable[A] =
    new TakeLeftByTimespanObservable(self, timespan)

  /** Creates a new Observable that emits every n-th event from the source,
    * dropping intermediary events.
    */
  final def takeEveryNth(n: Int): Observable[A] =
    self.liftByOperator(new TakeEveryNthOperator(n))

  /** Creates a new observable that mirrors the source until
    * the given `trigger` emits either an element or `onComplete`,
    * after which it is completed.
    *
    * The resulting observable is completed as soon as `trigger`
    * emits either an `onNext` or `onComplete`. If `trigger`
    * emits an `onError`, then the resulting observable is also
    * completed with error.
    *
    * @param trigger is an observable that will cancel the
    *        streaming as soon as it emits an event
    */
  final def takeUntil(trigger: Observable[Any]): Observable[A] =
    new TakeUntilObservable[A](self, trigger)

  /** Takes longest prefix of elements that satisfy the given predicate
    * and returns a new Observable that emits those elements.
    */
  final def takeWhile(p: A => Boolean): Observable[A] =
    self.liftByOperator(new TakeByPredicateOperator(p))

  /** Takes longest prefix of elements while given [[monix.execution.cancelables.BooleanCancelable BooleanCancelable]]
    * is not canceled and returns a new Observable that emits those elements.
    */
  final def takeWhileNotCanceled(c: BooleanCancelable): Observable[A] =
    self.liftByOperator(new TakeWhileNotCanceledOperator(c))

  /** Returns an Observable that emits only the first item emitted by
    * the source Observable during sequential time windows of a
    * specified duration.
    *
    * This differs from [[Observable!.throttleLast]] in that this only
    * tracks passage of time whereas `throttleLast` ticks at scheduled
    * intervals.
    *
    * @param interval time to wait before emitting another item after
    *        emitting the last item
    */
  final def throttleFirst(interval: FiniteDuration): Observable[A] =
    self.liftByOperator(new ThrottleFirstOperator[A](interval))

  /** Emit the most recent items emitted by the source within
    * periodic time intervals.
    *
    * Alias for [[sample]].
    *
    * @param period duration of windows within which the last item
    *        emitted by the source Observable will be emitted
    */
  final def throttleLast(period: FiniteDuration): Observable[A] =
    sample(period)

  /** Emit the most recent items emitted by the source within
    * periodic time intervals.
    *
    * Use the `sample` operator to periodically look at an observable
    * to see what item it has most recently emitted since the previous
    * sampling. Note that if the source observable has emitted no
    * items since the last time it was sampled, the observable that
    * results from the `sample` operator will emit no item for that
    * sampling period.
    *
    * @see [[sampleBy]] for fine control
    * @see [[sampleRepeated]] for repeating the last value on silence
    * @param period the timespan at which sampling occurs
    */
  final def sample(period: FiniteDuration): Observable[A] =
    self.sampleBy(Observable.intervalAtFixedRate(period, period))

  /** Returns an observable that, when the specified sampler
    * emits an item or completes, emits the most recently emitted item
    * (if any) emitted by the source since the previous
    * emission from the sampler.
    *
    * Use the `sampleBy` operator to periodically look at an observable
    * to see what item it has most recently emitted since the previous
    * sampling. Note that if the source observable has emitted no
    * items since the last time it was sampled, the observable that
    * results from the `sampleBy` operator will emit no item.
    *
    * @see [[sample]] for periodic sampling
    * @see [[sampleRepeatedBy]] for repeating the last value on silence
    * @param sampler - the observable to use for sampling the source
    */
  final def sampleBy[B](sampler: Observable[B]): Observable[A] =
    new ThrottleLastObservable[A, B](self, sampler, shouldRepeatOnSilence = false)

  /** Only emit an item from an observable if a particular timespan has
    * passed without it emitting another item.
    *
    * Note: If the source observable keeps emitting items more
    * frequently than the length of the time window, then no items will
    * be emitted by the resulting observable.
    *
    * Alias for [[debounce]].
    *
    * @param timeout the length of the window of time that must pass after
    *        the emission of an item from the source observable in
    *        which that observable emits no items in order for the
    *        item to be emitted by the resulting observable
    * @see [[echoOnce]] for a similar operator that also mirrors
    *     the source observable
    */
  final def throttleWithTimeout(timeout: FiniteDuration): Observable[A] =
    debounce(timeout)

  /** Only emit an item from an observable if a particular timespan has
    * passed without it emitting another item.
    *
    * Note: If the source observable keeps emitting items more
    * frequently than the length of the time window, then no items will
    * be emitted by the resulting observable.
    *
    * @param timeout the length of the window of time that must pass after
    *        the emission of an item from the source observable in
    *        which that observable emits no items in order for the
    *        item to be emitted by the resulting observable
    * @see [[echoOnce]] for a similar operator that also mirrors
    *     the source observable
    */
  final def debounce(timeout: FiniteDuration): Observable[A] =
    new DebounceObservable(self, timeout, repeat = false)

  /** Returns an observable that mirrors the source but that will trigger a
    * [[monix.execution.exceptions.DownstreamTimeoutException DownstreamTimeoutException]]
    * in case the downstream subscriber takes more than the given timespan
    * to process an `onNext` message.
    *
    * Note that this ignores the time it takes for the upstream to send
    * `onNext` messages. For detecting slow producers see [[timeoutOnSlowUpstream]].
    *
    * @param timeout maximum duration for `onNext`.
    */
  final def timeoutOnSlowDownstream(timeout: FiniteDuration): Observable[A] =
    new DownstreamTimeoutObservable[A](self, timeout)

  /** Returns an observable that mirrors the source but applies a timeout
    * for each emitted item by the upstream. If the next item isn't
    * emitted within the specified timeout duration starting from its
    * predecessor, the source is terminated and the downstream gets
    * subscribed to the given backup.
    *
    * Note that this ignores the time it takes to process `onNext`.
    * If dealing with a slow consumer, see [[timeoutOnSlowDownstream]].
    *
    * @param timeout maximum duration between emitted items before
    *        a timeout occurs (ignoring the time it takes to process `onNext`)
    * @param backup is the alternative data source to subscribe to on timeout
    */
  final def timeoutOnSlowUpstreamTo[B >: A](timeout: FiniteDuration, backup: Observable[B]): Observable[B] =
    self.timeoutOnSlowUpstream(timeout).onErrorHandleWith {
      case UpstreamTimeoutException(`timeout`) => backup
      case other => Observable.raiseError(other)
    }

  /** Returns an observable that mirrors the source but applies a timeout
    * for each emitted item by the upstream. If the next item isn't
    * emitted within the specified timeout duration starting from its
    * predecessor, the resulting Observable terminates and notifies
    * observers of a TimeoutException.
    *
    * Note that this ignores the time it takes to process `onNext`.
    * If dealing with a slow consumer, see [[timeoutOnSlowDownstream]].
    *
    * @param timeout maximum duration between emitted items before
    *        a timeout occurs (ignoring the time it takes to process `onNext`)
    */
  final def timeoutOnSlowUpstream(timeout: FiniteDuration): Observable[A] =
    new UpstreamTimeoutObservable[A](self, timeout)

  /** While the destination observer is busy, buffers events, applying
    * the given overflowStrategy.
    *
    * @param overflowStrategy - $overflowStrategyParam
    */
  final def whileBusyBuffer[B >: A](overflowStrategy: OverflowStrategy.Synchronous[B]): Observable[B] =
    asyncBoundary(overflowStrategy)

  /** $asyncBoundaryDescription
    *
    * @param overflowStrategy - $overflowStrategyParam
    */
  final def asyncBoundary[B >: A](overflowStrategy: OverflowStrategy[B]): Observable[B] =
    liftByOperator(new AsyncBoundaryOperator[B](overflowStrategy))

  /** While the destination observer is busy, drop the incoming events.
    */
  final def whileBusyDropEvents: Observable[A] =
    self.liftByOperator(new WhileBusyDropEventsOperator[A])

  /** While the destination observer is busy, drop the incoming events.
    * When the downstream recovers, we can signal a special event
    * meant to inform the downstream observer how many events where
    * dropped.
    *
    * @param onOverflow - $onOverflowParam
    */
  final def whileBusyDropEventsAndSignal[B >: A](onOverflow: Long => B): Observable[B] =
    self.liftByOperator(new WhileBusyDropEventsAndSignalOperator[B](onOverflow))

  /** Combines the elements emitted by the source with the latest elements
    * emitted by two observables.
    *
    * Similar with `combineLatest`, but only emits items when the single source
    * emits an item (not when any of the Observables that are passed to the operator
    * do, as combineLatest does).
    *
    * @param o1 is the first observable that gets paired with the source
    * @param o2 is the second observable that gets paired with the source
    * @param f is a mapping function over the generated pairs
    */
  final def withLatestFrom2[B1, B2, R](o1: Observable[B1], o2: Observable[B2])(f: (A, B1, B2) => R): Observable[R] =
    self.withLatestFrom(Observable.combineLatest2(o1, o2)) { (a, tuple) =>
      f(a, tuple._1, tuple._2)
    }

  /** Combines the elements emitted by the source with the latest elements
    * emitted by three observables.
    *
    * Similar with `combineLatest`, but only emits items when the single source
    * emits an item (not when any of the Observables that are passed to the operator
    * do, as combineLatest does).
    *
    * @param o1 is the first observable that gets paired with the source
    * @param o2 is the second observable that gets paired with the source
    * @param o3 is the third observable that gets paired with the source
    * @param f is a mapping function over the generated pairs
    */
  final def withLatestFrom3[B1, B2, B3, R](o1: Observable[B1], o2: Observable[B2], o3: Observable[B3])
    (f: (A, B1, B2, B3) => R): Observable[R] = {

    self.withLatestFrom(Observable.combineLatest3(o1, o2, o3)) { (a, o) =>
      f(a, o._1, o._2, o._3)
    }
  }

  /** Combines the elements emitted by the source with the latest element
    * emitted by another observable.
    *
    * Similar with `combineLatest`, but only emits items when the single source
    * emits an item (not when any of the Observables that are passed to the operator
    * do, as combineLatest does).
    *
    * @param other is an observable that gets paired with the source
    * @param f is a mapping function over the generated pairs
    */
  final def withLatestFrom[B, R](other: Observable[B])(f: (A, B) => R): Observable[R] =
    new WithLatestFromObservable[A, B, R](self, other, f)

  /** Combines the elements emitted by the source with the latest elements
    * emitted by four observables.
    *
    * Similar with `combineLatest`, but only emits items when the single source
    * emits an item (not when any of the Observables that are passed to the operator
    * do, as combineLatest does).
    *
    * @param o1 is the first observable that gets paired with the source
    * @param o2 is the second observable that gets paired with the source
    * @param o3 is the third observable that gets paired with the source
    * @param o4 is the fourth observable that gets paired with the source
    * @param f is a mapping function over the generated pairs
    */
  final def withLatestFrom4[B1, B2, B3, B4, R](
    o1: Observable[B1], o2: Observable[B2], o3: Observable[B3], o4: Observable[B4])
    (f: (A, B1, B2, B3, B4) => R): Observable[R] = {

    self.withLatestFrom(Observable.combineLatest4(o1, o2, o3, o4)) { (a, o) =>
      f(a, o._1, o._2, o._3, o._4)
    }
  }

  /** Combines the elements emitted by the source with the latest elements
    * emitted by five observables.
    *
    * Similar with `combineLatest`, but only emits items when the single source
    * emits an item (not when any of the Observables that are passed to the operator
    * do, as combineLatest does).
    *
    * @param o1 is the first observable that gets paired with the source
    * @param o2 is the second observable that gets paired with the source
    * @param o3 is the third observable that gets paired with the source
    * @param o4 is the fourth observable that gets paired with the source
    * @param o5 is the fifth observable that gets paired with the source
    * @param f is a mapping function over the generated pairs
    */
  final def withLatestFrom5[B1, B2, B3, B4, B5, R](
    o1: Observable[B1], o2: Observable[B2], o3: Observable[B3],
    o4: Observable[B4], o5: Observable[B5])
    (f: (A, B1, B2, B3, B4, B5) => R): Observable[R] = {

    self.withLatestFrom(Observable.combineLatest5(o1, o2, o3, o4, o5)) { (a, o) =>
      f(a, o._1, o._2, o._3, o._4, o._5)
    }
  }

  /** Combines the elements emitted by the source with the latest elements
    * emitted by six observables.
    *
    * Similar with `combineLatest`, but only emits items when the single source
    * emits an item (not when any of the Observables that are passed to the operator
    * do, as combineLatest does).
    *
    * @param o1 is the first observable that gets paired with the source
    * @param o2 is the second observable that gets paired with the source
    * @param o3 is the third observable that gets paired with the source
    * @param o4 is the fourth observable that gets paired with the source
    * @param o5 is the fifth observable that gets paired with the source
    * @param o6 is the sixth observable that gets paired with the source
    * @param f is a mapping function over the generated pairs
    */
  final def withLatestFrom6[B1, B2, B3, B4, B5, B6, R](
    o1: Observable[B1], o2: Observable[B2], o3: Observable[B3],
    o4: Observable[B4], o5: Observable[B5], o6: Observable[B6])
    (f: (A, B1, B2, B3, B4, B5, B6) => R): Observable[R] = {

    self.withLatestFrom(Observable.combineLatest6(o1, o2, o3, o4, o5, o6)) { (a, o) =>
      f(a, o._1, o._2, o._3, o._4, o._5, o._6)
    }
  }

  /** Creates a new observable from this observable and another given
    * observable by combining their items in pairs in a strict sequence.
    *
    * So the first item emitted by the new observable will be the tuple of the
    * first items emitted by each of the source observables; the second item
    * emitted by the new observable will be a tuple with the second items
    * emitted by each of those observables; and so forth.
    *
    * See [[combineLatest]] for a more relaxed alternative that doesn't
    * combine items in strict sequence.
    *
    * @param other is an observable that gets paired with the source
    * @return a new observable sequence that emits the paired items
    *         of the source observables
    */
  final def zip[B](other: Observable[B]): Observable[(A, B)] =
    new Zip2Observable[A, B, (A, B)](self, other)((a, b) => (a, b))

  /** Creates a new observable from this observable and another given
    * observable by combining their items in pairs in a strict sequence.
    *
    * So the first item emitted by the new observable will be the result
    * of the function applied to the first item emitted by each of
    * the source observables; the second item emitted by the new observable
    * will be the result of the function applied to the second item
    * emitted by each of those observables; and so forth.
    *
    * See [[combineLatestMap]] for a more relaxed alternative that doesn't
    * combine items in strict sequence.
    *
    * @param other is an observable that gets paired with the source
    * @param f is a mapping function over the generated pairs
    */
  final def zipMap[B, R](other: Observable[B])(f: (A, B) => R): Observable[R] =
    new Zip2Observable[A, B, R](self, other)(f)

  /** Zips the emitted elements of the source with their indices. */
  final def zipWithIndex: Observable[(A, Long)] =
    self.liftByOperator(new ZipWithIndexOperator[A])

  /** Creates a new observable from this observable that will emit a specific `separator`
    * between every pair of elements.
    *
    * @param separator is the separator
    */
  final def intersperse[B >: A](separator: B): Observable[B] =
    new IntersperseObservable(self, None, separator, None)

  /** Creates a new observable from this observable that will emit the `start` element
    * followed by the upstream elements paired with the `separator`, and lastly the `end` element.
    *
    * @param start is the first element emitted
    * @param separator is the separator
    * @param end the last element emitted
    */
  final def intersperse[B >: A](start: B, separator: B, end: B): Observable[B] =
    new IntersperseObservable(self, Some(start), separator, Some(end))

  /** Converts this `Observable` into an `org.reactivestreams.Publisher`.
    *
    * Meant for interoperability with other Reactive Streams
    * implementations.
    *
    * Usage sample:
    *
    * {{{
    *   import monix.eval.Task
    *   import monix.execution.rstreams.SingleAssignmentSubscription
    *   import org.reactivestreams.{Publisher, Subscriber, Subscription}
    *
    *   def sum(source: Publisher[Int], requestSize: Int): Task[Long] =
    *     Task.create { (_, cb) =>
    *       val sub = SingleAssignmentSubscription()
    *
    *       source.subscribe(new Subscriber[Int] {
    *         private[this] var requested = 0L
    *         private[this] var sum = 0L
    *
    *         def onSubscribe(s: Subscription): Unit = {
    *           sub := s
    *           requested = requestSize
    *           s.request(requestSize)
    *         }
    *
    *         def onNext(t: Int): Unit = {
    *           sum += t
    *           if (requestSize != Long.MaxValue) requested -= 1
    *
    *           if (requested <= 0) {
    *             requested = requestSize
    *             sub.request(request)
    *           }
    *         }
    *
    *         def onError(t: Throwable): Unit =
    *           cb.onError(t)
    *         def onComplete(): Unit =
    *           cb.onSuccess(sum)
    *       })
    *
    *       // Cancelable that can be used by Task
    *       sub
    *     }
    *
    *   val pub = Observable(1, 2, 3, 4).toReactivePublisher
    *
    *   // Yields 10
    *   sum(pub, requestSize = 128)
    * }}}
    *
    * See the [[http://www.reactive-streams.org/ Reactive Streams]]
    * protocol for details.
    */
  final def toReactivePublisher[B >: A](implicit s: Scheduler): RPublisher[B] =
    new RPublisher[B] {
      def subscribe(subscriber: RSubscriber[_ >: B]): Unit = {
        val subscription = SingleAssignCancelable()
        subscription := unsafeSubscribeFn(SafeSubscriber(
          Subscriber.fromReactiveSubscriber(subscriber, subscription)
        ))
      }
    }

  /** Converts this observable into a multicast observable, useful for
    * turning a cold observable into a hot one (i.e. whose source is
    * shared by all observers).
    */
  final def multicast[B >: A, R](pipe: Pipe[B, R])(implicit s: Scheduler): ConnectableObservable[R] =
    ConnectableObservable.multicast(this, pipe)

  /** Returns a new Observable that multi-casts (shares) the original
    * Observable.
    */
  final def share(implicit s: Scheduler): Observable[A] =
    publish.refCount

  /** Converts this observable into a multicast observable, useful for
    * turning a cold observable into a hot one (i.e. whose source is
    * shared by all observers). The underlying subject used is a
    * [[monix.reactive.subjects.PublishSubject PublishSubject]].
    */
  final def publish(implicit s: Scheduler): ConnectableObservable[A] =
    unsafeMulticast(PublishSubject[A]())

  /** Caches the emissions from the source Observable and replays them
    * in order to any subsequent Subscribers. This operator has
    * similar behavior to [[Observable!.replay(implicit* replay]]
    * except that this auto-subscribes to the source Observable rather
    * than returning a
    * [[monix.reactive.observables.ConnectableObservable ConnectableObservable]]
    * for which you must call
    * [[monix.reactive.observables.ConnectableObservable.connect connect]]
    * to activate the subscription.
    *
    * When you call cache, it does not yet subscribe to the source
    * Observable and so does not yet begin caching items. This only
    * happens when the first Subscriber calls the resulting
    * Observable's `subscribe` method.
    *
    * Note: You sacrifice the ability to cancel the origin when you
    * use the cache operator so be careful not to use this on
    * Observables that emit an infinite or very large number of items
    * that will use up memory.
    *
    * @return an Observable that, when first subscribed to, caches all of its
    *         items and notifications for the benefit of subsequent subscribers
    */
  final def cache: Observable[A] =
    CachedObservable.create(self)

  /** Caches the emissions from the source Observable and replays them
    * in order to any subsequent Subscribers. This operator has
    * similar behavior to [[Observable!.replay(implicit* replay]]
    * except that this auto-subscribes to the source Observable rather
    * than returning a
    * [[monix.reactive.observables.ConnectableObservable ConnectableObservable]]
    * for which you must call
    * [[monix.reactive.observables.ConnectableObservable.connect connect]]
    * to activate the subscription.
    *
    * When you call cache, it does not yet subscribe to the source
    * Observable and so does not yet begin caching items. This only
    * happens when the first Subscriber calls the resulting
    * Observable's `subscribe` method.
    *
    * @param maxCapacity is the maximum buffer size after which old events
    *        start being dropped (according to what happens when using
    *        [[monix.reactive.subjects.ReplaySubject.createLimited[A](capacity:Int,initial* ReplaySubject.createLimited]])
    *
    * @return an Observable that, when first subscribed to, caches all of its
    *         items and notifications for the benefit of subsequent subscribers
    */
  final def cache(maxCapacity: Int): Observable[A] =
    CachedObservable.create(self, maxCapacity)

  /** Converts this observable into a multicast observable, useful for
    * turning a cold observable into a hot one (i.e. whose source is
    * shared by all observers). The underlying subject used is a
    * [[monix.reactive.subjects.BehaviorSubject BehaviorSubject]].
    */
  final def behavior[B >: A](initialValue: B)(implicit s: Scheduler): ConnectableObservable[B] =
    unsafeMulticast(BehaviorSubject[B](initialValue))

  /** Converts this observable into a multicast observable, useful for
    * turning a cold observable into a hot one (i.e. whose source is
    * shared by all observers). The underlying subject used is a
    * [[monix.reactive.subjects.ReplaySubject ReplaySubject]].
    */
  final def replay(implicit s: Scheduler): ConnectableObservable[A] =
    unsafeMulticast(ReplaySubject[A]())

  /** Converts this observable into a multicast observable, useful for
    * turning a cold observable into a hot one (i.e. whose source is
    * shared by all observers). The underlying subject used is a
    * [[monix.reactive.subjects.ReplaySubject ReplaySubject]].
    *
    * @param bufferSize is the size of the buffer limiting the number
    *        of items that can be replayed (on overflow the head
    *        starts being dropped)
    */
  final def replay(bufferSize: Int)(implicit s: Scheduler): ConnectableObservable[A] =
    unsafeMulticast(ReplaySubject.createLimited[A](bufferSize))

  /** Converts this observable into a multicast observable, useful for
    * turning a cold observable into a hot one (i.e. whose source is
    * shared by all observers).
    *
    * This operator is unsafe because `Subject` objects are stateful
    * and have to obey the `Observer` contract, meaning that they
    * shouldn't be subscribed multiple times, so they are error
    * prone. Only use if you know what you're doing, otherwise prefer
    * the safe [[Observable!.multicast multicast]] operator.
    */
  final def unsafeMulticast[B >: A, R](processor: Subject[B, R])(implicit s: Scheduler): ConnectableObservable[R] =
    ConnectableObservable.unsafeMulticast(this, processor)

  /** Converts this observable into a multicast observable, useful for
    * turning a cold observable into a hot one (i.e. whose source is
    * shared by all observers). The underlying subject used is a
    * [[monix.reactive.subjects.AsyncSubject AsyncSubject]].
    */
  final def publishLast(implicit s: Scheduler): ConnectableObservable[A] =
    unsafeMulticast(AsyncSubject[A]())

  /** Creates a new [[monix.execution.CancelableFuture CancelableFuture]]
    * that upon execution will signal the first generated element of the
    * source observable. Returns an `Option` because the source can be empty.
    */
  final def runAsyncGetFirst(implicit s: Scheduler): CancelableFuture[Option[A]] =
    firstOptionL.runAsync

  /** Creates a new [[monix.execution.CancelableFuture CancelableFuture]]
    * that upon execution will signal the last generated element of the
    * source observable. Returns an `Option` because the source can be empty.
    */
  final def runAsyncGetLast(implicit s: Scheduler): CancelableFuture[Option[A]] =
    lastOptionL.runAsync

  /** Returns a [[monix.eval.Task Task]] that upon execution
    * will signal the last generated element of the source observable.
    *
    * Returns an `Option` because the source can be empty.
    */
  final def lastOptionL: Task[Option[A]] =
    map(Some.apply).lastOrElseL(None)

  /** Creates a new [[monix.eval.Task Task]] that upon execution
    * will signal the last generated element of the source observable.
    *
    * In case the stream was empty, then the given default gets
    * evaluated and emitted.
    */
  final def lastOrElseL[B >: A](default: => B): Task[B] =
    Task.cancelable0 { (s, cb) =>
      unsafeSubscribeFn(new Subscriber.Sync[A] {
        implicit val scheduler: Scheduler = s
        private[this] var value: A = _
        private[this] var isEmpty = true

        def onNext(elem: A): Ack = {
          if (isEmpty) isEmpty = false
          value = elem
          Continue
        }

        def onError(ex: Throwable): Unit = {
          cb.onError(ex)
        }

        def onComplete(): Unit = {
          if (isEmpty)
            cb(Eager(default))
          else
            cb.onSuccess(value)
        }
      })
    }

  /** Creates a task that emits the total number of `onNext`
    * events that were emitted by the source.
    */
  final def countL: Task[Long] =
    countF.headL

  /** Creates a new Observable that emits the total number of `onNext`
    * events that were emitted by the source.
    *
    * Note that this Observable emits only one item after the source
    * is complete.  And in case the source emits an error, then only
    * that error will be emitted.
    */
  final def countF: Observable[Long] =
    self.liftByOperator(CountOperator)

  /** Returns a task which emits the first item for which
    * the predicate holds.
    *
    * @param p is a function that evaluates the items emitted by the
    *        source observable, returning `true` if they pass the filter
    * @return a task that emits the first item in the source
    *         observable for which the filter evaluates as `true`
    */
  final def findL(p: A => Boolean): Task[Option[A]] =
    findF(p).headOptionL

  /** Given evidence that type `A` has a `cats.Monoid` implementation,
    * folds the stream with the provided monoid definition.
    *
    * For streams emitting numbers, this effectively sums them up.
    * For strings, this concatenates them.
    *
    * Example:
    *
    * {{{
    *   // Yields 10
    *   Observable(1, 2, 3, 4).foldL
    *
    *   // Yields "1234"
    *   Observable("1", "2", "3", "4").foldL
    * }}}
    *
    * Note, in case you don't have a `Monoid` instance in scope,
    * but you feel like you should, try this import:
    *
    * {{{
    *   import cats.instances.all._
    * }}}
    *
    * @see [[foldF]] for the version that returns an observable
    *      instead of a task.
    *
    * @param A is the `cats.Monoid` type class instance that's needed
    *          in scope for folding the source
    *
    * @return the result of combining all elements of the source,
    *         or the defined `Monoid.empty` element in case the
    *         stream is empty
    */
  final def foldL[AA >: A](implicit A: Monoid[AA]): Task[AA] =
    foldF(A).headL

  /** Given evidence that type `A` has a `cats.Monoid` implementation,
    * folds the stream with the provided monoid definition.
    *
    * For streams emitting numbers, this effectively sums them up.
    * For strings, this concatenates them.
    *
    * Example:
    *
    * {{{
    *   // Yields 10
    *   Observable(1, 2, 3, 4).foldF
    *
    *   // Yields "1234"
    *   Observable("1", "2", "3", "4").foldF
    * }}}
    *
    * Note, in case you don't have a `Monoid` instance in scope,
    * but you feel like you should, try this import:
    *
    * {{{
    *   import cats.instances.all._
    * }}}
    *
    * @see [[Observable.foldL foldL]] for the version that returns a
    *      task instead of an observable.
    *
    * @param A is the `cats.Monoid` type class instance that's needed
    *          in scope for folding the source
    *
    * @return the result of combining all elements of the source,
    *         or the defined `Monoid.empty` element in case the
    *         stream is empty
    */
  final def foldF[AA >: A](implicit A: Monoid[AA]): Observable[AA] =
    foldLeftF(A.empty)(A.combine)

  /** Folds the source observable, from start to finish, until the
    * source completes, or until the operator short-circuits the
    * process by returning `false`.
    *
    * Note that a call to [[foldLeftL]] is equivalent to this function
    * being called with an operator always returning `Left` results.
    *
    * Example: {{{
    *   // Sums first 10 items
    *   Observable.range(0, 1000).foldWhileLeftL((0, 0)) {
    *     case ((sum, count), e) =>
    *       val next = (sum + e, count + 1)
    *       if (count + 1 < 10) Left(next) else Right(next)
    *   }
    *
    *   // Implements exists(predicate)
    *   Observable(1, 2, 3, 4, 5).foldWhileLeftL(false) {
    *     (default, e) =>
    *       if (e == 3) Right(true) else Left(default)
    *   }
    *
    *   // Implements forall(predicate)
    *   Observable(1, 2, 3, 4, 5).foldWhileLeftL(true) {
    *     (default, e) =>
    *       if (e != 3) Right(false) else Left(default)
    *   }
    * }}}
    *
    * @see [[foldWhileLeftF]] for a version that returns an observable
    *      instead of a task.
    *
    * @param seed is the initial state, specified as a possibly lazy value;
    *        it gets evaluated when the subscription happens and if it
    *        triggers an error then the subscriber will get immediately
    *        terminated with an error
    *
    * @param op is the binary operator returning either `Left`,
    *        signaling that the state should be evolved or a `Right`,
    *        signaling that the process can be short-circuited and
    *        the result returned immediately
    *
    * @return the result of inserting `op` between consecutive
    *         elements of this observable, going from left to right with
    *         the `seed` as the start value, or `seed` if the observable
    *         is empty
    */
  final def foldWhileLeftL[S](seed: => S)(op: (S, A) => Either[S, S]): Task[S] =
    foldWhileLeftF(seed)(op).headL

  /** Folds the source observable, from start to finish, until the
    * source completes, or until the operator short-circuits the
    * process by returning `false`.
    *
    * Note that a call to [[foldLeftF]] is equivalent to this function
    * being called with an operator always returning `true` as the first
    * member of its result.
    *
    * Example: {{{
    *   // Sums first 10 items
    *   Observable.range(0, 1000).foldWhileLeftF((0, 0)) {
    *     case ((sum, count), e) =>
    *       val next = (sum + e, count + 1)
    *       if (count + 1 < 10) Left(next) else Right(next)
    *   }
    *
    *   // Implements exists(predicate)
    *   Observable(1, 2, 3, 4, 5).foldWhileLeftF(false) {
    *     (default, e) =>
    *       if (e == 3) Right(true) else Left(default)
    *   }
    *
    *   // Implements forall(predicate)
    *   Observable(1, 2, 3, 4, 5).foldWhileLeftF(true) {
    *     (default, e) =>
    *       if (e != 3) Right(false) else Left(default)
    *   }
    * }}}
    *
    * @see [[Observable.foldWhileLeftL foldWhileLeftL]] for a version
    *      that returns a task instead of an observable.
    *
    * @param seed is the initial state, specified as a possibly lazy value;
    *        it gets evaluated when the subscription happens and if it
    *        triggers an error then the subscriber will get immediately
    *        terminated with an error
    *
    * @param op is the binary operator returning either `Left`,
    *        signaling that the state should be evolved or a `Right`,
    *        signaling that the process can be short-circuited and
    *        the result returned immediately
    *
    * @return the result of inserting `op` between consecutive
    *         elements of this observable, going from left to right with
    *         the `seed` as the start value, or `seed` if the observable
    *         is empty
    */
  final def foldWhileLeftF[S](seed: => S)(op: (S, A) => Either[S, S]): Observable[S] =
    new FoldWhileLeftObservable[A, S](self, seed _, op)

  /** Alias for [[firstL]]. */
  final def headL: Task[A] = firstL

  /** Creates a new [[monix.eval.Task Task]] that upon execution
    * will signal the first generated element of the source observable.
    *
    * In case the stream was empty, then the `Task` gets completed
    * in error with a `NoSuchElementException`.
    */
  final def firstL: Task[A] =
    firstOrElseL(throw new NoSuchElementException("firstL on empty observable"))

  /** Creates a new [[monix.eval.Task Task]] that upon execution
    * will signal the first generated element of the source observable.
    *
    * In case the stream was empty, then the given default
    * gets evaluated and emitted.
    */
  final def firstOrElseL[B >: A](default: => B): Task[B] =
    Task.cancelable0 { (s, cb) =>
      unsafeSubscribeFn(new Subscriber.Sync[A] {
        implicit val scheduler: Scheduler = s
        private[this] var isDone = false

        def onNext(elem: A): Ack = {
          cb.onSuccess(elem)
          isDone = true
          Stop
        }

        def onError(ex: Throwable): Unit =
          if (!isDone) {
            isDone = true
            cb.onError(ex)
          }

        def onComplete(): Unit =
          if (!isDone) {
            isDone = true
            cb(Eager(default))
          }
      })
    }

  /** Returns a `Task` that emits a single boolean, either true, in
    * case the given predicate holds for all the items emitted by the
    * source, or false in case at least one item is not verifying the
    * given predicate.
    *
    * @param p is a function that evaluates the items emitted by the source
    *        observable, returning `true` if they pass the filter
    * @return a task that emits only true or false in case the given
    *         predicate holds or not for all the items
    */
  final def forAllL(p: A => Boolean): Task[Boolean] =
    existsL(e => !p(e)).map(r => !r)

  /** Returns a `Task` which emits either `true`, in case the given predicate
    * holds for at least one item, or `false` otherwise.
    *
    * @param p is a function that evaluates the items emitted by the
    *        source, returning `true` if they pass the filter
    * @return a task that emits `true` or `false` in case
    *         the given predicate holds or not for at least one item
    */
  final def existsL(p: A => Boolean): Task[Boolean] =
    findF(p).foldLeftL(false)((_, _) => true)

  /** Returns an Observable which only emits the first item for which
    * the predicate holds.
    *
    * @param p is a function that evaluates the items emitted by the
    *        source Observable, returning `true` if they pass the filter
    * @return an Observable that emits only the first item in the original
    *         Observable for which the filter evaluates as `true`
    */
  final def findF(p: A => Boolean): Observable[A] =
    filter(p).headF

  /** Only emits those items for which the given predicate holds.
    *
    * @param p a function that evaluates the items emitted by the source
    *        returning `true` if they pass the filter
    * @return a new observable that emits only those items in the source
    *         for which the filter evaluates as `true`
    */
  final def filter(p: A => Boolean): Observable[A] =
    self.liftByOperator(new FilterOperator(p))

  /** Only emits the first element emitted by the source observable,
    * after which it's completed immediately.
    */
  final def headF: Observable[A] = take(1)

  /** Selects the first `n` elements (from the start).
    *
    * @param  n the number of elements to take
    * @return a new Observable that emits only the first
    *         `n` elements from the source
    */
  final def take(n: Long): Observable[A] =
    if (n <= 0) Observable.empty else self.liftByOperator(new TakeLeftOperator(n))

  /** Applies a binary operator to a start value and all elements of
    * the source, going left to right and returns a new `Task` that
    * upon evaluation will eventually emit the final result.
    */
  final def foldLeftL[R](seed: => R)(op: (R, A) => R): Task[R] =
    foldLeftF(seed)(op).headL

  /** Applies a binary operator to a start value and all elements of
    * this Observable, going left to right and returns a new
    * Observable that emits only one item before `onComplete`.
    *
    * @param seed is the initial state, specified as a possibly lazy value;
    *        it gets evaluated when the subscription happens and if it triggers
    *        an error then the subscriber will get immediately terminated
    *        with an error
    *
    * @param op is an operator that will fold the signals of the source
    *        observable, returning the next state
    */
  final def foldLeftF[R](seed: => R)(op: (R, A) => R): Observable[R] =
    new FoldLeftObservable[A, R](self, seed _, op)

  /** Alias for [[firstOrElseL]]. */
  final def headOrElseL[B >: A](default: => B): Task[B] = firstOrElseL(default)

  /** Returns a [[monix.eval.Task Task]] that upon execution
    * will signal the last generated element of the source observable.
    *
    * In case the stream was empty, then the `Task` gets completed
    * in error with a `NoSuchElementException`.
    */
  final def lastL: Task[A] =
    lastOrElseL(throw new NoSuchElementException("lastL"))

  /** Returns a task that emits `true` if the source observable is
    * empty, otherwise `false`.
    */
  final def isEmptyL: Task[Boolean] =
    isEmptyF.headL

  /** Returns an Observable that emits true if the source Observable is
    * empty, otherwise false.
    */
  final def isEmptyF: Observable[Boolean] =
    self.liftByOperator(IsEmptyOperator)

  /** Creates a new [[monix.eval.Task Task]] that will consume the
    * source observable and upon completion of the source it will
    * complete with `Unit`.
    */
  final def completedL: Task[Unit] =
    Task.cancelable0 { (s, cb) =>
      unsafeSubscribeFn(new Subscriber.Sync[A] {
        implicit val scheduler: Scheduler = s
        private[this] var isDone = false

        def onNext(elem: A): Ack = Continue

        def onError(ex: Throwable): Unit =
          if (!isDone) {
            isDone = true; cb.onError(ex)
          }

        def onComplete(): Unit =
          if (!isDone) {
            isDone = true; cb.onSuccess(())
          }
      })
    }

  /** Given a `cats.Order` over the stream's elements, returns the
    * maximum element in the stream.
    *
    * Example:
    * {{{
    *   // Yields Some(20)
    *   Observable(10, 7, 6, 8, 20, 3, 5).maxL
    *
    *   // Yields Observable.empty
    *   Observable.empty.maxL
    * }}}
    *
    * $catsOrderDesc
    *
    * @see [[Observable.maxF maxF]] for the version that returns an
    *      observable instead of a `Task`.
    *
    * @param A is the `cats.Order` type class instance that's going
    *          to be used for comparing elements
    *
    * @return the maximum element of the source stream, relative
    *         to the defined `Order`
    */
  final def maxL[AA >: A](implicit A: Order[AA]): Task[Option[AA]] =
    maxF(A).headOptionL

  /** Given a `cats.Order` over the stream's elements, returns the
    * maximum element in the stream.
    *
    * Example:
    * {{{
    *   // Yields Observable(20)
    *   Observable(10, 7, 6, 8, 20, 3, 5).maxF
    *
    *   // Yields Observable.empty
    *   Observable.empty.maxF
    * }}}
    *
    * $catsOrderDesc
    *
    * @see [[Observable.maxL maxL]] for the version that returns a
    *      [[monix.eval.Task Task]] instead of an observable.
    *
    * @param A is the `cats.Order` type class instance that's going
    *          to be used for comparing elements
    *
    * @return the maximum element of the source stream, relative
    *         to the defined `Order`
    */
  final def maxF[AA >: A](implicit A: Order[AA]): Observable[AA] =
    self.liftByOperator(new MaxOperator[AA]()(A))

  /** Alias for [[firstOptionL]]. */
  final def headOptionL: Task[Option[A]] = firstOptionL

  /** Creates a new [[monix.eval.Task Task]] that upon execution
    * will signal the first generated element of the source observable.
    *
    * Returns an `Option` because the source can be empty.
    */
  final def firstOptionL: Task[Option[A]] =
    map(Some.apply).firstOrElseL(None)

  /** Takes the elements of the source observable and emits the
    * element that has the maximum key value, where the key is
    * generated by the given function.
    *
    * Example:
    * {{{
    *   case class Person(name: String, age: Int)
    *
    *   // Yields Some(Person("Alex", 34))
    *   Observable(Person("Alex", 34), Person("Alice", 27))
    *     .maxByL(_.age)
    * }}}
    *
    * $catsOrderDesc
    *
    * @see [[Observable.maxByF maxByF]] for the version that returns an
    *      observable instead of a `Task`.
    *
    * @param key is the function that returns the key for which the
    *        given ordering is defined
    *
    * @param K is the `cats.Order` type class instance that's going
    *          to be used for comparing elements
    *
    * @return the maximum element of the source stream, relative
    *         to its key generated by the given function and the
    *         given ordering
    */
  final def maxByL[K](key: A => K)(implicit K: Order[K]): Task[Option[A]] =
    maxByF(key)(K).headOptionL

  /** Takes the elements of the source observable and emits the
    * element that has the maximum key value, where the key is
    * generated by the given function.
    *
    * Example:
    * {{{
    *   case class Person(name: String, age: Int)
    *
    *   // Yields Observable(Person("Alex", 34))
    *   Observable(Person("Alex", 34), Person("Alice", 27))
    *     .maxByF(_.age)
    * }}}
    *
    * $catsOrderDesc
    *
    * @see [[Observable.maxByL maxByL]] for the version that returns a
    *      [[monix.eval.Task Task]] instead of an observable.
    *
    * @param key is the function that returns the key for which the
    *        given ordering is defined
    *
    * @param K is the `cats.Order` type class instance that's going
    *          to be used for comparing elements
    *
    * @return the maximum element of the source stream, relative
    *         to its key generated by the given function and the
    *         given ordering
    */
  final def maxByF[K](key: A => K)(implicit K: Order[K]): Observable[A] =
    self.liftByOperator(new MaxByOperator[A, K](key)(K))

  /** Given a `cats.Order` over the stream's elements, returns the
    * minimum element in the stream.
    *
    * Example:
    * {{{
    *   // Yields Some(3)
    *   Observable(10, 7, 6, 8, 20, 3, 5).minL
    *
    *   // Yields None
    *   Observable.empty.minL
    * }}}
    *
    * $catsOrderDesc
    *
    * @see [[Observable.minF minF]] for the version that returns an
    *      observable instead of a `Task`.
    *
    * @param A is the `cats.Order` type class instance that's going
    *          to be used for comparing elements
    *
    * @return the minimum element of the source stream, relative
    *         to the defined `Order`
    */
  final def minL[AA >: A](implicit A: Order[AA]): Task[Option[AA]] =
    minF(A).headOptionL

  /** Given a `cats.Order` over the stream's elements, returns the
    * minimum element in the stream.
    *
    * Example:
    * {{{
    *   // Yields Observable(3)
    *   Observable(10, 7, 6, 8, 20, 3, 5).minF
    *
    *   // Yields Observable.empty
    *   Observable.empty.minF
    * }}}
    *
    * $catsOrderDesc
    *
    * @see [[Observable.minL minL]] for the version that returns a
    *      [[monix.eval.Task Task]] instead of an observable.
    *
    * @param A is the `cats.Order` type class instance that's going
    *          to be used for comparing elements
    *
    * @return the minimum element of the source stream, relative
    *         to the defined `Order`
    */
  final def minF[AA >: A](implicit A: Order[AA]): Observable[AA] =
    self.liftByOperator(new MinOperator()(A))

  /** Takes the elements of the source observable and emits the
    * element that has the minimum key value, where the key is
    * generated by the given function.
    *
    * Example:
    * {{{
    *   case class Person(name: String, age: Int)
    *
    *   // Yields Some(Person("Alice", 27))
    *   Observable(Person("Alex", 34), Person("Alice", 27))
    *     .minByL(_.age)
    * }}}
    *
    * $catsOrderDesc
    *
    * @param key is the function that returns the key for which the
    *        given ordering is defined
    *
    * @param K is the `cats.Order` type class instance that's going
    *          to be used for comparing elements
    *
    * @return the minimum element of the source stream, relative
    *         to its key generated by the given function and the
    *         given ordering
    */
  final def minByL[K](key: A => K)(implicit K: Order[K]): Task[Option[A]] =
    minByF(key)(K).headOptionL

  /** Takes the elements of the source observable and emits the
    * element that has the minimum key value, where the key is
    * generated by the given function.
    *
    * Example:
    * {{{
    *   case class Person(name: String, age: Int)
    *
    *   // Yields Observable(Person("Alice", 27))
    *   Observable(Person("Alex", 34), Person("Alice", 27))
    *     .minByF(_.age)
    * }}}
    *
    * $catsOrderDesc
    *
    * @param key is the function that returns the key for which the
    *        given ordering is defined
    *
    * @param K is the `cats.Order` type class instance that's going
    *          to be used for comparing elements
    *
    * @return the minimum element of the source stream, relative
    *         to its key generated by the given function and the
    *         given ordering
    */
  final def minByF[K](key: A => K)(implicit K: Order[K]): Observable[A] =
    self.liftByOperator(new MinByOperator[A, K](key))

  /** Returns a task that emits `false` if the source observable is
    * empty, otherwise `true`.
    */
  final def nonEmptyL: Task[Boolean] =
    nonEmptyF.headL

  /** Returns an Observable that emits false if the source Observable is
    * empty, otherwise true.
    */
  final def nonEmptyF: Observable[Boolean] =
    self.liftByOperator(IsEmptyOperator).map(b => !b)

  /** Given a source that emits numeric values, the `sum` operator sums
    * up all values and returns the result.
    */
  final def sumL[B >: A](implicit B: Numeric[B]): Task[B] =
    sumF(B).headL

  /** Given a source that emits numeric values, the `sum` operator sums
    * up all values and at onComplete it emits the total.
    */
  final def sumF[AA >: A](implicit A: Numeric[AA]): Observable[AA] =
    foldLeftF(A.zero)(A.plus)

  /** Returns a `Task` that upon evaluation will collect all items from
    * the source in a Scala `List` and return this list instead.
    *
    * WARNING: for infinite streams the process will eventually blow up
    * with an out of memory error.
    */
  final def toListL: Task[List[A]] =
    foldLeftL(mutable.ListBuffer.empty[A])(_ += _).map(_.toList)

  /** Makes the source `Observable` uninterruptible such that a `cancel`
    * signal has no effect.
    *
    * {{{
    *   val cancelable = Observable
    *     .eval(println("Hello!"))
    *     .delayExecution(10.seconds)
    *     .subscribe()
    *
    *   // No longer works
    *   cancelable.cancel()
    *
    *   // After 10 seconds
    *   //=> Hello!
    * }}}
    */
  final def uncancelable: Observable[A] =
    new UncancelableObservable[A](self)

  /** Creates a new [[monix.eval.Task Task]] that will consume the
    * source observable, executing the given callback for each element.
    */
  final def foreachL(cb: A => Unit): Task[Unit] =
    Task.cancelable0 { (s, onFinish) =>
      unsafeSubscribeFn(new ForeachSubscriber[A](cb, onFinish, s))
    }

  /** Subscribes to the source `Observable` and foreach element emitted
    * by the source it executes the given callback.
    */
  final def foreach(cb: A => Unit)(implicit s: Scheduler): CancelableFuture[Unit] = {
    val p = Promise[Unit]()
    val onFinish = Callback.fromPromise(p)
    val c = unsafeSubscribeFn(new ForeachSubscriber[A](cb, onFinish, s))
    CancelableFuture(p.future, c)
  }
}


/** Observable builders.
  *
  * @define multicastDesc Creates an input channel and an output observable
  *         pair for building a [[MulticastStrategy multicast]] data-source.
  *
  *         Useful for building [[MulticastStrategy multicast]] observables
  *         from data-sources that cannot be back-pressured.
  *
  *         Prefer [[Observable.create]] when possible.
  *
  * @define fromIteratorDesc Converts any `Iterator` into an observable.
  *
  *         WARNING: reading from an `Iterator` is a destructive process.
  *         Therefore only a single subscriber is supported, the result being
  *         a single-subscriber observable. If multiple subscribers are attempted,
  *         all subscribers, except for the first one, will be terminated with a
  *         [[monix.execution.exceptions.APIContractViolationException APIContractViolationException]].
  *
  *         Therefore, if you need a factory of data sources, from a cold source
  *         from which you can open how many iterators you want,
  *         you can use [[Observable.defer]] to build such a factory. Or you can share
  *         the resulting observable by converting it into a
  *         [[monix.reactive.observables.ConnectableObservable ConnectableObservable]]
  *         by means of [[Observable!.multicast multicast]].
  *
  * @define fromInputStreamDesc Converts a `java.io.InputStream` into an
  *         observable that will emit `Array[Byte]` elements.
  *
  *         WARNING: reading from the input stream is a destructive process.
  *         Therefore only a single subscriber is supported, the result being
  *         a single-subscriber observable. If multiple subscribers are attempted,
  *         all subscribers, except for the first one, will be terminated with a
  *         [[monix.execution.exceptions.APIContractViolationException APIContractViolationException]].
  *
  *         Therefore, if you need a factory of data sources, from a cold source such
  *         as a `java.io.File` from which you can open how many file handles you want,
  *         you can use [[Observable.defer]] to build such a factory. Or you can share
  *         the resulting observable by converting it into a
  *         [[monix.reactive.observables.ConnectableObservable ConnectableObservable]]
  *         by means of [[Observable!.multicast multicast]].
  *
  * @define fromCharsReaderDesc Converts a `java.io.Reader` into an observable
  *         that will emit `Array[Char]` elements.
  *
  *         WARNING: reading from a reader is a destructive process.
  *         Therefore only a single subscriber is supported, the result being
  *         a single-subscriber observable. If multiple subscribers are attempted,
  *         all subscribers, except for the first one, will be terminated with a
  *         [[monix.execution.exceptions.APIContractViolationException APIContractViolationException]].
  *
  *         Therefore, if you need a factory of data sources, from a cold source such
  *         as a `java.io.File` from which you can open how many file handles you want,
  *         you can use [[Observable.defer]] to build such a factory. Or you can share
  *         the resulting observable by converting it into a
  *         [[monix.reactive.observables.ConnectableObservable ConnectableObservable]]
  *         by means of [[Observable!.multicast multicast]].
  *
  * @define blocksDefaultSchedulerDesc This operation will start processing on the current
  *         thread (on `subscribe()`), so in order to not block, it might be better to also do an
  *         [[Observable.executeAsync executeAsync]], or you may want to use the
  *         [[monix.execution.ExecutionModel.AlwaysAsyncExecution AlwaysAsyncExecution]]
  *         model, which can be configured per `Scheduler`, see
  *         [[monix.execution.Scheduler.withExecutionModel Scheduler.withExecutionModel]],
  *         or per `Observable`, see [[Observable.executeWithModel]].
  */
object Observable {
  /** An `Operator` is a function for transforming observers,
    * that can be used for lifting observables.
    *
    * See [[Observable.liftByOperator]].
    */
  type Operator[-I, +O] = Subscriber[O] => Subscriber[I]

  /** Given a sequence of elements, builds an observable from it. */
  def apply[A](elems: A*): Observable[A] =
    Observable.fromIterable(elems)

  /** Lifts an element into the `Observable` context.
    *
    * Alias for [[now]].
    */
  def pure[A](elem: A): Observable[A] =
    new builders.NowObservable(elem)

  /** Alias for [[eval]]. */
  def delay[A](a: => A): Observable[A] = eval(a)

  /** Given a non-strict value, converts it into an Observable
    * that emits a single element and that memoizes the value
    * for subsequent invocations.
    */
  def evalOnce[A](f: => A): Observable[A] =
    new builders.EvalOnceObservable(f)

  /** Returns an `Observable` that on execution emits the given strict value.
    */
  def now[A](elem: A): Observable[A] =
    new builders.NowObservable(elem)

  /** Creates an Observable that emits an error.
    */
  def raiseError[A](ex: Throwable): Observable[A] =
    new builders.ErrorObservable(ex)

  /** Given a non-strict value, converts it into an Observable
    * that upon subscription, evaluates the expression and
    * emits a single element.
    */
  def eval[A](a: => A): Observable[A] =
    new builders.EvalAlwaysObservable(a)

  /** Lifts a non-strict value into an observable that emits a single element,
    * but upon subscription delay its evaluation by the specified timespan
    */
  def evalDelayed[A](delay: FiniteDuration, a: => A): Observable[A] =
    eval(a).delayExecution(delay)

  /** Creates an Observable that doesn't emit anything and that never
    * completes.
    */
  def never[A]: Observable[A] =
    builders.NeverObservable

  /** Keeps calling `f` and concatenating the resulting observables
    * for each `scala.util.Left` event emitted by the source, concatenating
    * the resulting observables and pushing every `scala.util.Right[B]`
    * events downstream.
    *
    * Based on Phil Freeman's
    * [[http://functorial.com/stack-safety-for-free/index.pdf Stack Safety for Free]].
    *
    * It helps to wrap your head around it if you think of it as being
    * equivalent to this inefficient and unsafe implementation (for `Observable`):
    *
    * {{{
    *   def tailRecM[A, B](a: A)(f: (A) => Observable[Either[A, B]]): Observable[B] =
    *     f(a).flatMap {
    *       case Right(b) => pure(b)
    *       case Left(nextA) => tailRecM(nextA)(f)
    *}
    * }}}
    */
  def tailRecM[A, B](a: A)(f: (A) => Observable[Either[A, B]]): Observable[B] =
    new builders.TailRecMObservable[A, B](a, f)

  /** Given a subscribe function, lifts it into an [[Observable]].
    *
    * This function is unsafe to use because users have to know and apply
    * the Monix communication contract, related to thread-safety, communicating
    * demand (back-pressure) and error handling.
    *
    * Only use if you know what you're doing. Otherwise prefer [[create]].
    */
  def unsafeCreate[A](f: Subscriber[A] => Cancelable): Observable[A] =
    new builders.UnsafeCreateObservable(f)

  /** Creates an observable from a function that receives a
    * concurrent and safe
    * [[monix.reactive.observers.Subscriber.Sync Subscriber.Sync]].
    *
    * This builder represents the safe way of building observables
    * from data-sources that cannot be back-pressured.
    */
  def create[A](overflowStrategy: OverflowStrategy.Synchronous[A])
    (f: Subscriber.Sync[A] => Cancelable): Observable[A] =
    new builders.CreateObservable(overflowStrategy, f)

  /** $multicastDesc
    *
    * @param multicast is the multicast strategy to use (e.g. publish, behavior,
    *        reply, async)
    */
  def multicast[A](multicast: MulticastStrategy[A])
    (implicit s: Scheduler): (Observer.Sync[A], Observable[A]) = {

    val ref = ConcurrentSubject(multicast)
    (ref, ref)
  }

  /** $multicastDesc
    *
    * @param multicast is the multicast strategy to use (e.g. publish, behavior,
    *        reply, async)
    * @param overflow is the overflow strategy for the buffer that gets placed
    *        in front (since this will be a hot data-source that cannot be
    *        back-pressured)
    */
  def multicast[A](multicast: MulticastStrategy[A], overflow: OverflowStrategy.Synchronous[A])
    (implicit s: Scheduler): (Observer.Sync[A], Observable[A]) = {

    val ref = ConcurrentSubject(multicast, overflow)
    (ref, ref)
  }

  /** Converts to [[Observable]] from any `F[_]` that has an [[ObservableLike]]
    * instance.
    *
    * Supported types includes, but is not necessarily limited to:
    *
    *  - [[https://typelevel.org/cats/datatypes/eval.html cats.Eval]]
    *  - [[https://typelevel.org/cats-effect/datatypes/io.html cats.effect.IO]]
    *  - [[https://typelevel.org/cats-effect/datatypes/syncio.html cats.effect.SyncIO]]
    *  - [[https://typelevel.org/cats-effect/typeclasses/effect.html cats.effect.Effect (Async)]]
    *  - [[https://typelevel.org/cats-effect/typeclasses/concurrent-effect.html cats.effect.ConcurrentEffect]]
    *  - [[https://www.reactive-streams.org/ org.reactivestreams.Publisher]]
    *  - [[monix.eval.Coeval]]
    *  - [[monix.eval.Task]]
    *  - [[scala.Either]]
    *  - [[scala.util.Try]]
    *  - [[scala.concurrent.Future]]
    */
  def from[F[_], A](fa: F[A])(implicit F: ObservableLike[F]): Observable[A] =
    F.toObservable(fa)

  /**
    * Converts any `Iterable` into an [[Observable]].
    */
  def fromIterable[A](iterable: Iterable[A]): Observable[A] =
    new builders.IterableAsObservable[A](iterable)

  /** $fromIteratorDesc
    *
    * @param iterator to transform into an observable
    */
  def fromIterator[A](iterator: Iterator[A]): Observable[A] =
    new builders.IteratorAsObservable[A](iterator, Cancelable.empty)

  /** $fromIteratorDesc
    *
    * This variant of `fromIterator` takes an `onFinish` callback that
    * will be called when the streaming is finished, either with
    * `onComplete`, `onError`, when the downstream signals a `Stop` or
    * when the subscription gets canceled.
    *
    * This `onFinish` callback is guaranteed to be called only once.
    *
    * Useful for controlling resource deallocation (e.g. closing file
    * handles).
    *
    * @param iterator to transform into an observable
    * @param onFinish a callback that will be called for resource deallocation
    *        whenever the iterator is complete, or when the stream is
    *        canceled
    */
  def fromIterator[A](iterator: Iterator[A], onFinish: () => Unit): Observable[A] =
    new builders.IteratorAsObservable[A](iterator, Cancelable(onFinish))

  /** $fromInputStreamDesc
    *
    * $blocksDefaultSchedulerDesc
    *
    * @param in is the `InputStream` to convert into an observable
    */
  def fromInputStream(in: InputStream): Observable[Array[Byte]] =
    fromInputStream(in, chunkSize = 4096)

  /** $fromInputStreamDesc
    *
    * $blocksDefaultSchedulerDesc
    *
    * @param in is the `InputStream` to convert into an observable
    * @param chunkSize is the maximum length of the emitted arrays of bytes.
    *        It's also used when reading from the input stream.
    */
  def fromInputStream(in: InputStream, chunkSize: Int): Observable[Array[Byte]] =
    new builders.InputStreamObservable(in, chunkSize)

  /** $fromCharsReaderDesc
    *
    * $blocksDefaultSchedulerDesc
    *
    * @param in is the `Reader` to convert into an observable
    */
  def fromCharsReader(in: Reader): Observable[Array[Char]] =
    fromCharsReader(in, chunkSize = 4096)

  /** $fromCharsReaderDesc
    *
    * $blocksDefaultSchedulerDesc
    *
    * @param in is the `Reader` to convert into an observable
    * @param chunkSize is the maximum length of the emitted arrays of chars.
    *        It's also used when reading from the reader.
    */
  def fromCharsReader(in: Reader, chunkSize: Int): Observable[Array[Char]] =
    new builders.CharsReaderObservable(in, chunkSize)

  /** Converts a `java.io.BufferedReader` into an
    * observable that will emit `String` text lines from the input.
    *
    * Note that according to the specification of `BufferedReader`, a
    * line is considered to be terminated by any one of a line
    * feed (`\n`), a carriage return (`\r`), or a carriage return
    * followed immediately by a linefeed.
    *
    * WARNING: reading from a reader is a destructive process.
    * Therefore only a single subscriber is supported, the result being
    * a single-subscriber observable. If multiple subscribers are attempted,
    * all subscribers, except for the first one, will be terminated with a
    * [[monix.execution.exceptions.APIContractViolationException APIContractViolationException]].
    *
    * Therefore, if you need a factory of data sources, from a cold source such
    * as a `java.io.File` from which you can open how many file handles you want,
    * you can use [[Observable.defer]] to build such a factory. Or you can share
    * the resulting observable by converting it into a
    * [[monix.reactive.observables.ConnectableObservable ConnectableObservable]]
    * by means of [[Observable!.multicast multicast]].
    *
    * @param in is the `Reader` to convert into an observable
    */
  def fromLinesReader(in: BufferedReader): Observable[String] =
    new builders.LinesReaderObservable(in)

  /** Given a `org.reactivestreams.Publisher`, converts it into a
    * Monix / Rx Observable.
    *
    * See the [[http://www.reactive-streams.org/ Reactive Streams]]
    * protocol that Monix implements.
    *
    * @see [[Observable.toReactive]] for converting an `Observable` to
    *      a reactive publisher.
    *
    * @param publisher is the `org.reactivestreams.Publisher` reference to
    *        wrap into an [[Observable]]
    */
  def fromReactivePublisher[A](publisher: RPublisher[A]): Observable[A] =
    new builders.ReactiveObservable[A](publisher, 0)

  /** Given a `org.reactivestreams.Publisher`, converts it into a
    * Monix / Rx Observable.
    *
    * See the [[http://www.reactive-streams.org/ Reactive Streams]]
    * protocol that Monix implements.
    *
    * @see [[Observable.toReactive]] for converting an `Observable` to
    *      a reactive publisher.
    *
    * @param publisher is the `org.reactivestreams.Publisher` reference to
    *        wrap into an [[Observable]]
    *
    * @param requestCount a strictly positive number, representing the size
    *        of the buffer used and the number of elements requested on each
    *        cycle when communicating demand, compliant with the
    *        reactive streams specification. If `Int.MaxValue` is given,
    *        then no back-pressuring logic will be applied (e.g. an unbounded
    *        buffer is used and the source has a license to stream as many
    *        events as it wants).
    */
  def fromReactivePublisher[A](publisher: RPublisher[A], requestCount: Int): Observable[A] =
    new builders.ReactiveObservable[A](publisher, requestCount)

  /** Transforms a non-strict [[monix.eval.Coeval Coeval]] value
    * into an `Observable` that emits a single element.
    */
  def coeval[A](value: Coeval[A]): Observable[A] =
    value match {
      case Coeval.Now(a) => Observable.now(a)
      case Coeval.Error(e) => Observable.raiseError(e)
      case other => Observable.eval(other.value())
    }

  /** Converts a `cats.Eval` value into an `Observable`
    * that emits a single element.
    *
    * {{{
    *   import cats.Eval
    *
    *   val value = Eval.always("Hello!")
    *
    *   Observable.fromEval(value)
    * }}}
    */
  def fromEval[A](fa: Eval[A]): Observable[A] =
    fa match {
      case cats.Now(v) => Observable.now(v)
      case _ => Observable.eval(fa.value)
    }

  /** Converts a Scala `Try` into an `Observable`.
    *
    * {{{
    *   import scala.util.Try
    *
    *   val value = Try(1)
    *   Observable.fromTry(value)
    * }}}
    */
  def fromTry[A](a: Try[A]): Observable[A] =
    a match {
      case Success(v) => Observable.now(v)
      case Failure(e) => Observable.raiseError(e)
    }

  /**
    * Builds an `Observable` instance out of a Scala `Either`.
    */
  def fromEither[E <: Throwable, A](a: Either[E, A]): Observable[A] =
    a match {
      case Right(v) => Observable.now(v)
      case Left(ex) => Observable.raiseError(ex)
    }

  /**
    * Builds a [[Observable]] instance out of a Scala `Either`.
    */
  def fromEither[E, A](f: E => Throwable)(a: Either[E, A]): Observable[A] =
    a match {
      case Right(v) => Observable.now(v)
      case Left(ex) => Observable.raiseError(f(ex))
    }

  /** Converts a Scala `Future` provided into an [[Observable]].
    *
    * If the created instance is a
    * [[monix.execution.CancelableFuture CancelableFuture]],
    * then it will be used for the returned
    * [[monix.execution.Cancelable Cancelable]] on `subscribe`.
    */
  def fromFuture[A](factory: => Future[A]): Observable[A] =
    new builders.FutureAsObservable(factory)

  /** Converts generic `F[_]` effects to `Observable`.
    *
    * Currently supported data types:
    *
    *  - [[monix.eval.Task]]
    *  - [[monix.eval.Coeval]]
    *  - [[scala.concurrent.Future]]
    *  - [[https://typelevel.org/cats-effect/datatypes/io.html cats.effect.IO]]
    *  - any [[https://typelevel.org/cats-effect/typeclasses/effect.html cats.effect.Effect]]
    *  - any [[https://typelevel.org/cats-effect/typeclasses/concurrent-effect.html cats.effect.ConcurrentEffect]]
    *
    *  Sample:
    *
    *  {{{
    *    import cats.implicits._
    *    import cats.effect.IO
    *
    *    val task = IO.sleep(5.seconds) *> IO(println("Hello!"))
    *
    *    Observable.fromTaskLike(task)
    *  }}}
    */
  def fromTaskLike[F[_], A](fa: F[A])(implicit F: TaskLike[F]): Observable[A] =
    fromTask(F.toTask(fa))

  /** Converts any `cats.effect.IO` value that implements into an
    * `Observable` that emits a single element.
    *
    * {{{
    *   import cats.effect.IO
    *
    *   val io = IO("Hello!")
    *
    *   Observable.fromIO(io)
    * }}}
    */
  def fromIO[A](fa: IO[A]): Observable[A] =
    fromTask(Task.fromIO(fa))

  /** Converts any [[monix.eval.Task Task]] into an [[Observable]].
    *
    * {{{
    *   val task = Task.eval("Hello!")
    *
    *   Observable.fromTask(task)
    * }}}
    */
  def fromTask[A](task: Task[A]): Observable[A] =
    new builders.TaskAsObservable(task)

  /** Alias for [[defer]]. */
  def suspend[A](fa: => Observable[A]): Observable[A] = defer(fa)

  /** Returns a new observable that creates a sequence from the
    * given factory on each subscription.
    */
  def defer[A](fa: => Observable[A]): Observable[A] =
    new builders.DeferObservable(fa _)

  /** Builds a new observable from a strict `head` and a lazily
    * evaluated tail.
    */
  def cons[A](head: A, tail: Observable[A]): Observable[A] =
    new builders.ConsObservable[A](head, tail)

  /** Creates a new observable from this observable and another given
    * observable by interleaving their items into a strictly alternating sequence.
    *
    * So the first item emitted by the new observable will be the item emitted by
    * `self`, the second item will be emitted by the other observable, and so forth;
    * when either `self` or `other` calls `onCompletes`, the items will then be
    * directly coming from the observable that has not completed; when `onError` is
    * called by either `self` or `other`, the new observable will call `onError` and halt.
    *
    * See [[merge]] for a more relaxed alternative that doesn't
    * emit items in strict alternating sequence.
    */
  def interleave2[A](oa1: Observable[A], oa2: Observable[A]): Observable[A] =
    new builders.Interleave2Observable(oa1, oa2)

  /** Creates an Observable that emits auto-incremented natural numbers
    * (longs) spaced by a given time interval. Starts from 0 with no
    * delay, after which it emits incremented numbers spaced by the
    * `period` of time. The given `period` of time acts as a fixed
    * delay between successive events.
    *
    * @param initialDelay is the delay to wait before emitting the first event
    * @param delay the time to wait between 2 successive events
    */
  def intervalWithFixedDelay(initialDelay: FiniteDuration, delay: FiniteDuration): Observable[Long] =
    new builders.IntervalFixedDelayObservable(initialDelay, delay)

  /** Creates an Observable that emits auto-incremented natural numbers
    * (longs) spaced by a given time interval. Starts from 0 with no
    * delay, after which it emits incremented numbers spaced by the
    * `period` of time. The given `period` of time acts as a fixed
    * delay between successive events.
    *
    * @param delay the delay between 2 successive events
    */
  def interval(delay: FiniteDuration): Observable[Long] =
    intervalWithFixedDelay(delay)

  /** Creates an Observable that emits auto-incremented natural numbers
    * (longs) spaced by a given time interval. Starts from 0 with no
    * delay, after which it emits incremented numbers spaced by the
    * `period` of time. The given `period` of time acts as a fixed
    * delay between successive events.
    *
    * @param delay the delay between 2 successive events
    */
  def intervalWithFixedDelay(delay: FiniteDuration): Observable[Long] =
    new builders.IntervalFixedDelayObservable(Duration.Zero, delay)

  /** Creates an Observable that emits auto-incremented natural numbers
    * (longs) at a fixed rate, as given by the specified `period`. The
    * time it takes to process an `onNext` event gets subtracted from
    * the specified `period` and thus the created observable tries to
    * emit events spaced by the given time interval, regardless of how
    * long the processing of `onNext` takes.
    *
    * @param period the period between 2 successive `onNext` events
    */
  def intervalAtFixedRate(period: FiniteDuration): Observable[Long] =
    new builders.IntervalFixedRateObservable(Duration.Zero, period)

  /** Creates an Observable that emits auto-incremented natural numbers
    * (longs) at a fixed rate, as given by the specified `period`. The
    * time it takes to process an `onNext` event gets subtracted from
    * the specified `period` and thus the created observable tries to
    * emit events spaced by the given time interval, regardless of how
    * long the processing of `onNext` takes.
    *
    * This version of the `intervalAtFixedRate` allows specifying an
    * `initialDelay` before events start being emitted.
    *
    * @param initialDelay is the initial delay before emitting the first event
    * @param period the period between 2 successive `onNext` events
    */
  def intervalAtFixedRate(initialDelay: FiniteDuration, period: FiniteDuration): Observable[Long] =
    new builders.IntervalFixedRateObservable(initialDelay, period)

  /** Creates an Observable that continuously emits the given ''item'' repeatedly.
    */
  def repeat[A](elems: A*): Observable[A] =
    new builders.RepeatObservable(elems: _*)

  /** Repeats the execution of the given `task`, emitting
    * the results indefinitely.
    */
  def repeatEval[A](task: => A): Observable[A] =
    new builders.RepeatEvalObservable(task)

  /** Repeats the evaluation of given effectful value, emitting
    * the results indefinitely.
    *
    */
  def repeatEvalF[F[_], A](fa: F[A])(implicit F: TaskLike[F]): Observable[A] =
    repeat(()).mapEval(_ => fa)(F)

  /** Creates an Observable that emits items in the given range.
    *
    * @param from the range start
    * @param until the range end
    * @param step increment step, either positive or negative
    */
  def range(from: Long, until: Long, step: Long = 1L): Observable[Long] =
    new builders.RangeObservable(from, until, step)

  /** Given an initial state and a generator function that produces the
    * next state and the next element in the sequence, creates an
    * observable that keeps generating elements produced by our
    * generator function.
    */
  def fromStateAction[S, A](f: S => (A, S))(seed: => S): Observable[A] =
    new builders.StateActionObservable(seed, f)

  /** Given an initial state and a generator function that produces the
    * next state and the next element in the sequence, creates an
    * observable that keeps generating elements produced by our
    * generator function.
    */
  def fromAsyncStateAction[S, A](f: S => Task[(A, S)])(seed: => S): Observable[A] =
    new builders.AsyncStateActionObservable(seed, f)

  /** Wraps this Observable into a `org.reactivestreams.Publisher`.
    * See the [[http://www.reactive-streams.org/ Reactive Streams]]
    * protocol that Monix implements.
    */
  def toReactive[A](source: Observable[A])(implicit s: Scheduler): RPublisher[A] =
    source.toReactivePublisher[A](s)

  /** Create an Observable that repeatedly emits the given `item`, until
    * the underlying Observer cancels.
    */
  def timerRepeated[A](initialDelay: FiniteDuration, period: FiniteDuration, unit: A): Observable[A] =
    new builders.RepeatedValueObservable[A](initialDelay, period, unit)

  /** Concatenates the given list of ''observables'' into a single observable.
    */
  def flatten[A](sources: Observable[A]*): Observable[A] =
    Observable.fromIterable(sources).concat

  /** Concatenates the given list of ''observables'' into a single
    * observable.  Delays errors until the end.
    */
  def flattenDelayError[A](sources: Observable[A]*): Observable[A] =
    Observable.fromIterable(sources).concatDelayErrors

  /** Merges the given list of ''observables'' into a single observable.
    */
  def merge[A](sources: Observable[A]*)
    (implicit os: OverflowStrategy[A] = OverflowStrategy.Default): Observable[A] =
    Observable.fromIterable(sources).mergeMap(identity)(os)

  /** Merges the given list of ''observables'' into a single observable.
    * Delays errors until the end.
    */
  def mergeDelayError[A](sources: Observable[A]*)
    (implicit os: OverflowStrategy[A] = OverflowStrategy.Default): Observable[A] =
    Observable.fromIterable(sources).mergeMapDelayErrors(identity)(os)

  /** Concatenates the given list of ''observables'' into a single
    * observable.
    */
  def concat[A](sources: Observable[A]*): Observable[A] =
    Observable.fromIterable(sources).concatMap[A](identity)

  /** Concatenates the given list of ''observables'' into a single observable.
    * Delays errors until the end.
    */
  def concatDelayError[A](sources: Observable[A]*): Observable[A] =
    Observable.fromIterable(sources).concatMapDelayErrors[A](identity)

  /** Given a sequence of observables, builds an observable
    * that emits the elements of the most recently emitted
    * observable.
    */
  def switch[A](sources: Observable[A]*): Observable[A] =
    Observable.fromIterable(sources).switch

  /** Creates a new observable from two observable sequences
    * by combining their items in pairs in a strict sequence.
    *
    * So the first item emitted by the new observable will be the result
    * of the function applied to the first items emitted by each of
    * the source observables; the second item emitted by the new observable
    * will be the result of the function applied to the second items
    * emitted by each of those observables; and so forth.
    *
    * See [[combineLatestMap2]] for a more relaxed alternative that doesn't
    * combine items in strict sequence.
    */
  def zip2[A1, A2](oa1: Observable[A1], oa2: Observable[A2]): Observable[(A1, A2)] =
    new builders.Zip2Observable[A1, A2, (A1, A2)](oa1, oa2)((a1, a2) => (a1, a2))

  /** Creates a new observable from two observable sequences
    * by combining their items in pairs in a strict sequence.
    *
    * So the first item emitted by the new observable will be the result
    * of the function applied to the first items emitted by each of
    * the source observables; the second item emitted by the new observable
    * will be the result of the function applied to the second items
    * emitted by each of those observables; and so forth.
    *
    * See [[combineLatestMap2]] for a more relaxed alternative that doesn't
    * combine items in strict sequence.
    *
    * @param f is the mapping function applied over the generated pairs
    */
  def zipMap2[A1, A2, R](oa1: Observable[A1], oa2: Observable[A2])(f: (A1, A2) => R): Observable[R] =
    new builders.Zip2Observable[A1, A2, R](oa1, oa2)(f)

  /** Creates a new observable from three observable sequences
    * by combining their items in pairs in a strict sequence.
    *
    * So the first item emitted by the new observable will be the result
    * of the function applied to the first items emitted by each of
    * the source observables; the second item emitted by the new observable
    * will be the result of the function applied to the second items
    * emitted by each of those observables; and so forth.
    *
    * See [[combineLatestMap3]] for a more relaxed alternative that doesn't
    * combine items in strict sequence.
    */
  def zip3[A1, A2, A3](oa1: Observable[A1], oa2: Observable[A2], oa3: Observable[A3]): Observable[(A1, A2, A3)] =
    new builders.Zip3Observable(oa1, oa2, oa3)((a1, a2, a3) => (a1, a2, a3))

  /** Creates a new observable from three observable sequences
    * by combining their items in pairs in a strict sequence.
    *
    * So the first item emitted by the new observable will be the result
    * of the function applied to the first items emitted by each of
    * the source observables; the second item emitted by the new observable
    * will be the result of the function applied to the second items
    * emitted by each of those observables; and so forth.
    *
    * See [[combineLatestMap3]] for a more relaxed alternative that doesn't
    * combine items in strict sequence.
    *
    * @param f is the mapping function applied over the generated pairs
    */
  def zipMap3[A1, A2, A3, R](oa1: Observable[A1], oa2: Observable[A2], oa3: Observable[A3])
    (f: (A1, A2, A3) => R): Observable[R] =
    new builders.Zip3Observable(oa1, oa2, oa3)(f)

  /** Creates a new observable from four observable sequences
    * by combining their items in pairs in a strict sequence.
    *
    * So the first item emitted by the new observable will be the result
    * of the function applied to the first items emitted by each of
    * the source observables; the second item emitted by the new observable
    * will be the result of the function applied to the second items
    * emitted by each of those observables; and so forth.
    *
    * See [[combineLatestMap4]] for a more relaxed alternative that doesn't
    * combine items in strict sequence.
    */
  def zip4[A1, A2, A3, A4]
  (oa1: Observable[A1], oa2: Observable[A2], oa3: Observable[A3], oa4: Observable[A4]): Observable[(A1, A2, A3, A4)] =
    new builders.Zip4Observable(oa1, oa2, oa3, oa4)((a1, a2, a3, a4) => (a1, a2, a3, a4))

  /** Creates a new observable from four observable sequences
    * by combining their items in pairs in a strict sequence.
    *
    * So the first item emitted by the new observable will be the result
    * of the function applied to the first items emitted by each of
    * the source observables; the second item emitted by the new observable
    * will be the result of the function applied to the second items
    * emitted by each of those observables; and so forth.
    *
    * See [[combineLatestMap4]] for a more relaxed alternative that doesn't
    * combine items in strict sequence.
    *
    * @param f is the mapping function applied over the generated pairs
    */
  def zipMap4[A1, A2, A3, A4, R]
  (oa1: Observable[A1], oa2: Observable[A2], oa3: Observable[A3], oa4: Observable[A4])
    (f: (A1, A2, A3, A4) => R): Observable[R] =
    new builders.Zip4Observable(oa1, oa2, oa3, oa4)(f)

  /** Creates a new observable from five observable sequences
    * by combining their items in pairs in a strict sequence.
    *
    * So the first item emitted by the new observable will be the result
    * of the function applied to the first items emitted by each of
    * the source observables; the second item emitted by the new observable
    * will be the result of the function applied to the second items
    * emitted by each of those observables; and so forth.
    *
    * See [[combineLatestMap5]] for a more relaxed alternative that doesn't
    * combine items in strict sequence.
    */
  def zip5[A1, A2, A3, A4, A5](
    oa1: Observable[A1], oa2: Observable[A2], oa3: Observable[A3],
    oa4: Observable[A4], oa5: Observable[A5]): Observable[(A1, A2, A3, A4, A5)] =
    new builders.Zip5Observable(oa1, oa2, oa3, oa4, oa5)((a1, a2, a3, a4, a5) => (a1, a2, a3, a4, a5))

  /** Creates a new observable from five observable sequences
    * by combining their items in pairs in a strict sequence.
    *
    * So the first item emitted by the new observable will be the result
    * of the function applied to the first items emitted by each of
    * the source observables; the second item emitted by the new observable
    * will be the result of the function applied to the second items
    * emitted by each of those observables; and so forth.
    *
    * See [[combineLatestMap5]] for a more relaxed alternative that doesn't
    * combine items in strict sequence.
    *
    * @param f is the mapping function applied over the generated pairs
    */
  def zipMap5[A1, A2, A3, A4, A5, R]
  (oa1: Observable[A1], oa2: Observable[A2], oa3: Observable[A3],
    oa4: Observable[A4], oa5: Observable[A5])
    (f: (A1, A2, A3, A4, A5) => R): Observable[R] =
    new builders.Zip5Observable(oa1, oa2, oa3, oa4, oa5)(f)

  /** Creates a new observable from five observable sequences
    * by combining their items in pairs in a strict sequence.
    *
    * So the first item emitted by the new observable will be the result
    * of the function applied to the first items emitted by each of
    * the source observables; the second item emitted by the new observable
    * will be the result of the function applied to the second items
    * emitted by each of those observables; and so forth.
    *
    * See [[combineLatestMap5]] for a more relaxed alternative that doesn't
    * combine items in strict sequence.
    */
  def zip6[A1, A2, A3, A4, A5, A6](
    oa1: Observable[A1], oa2: Observable[A2], oa3: Observable[A3],
    oa4: Observable[A4], oa5: Observable[A5], oa6: Observable[A6]): Observable[(A1, A2, A3, A4, A5, A6)] =
    new builders.Zip6Observable(oa1, oa2, oa3, oa4, oa5, oa6)((a1, a2, a3, a4, a5, a6) => (a1, a2, a3, a4, a5, a6))

  /** Creates a new observable from five observable sequences
    * by combining their items in pairs in a strict sequence.
    *
    * So the first item emitted by the new observable will be the result
    * of the function applied to the first items emitted by each of
    * the source observables; the second item emitted by the new observable
    * will be the result of the function applied to the second items
    * emitted by each of those observables; and so forth.
    *
    * See [[combineLatestMap5]] for a more relaxed alternative that doesn't
    * combine items in strict sequence.
    *
    * @param f is the mapping function applied over the generated pairs
    */
  def zipMap6[A1, A2, A3, A4, A5, A6, R]
  (oa1: Observable[A1], oa2: Observable[A2], oa3: Observable[A3],
    oa4: Observable[A4], oa5: Observable[A5], oa6: Observable[A6])
    (f: (A1, A2, A3, A4, A5, A6) => R): Observable[R] =
    new builders.Zip6Observable(oa1, oa2, oa3, oa4, oa5, oa6)(f)

  /** Given an observable sequence, it [[Observable!.zip zips]] them
    * together returning a new observable that generates sequences.
    */
  def zipList[A](sources: Observable[A]*): Observable[Seq[A]] = {
    if (sources.isEmpty) Observable.empty
    else {
      val seed = sources.head.map(t => Vector(t))
      sources.tail.foldLeft(seed) { (acc, obs) =>
        acc.zipMap(obs)((seq, elem) => seq :+ elem)
      }
    }
  }

  /** Creates an observable that doesn't emit anything, but immediately
    * calls `onComplete` instead.
    */
  def empty[A]: Observable[A] =
    builders.EmptyObservable

  /** Creates a combined observable from 2 source observables.
    *
    * This operator behaves in a similar way to [[zip2]],
    * but while `zip` emits items only when all of the zipped source
    * observables have emitted a previously unzipped item, `combine`
    * emits an item whenever any of the source Observables emits an
    * item (so long as each of the source Observables has emitted at
    * least one item).
    */
  def combineLatest2[A1, A2](oa1: Observable[A1], oa2: Observable[A2]): Observable[(A1, A2)] =
    new builders.CombineLatest2Observable[A1, A2, (A1, A2)](oa1, oa2)((a1, a2) => (a1, a2))

  /** Creates a combined observable from 2 source observables.
    *
    * This operator behaves in a similar way to [[zipMap2]],
    * but while `zip` emits items only when all of the zipped source
    * observables have emitted a previously unzipped item, `combine`
    * emits an item whenever any of the source Observables emits an
    * item (so long as each of the source Observables has emitted at
    * least one item).
    */
  def combineLatestMap2[A1, A2, R](oa1: Observable[A1], oa2: Observable[A2])
    (f: (A1, A2) => R): Observable[R] =
    new builders.CombineLatest2Observable[A1, A2, R](oa1, oa2)(f)

  /** Creates a combined observable from 3 source observables.
    *
    * This operator behaves in a similar way to [[zip3]],
    * but while `zip` emits items only when all of the zipped source
    * observables have emitted a previously unzipped item, `combine`
    * emits an item whenever any of the source Observables emits an
    * item (so long as each of the source Observables has emitted at
    * least one item).
    */
  def combineLatest3[A1, A2, A3](
    oa1: Observable[A1], oa2: Observable[A2], oa3: Observable[A3]): Observable[(A1, A2, A3)] =
    new builders.CombineLatest3Observable(oa1, oa2, oa3)((a1, a2, a3) => (a1, a2, a3))

  /** Creates a combined observable from 3 source observables.
    *
    * This operator behaves in a similar way to [[zipMap3]],
    * but while `zip` emits items only when all of the zipped source
    * observables have emitted a previously unzipped item, `combine`
    * emits an item whenever any of the source Observables emits an
    * item (so long as each of the source Observables has emitted at
    * least one item).
    */
  def combineLatestMap3[A1, A2, A3, R](a1: Observable[A1], a2: Observable[A2], a3: Observable[A3])
    (f: (A1, A2, A3) => R): Observable[R] =
    new builders.CombineLatest3Observable[A1, A2, A3, R](a1, a2, a3)(f)

  /** Creates a combined observable from 4 source observables.
    *
    * This operator behaves in a similar way to [[zip4]],
    * but while `zip` emits items only when all of the zipped source
    * observables have emitted a previously unzipped item, `combine`
    * emits an item whenever any of the source Observables emits an
    * item (so long as each of the source Observables has emitted at
    * least one item).
    */
  def combineLatest4[A1, A2, A3, A4](
    oa1: Observable[A1], oa2: Observable[A2], oa3: Observable[A3],
    oa4: Observable[A4]): Observable[(A1, A2, A3, A4)] =
    new builders.CombineLatest4Observable(oa1, oa2, oa3, oa4)((a1, a2, a3, a4) => (a1, a2, a3, a4))

  /** Creates a combined observable from 4 source observables.
    *
    * This operator behaves in a similar way to [[zipMap4]],
    * but while `zip` emits items only when all of the zipped source
    * observables have emitted a previously unzipped item, `combine`
    * emits an item whenever any of the source Observables emits an
    * item (so long as each of the source Observables has emitted at
    * least one item).
    */
  def combineLatestMap4[A1, A2, A3, A4, R]
  (a1: Observable[A1], a2: Observable[A2], a3: Observable[A3], a4: Observable[A4])
    (f: (A1, A2, A3, A4) => R): Observable[R] =
    new builders.CombineLatest4Observable[A1, A2, A3, A4, R](a1, a2, a3, a4)(f)

  /** Creates a combined observable from 5 source observables.
    *
    * This operator behaves in a similar way to [[zip5]],
    * but while `zip` emits items only when all of the zipped source
    * observables have emitted a previously unzipped item, `combine`
    * emits an item whenever any of the source Observables emits an
    * item (so long as each of the source Observables has emitted at
    * least one item).
    */
  def combineLatest5[A1, A2, A3, A4, A5](
    oa1: Observable[A1], oa2: Observable[A2], oa3: Observable[A3],
    oa4: Observable[A4], oa5: Observable[A5]): Observable[(A1, A2, A3, A4, A5)] =
    new builders.CombineLatest5Observable(oa1, oa2, oa3, oa4, oa5)((a1, a2, a3, a4, a5) => (a1, a2, a3, a4, a5))

  /** Creates a combined observable from 5 source observables.
    *
    * This operator behaves in a similar way to [[zipMap5]],
    * but while `zip` emits items only when all of the zipped source
    * observables have emitted a previously unzipped item, `combine`
    * emits an item whenever any of the source Observables emits an
    * item (so long as each of the source Observables has emitted at
    * least one item).
    */
  def combineLatestMap5[A1, A2, A3, A4, A5, R]
  (a1: Observable[A1], a2: Observable[A2], a3: Observable[A3], a4: Observable[A4], a5: Observable[A5])
    (f: (A1, A2, A3, A4, A5) => R): Observable[R] =
    new builders.CombineLatest5Observable[A1, A2, A3, A4, A5, R](a1, a2, a3, a4, a5)(f)

  /** Creates a combined observable from 6 source observables.
    *
    * This operator behaves in a similar way to [[zip6]],
    * but while `zip` emits items only when all of the zipped source
    * observables have emitted a previously unzipped item, `combine`
    * emits an item whenever any of the source Observables emits an
    * item (so long as each of the source Observables has emitted at
    * least one item).
    */
  def combineLatest6[A1, A2, A3, A4, A5, A6](
    oa1: Observable[A1], oa2: Observable[A2], oa3: Observable[A3],
    oa4: Observable[A4], oa5: Observable[A5], oa6: Observable[A6]): Observable[(A1, A2, A3, A4, A5, A6)] =
    new builders.CombineLatest6Observable(oa1, oa2, oa3, oa4, oa5, oa6)((a1, a2, a3, a4, a5, a6) => (a1, a2, a3, a4, a5, a6))

  /** Creates a combined observable from 6 source observables.
    *
    * This operator behaves in a similar way to [[zipMap6]],
    * but while `zip` emits items only when all of the zipped source
    * observables have emitted a previously unzipped item, `combine`
    * emits an item whenever any of the source Observables emits an
    * item (so long as each of the source Observables has emitted at
    * least one item).
    */
  def combineLatestMap6[A1, A2, A3, A4, A5, A6, R]
  (a1: Observable[A1], a2: Observable[A2], a3: Observable[A3],
    a4: Observable[A4], a5: Observable[A5], a6: Observable[A6])
    (f: (A1, A2, A3, A4, A5, A6) => R): Observable[R] =
    new builders.CombineLatest6Observable[A1, A2, A3, A4, A5, A6, R](a1, a2, a3, a4, a5, a6)(f)

  /** Given an observable sequence, it combines them together
    * (using [[combineLatestMap2 combineLatest]])
    * returning a new observable that generates sequences.
    */
  def combineLatestList[A](sources: Observable[A]*): Observable[Seq[A]] = {
    if (sources.isEmpty) Observable.empty
    else {
      val seed = sources.head.map(t => Vector(t))
      sources.tail.foldLeft(seed) { (acc, obs) =>
        acc.combineLatestMap(obs) { (seq, elem) => seq :+ elem }
      }
    }
  }

  /** Given a list of source Observables, emits all of the items from
    * the first of these Observables to emit an item or to complete,
    * and cancel the rest.
    */
  def firstStartedOf[A](source: Observable[A]*): Observable[A] =
    new builders.FirstStartedObservable(source: _*)

  /** Implicit type class instances for [[Observable]]. */
  implicit val catsInstances: CatsInstances =
    new CatsInstances

  /** Cats instances for [[Observable]]. */
  class CatsInstances extends MonadError[Observable, Throwable]
    with MonoidK[Observable]
    with CoflatMap[Observable]
    with TaskLift[Observable] {

    override val unit: Observable[Unit] =
      Observable.now(())
    override def pure[A](a: A): Observable[A] =
      Observable.now(a)
    override def combineK[A](x: Observable[A], y: Observable[A]): Observable[A] =
      x ++ y
    override def flatMap[A, B](fa: Observable[A])(f: (A) => Observable[B]): Observable[B] =
      fa.flatMap(f)
    override def flatten[A](ffa: Observable[Observable[A]]): Observable[A] =
      ffa.flatten
    override def tailRecM[A, B](a: A)(f: (A) => Observable[Either[A, B]]): Observable[B] =
      Observable.tailRecM(a)(f)
    override def coflatMap[A, B](fa: Observable[A])(f: (Observable[A]) => B): Observable[B] =
      Observable.eval(f(fa))
    override def ap[A, B](ff: Observable[(A) => B])(fa: Observable[A]): Observable[B] =
      for (f <- ff; a <- fa) yield f(a)
    override def map2[A, B, Z](fa: Observable[A], fb: Observable[B])(f: (A, B) => Z): Observable[Z] =
      for (a <- fa; b <- fb) yield f(a, b)
    override def map[A, B](fa: Observable[A])(f: (A) => B): Observable[B] =
      fa.map(f)
    override def raiseError[A](e: Throwable): Observable[A] =
      Observable.raiseError(e)
    override def handleError[A](fa: Observable[A])(f: (Throwable) => A): Observable[A] =
      fa.onErrorHandle(f)
    override def handleErrorWith[A](fa: Observable[A])(f: (Throwable) => Observable[A]): Observable[A] =
      fa.onErrorHandleWith(f)
    override def recover[A](fa: Observable[A])(pf: PartialFunction[Throwable, A]): Observable[A] =
      fa.onErrorRecover(pf)
    override def recoverWith[A](fa: Observable[A])(pf: PartialFunction[Throwable, Observable[A]]): Observable[A] =
      fa.onErrorRecoverWith(pf)
    override def empty[A]: Observable[A] =
      Observable.empty[A]
    override def taskLift[A](task: Task[A]): Observable[A] =
      Observable.fromTask(task)
  }

  /** [[cats.NonEmptyParallel]] instance for [[Observable]]. */
  implicit val observableNonEmptyParallel: NonEmptyParallel[Observable, CombineObservable.Type] =
    new NonEmptyParallel[Observable, CombineObservable.Type] {
      import CombineObservable.unwrap
      import CombineObservable.{apply => wrap}

      override def flatMap: FlatMap[Observable] = implicitly[FlatMap[Observable]]
      override def apply: Apply[CombineObservable.Type] = CombineObservable.combineObservableApplicative

      override val sequential = new (CombineObservable.Type ~> Observable) {
        def apply[A](fa: CombineObservable.Type[A]): Observable[A] = unwrap(fa)
      }
      override val parallel = new (Observable ~> CombineObservable.Type) {
        def apply[A](fa: Observable[A]): CombineObservable.Type[A] = wrap(fa)
      }
  }

  // -- DEPRECATIONS

  /** DEPRECATED — please use [[Observable!.executeAsync .executeAsync]].
    *
    * The reason for the deprecation is the repurposing of the word "fork"
    * in [[monix.eval.Task Task]].
    */
  @deprecated("Please use Observable!.executeAsync", "3.0.0")
  def fork[A](fa: Observable[A]): Observable[A] = {
    // $COVERAGE-OFF$
    fa.executeAsync
    // $COVERAGE-ON$
  }

  /** DEPRECATED — please use [[Observable!.executeOn .executeOn]].
    *
    * The reason for the deprecation is the repurposing of the word "fork"
    * in [[monix.eval.Task Task]].
    */
  @deprecated("Please use Observable!.executeOn", "3.0.0")
  def fork[A](fa: Observable[A], scheduler: Scheduler): Observable[A] =
    fa.executeOn(scheduler)

  implicit final class DeprecatedExtensions[A](val self: Observable[A]) extends AnyVal {
    /** DEPRECATED - renamed to [[Observable.executeAsync executeAsync]].
      *
      * The reason for the deprecation is the repurposing of the word "fork"
      * in [[monix.eval.Task Task]].
      */
    @deprecated("Renamed to Observable!.executeAsync", "3.0.0")
    def executeWithFork: Observable[A] = {
      // $COVERAGE-OFF$
      self.executeAsync
      // $COVERAGE-ON$
    }
    /** DEPRECATED - renamed to [[Observable.delayExecution delayExecution]].
      *
      * The reason for the deprecation is making the name more consistent
      * with [[monix.eval.Task Task]].
      */
    @deprecated("Renamed to Observable!.delayExecution", "3.0.0")
    def delaySubscription(timespan: FiniteDuration): Observable[A] =
      self.delayExecution(timespan)
    /** DEPRECATED - renamed to [[Observable.delayExecutionWith delayExecutionWith]].
      *
      * The reason for the deprecation is making the name more consistent
      * with [[Observable.delayExecution delayExecution]].
      */
    @deprecated("Renamed to Observable!.delayExecutionWith", "3.0.0")
    def delaySubscriptionWith(trigger: Observable[Any]): Observable[A] =
      self.delayExecutionWith(trigger)
  }
}
