/*
 * Decompiled with CFR 0.152.
 */
package io.moquette.broker;

import io.moquette.broker.BrokerConfiguration;
import io.moquette.broker.DebugUtils;
import io.moquette.broker.InflightResender;
import io.moquette.broker.NettyUtils;
import io.moquette.broker.PostOffice;
import io.moquette.broker.Session;
import io.moquette.broker.SessionCorruptedException;
import io.moquette.broker.SessionRegistry;
import io.moquette.broker.security.IAuthenticator;
import io.moquette.broker.subscriptions.Topic;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufHolder;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelPipeline;
import io.netty.handler.codec.mqtt.MqttConnAckMessage;
import io.netty.handler.codec.mqtt.MqttConnectMessage;
import io.netty.handler.codec.mqtt.MqttConnectPayload;
import io.netty.handler.codec.mqtt.MqttConnectReturnCode;
import io.netty.handler.codec.mqtt.MqttFixedHeader;
import io.netty.handler.codec.mqtt.MqttMessage;
import io.netty.handler.codec.mqtt.MqttMessageBuilders;
import io.netty.handler.codec.mqtt.MqttMessageIdVariableHeader;
import io.netty.handler.codec.mqtt.MqttMessageType;
import io.netty.handler.codec.mqtt.MqttPubAckMessage;
import io.netty.handler.codec.mqtt.MqttPublishMessage;
import io.netty.handler.codec.mqtt.MqttPublishVariableHeader;
import io.netty.handler.codec.mqtt.MqttQoS;
import io.netty.handler.codec.mqtt.MqttSubAckMessage;
import io.netty.handler.codec.mqtt.MqttSubscribeMessage;
import io.netty.handler.codec.mqtt.MqttUnsubAckMessage;
import io.netty.handler.codec.mqtt.MqttUnsubscribeMessage;
import io.netty.handler.codec.mqtt.MqttVersion;
import io.netty.handler.timeout.IdleStateHandler;
import io.netty.util.concurrent.GenericFutureListener;
import java.net.InetSocketAddress;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class MQTTConnection {
    private static final Logger LOG = LoggerFactory.getLogger(MQTTConnection.class);
    final Channel channel;
    private final BrokerConfiguration brokerConfig;
    private final IAuthenticator authenticator;
    private final SessionRegistry sessionRegistry;
    private final PostOffice postOffice;
    private volatile boolean connected;
    private final AtomicInteger lastPacketId = new AtomicInteger(0);
    private Session bindedSession;

    MQTTConnection(Channel channel, BrokerConfiguration brokerConfig, IAuthenticator authenticator, SessionRegistry sessionRegistry, PostOffice postOffice) {
        this.channel = channel;
        this.brokerConfig = brokerConfig;
        this.authenticator = authenticator;
        this.sessionRegistry = sessionRegistry;
        this.postOffice = postOffice;
        this.connected = false;
    }

    void handleMessage(MqttMessage msg) {
        MqttMessageType messageType = msg.fixedHeader().messageType();
        LOG.debug("Received MQTT message, type: {}", (Object)messageType);
        switch (messageType) {
            case CONNECT: {
                this.processConnect((MqttConnectMessage)msg);
                break;
            }
            case SUBSCRIBE: {
                this.processSubscribe((MqttSubscribeMessage)msg);
                break;
            }
            case UNSUBSCRIBE: {
                this.processUnsubscribe((MqttUnsubscribeMessage)msg);
                break;
            }
            case PUBLISH: {
                this.processPublish((MqttPublishMessage)msg);
                break;
            }
            case PUBREC: {
                this.processPubRec(msg);
                break;
            }
            case PUBCOMP: {
                this.processPubComp(msg);
                break;
            }
            case PUBREL: {
                this.processPubRel(msg);
                break;
            }
            case DISCONNECT: {
                this.processDisconnect(msg);
                break;
            }
            case PUBACK: {
                this.processPubAck(msg);
                break;
            }
            case PINGREQ: {
                MqttFixedHeader pingHeader = new MqttFixedHeader(MqttMessageType.PINGRESP, false, MqttQoS.AT_MOST_ONCE, false, 0);
                MqttMessage pingResp = new MqttMessage(pingHeader);
                this.channel.writeAndFlush((Object)pingResp).addListener((GenericFutureListener)ChannelFutureListener.CLOSE_ON_FAILURE);
                break;
            }
            default: {
                LOG.error("Unknown MessageType: {}", (Object)messageType);
            }
        }
    }

    private void processPubComp(MqttMessage msg) {
        int messageID = ((MqttMessageIdVariableHeader)msg.variableHeader()).messageId();
        this.postOffice.routeCommand(this.bindedSession.getClientID(), "PUBCOMP", () -> {
            this.bindedSession.processPubComp(messageID);
            return this.bindedSession.getClientID();
        });
    }

    private void processPubRec(MqttMessage msg) {
        int messageID = ((MqttMessageIdVariableHeader)msg.variableHeader()).messageId();
        this.postOffice.routeCommand(this.bindedSession.getClientID(), "PUBREC", () -> {
            this.bindedSession.processPubRec(messageID);
            return null;
        });
    }

    static MqttMessage pubrel(int messageID) {
        MqttFixedHeader pubRelHeader = new MqttFixedHeader(MqttMessageType.PUBREL, false, MqttQoS.AT_LEAST_ONCE, false, 0);
        return new MqttMessage(pubRelHeader, (Object)MqttMessageIdVariableHeader.from((int)messageID));
    }

    private void processPubAck(MqttMessage msg) {
        int messageID = ((MqttMessageIdVariableHeader)msg.variableHeader()).messageId();
        String clientId = this.getClientId();
        this.postOffice.routeCommand(clientId, "PUB ACK", () -> {
            this.bindedSession.pubAckReceived(messageID);
            return null;
        });
    }

    PostOffice.RouteResult processConnect(MqttConnectMessage msg) {
        MqttConnectPayload payload = msg.payload();
        String clientId = payload.clientIdentifier();
        String username = payload.userName();
        LOG.trace("Processing CONNECT message. CId: {} username: {}", (Object)clientId, (Object)username);
        if (this.isNotProtocolVersion(msg, MqttVersion.MQTT_3_1) && this.isNotProtocolVersion(msg, MqttVersion.MQTT_3_1_1)) {
            LOG.warn("MQTT protocol version is not valid. CId: {}", (Object)clientId);
            this.abortConnection(MqttConnectReturnCode.CONNECTION_REFUSED_UNACCEPTABLE_PROTOCOL_VERSION);
            return PostOffice.RouteResult.failed(clientId);
        }
        boolean cleanSession = msg.variableHeader().isCleanSession();
        if (clientId == null || clientId.length() == 0) {
            if (!this.brokerConfig.isAllowZeroByteClientId()) {
                LOG.info("Broker doesn't permit MQTT empty client ID. Username: {}", (Object)username);
                this.abortConnection(MqttConnectReturnCode.CONNECTION_REFUSED_IDENTIFIER_REJECTED);
                return PostOffice.RouteResult.failed(clientId);
            }
            if (!cleanSession) {
                LOG.info("MQTT client ID cannot be empty for persistent session. Username: {}", (Object)username);
                this.abortConnection(MqttConnectReturnCode.CONNECTION_REFUSED_IDENTIFIER_REJECTED);
                return PostOffice.RouteResult.failed(clientId);
            }
            clientId = UUID.randomUUID().toString().replace("-", "");
            LOG.debug("Client has connected with integration generated id: {}, username: {}", (Object)clientId, (Object)username);
        }
        if (!this.login(msg, clientId)) {
            this.abortConnection(MqttConnectReturnCode.CONNECTION_REFUSED_BAD_USER_NAME_OR_PASSWORD);
            this.channel.close().addListener((GenericFutureListener)ChannelFutureListener.CLOSE_ON_FAILURE);
            return PostOffice.RouteResult.failed(clientId);
        }
        String sessionId = clientId;
        return this.postOffice.routeCommand(clientId, "CONN", () -> {
            this.executeConnect(msg, sessionId);
            return null;
        });
    }

    private void executeConnect(final MqttConnectMessage msg, String clientId) {
        SessionRegistry.SessionCreationResult result;
        try {
            LOG.trace("Binding MQTTConnection to session");
            result = this.sessionRegistry.createOrReopenSession(msg, clientId, this.getUsername());
            result.session.bind(this);
            this.bindedSession = result.session;
        }
        catch (SessionCorruptedException scex) {
            LOG.warn("MQTT session for client ID {} cannot be created", (Object)clientId);
            this.abortConnection(MqttConnectReturnCode.CONNECTION_REFUSED_SERVER_UNAVAILABLE);
            return;
        }
        boolean msgCleanSessionFlag = msg.variableHeader().isCleanSession();
        boolean isSessionAlreadyPresent = !msgCleanSessionFlag && result.alreadyStored;
        final String clientIdUsed = clientId;
        MqttConnAckMessage ackMessage = MqttMessageBuilders.connAck().returnCode(MqttConnectReturnCode.CONNECTION_ACCEPTED).sessionPresent(isSessionAlreadyPresent).build();
        this.channel.writeAndFlush((Object)ackMessage).addListener((GenericFutureListener)new ChannelFutureListener(){

            public void operationComplete(ChannelFuture future) throws Exception {
                if (future.isSuccess()) {
                    LOG.trace("CONNACK sent, channel: {}", (Object)MQTTConnection.this.channel);
                    if (!result.session.completeConnection()) {
                        MqttMessage disconnectMsg = MqttMessageBuilders.disconnect().build();
                        MQTTConnection.this.channel.writeAndFlush((Object)disconnectMsg).addListener((GenericFutureListener)CLOSE);
                        LOG.warn("CONNACK is sent but the session created can't transition in CONNECTED state");
                    } else {
                        NettyUtils.clientID(MQTTConnection.this.channel, clientIdUsed);
                        MQTTConnection.this.connected = true;
                        if (result.mode == SessionRegistry.CreationModeEnum.REOPEN_EXISTING) {
                            result.session.sendQueuedMessagesWhileOffline();
                        }
                        MQTTConnection.this.initializeKeepAliveTimeout(MQTTConnection.this.channel, msg, clientIdUsed);
                        MQTTConnection.this.setupInflightResender(MQTTConnection.this.channel);
                        MQTTConnection.this.postOffice.dispatchConnection(msg);
                        LOG.trace("dispatch connection: {}", (Object)msg);
                    }
                } else {
                    MQTTConnection.this.bindedSession.disconnect();
                    MQTTConnection.this.sessionRegistry.remove(MQTTConnection.this.bindedSession.getClientID());
                    LOG.error("CONNACK send failed, cleanup session and close the connection", future.cause());
                    MQTTConnection.this.channel.close();
                }
            }
        });
    }

    private void setupInflightResender(Channel channel) {
        channel.pipeline().addFirst("inflightResender", (ChannelHandler)new InflightResender(5000L, TimeUnit.MILLISECONDS));
    }

    private void initializeKeepAliveTimeout(Channel channel, MqttConnectMessage msg, String clientId) {
        int keepAlive = msg.variableHeader().keepAliveTimeSeconds();
        NettyUtils.keepAlive(channel, keepAlive);
        NettyUtils.cleanSession(channel, msg.variableHeader().isCleanSession());
        NettyUtils.clientID(channel, clientId);
        int idleTime = Math.round((float)keepAlive * 1.5f);
        this.setIdleTime(channel.pipeline(), idleTime);
        LOG.debug("Connection has been configured CId={}, keepAlive={}, removeTemporaryQoS2={}, idleTime={}", new Object[]{clientId, keepAlive, msg.variableHeader().isCleanSession(), idleTime});
    }

    private void setIdleTime(ChannelPipeline pipeline, int idleTime) {
        if (pipeline.names().contains("idleStateHandler")) {
            pipeline.remove("idleStateHandler");
        }
        pipeline.addFirst("idleStateHandler", (ChannelHandler)new IdleStateHandler(idleTime, 0, 0));
    }

    private boolean isNotProtocolVersion(MqttConnectMessage msg, MqttVersion version) {
        return msg.variableHeader().version() != version.protocolLevel();
    }

    private void abortConnection(MqttConnectReturnCode returnCode) {
        MqttConnAckMessage badProto = MqttMessageBuilders.connAck().returnCode(returnCode).sessionPresent(false).build();
        this.channel.writeAndFlush((Object)badProto).addListener((GenericFutureListener)ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
        this.channel.close().addListener((GenericFutureListener)ChannelFutureListener.CLOSE_ON_FAILURE);
    }

    private boolean login(MqttConnectMessage msg, String clientId) {
        if (msg.variableHeader().hasUserName()) {
            byte[] pwd = null;
            if (msg.variableHeader().hasPassword()) {
                pwd = msg.payload().passwordInBytes();
            } else if (!this.brokerConfig.isAllowAnonymous()) {
                LOG.info("Client didn't supply any password and MQTT anonymous mode is disabled CId={}", (Object)clientId);
                return false;
            }
            String login = msg.payload().userName();
            if (!this.authenticator.checkValid(clientId, login, pwd)) {
                LOG.info("Authenticator has rejected the MQTT credentials CId={}, username={}", (Object)clientId, (Object)login);
                return false;
            }
            NettyUtils.userName(this.channel, login);
        } else if (!this.brokerConfig.isAllowAnonymous()) {
            LOG.info("Client didn't supply any credentials and MQTT anonymous mode is disabled. CId={}", (Object)clientId);
            return false;
        }
        return true;
    }

    void handleConnectionLost() {
        String clientID = NettyUtils.clientID(this.channel);
        if (clientID == null || clientID.isEmpty()) {
            return;
        }
        LOG.debug("Notifying connection lost event");
        this.postOffice.routeCommand(clientID, "CONN LOST", () -> {
            if (this.isBoundToSession() || this.isSessionUnbound()) {
                LOG.debug("Cleaning {}", (Object)clientID);
                this.processConnectionLost(clientID);
            } else {
                LOG.debug("NOT Cleaning {}, bound to other connection.", (Object)clientID);
            }
            return null;
        });
    }

    private void processConnectionLost(String clientID) {
        if (this.bindedSession.hasWill()) {
            this.postOffice.fireWill(this.bindedSession.getWill());
        }
        if (this.bindedSession.isClean()) {
            LOG.debug("Remove session for client {}", (Object)clientID);
            this.sessionRegistry.remove(this.bindedSession.getClientID());
        } else {
            this.bindedSession.disconnect();
        }
        this.connected = false;
        String userName = NettyUtils.userName(this.channel);
        this.postOffice.dispatchConnectionLost(clientID, userName);
        LOG.trace("dispatch disconnection: userName={}", (Object)userName);
    }

    boolean isConnected() {
        return this.connected;
    }

    void dropConnection() {
        this.channel.close().addListener((GenericFutureListener)ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
    }

    PostOffice.RouteResult processDisconnect(MqttMessage msg) {
        String clientID = NettyUtils.clientID(this.channel);
        LOG.trace("Start DISCONNECT");
        if (!this.connected) {
            LOG.info("DISCONNECT received on already closed connection");
            return PostOffice.RouteResult.success(clientID, CompletableFuture.completedFuture(null));
        }
        return this.postOffice.routeCommand(clientID, "DISCONN", () -> {
            if (!this.isBoundToSession()) {
                LOG.debug("NOT processing disconnect {}, not bound.", (Object)clientID);
                return null;
            }
            this.bindedSession.disconnect();
            this.connected = false;
            this.channel.close().addListener((GenericFutureListener)ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
            String userName = NettyUtils.userName(this.channel);
            this.postOffice.clientDisconnected(clientID, userName);
            LOG.trace("dispatch disconnection userName={}", (Object)userName);
            return null;
        });
    }

    PostOffice.RouteResult processSubscribe(MqttSubscribeMessage msg) {
        String clientID = NettyUtils.clientID(this.channel);
        if (!this.connected) {
            LOG.warn("SUBSCRIBE received on already closed connection");
            this.dropConnection();
            return PostOffice.RouteResult.success(clientID, CompletableFuture.completedFuture(null));
        }
        String username = NettyUtils.userName(this.channel);
        return this.postOffice.routeCommand(clientID, "SUB", () -> {
            if (this.isBoundToSession()) {
                this.postOffice.subscribeClientToTopics(msg, clientID, username, this);
            }
            return null;
        });
    }

    void sendSubAckMessage(int messageID, MqttSubAckMessage ackMessage) {
        LOG.trace("Sending SUBACK response messageId: {}", (Object)messageID);
        this.channel.writeAndFlush((Object)ackMessage).addListener((GenericFutureListener)ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
    }

    private void processUnsubscribe(MqttUnsubscribeMessage msg) {
        List topics = msg.payload().topics();
        String clientID = NettyUtils.clientID(this.channel);
        int messageId = msg.variableHeader().messageId();
        this.postOffice.routeCommand(clientID, "UNSUB", () -> {
            if (!this.isBoundToSession()) {
                return null;
            }
            LOG.trace("Processing UNSUBSCRIBE message. topics: {}", (Object)topics);
            this.postOffice.unsubscribe(topics, this, messageId);
            return null;
        });
    }

    void sendUnsubAckMessage(List<String> topics, String clientID, int messageID) {
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.UNSUBACK, false, MqttQoS.AT_MOST_ONCE, false, 0);
        MqttUnsubAckMessage ackMessage = new MqttUnsubAckMessage(fixedHeader, MqttMessageIdVariableHeader.from((int)messageID));
        LOG.trace("Sending UNSUBACK message. messageId: {}, topics: {}", (Object)messageID, topics);
        this.channel.writeAndFlush((Object)ackMessage).addListener((GenericFutureListener)ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
        LOG.trace("Client unsubscribed from topics <{}>", topics);
    }

    PostOffice.RouteResult processPublish(MqttPublishMessage msg) {
        MqttQoS qos = msg.fixedHeader().qosLevel();
        String username = NettyUtils.userName(this.channel);
        String topicName = msg.variableHeader().topicName();
        String clientId = this.getClientId();
        int messageID = msg.variableHeader().packetId();
        LOG.trace("Processing PUBLISH message, topic: {}, messageId: {}, qos: {}", new Object[]{topicName, messageID, qos});
        Topic topic = new Topic(topicName);
        if (!topic.isValid()) {
            LOG.debug("Drop connection because of invalid topic format");
            this.dropConnection();
        }
        msg.retain();
        switch (qos) {
            case AT_MOST_ONCE: {
                return this.postOffice.routeCommand(clientId, "PUB QoS0", () -> {
                    if (!this.isBoundToSession()) {
                        return null;
                    }
                    this.postOffice.receivedPublishQos0(topic, username, clientId, msg);
                    return null;
                }).ifFailed(() -> ((MqttPublishMessage)msg).release());
            }
            case AT_LEAST_ONCE: {
                return this.postOffice.routeCommand(clientId, "PUB QoS1", () -> {
                    if (!this.isBoundToSession()) {
                        return null;
                    }
                    this.postOffice.receivedPublishQos1(this, topic, username, messageID, msg);
                    return null;
                }).ifFailed(() -> ((MqttPublishMessage)msg).release());
            }
            case EXACTLY_ONCE: {
                PostOffice.RouteResult firstStepResult = this.postOffice.routeCommand(clientId, "PUB QoS2", () -> {
                    if (!this.isBoundToSession()) {
                        return null;
                    }
                    this.bindedSession.receivedPublishQos2(messageID, msg);
                    return null;
                });
                if (!firstStepResult.isSuccess()) {
                    msg.release();
                    LOG.trace("Failed to enqueue PUB QoS2 to session loop for  {}", (Object)clientId);
                    return firstStepResult;
                }
                firstStepResult.completableFuture().thenRun(() -> this.postOffice.receivedPublishQos2(this, msg, username).completableFuture());
                return firstStepResult;
            }
        }
        LOG.error("Unknown QoS-Type:{}", (Object)qos);
        return PostOffice.RouteResult.failed(clientId, "Unknown QoS-");
    }

    void sendPubRec(int messageID) {
        LOG.trace("sendPubRec invoked, messageID: {}", (Object)messageID);
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PUBREC, false, MqttQoS.AT_MOST_ONCE, false, 0);
        MqttPubAckMessage pubRecMessage = new MqttPubAckMessage(fixedHeader, MqttMessageIdVariableHeader.from((int)messageID));
        this.sendIfWritableElseDrop((MqttMessage)pubRecMessage);
    }

    private void processPubRel(MqttMessage msg) {
        int messageID = ((MqttMessageIdVariableHeader)msg.variableHeader()).messageId();
        this.postOffice.routeCommand(this.bindedSession.getClientID(), "PUBREL", () -> {
            this.executePubRel(messageID);
            return null;
        });
    }

    private void executePubRel(int messageID) {
        this.bindedSession.receivedPubRelQos2(messageID);
        this.sendPubCompMessage(messageID);
    }

    void sendPublish(MqttPublishMessage publishMsg) {
        int packetId = publishMsg.variableHeader().packetId();
        String topicName = publishMsg.variableHeader().topicName();
        MqttQoS qos = publishMsg.fixedHeader().qosLevel();
        if (LOG.isTraceEnabled()) {
            LOG.trace("Sending PUBLISH({}) message. MessageId={}, topic={}, payload={}", new Object[]{qos, packetId, topicName, DebugUtils.payload2Str(publishMsg.payload())});
        } else {
            LOG.debug("Sending PUBLISH({}) message. MessageId={}, topic={}", new Object[]{qos, packetId, topicName});
        }
        this.sendIfWritableElseDrop((MqttMessage)publishMsg);
    }

    void sendIfWritableElseDrop(MqttMessage msg) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("OUT {}", (Object)msg.fixedHeader().messageType());
        }
        if (this.channel.isWritable()) {
            MqttMessage retainedDup = msg;
            if (msg instanceof ByteBufHolder) {
                retainedDup = ((ByteBufHolder)msg).retainedDuplicate();
            }
            ChannelFuture channelFuture = this.brokerConfig.isImmediateBufferFlush() ? this.channel.writeAndFlush((Object)retainedDup) : this.channel.write((Object)retainedDup);
            channelFuture.addListener((GenericFutureListener)ChannelFutureListener.FIRE_EXCEPTION_ON_FAILURE);
        }
    }

    public void writabilityChanged() {
        if (this.channel.isWritable()) {
            LOG.debug("Channel is again writable");
            this.bindedSession.writabilityChanged();
        }
    }

    void sendPubAck(int messageID) {
        LOG.trace("sendPubAck for messageID: {}", (Object)messageID);
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PUBACK, false, MqttQoS.AT_MOST_ONCE, false, 0);
        MqttPubAckMessage pubAckMessage = new MqttPubAckMessage(fixedHeader, MqttMessageIdVariableHeader.from((int)messageID));
        this.sendIfWritableElseDrop((MqttMessage)pubAckMessage);
    }

    private void sendPubCompMessage(int messageID) {
        LOG.trace("Sending PUBCOMP message messageId: {}", (Object)messageID);
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PUBCOMP, false, MqttQoS.AT_MOST_ONCE, false, 0);
        MqttMessage pubCompMessage = new MqttMessage(fixedHeader, (Object)MqttMessageIdVariableHeader.from((int)messageID));
        this.sendIfWritableElseDrop(pubCompMessage);
    }

    String getClientId() {
        return NettyUtils.clientID(this.channel);
    }

    String getUsername() {
        return NettyUtils.userName(this.channel);
    }

    public void sendPublishWithPacketId(Topic topic, MqttQoS qos, ByteBuf payload, boolean retained) {
        int packetId = this.nextPacketId();
        MqttPublishMessage publishMsg = MQTTConnection.createPublishMessage(topic.toString(), qos, payload, packetId, retained);
        this.sendPublish(publishMsg);
    }

    void sendPublishQos0(Topic topic, MqttQoS qos, ByteBuf payload, boolean retained) {
        MqttPublishMessage publishMsg = MQTTConnection.createPublishMessage(topic.toString(), qos, payload, 0, retained);
        this.sendPublish(publishMsg);
    }

    static MqttPublishMessage createRetainedPublishMessage(String topic, MqttQoS qos, ByteBuf message) {
        return MQTTConnection.createPublishMessage(topic, qos, message, 0, true);
    }

    static MqttPublishMessage createNonRetainedPublishMessage(String topic, MqttQoS qos, ByteBuf message) {
        return MQTTConnection.createPublishMessage(topic, qos, message, 0, false);
    }

    static MqttPublishMessage createRetainedPublishMessage(String topic, MqttQoS qos, ByteBuf message, int messageId) {
        return MQTTConnection.createPublishMessage(topic, qos, message, messageId, true);
    }

    static MqttPublishMessage createNotRetainedPublishMessage(String topic, MqttQoS qos, ByteBuf message, int messageId) {
        return MQTTConnection.createPublishMessage(topic, qos, message, messageId, false);
    }

    private static MqttPublishMessage createPublishMessage(String topic, MqttQoS qos, ByteBuf message, int messageId, boolean retained) {
        MqttFixedHeader fixedHeader = new MqttFixedHeader(MqttMessageType.PUBLISH, false, qos, retained, 0);
        MqttPublishVariableHeader varHeader = new MqttPublishVariableHeader(topic, messageId);
        return new MqttPublishMessage(fixedHeader, varHeader, message);
    }

    public void resendNotAckedPublishes() {
        this.bindedSession.resendInflightNotAcked();
    }

    int nextPacketId() {
        return this.lastPacketId.updateAndGet(v -> v == 65535 ? 1 : v + 1);
    }

    public String toString() {
        return "MQTTConnection{channel=" + this.channel + ", connected=" + this.connected + '}';
    }

    InetSocketAddress remoteAddress() {
        return (InetSocketAddress)this.channel.remoteAddress();
    }

    public void readCompleted() {
        LOG.debug("readCompleted client CId: {}", (Object)this.getClientId());
        if (this.getClientId() != null) {
            this.bindedSession.flushAllQueuedMessages();
        }
    }

    public void flush() {
        this.channel.flush();
    }

    private boolean isBoundToSession() {
        return this.bindedSession != null && this.bindedSession.isBoundTo(this);
    }

    private boolean isSessionUnbound() {
        return this.bindedSession != null && this.bindedSession.isBoundTo(null);
    }

    public void bindSession(Session session) {
        this.bindedSession = session;
    }
}

