package nashid.verify.sdk.mrtd2.activity

import android.content.Intent
import android.graphics.Bitmap
import android.os.Bundle
import android.util.Log
import android.util.TypedValue
import android.view.animation.AnimationUtils
import android.widget.LinearLayout
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.app.AlertDialog
import androidx.lifecycle.ViewModelProvider
import nashid.mv.liveness.LivenessMainActivity
import nashid.verify.sdk.OtherConstant
import nashid.verify.sdk.R
import nashid.verify.sdk.Utility
import nashid.verify.sdk.databinding.ActivitySkipNfcBinding
import nashid.verify.sdk.internetutility.ErrorUtility
import nashid.verify.sdk.mrtd2.activity.viewmodel.SkipNFCLivenessViewModel
import nashid.verify.sdk.mrtd2.rooted.RootedCheck
import nashid.verify.sdk.utility.PermissionAndLocationHelper

internal class SkipNFCLivenessActivity : BaseActivity() {
    private lateinit var binding: ActivitySkipNfcBinding
    private lateinit var viewModel: SkipNFCLivenessViewModel

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        // Set up data binding
        binding = ActivitySkipNfcBinding.inflate(layoutInflater)
        setContentView(binding.root)
        viewModel = ViewModelProvider(this)[SkipNFCLivenessViewModel::class.java]

        if (RootedCheck.getInstance().isRootedDevice(applicationContext)) {
            RootedCheck.getInstance().showRootedDeviceDialog(
                this,
                getString(R.string.root_dialog),
                getString(R.string.root_desc),
                getString(R.string.root_btn),
            )
        } else {
            initView()
        }
    }

    private fun initView() {
        setLayoutAndTextSize()
        handleIntentExtras()
        observeViewModel()
        viewModel.handleInternetConnectionData(isInternetAvailable)
    }

    private fun observeViewModel() {
        viewModel.getFinishActivity().observe(this) { handleFinishActivity(it) }
        viewModel.getIsLiveness().observe(this) { handleLiveness(it) }
        viewModel.getOpenActivity().observe(this) { handleOpenActivity(it) }
        viewModel.getInternetConnection().observe(this) { handleInternetConnection(it) }
        viewModel.getHandleLocationData().observe(this) { handleLocationData(it) }
        viewModel.getLivenessBitmap().observe(this) {
            it?.let { it1 ->
                setLivenessBitmap(
                    it1,
                )
            }
        }
        viewModel.getScanBitmap().observe(this) { it?.let { it1 -> setScanBitmap(it1) } }
        viewModel.getFaceMatchApiFail().observe(this) {
        }
    }

    private fun openAlertDialog(errorMessage: String) {
        if (errorMessage.isNotEmpty()) {
            AlertDialog
                .Builder(this)
                .setTitle(getString(R.string.face_error_title))
                .setMessage(errorMessage)
                .setCancelable(false)
                .setPositiveButton(getString(R.string.face_error_retry)) { dialog, which ->
                    viewModel.extractFacesFromPhoto()
                }.setNegativeButton(getString(R.string.face_error_cancel)) { dialog, which ->
                    dialog.dismiss()
                    viewModel.getImageMatchedResult("0")
                }.show()
        }
    }

    private fun setLivenessBitmap(bitmap: Bitmap) {
        binding.imgTest123.setImageBitmap(bitmap)
    }

    private fun setScanBitmap(bitmap: Bitmap) {
        binding.imgTest12.setImageBitmap(bitmap)
    }

    private fun handleFinishActivity(shouldFinish: Boolean) {
        if (shouldFinish) {
            finish()
            overridePendingTransition(android.R.anim.fade_in, android.R.anim.fade_out)
        }
    }

    private fun handleLiveness(isLiveness: Boolean) {
        if (isLiveness) {
            openLivenessScreen()
        }
    }

    private fun handleOpenActivity(activityClass: Class<*>?) {
        activityClass?.let {
            if (it.name.equals("SelectDocumentActivity", ignoreCase = true)) {
                openSelectDocumentScreen()
            } else {
                val intent = Intent(this, ComparisionSuccessful::class.java)
                intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK)
                startActivity(intent)
                overridePendingTransition(android.R.anim.fade_in, android.R.anim.fade_out)
                finish()
            }
        }
    }

    private fun handleInternetConnection(isAvailable: Boolean) {
        if (!isAvailable) {
            ErrorUtility
                .getInstance()
                .showNoInternetDialog(this, !this.isFinishing && !this.isDestroyed)
        }
    }

    private fun handleLocationData(callLocationData: Boolean) {
        if (callLocationData) {
            handleLocationData()
        }
    }

    private fun handleIntentExtras() {
        if (intent.hasExtra(getString(R.string.doc_key))) {
            val selectedDoc = intent.getStringExtra(getString(R.string.doc_key))
            if (selectedDoc != null) {
                viewModel.setSelectedDoc(selectedDoc)
            }
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        ErrorUtility.getInstance().unregisterConnectivityManager(this)
    }

    private fun setLayoutAndTextSize() {
        val textSizeConverter = TextSizeConverter(applicationContext)
        textSizeConverter.changeStatusBarColor(this)
        val layoutParams2 = binding.imgLoader.layoutParams
        layoutParams2.width = textSizeConverter.getWidth(24)
        layoutParams2.height = textSizeConverter.getHeight(24)
        binding.imgLoader.layoutParams = layoutParams2

        val marginLayoutParam = binding.txtValidating.layoutParams as LinearLayout.LayoutParams
        marginLayoutParam.setMargins(0, textSizeConverter.getPaddingORMarginValue(32), 0, 0)
        binding.txtValidating.layoutParams = marginLayoutParam
        binding.txtValidating.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(20),
        )

        val rotation = AnimationUtils.loadAnimation(this, R.anim.rotate)
        rotation.fillAfter = true
        binding.imgLoader.startAnimation(rotation)
    }

    private fun openLivenessScreen() {
        val intent = Intent(this, LivenessMainActivity::class.java)
        someActivityResultLauncher.launch(intent)
        overridePendingTransition(android.R.anim.fade_in, android.R.anim.fade_out)
    }

    private val someActivityResultLauncher =
        registerForActivityResult(
            ActivityResultContracts.StartActivityForResult(),
        ) { result ->
            Log.d(tag, "onActivityResult: ")
            if (Utility.getInstance().liveImage != null) {
                viewModel.validateFaceMatchAction()
            } else {
                handleLocationPermission()
            }
        }

    private fun handleLocationPermission() {
        if (OtherConstant.getInstance().appName.equals(
                getString(R.string.overify),
                ignoreCase = true,
            )
        ) {
            requestLocationData()
        } else {
            viewModel.validateJustCompletion()
        }
    }

    override fun onBackPressed() {
        Log.d(tag, "onBackPressed: ${viewModel.getIsApiCalled().value}")
        if (viewModel.getIsApiCalled().value == false) onBackPressedDispatcher.onBackPressed()
    }

    private fun requestLocationData() {
        PermissionAndLocationHelper.requestPermissionAndLocation(
            this,
            object : PermissionAndLocationHelper.ResultCallback {
                override fun onResult(
                    latitude: Double,
                    longitude: Double,
                    address: String,
                ) {
                    viewModel.callAMLCrop(
                        this@SkipNFCLivenessActivity,
                        latitude,
                        longitude,
                        address,
                        true,
                    )
                }
            },
        )
    }

    private fun openSelectDocumentScreen() {
        val intent = Intent(this, SelectDocumentActivity::class.java)
        intent.flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK
        startActivity(intent)
    }

    override fun onAvailable() {
        if (viewModel.getIsApiCalled().value == false) {
            viewModel.handleInternetConnectionData(
                isInternetAvailable,
            )
        }
    }

    private fun handleLocationData() {
        PermissionAndLocationHelper.requestPermissionAndLocation(
            this,
            object : PermissionAndLocationHelper.ResultCallback {
                override fun onResult(
                    latitude: Double,
                    longitude: Double,
                    address: String,
                ) {
                    viewModel.callAMLCrop(
                        this@SkipNFCLivenessActivity,
                        latitude,
                        longitude,
                        address,
                        false,
                    )
                }
            },
        )
    }
}
