package nashid.verify.sdk.mrtd2.activity.viewmodel

import android.app.Activity
import android.app.Application
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.drawable.Drawable
import androidx.annotation.Nullable
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.MutableLiveData
import com.bumptech.glide.Glide
import com.bumptech.glide.request.target.CustomTarget
import nashid.verify.sdk.OtherConstant
import nashid.verify.sdk.R
import nashid.verify.sdk.Utility
import nashid.verify.sdk.mrtd2.resultcallback.callbackclass.DocumentImages
import nashid.verify.sdk.networking.APIClient
import nashid.verify.sdk.networking.APIInterface
import nashid.verify.sdk.networking.models.AutoCrop
import nashid.verify.sdk.utility.DataUtils
import okhttp3.MediaType.Companion.toMediaTypeOrNull
import okhttp3.MultipartBody
import okhttp3.RequestBody.Companion.asRequestBody
import okhttp3.RequestBody.Companion.toRequestBody
import retrofit2.Call
import retrofit2.Callback
import retrofit2.Response
import java.io.ByteArrayOutputStream
import java.io.File
import java.io.FileOutputStream
import java.io.IOException
import java.text.SimpleDateFormat
import java.util.Locale

class ScanningViewModel(
    application: Application,
) : AndroidViewModel(application) {
    val personalHashmap = MutableLiveData<HashMap<String, String>>()
    val name = MutableLiveData<String>()
    val countryCode = MutableLiveData<String>()
    val frontPageImage = MutableLiveData<Bitmap>()
    val backPageImage = MutableLiveData<Bitmap?>()
    val profileImage = MutableLiveData<Bitmap>()

    init {
        // Initialize personalHashmap with empty hash map
        personalHashmap.value = HashMap()
    }

    fun initData(context: Activity) {
        val utility = Utility.getInstance()
        name.value = utility.name
        countryCode.value = "${utility.nationality} | ${utility.passportNumber?.replace("<", "")}"

        var bitmap =
            BitmapFactory.decodeByteArray(utility.scannedImage, 0, utility.scannedImage!!.size)
        if (bitmap == null) {
            bitmap =
                DataUtils.getCropImageBitmap(
                    context,
                    utility.scannedImage,
                )
        }
        callAutoCrop(context, bitmap) { url ->
            loadImage(context, url) { frontPageImage.value = it }
            if (utility.selectedDoc != null && !utility.selectedDoc.equals("E-PASSPORT", true)) {
                var backBitmap = BitmapFactory.decodeByteArray(utility.scannedIdFrontView, 0, utility.scannedIdFrontView!!.size)
                if (backBitmap == null) {
                    backBitmap = DataUtils.getCropImageBitmap(context, utility.scannedIdFrontView)
                }
                callAutoCrop(context, backBitmap) { backUrl ->
                    loadImage(context, backUrl) { backPageImage.value = it }
                }
            }
        }

        personalHashmap.value?.apply {
            utility.documentType?.let { put(context.getString(R.string.scan_result_doctype), it) }
            put(context.getString(R.string.scan_result_doc_number), utility.passportNumber?.replace("<", "") ?: "")
            utility.name?.let { put(context.getString(R.string.scan_result_name), it) }
            utility.nationality?.let {
                put(
                    context.getString(R.string.scan_result_nationality),
                    it,
                )
            }
            utility.country?.let { put(context.getString(R.string.scan_result_country), it) }
            put(context.getString(R.string.scan_result_birth_date), formatDate(utility.dateOfBirth))
            put(context.getString(R.string.scan_result_expiry_date), formatDate(utility.expiryDate))
            put(context.getString(R.string.scan_result_gender), if (utility.gender.equals("f", true)) "Female" else "Male")
            put(context.getString(R.string.scan_result_mrz), "${utility.mrzLine1}\n${utility.mrzLine2}\n${utility.mrzLine3}")
        }

        profileImage.value = DocumentImages.instance.scanPhoto
    }

    private fun formatDate(date: String?): String =
        try {
            val sdf = SimpleDateFormat("yyMMdd", Locale.ENGLISH)
            val parsedDate = sdf.parse(date)
            SimpleDateFormat("dd/MM/yyyy", Locale.ENGLISH).format(parsedDate)
        } catch (e: Exception) {
            date ?: ""
        }

    private fun callAutoCrop(
        context: Activity,
        bitmap: Bitmap?,
        callback: (String) -> Unit,
    ) {
        val stream = ByteArrayOutputStream()
        bitmap!!.compress(Bitmap.CompressFormat.PNG, 100, stream)
        val byteArray = stream.toByteArray()
        val filename1 = File(context.filesDir, "cropped.png")
        if (filename1.exists()) {
            filename1.delete()
        }
        try {
            FileOutputStream(filename1).use { fos -> fos.write(byteArray) }
        } catch (e: IOException) {
            e.printStackTrace()
        }
        val fileRequestBody = filename1.asRequestBody("image/jpeg".toMediaTypeOrNull())

        val filePart = MultipartBody.Part.createFormData("file", "cropped", fileRequestBody)
        val param1 =
            "text/plain".toMediaTypeOrNull()?.let { APIClient.autoCropApiKey().toRequestBody(it) }
        val param2 =
            "text/plain"
                .toMediaTypeOrNull()
                ?.let {
                    APIClient
                        .autoCropApiSecretKey()
                        .toRequestBody(it)
                }
        APIClient
            .getClient("image crop", OtherConstant.getInstance().autoCropBaseUrl)
            .create<APIInterface>(
                APIInterface::class.java,
            ).callAutoCrop(param1, param2, filePart)
            .enqueue(
                object : Callback<AutoCrop?> {
                    override fun onResponse(
                        call: Call<AutoCrop?>,
                        response: Response<AutoCrop?>,
                    ) {
                        callback(response.body()?.presigned_url ?: "")
//                        autoCroppedCallbacks?.onSuccess(response.body()!!.presigned_url)
                    }

                    override fun onFailure(
                        call: Call<AutoCrop?>,
                        t: Throwable,
                    ) {
                        callback("")
                    }
                },
            )
    }

    private fun loadImage(
        context: Activity,
        url: String,
        callback: (Bitmap) -> Unit,
    ) {
        Glide
            .with(context.applicationContext)
            .asBitmap()
            .load(url)
            .into(
                object : CustomTarget<Bitmap?>() {
                    override fun onResourceReady(
                        resource: Bitmap,
                        transition: com.bumptech.glide.request.transition.Transition<in Bitmap?>?,
                    ) {
                        callback(resource)
                    }

                    override fun onLoadCleared(
                        @Nullable placeholder: Drawable?,
                    ) {}
                },
            )
    }
}
