/*
 * Copyright 2020 Google LLC. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package nashid.verify.sdk.mrtd2.activity

import android.Manifest
import android.animation.Animator
import android.animation.AnimatorListenerAdapter
import android.annotation.SuppressLint
import android.app.Dialog
import android.content.Intent
import android.content.pm.PackageManager
import android.graphics.Bitmap
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.graphics.PixelFormat
import android.graphics.PorterDuff
import android.graphics.PorterDuffXfermode
import android.graphics.RectF
import android.graphics.drawable.ColorDrawable
import android.graphics.drawable.Drawable
import android.os.Build
import android.os.Bundle
import android.os.CountDownTimer
import android.util.Log
import android.util.Size
import android.util.TypedValue
import android.view.Surface
import android.view.SurfaceHolder
import android.view.View
import android.view.View.GONE
import android.view.View.VISIBLE
import android.view.ViewTreeObserver
import android.view.Window
import android.view.WindowInsets
import android.view.WindowInsetsController
import android.view.animation.AnimationUtils
import android.widget.AdapterView
import android.widget.CompoundButton
import android.widget.LinearLayout
import android.widget.RelativeLayout
import androidx.camera.core.CameraInfoUnavailableException
import androidx.camera.core.CameraSelector
import androidx.camera.core.ImageAnalysis
import androidx.camera.core.ImageCapture
import androidx.camera.core.ImageCapture.OnImageCapturedCallback
import androidx.camera.core.ImageProxy
import androidx.camera.core.Preview
import androidx.camera.lifecycle.ProcessCameraProvider
import androidx.core.content.ContextCompat
import androidx.core.content.res.ResourcesCompat
import androidx.lifecycle.ViewModelProvider
import androidx.vectordrawable.graphics.drawable.Animatable2Compat
import com.airbnb.lottie.LottieDrawable
import com.bumptech.glide.Glide
import com.bumptech.glide.load.DataSource
import com.bumptech.glide.load.engine.GlideException
import com.bumptech.glide.load.resource.gif.GifDrawable
import com.bumptech.glide.request.RequestListener
import com.bumptech.glide.request.target.Target
import com.caverock.androidsvg.SVG
import com.caverock.androidsvg.SVGParseException
import com.google.mlkit.vision.common.InputImage
import com.google.mlkit.vision.common.internal.ImageConvertUtils
import nashid.verify.sdk.BuildConfig
import nashid.verify.sdk.Constant
import nashid.verify.sdk.Logger
import nashid.verify.sdk.R
import nashid.verify.sdk.Utility.Companion.getInstance
import nashid.verify.sdk.databinding.ActivityVisionCameraxLivePreviewBinding
import nashid.verify.sdk.databinding.DialogEmailPhoneConfirmationBinding
import nashid.verify.sdk.internetutility.ErrorUtility
import nashid.verify.sdk.mrtd2.activity.viewmodel.scan.CameraXLivePreviewViewModel
import nashid.verify.sdk.mrtd2.rooted.RootedCheck
import nashid.verify.sdk.utility.PermissionHelper.PermissionCallback
import nashid.verify.sdk.utility.PermissionHelper.requestPermissions
import java.io.ByteArrayOutputStream
import java.util.Objects
import java.util.concurrent.Executor
import java.util.concurrent.Executors
import kotlin.math.min

class CameraXLivePreviewActivity :
    BaseActivity(),
    AdapterView.OnItemSelectedListener,
    CompoundButton.OnCheckedChangeListener,
    SurfaceHolder.Callback {
    private val executor: Executor = Executors.newSingleThreadExecutor()
    private var cameraProvider: ProcessCameraProvider? = null
    private var previewUseCase: Preview? = null
    private var analysisUseCase: ImageAnalysis? = null
    private var lensFacing = CameraSelector.LENS_FACING_BACK
    private var cameraSelector: CameraSelector? = null
    private var imageCapture: ImageCapture? = null
    private lateinit var holder: SurfaceHolder
    private lateinit var canvas: Canvas
    private var boxWidth = 0
    private var boxHeight = 0
    private var mPImage: ImageProxy? = null
    private var left = 0
    private var right = 0
    private var top = 0
    private var bottom = 0
    private var diameter = 0
    private lateinit var binding: ActivityVisionCameraxLivePreviewBinding
    private var croppedBitmap: Bitmap? = null
    private val screenDivideValue = 1.8
    private val logger = Logger.withTag(this.javaClass.simpleName)
    private var textSizeConverter: TextSizeConverter? = null
    private var viewModel: CameraXLivePreviewViewModel? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivityVisionCameraxLivePreviewBinding.inflate(layoutInflater)
        setContentView(binding.root)

        viewModel = ViewModelProvider(this)[CameraXLivePreviewViewModel::class.java]

        RootedCheck.getInstance().setFlag(this@CameraXLivePreviewActivity)
        if (RootedCheck.getInstance().isRootedDevice(applicationContext)) {
            RootedCheck.getInstance().showRootedDeviceDialog(
                this,
                getString(R.string.root_dialog),
                getString(R.string.root_desc),
                getString(R.string.root_btn),
            )
        } else {
            initView()
        }
    }

    private fun setLayoutAndTextSize() {
        textSizeConverter = TextSizeConverter(applicationContext)
        textSizeConverter?.let { textSizeConverter ->
            var layoutParams2 = binding.layoutHeader.imgBack.layoutParams
            layoutParams2.width = textSizeConverter.getWidth(24)
            layoutParams2.height = textSizeConverter.getHeight(24)
            binding.layoutHeader.imgBack.layoutParams = layoutParams2
            layoutParams2 = binding.layoutStepMain.lytStep.layoutParams
            layoutParams2.width = LinearLayout.LayoutParams.MATCH_PARENT
            layoutParams2.height = textSizeConverter.getHeight(26)
            binding.layoutStepMain.lytStep.layoutParams = layoutParams2
            binding.layoutHeader.txtHelp.text = getString(R.string.header_doc_scan)
            binding.layoutHeader.txtHelp.setTextColor(Color.WHITE)
            binding.layoutHeader.txtHelp.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(17),
            )
            binding.txtGlare.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(16),
            )
            var padding = textSizeConverter.getPaddingORMarginValue(16)
            binding.layoutParentHeader.setPadding(padding, padding, padding, padding)
            val padding1 = textSizeConverter.getPaddingORMarginValue(4)
            binding.txtGlare.setPadding(padding1, padding1, padding1, padding1)
            val layoutParams = textSizeConverter.linearLayoutParam
            layoutParams.setMargins(0, textSizeConverter.getPaddingORMarginValue(4), 0, 0)
            binding.layoutHeader.lytHeaderMain.layoutParams = layoutParams
            val scanInstruction =
                RelativeLayout.LayoutParams(
                    RelativeLayout.LayoutParams.MATCH_PARENT,
                    RelativeLayout.LayoutParams.WRAP_CONTENT,
                )
            scanInstruction.addRule(RelativeLayout.CENTER_IN_PARENT, RelativeLayout.TRUE)
            scanInstruction.setMargins(
                textSizeConverter.getPaddingORMarginValue(40),
                0,
                textSizeConverter.getPaddingORMarginValue(40),
                0,
            )
            binding.lytScanInstruction.layoutParams = scanInstruction
            var marginLayoutParam =
                binding.layoutStepMain.lytStep.layoutParams as LinearLayout.LayoutParams
            marginLayoutParam.setMargins(0, textSizeConverter.getPaddingORMarginValue(24), 0, 0)
            binding.layoutStepMain.lytStep.layoutParams = marginLayoutParam
            val stepIconSize = 22
            var stepLayoutParam = binding.layoutStepMain.txtStep1.layoutParams
            stepLayoutParam.width = textSizeConverter.getWidth(stepIconSize)
            stepLayoutParam.height = textSizeConverter.getHeight(stepIconSize)
            binding.layoutStepMain.txtStep1.layoutParams = stepLayoutParam
            stepLayoutParam = binding.layoutStepMain.txtStep2.layoutParams
            stepLayoutParam.width = textSizeConverter.getWidth(stepIconSize)
            stepLayoutParam.height = textSizeConverter.getHeight(stepIconSize)
            binding.layoutStepMain.txtStep2.layoutParams = stepLayoutParam
            stepLayoutParam = binding.layoutStepMain.txtStep3.layoutParams
            stepLayoutParam.width = textSizeConverter.getWidth(stepIconSize)
            stepLayoutParam.height = textSizeConverter.getHeight(stepIconSize)
            binding.layoutStepMain.txtStep3.layoutParams = stepLayoutParam
            binding.statusTxt.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(16),
            )
            marginLayoutParam = binding.statusTxt.layoutParams as LinearLayout.LayoutParams
            marginLayoutParam.setMargins(0, textSizeConverter.getPaddingORMarginValue(24), 0, 0)
            binding.statusTxt.layoutParams = marginLayoutParam
            marginLayoutParam = binding.lytBottom.layoutParams as LinearLayout.LayoutParams
            marginLayoutParam.setMargins(0, 0, 0, textSizeConverter.getPaddingORMarginValue(26))
            binding.lytBottom.layoutParams = marginLayoutParam
            stepLayoutParam = binding.imgBottom.layoutParams
            stepLayoutParam.width = textSizeConverter.getWidth(16)
            stepLayoutParam.height = textSizeConverter.getHeight(16)
            binding.imgBottom.layoutParams = stepLayoutParam
            var svg: SVG?
            try {
                svg = SVG.getFromResource(resources, R.raw.scan_bottom)
                binding.imgBottom.setSVG(svg)
            } catch (e: SVGParseException) {
                e.printStackTrace()
            }
            binding.txtBottom.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(14),
            )
            binding.imgScanComplete.layoutParams.width = textSizeConverter.getWidth(56)
            binding.imgScanComplete.layoutParams.height = textSizeConverter.getHeight(56)
            binding.imgScanComplete.invalidate()
            binding.imgScanComplete.setAnimation(R.raw.scan_completed)
            marginLayoutParam = binding.txtScanCompleted.layoutParams as LinearLayout.LayoutParams
            marginLayoutParam.setMargins(0, textSizeConverter.getPaddingORMarginValue(24), 0, 0)
            binding.txtScanCompleted.layoutParams = marginLayoutParam
            binding.txtScanCompleted.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(20),
            )
            layoutParams2 = binding.imgScanFail.layoutParams
            layoutParams2.width = textSizeConverter.getWidth(56)
            layoutParams2.height = textSizeConverter.getHeight(56)
            binding.imgScanFail.layoutParams = layoutParams2
            binding.txtScanInstruction.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(16),
            )
            marginLayoutParam = binding.txtScanFail.layoutParams as LinearLayout.LayoutParams
            marginLayoutParam.setMargins(0, textSizeConverter.getPaddingORMarginValue(24), 0, 0)
            binding.txtScanFail.layoutParams = marginLayoutParam
            binding.txtScanFail.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(20),
            )
            marginLayoutParam =
                binding.btnRescanningDocuments.layoutParams as LinearLayout.LayoutParams
            marginLayoutParam.setMargins(
                textSizeConverter.getPaddingORMarginValue(16),
                0,
                textSizeConverter.getPaddingORMarginValue(16),
                textSizeConverter.getPaddingORMarginValue(44),
            )
            binding.btnRescanningDocuments.layoutParams = marginLayoutParam
            padding = textSizeConverter.getPaddingORMarginValue(12)
            binding.txtRescanningDocuments.setPadding(0, padding, 0, padding)
            binding.txtScanInstruction.setPadding(
                textSizeConverter.getPaddingORMarginValue(34),
                padding,
                textSizeConverter.getPaddingORMarginValue(34),
                padding,
            )
            binding.txtRescanningDocuments.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(16),
            )
            binding.btnRescanningDocuments.radius = textSizeConverter.calculateRadius(8).toFloat()
            binding.cardInstruction.radius = textSizeConverter.calculateRadius(4).toFloat()
            stepLayoutParam = binding.imgError1.layoutParams
            stepLayoutParam.width = textSizeConverter.getWidth(14)
            stepLayoutParam.height = textSizeConverter.getHeight(14)
            binding.imgError1.layoutParams = stepLayoutParam
            stepLayoutParam = binding.imgError2.layoutParams
            stepLayoutParam.width = textSizeConverter.getWidth(14)
            stepLayoutParam.height = textSizeConverter.getHeight(14)
            binding.imgError2.layoutParams = stepLayoutParam
            stepLayoutParam = binding.imgError3.layoutParams
            stepLayoutParam.width = textSizeConverter.getWidth(14)
            stepLayoutParam.height = textSizeConverter.getHeight(14)
            binding.imgError3.layoutParams = stepLayoutParam
            marginLayoutParam = binding.imgError1.layoutParams as LinearLayout.LayoutParams
            marginLayoutParam.setMargins(textSizeConverter.getPaddingORMarginValue(28), 0, 0, 0)
            binding.imgError1.layoutParams = marginLayoutParam
            marginLayoutParam = binding.imgError2.layoutParams as LinearLayout.LayoutParams
            marginLayoutParam.setMargins(textSizeConverter.getPaddingORMarginValue(28), 0, 0, 0)
            binding.imgError2.layoutParams = marginLayoutParam
            marginLayoutParam = binding.imgError3.layoutParams as LinearLayout.LayoutParams
            marginLayoutParam.setMargins(textSizeConverter.getPaddingORMarginValue(28), 0, 0, 0)
            binding.imgError3.layoutParams = marginLayoutParam
            marginLayoutParam = binding.lytError1.layoutParams as LinearLayout.LayoutParams
            marginLayoutParam.setMargins(0, textSizeConverter.getPaddingORMarginValue(76), 0, 0)
            binding.lytError1.layoutParams = marginLayoutParam
            marginLayoutParam = binding.lytError2.layoutParams as LinearLayout.LayoutParams
            marginLayoutParam.setMargins(
                0,
                textSizeConverter.getPaddingORMarginValue(20),
                0,
                textSizeConverter.getPaddingORMarginValue(20),
            )
            binding.lytError2.layoutParams = marginLayoutParam
            marginLayoutParam = binding.lytError3.layoutParams as LinearLayout.LayoutParams
            marginLayoutParam.setMargins(0, 0, 0, textSizeConverter.getPaddingORMarginValue(70))
            binding.lytError3.layoutParams = marginLayoutParam
            binding.txtError1.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(16),
            )
            binding.txtError2.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(16),
            )
            binding.txtError3.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(16),
            )
            binding.txtError1.setPadding(textSizeConverter.getPaddingORMarginValue(6), 0, 0, 0)
            binding.txtError2.setPadding(textSizeConverter.getPaddingORMarginValue(6), 0, 0, 0)
            binding.txtError3.setPadding(textSizeConverter.getPaddingORMarginValue(6), 0, 0, 0)
            try {
                svg = SVG.getFromResource(resources, R.raw.scan_bottom)
                binding.imgError1.setSVG(svg)
                binding.imgError2.setSVG(svg)
                binding.imgError3.setSVG(svg)
            } catch (e: SVGParseException) {
                e.printStackTrace()
            }

            binding.txtTurnDoc.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(20),
            )

            binding.imgBackcrdScan.viewTreeObserver.addOnPreDrawListener(
                object : ViewTreeObserver.OnPreDrawListener {
                    override fun onPreDraw(): Boolean {
                        val height = binding.imgBackcrdScan.height
                        binding.imgBackcrdScan.viewTreeObserver.removeOnPreDrawListener(this)
                        val marginLayoutParam =
                            binding.txtTurnDoc.layoutParams as RelativeLayout.LayoutParams
                        marginLayoutParam.setMargins(0, textSizeConverter.getPaddingORMarginValue(-(height / 12)), 0, 0)
                        binding.txtTurnDoc.layoutParams = marginLayoutParam
                        return true
                    }
                },
            )
        }
    }

    private fun setRescanningDocumentVisibility() {
        viewModel!!.cameraXLiveData.setIsOverlayVisible(false)
        viewModel!!.cameraXLiveData.setIsScanVisible(false)
        viewModel!!.cameraXLiveData.setIsScanFailVisible(true)
        viewModel!!.cameraXLiveData.setIsTransparentVisible(true)
        viewModel!!.cameraXLiveData.setIsScanCompleteVisible(false)
    }

    private var cTimer: CountDownTimer? = null

    private fun startTimer() {
        cTimer =
            object : CountDownTimer(DELAY_MILLIS, 1000) {
                override fun onTick(millisUntilFinished: Long) {}

                override fun onFinish() {
                    if (isInternetAvailable) {
                        if (java.lang.Boolean.TRUE == viewModel!!.cameraXLiveData.getIsFrontView().value) {
                            setRescanningDocumentVisibility()
                        } else if (getInstance().passportNumber == null && getInstance().dateOfBirth == null && getInstance().expiryDate == null) {
                            setRescanningDocumentVisibility()
                        } else {
                            cTimer!!.start()
                        }
                    } else {
                        cTimer!!.start()
                    }
                }
            }.start()
    }

    private var instructionTimer: CountDownTimer? = null

    private fun startInstructionTimer() {
        instructionTimer =
            object : CountDownTimer(DELAY_INSTRUCTION_MILLIS, 1000) {
                override fun onTick(millisUntilFinished: Long) {}

                override fun onFinish() {
                    if (binding.lytScanInstruction.visibility == View.VISIBLE) {
                        instructionTimer!!.start()
                    } else {
                        binding.lytScanInstruction.visibility = View.VISIBLE
                    }
                }
            }.start()
    }

    private fun observeViewModel() {
        viewModel!!
            .cameraXLiveData
            .getIsInternetAvailable()
            .observe(this) { isAvailable: Boolean -> handleInternetConnection(isAvailable) }
        viewModel!!
            .cameraXLiveData
            .getCloseImageProxy()
            .observe(this) { needToClose: Boolean -> handleCloseImageProxy(needToClose) }
        viewModel!!
            .cameraXLiveData
            .getCaptureAnImage()
            .observe(this) { needToCallCapture: Boolean -> handleCaptureAnImage(needToCallCapture) }
        viewModel!!
            .cameraXLiveData
            .getCTimer()
            .observe(this) { needToCloseTimer: Boolean -> handleCTimer(needToCloseTimer) }
        viewModel!!
            .cameraXLiveData
            .getCloseAnalysisUseCase()
            .observe(this) { needToClearAnalyzer: Boolean ->
                handleAnalysisUseCase(needToClearAnalyzer)
            }
        viewModel!!
            .cameraXLiveData
            .getUpdateCustomLoader()
            .observe(this) { needToUpdateCustomLoader: Boolean ->
                updateCustomLoader(needToUpdateCustomLoader)
            }
        viewModel!!
            .cameraXLiveData
            .getUnbindCameraProvider()
            .observe(this) { isUnbind: Boolean -> unBindCameraProvider(isUnbind) }
        viewModel!!.cameraXLiveData.getIsOverlayVisible().observe(this) { visibility: Boolean ->
            Log.d(tag, "observeViewModel: overlay")
            binding.overlay.visibility = if (visibility) View.VISIBLE else View.GONE
        }
        viewModel!!.cameraXLiveData.getIsTransparentVisible().observe(this) { visibility: Boolean ->
            Log.d(tag, "observeViewModel: transparent")
            binding.lytTransparent.visibility = if (visibility) View.VISIBLE else View.GONE
        }
        viewModel!!.cameraXLiveData.getGlareTextVisibility().observe(this) { visibility: Boolean ->
            Log.d(tag, "observeViewModel: glare")
            binding.txtGlare.visibility = if (visibility) View.VISIBLE else View.GONE
        }
        viewModel!!
            .cameraXLiveData
            .getIsLayoutBottomVisible()
            .observe(this) { visibility: Boolean ->
                Log.d(tag, "observeViewModel: bottom")
                binding.lytBottom.visibility = if (visibility) View.VISIBLE else View.GONE
            }
        viewModel!!.cameraXLiveData.getIsScanVisible().observe(this) { visibility: Boolean ->
            Log.d(tag, "observeViewModel: scan")
            binding.scan.visibility = if (visibility) View.VISIBLE else View.GONE
        }
        viewModel!!
            .cameraXLiveData
            .getIsScanCompleteVisible()
            .observe(this) { visibility: Boolean ->
                Log.d(tag, "observeViewModel: scan complete")
                binding.lyoutScanComplete.visibility = if (visibility) View.VISIBLE else View.GONE
            }
        viewModel!!.cameraXLiveData.getIsScanFailVisible().observe(this) { visibility: Boolean ->
            Log.d(tag, "observeViewModel: scan fail")
            binding.lyoutScanFail.visibility = if (visibility) View.VISIBLE else View.GONE
        }
        viewModel!!
            .cameraXLiveData
            .getIsBackCardScanVisible()
            .observe(this) { visibility: Boolean ->
                binding.imgBackcrdScan.visibility = if (visibility) View.VISIBLE else View.INVISIBLE
            }
        viewModel!!
            .cameraXLiveData
            .getStatusText()
            .observe(this) { text: String -> binding.statusTxt.text = text }
        viewModel!!
            .cameraXLiveData
            .getGlareText()
            .observe(this) { text: String -> binding.txtGlare.text = text }
        viewModel!!
            .cameraXLiveData
            .getBottomText()
            .observe(this) { text: String -> binding.txtBottom.text = text }
    }

    private fun unBindCameraProvider(isUnbind: Boolean) {
        if (isUnbind) {
            assert(cameraProvider != null)
            cameraProvider!!.unbindAll()
        }
    }

    private fun updateCustomLoader(needToUpdateCustomLoader: Boolean) {
        if (needToUpdateCustomLoader) {
            binding.customLoader.root.visibility = View.VISIBLE
            binding.customLoader.loaderContainer.setBackgroundColor(Color.WHITE)
            val layoutParams2 = binding.customLoader.imgLoader.layoutParams
            layoutParams2.width = textSizeConverter!!.getWidth(24)
            layoutParams2.height = textSizeConverter!!.getHeight(24)
            binding.customLoader.imgLoader.layoutParams = layoutParams2
            val marginLayoutParam =
                binding.customLoader.txtValidating.layoutParams as LinearLayout.LayoutParams
            marginLayoutParam.setMargins(0, textSizeConverter!!.getPaddingORMarginValue(32), 0, 0)
            binding.customLoader.txtValidating.layoutParams = marginLayoutParam
            binding.customLoader.txtValidating.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter!!.getTextSize(20),
            )
            val rotation =
                AnimationUtils.loadAnimation(this@CameraXLivePreviewActivity, R.anim.rotate)
            rotation.fillAfter = true
            binding.customLoader.imgLoader.startAnimation(rotation)
        }
    }

    private fun handleAnalysisUseCase(needToClearAnalyzer: Boolean) {
        if (needToClearAnalyzer) {
            analysisUseCase!!.clearAnalyzer()
        }
    }

    private fun handleCaptureAnImage(needToCallCapture: Boolean) {
        if (needToCallCapture) {
            captureAnImage()
        }
    }

    private fun handleCTimer(needToCloseTimer: Boolean) {
        if (cTimer != null) {
            if (needToCloseTimer) {
                cTimer!!.cancel()
            } else {
                cTimer!!.start()
            }
        }
    }

    private fun handleCloseImageProxy(needToClose: Boolean) {
        if (needToClose) {
            if (mPImage != null) {
                mPImage!!.close()
            }
        }
    }

    private fun handleInternetConnection(isAvailable: Boolean) {
        if (!isAvailable) {
            ErrorUtility
                .getInstance()
                .showNoInternetDialog(this, !this.isFinishing && !this.isDestroyed)
        }
    }

    private fun handleIntentExtras() {
        if (intent.hasExtra(getString(R.string.doc_key))) {
            val selectedDoc = intent.getStringExtra(getString(R.string.doc_key))
            if (selectedDoc != null) {
                viewModel!!.setSelectedDoc(selectedDoc, resources)
            }
        }
    }

    private fun setUpCameraSelector() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            window.setDecorFitsSystemWindows(false)
            window.insetsController?.apply {
                hide(WindowInsets.Type.statusBars())
                systemBarsBehavior = WindowInsetsController.BEHAVIOR_SHOW_TRANSIENT_BARS_BY_SWIPE
            }
        } else {
            @Suppress("DEPRECATION")
            window.decorView.systemUiVisibility =
                View.SYSTEM_UI_FLAG_LAYOUT_STABLE or
                View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
        }
        window.statusBarColor = Color.TRANSPARENT
        cameraSelector = CameraSelector.Builder().requireLensFacing(lensFacing).build()
    }

    private fun initView() {
        handleIntentExtras()
        logger.log("View setting..")
        setUpCameraSelector()
        setLayoutAndTextSize()
        setUpOverlayAndAnimation()
        viewModel!!.cameraXLiveData.setIsInternetAvailable(isInternetAvailable)
        observeViewModel()
    }

    private fun setUpOverlayAndAnimation() {
        binding.txtBottom.setPadding(textSizeConverter!!.getPaddingORMarginValue(6), 0, 0, 0)
        binding.overlay.setZOrderMediaOverlay(true)
        holder = binding.overlay.holder
        holder.setFormat(PixelFormat.TRANSPARENT)
        holder.addCallback(this)
        try {
            val back = SVG.getFromResource(resources, R.raw.back_white)
            binding.layoutHeader.imgBack.setSVG(back)
            binding.layoutHeader.lytHeaderMain.setBackgroundColor(Color.TRANSPARENT)
            if (storage.getPreferredLocale().equals("ar", ignoreCase = true)) {
                binding.layoutHeader.imgBack.rotation = 180f
            }
        } catch (e: SVGParseException) {
            e.printStackTrace()
        }
        binding.lottieAnimationView.setAnimation(R.raw.scan)
        binding.lottieAnimationView.playAnimation()
        binding.lottieAnimationView.repeatCount = LottieDrawable.INFINITE
        requestPermissions(
            this@CameraXLivePreviewActivity,
            object : PermissionCallback {
                override fun onResult(
                    allGranted: Boolean,
                    grantedList: List<String>,
                    deniedList: List<String>,
                ) {
                    if (allGranted) {
                        startCameraInit()
                    }
                }
            },
        )
    }

    private fun startCameraInit() {
        logger.log("start camera..")
        initCameraViewModelProvider()
        initClick()
        startTimer()
    }

    private fun initCameraViewModelProvider() {
        viewModel!!.processCameraProvider.observe(this) { provider: ProcessCameraProvider? ->
            cameraProvider = provider
            bindAllCameraUseCases()
            bindAnalysisUseCase()
        }
    }

    private fun initClick() {
        binding.layoutHeader.imgBack.setOnClickListener { onBackPressedDispatcher.onBackPressed() }
        binding.btnRescanningDocuments.setOnClickListener {
            viewModel!!.cameraXLiveData.setIsOverlayVisible(true)
            viewModel!!.cameraXLiveData.setIsScanVisible(true)
            viewModel!!.cameraXLiveData.setIsScanFailVisible(false)
            viewModel!!.cameraXLiveData.setIsTransparentVisible(false)
            viewModel!!.cameraXLiveData.setIsScanCompleteVisible(false)
            viewModel!!.cameraXLiveData.setIsBackCardScanVisible(false)
            cTimer!!.start()
            if (mPImage != null) mPImage!!.close()
        }
    }

    @Synchronized
    override fun onItemSelected(
        parent: AdapterView<*>?,
        view: View,
        pos: Int,
        id: Long,
    ) {
        bindAnalysisUseCase()
    }

    override fun onNothingSelected(parent: AdapterView<*>?) {}

    override fun onCheckedChanged(
        buttonView: CompoundButton,
        isChecked: Boolean,
    ) {
        if (ContextCompat.checkSelfPermission(
                this,
                Manifest.permission.CAMERA,
            ) == PackageManager.PERMISSION_GRANTED
        ) {
            if (cameraProvider == null) {
                return
            }
            val newLensFacing =
                if (lensFacing == CameraSelector.LENS_FACING_FRONT) CameraSelector.LENS_FACING_BACK else CameraSelector.LENS_FACING_FRONT
            val newCameraSelector =
                CameraSelector.Builder().requireLensFacing(newLensFacing).build()
            try {
                if (cameraProvider!!.hasCamera(newCameraSelector)) {
                    lensFacing = newLensFacing
                    cameraSelector = newCameraSelector
                    bindAllCameraUseCases()
                }
            } catch (e: CameraInfoUnavailableException) {
                e.printStackTrace()
            }
        }
    }

    private fun bindAllCameraUseCases() {
        if (cameraProvider != null) {
            cameraProvider!!.unbindAll()
            bindPreviewUseCase()
        }
    }

    @SuppressLint("UnsafeOptInUsageError")
    private fun bindPreviewUseCase() {
        if (ContextCompat.checkSelfPermission(
                this,
                Manifest.permission.CAMERA,
            ) == PackageManager.PERMISSION_GRANTED
        ) {
            if (cameraProvider == null) {
                return
            }
            if (previewUseCase != null) {
                cameraProvider!!.unbind(previewUseCase)
            }
            val builder = Preview.Builder()
            previewUseCase = builder.build()
            previewUseCase!!.setSurfaceProvider(binding.previewView.surfaceProvider)
            val rotation =
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
                    display?.rotation ?: Surface.ROTATION_0
                } else {
                    @Suppress("DEPRECATION")
                    windowManager.defaultDisplay.rotation
                }
            imageCapture =
                ImageCapture
                    .Builder()
                    .setTargetRotation(rotation)
                    .build()
            cameraProvider!!.bindToLifecycle(this, cameraSelector!!, imageCapture, previewUseCase)
        }
    }

    @SuppressLint("UnsafeOptInUsageError")
    private fun bindAnalysisUseCase() {
        if (ContextCompat.checkSelfPermission(
                this,
                Manifest.permission.CAMERA,
            ) == PackageManager.PERMISSION_GRANTED
        ) {
            setUpAnalysisUseCase()
        }
    }

    @SuppressLint("RestrictedApi")
    private fun setUpAnalysisUseCase() {
        runOnUiThread { viewModel!!.cameraXLiveData.setIsScanVisible(true) }
        if (cameraProvider == null) {
            return
        }
        if (analysisUseCase != null) {
            cameraProvider!!.unbind(analysisUseCase)
        }
        analysisUseCase =
            ImageAnalysis
                .Builder()
                .setDefaultResolution(Size(1280, 720))
                .setBackpressureStrategy(ImageAnalysis.STRATEGY_KEEP_ONLY_LATEST)
                .build()
        analysisUseCase!!.setAnalyzer(executor) { pImage: ImageProxy ->
            if (isInternetAvailable) {
                analyzeImage(pImage)
            } else {
                pImage.close()
            }
        }
        cameraProvider!!.bindToLifecycle(this, cameraSelector!!, imageCapture, analysisUseCase)
    }

    private var bitmapImage: Bitmap? = null
    private var originalBitmapImage: Bitmap? = null

    private fun analyzeImage(pImage: ImageProxy) {
        @SuppressLint("UnsafeOptInUsageError")
        val mediaImage = pImage.image
        if (mediaImage == null) {
            pImage.close()
            return
        }
        Log.d(Companion.TAG, "analyze:image ")
        val image = InputImage.fromMediaImage(mediaImage, pImage.imageInfo.rotationDegrees)
        try {
            bitmapImage = ImageConvertUtils.getInstance().convertToUpRightBitmap(image)
            originalBitmapImage = bitmapImage
        } catch (e: Exception) {
            e.printStackTrace()
        }
        cropBitmapByPercentage()
        if (java.lang.Boolean.TRUE == viewModel!!.cameraXLiveData.getFlag().value) {
            passImageToRead(pImage)
        } else {
            assert(analysisUseCase != null)
            analysisUseCase!!.clearAnalyzer()
        }
    }

    @SuppressLint("SetTextI18n")
    private fun passImageToRead(pImage: ImageProxy) {
        mPImage = pImage
        if (BuildConfig.BUILD_TYPE.equals("debug", ignoreCase = true)) {
            runOnUiThread {
                Log.d(Companion.TAG, "run: " + croppedBitmap!!.height + " " + croppedBitmap!!.width)
                binding.testimg1.visibility = View.VISIBLE
                binding.testimg1.setImageBitmap(croppedBitmap)
            }
        }
        if (binding.lyoutScanFail.visibility == View.GONE) {
            val imageLight = getString(R.string.image_light_normal)
            if (imageLight.equals(getString(R.string.image_light_normal), ignoreCase = true)) {
                var hasGlare = GlareDetector.isGlareDetected(croppedBitmap)
                hasGlare = false
                if (!hasGlare) {
                    runOnUiThread { viewModel!!.cameraXLiveData.setGlareTextVisibility(false) }
                    //                    runOnUiThread(() -> binding.txtGlare.setVisibility(View.GONE));
                    filterScannedText()
                } else {
                    runOnUiThread {
                        viewModel!!.cameraXLiveData.setGlareTextVisibility(true)
                        viewModel!!.cameraXLiveData.setGlareText(getString(R.string.glare))
                        runOnUiThread { pImage.close() }
                    }
                }
            } else {
                runOnUiThread {
                    viewModel!!.cameraXLiveData.setGlareTextVisibility(true)
                    viewModel!!.cameraXLiveData.setGlareText(getString(R.string.image_light) + imageLight)
                    runOnUiThread { pImage.close() }
                }
            }
        }
    }

    private fun cropBitmapByPercentage() {
        val height = bitmapImage!!.height
        val width = bitmapImage!!.width
        diameter = min(height, width)
        val offset = (Constant.visionAPIOffset * diameter).toInt()
        Log.d(Companion.TAG, "analyzeImage: $width   $diameter")
        diameter -= offset
        val value = 4
        Log.d(Companion.TAG, "analyzeImage:after removing offset $width   $diameter")
        left = width - diameter
        top = (height / screenDivideValue - diameter / value).toInt()
        right = diameter
        bottom = (height / screenDivideValue + diameter / value).toInt()
        boxHeight = bottom - top
        boxWidth = right - left
        val cutPercentage = 16
        val cutAmount = (width * (cutPercentage / 100.0)).toInt()
        left += cutAmount
        right -= cutAmount
        val newBoxWidth = right - left
        if (java.lang.Boolean.FALSE == viewModel!!.cameraXLiveData.getIsFrontView().value) {
            croppedBitmap = Bitmap.createBitmap(bitmapImage!!, left, top, newBoxWidth, boxHeight)
        } else {
            val cutPercentageTopBottom = 5
            val cutAmountTopBottom = (height * (cutPercentageTopBottom / 100.0)).toInt()
            top += cutAmountTopBottom
            bottom -= cutAmountTopBottom
            val newBoxHeight = bottom - top
            croppedBitmap = Bitmap.createBitmap(bitmapImage!!, left, top, newBoxWidth, newBoxHeight)
        }
        val stream = ByteArrayOutputStream()
        croppedBitmap!!.compress(Bitmap.CompressFormat.PNG, 100, stream)
    }

    private fun filterScannedText() {
        try {
//            this only happen id selected document is idCard
//            The following lines are used to divide the photo into 4 parts. The top left side of the photo is used to read the text from "ROYAL OMAN POLICE D.G.OF CIVIL STATUS." After successfully reading the data, the bitmap is divided horizontally, and the MRZ code is read from the bottom part of the image. This is done to increase the speed and accuracy of reading the data.
            if (!Objects
                    .requireNonNull(viewModel!!.cameraXLiveData.getSelectedDoc().value)
                    .equals(getString(R.string.e_passport), ignoreCase = true)
            ) {
                if (java.lang.Boolean.FALSE == viewModel!!.cameraXLiveData.getAlgoHeaderDetect().value && java.lang.Boolean.TRUE == viewModel!!.cameraXLiveData.getIsFrontView().value) {
                    croppedBitmap =
                        Bitmap.createBitmap(
                            croppedBitmap!!,
                            0,
                            0,
                            croppedBitmap!!.width / 2,
                            croppedBitmap!!.height / 2,
                        )
                } else {
                    if (java.lang.Boolean.TRUE == viewModel!!.cameraXLiveData.getIsFrontView().value) {
                        croppedBitmap =
                            Bitmap.createBitmap(
                                croppedBitmap!!,
                                0,
                                croppedBitmap!!.height / 2,
                                croppedBitmap!!.width,
                                croppedBitmap!!.height / 2,
                            )
                    }
                }
            }
            //            end here
            if (BuildConfig.BUILD_TYPE.equals("debug", ignoreCase = true)) {
                runOnUiThread {
                    binding.testimg2.visibility = View.VISIBLE
                    binding.testimg2.setImageBitmap(croppedBitmap)
                }
            }
            viewModel!!.processTextFromBitmap(croppedBitmap)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    private fun captureAnImage() {
        logger.log("capturing photo")
        imageCapture!!.takePicture(
            executor,
            object : OnImageCapturedCallback() {
                override fun onCaptureSuccess(image: ImageProxy) {
                    binding.imgScanComplete.setMaxFrame(80)
                    super.onCaptureSuccess(image)
                    if (Objects
                            .requireNonNull(viewModel!!.cameraXLiveData.getSelectedDoc().value)
                            .equals(getString(R.string.e_passport), ignoreCase = true)
                    ) {
                        capturePassportImage()
                    } else {
                        val stream = reCropDocBitmap()
                        if (java.lang.Boolean.TRUE == viewModel!!.cameraXLiveData.getIsFrontView().value && java.lang.Boolean.FALSE == viewModel!!.cameraXLiveData.getIsFrontViewScanned().value) {
                            captureFrontViewIdCard(stream)
                        } else {
                            captureBackViewIdCard(stream)
                        }
                    }
                }
            },
        )
    }

    private fun captureFrontViewIdCard(stream: ByteArrayOutputStream) {
        val byteArray = stream.toByteArray()
        getInstance().scannedImage = byteArray
        runOnUiThread {
            viewModel!!.cameraXLiveData.setStatusText("")
            viewModel!!.cameraXLiveData.setIsOverlayVisible(false)
            viewModel!!.cameraXLiveData.setIsScanVisible(false)
            viewModel!!.cameraXLiveData.setIsTransparentVisible(true)
            viewModel!!.cameraXLiveData.setIsScanCompleteVisible(true)
            binding.imgScanComplete.playAnimation()
            binding.imgScanComplete.addAnimatorListener(
                object : AnimatorListenerAdapter() {
                    override fun onAnimationEnd(animation: Animator) {
                        super.onAnimationEnd(animation)
                        if (viewModel!!.cameraXLiveData.getIdNo().value == null && viewModel!!.cameraXLiveData.dOB.value == null && viewModel!!.cameraXLiveData.getExpiryDate().value == null) {
                            viewModel!!.cameraXLiveData.setIsScanCompleteVisible(false)
                            viewModel!!.cameraXLiveData.setIsBackCardScanVisible(true)
                            viewModel!!.cameraXLiveData.setStatusText("")
                            logger.log("loading card flip animation")
                            val requestBuilder =
                                Glide
                                    .with(this@CameraXLivePreviewActivity)
                                    .asGif()
                                    .load(R.drawable.card_flip)
                                    .listener(
                                        object : RequestListener<GifDrawable> {
                                            override fun onLoadFailed(
                                                e: GlideException?,
                                                model: Any?,
                                                target: Target<GifDrawable>,
                                                isFirstResource: Boolean,
                                            ): Boolean = false

                                            override fun onResourceReady(
                                                resource: GifDrawable,
                                                model: Any,
                                                target: Target<GifDrawable>,
                                                dataSource: DataSource,
                                                isFirstResource: Boolean,
                                            ): Boolean {
                                                binding.txtTurnDoc.visibility = VISIBLE
                                                binding.txtTurnDoc.setText(getString(R.string.turn_document))
                                                logger.log("resource ready to load")
                                                resource.setLoopCount(1)
                                                resource.registerAnimationCallback(
                                                    object :
                                                        Animatable2Compat.AnimationCallback() {
                                                        override fun onAnimationEnd(drawable: Drawable) {
                                                            logger.log("animation end")
                                                            binding.txtTurnDoc.visibility = GONE
                                                            viewModel!!.cameraXLiveData.setStatusText(getString(R.string.scan_id_top_back_text))
                                                            viewModel!!.cameraXLiveData.setIsOverlayVisible(true)
                                                            viewModel!!.cameraXLiveData.setIsScanVisible(true)
                                                            viewModel!!.cameraXLiveData.setIsTransparentVisible(
                                                                false,
                                                            )
                                                            viewModel!!.cameraXLiveData.setIsScanCompleteVisible(
                                                                false,
                                                            )
                                                            viewModel!!.cameraXLiveData.setIsBackCardScanVisible(
                                                                false,
                                                            )
                                                            Log.d(
                                                                Companion.TAG,
                                                                "onAnimationEnd: start time " + viewModel!!.printCurrentTimeInLogs(),
                                                            )
                                                            mPImage!!.close()
                                                        }
                                                    },
                                                )
                                                return false
                                            }
                                        },
                                    )
                            requestBuilder.into(binding.imgBackcrdScan)
                        }
                    }
                },
            )
        }
        runOnUiThread {
            viewModel!!.cameraXLiveData.setIdLine1(true)
            viewModel!!.cameraXLiveData.setIdLine2(true)
            viewModel!!.cameraXLiveData.setIdLine3(true)
            viewModel!!.cameraXLiveData.setIsFrontViewScanned(true)
        }
    }

    private fun captureBackViewIdCard(stream: ByteArrayOutputStream) {
        runOnUiThread { viewModel!!.cameraXLiveData.setStatusText("") }
        val byteArray = stream.toByteArray()
        getInstance().scannedIdFrontView = byteArray
        runOnUiThread {
            viewModel!!.cameraXLiveData.setIsOverlayVisible(false)
            viewModel!!.cameraXLiveData.setIsScanVisible(false)
            viewModel!!.cameraXLiveData.setIsTransparentVisible(true)
            viewModel!!.cameraXLiveData.setIsBackCardScanVisible(false)
            viewModel!!.cameraXLiveData.setIsScanCompleteVisible(true)
            binding.imgScanComplete.playAnimation()
            Log.d(Companion.TAG, "onAnimationEnd: end time " + viewModel!!.printCurrentTimeInLogs())
            binding.imgScanComplete.addAnimatorListener(
                object : AnimatorListenerAdapter() {
                    override fun onAnimationEnd(animation: Animator) {
                        super.onAnimationEnd(animation)
                        viewModel!!.handleAnimationCompleted(this@CameraXLivePreviewActivity)
                    }
                },
            )
        }
    }

    private fun reCropDocBitmap(): ByteArrayOutputStream {
        val height = originalBitmapImage!!.height
        val width = originalBitmapImage!!.width
        diameter = min(height, width)
        val offset = (Constant.visionAPIOffset * diameter).toInt()
        Log.d(Companion.TAG, "analyzeImage: $width   $diameter")
        diameter -= offset
        val value = 4
        Log.d(Companion.TAG, "analyzeImage:after removing offset $width   $diameter")
        left = width - diameter
        top = (height / screenDivideValue - diameter / value).toInt()
        right = diameter
        bottom = (height / screenDivideValue + diameter / value).toInt()
        boxHeight = bottom - top
        boxWidth = right - left
        croppedBitmap = Bitmap.createBitmap(originalBitmapImage!!, left, top, boxWidth, boxHeight)
        val stream = ByteArrayOutputStream()
        croppedBitmap!!.compress(Bitmap.CompressFormat.PNG, 100, stream)
        return stream
    }

    private fun capturePassportImage() {
        val stream = ByteArrayOutputStream()
        croppedBitmap!!.compress(Bitmap.CompressFormat.PNG, 100, stream)
        val byteArray = stream.toByteArray()
        getInstance().scannedImage = byteArray
        runOnUiThread {
            viewModel!!.cameraXLiveData.setStatusText("")
            viewModel!!.cameraXLiveData.setIsOverlayVisible(false)
            viewModel!!.cameraXLiveData.setIsScanVisible(false)
            viewModel!!.cameraXLiveData.setIsTransparentVisible(true)
            viewModel!!.cameraXLiveData.setIsScanCompleteVisible(true)
            viewModel!!.cameraXLiveData.setIsLayoutBottomVisible(false)
            binding.imgScanComplete.playAnimation()
            binding.imgScanComplete.addAnimatorListener(
                object : AnimatorListenerAdapter() {
                    override fun onAnimationEnd(animation: Animator) {
                        super.onAnimationEnd(animation)
                        logger.log("passport scan completed")
                        viewModel!!.handleAnimationCompleted(this@CameraXLivePreviewActivity)
                    }
                },
            )
        }
    }

    override fun surfaceCreated(holder: SurfaceHolder) {
        drawFocusRect(ContextCompat.getColor(applicationContext, R.color.overlay_color))
    }

    private fun drawFocusRect(color: Int) {
        try {
            val height = binding.previewView.height
            val width = binding.previewView.width
            diameter = width
            if (height < width) {
                diameter = height
            }
            val offset = (Constant.drawSquareOffset * diameter).toInt()
            diameter -= offset
            canvas = holder.lockCanvas()
            canvas.drawColor(
                ContextCompat.getColor(applicationContext, R.color.overlay_color),
                PorterDuff.Mode.ADD,
            )
            val paint = Paint()
            paint.style = Paint.Style.STROKE
            paint.color = color
            paint.strokeWidth = 3f
            left = width - diameter
            top = (height / screenDivideValue - diameter / 3).toInt()
            right = diameter
            bottom = (height / screenDivideValue + diameter / 3).toInt()
            boxHeight = bottom - top
            boxWidth = right - left
            val paint1 = Paint(Paint.DITHER_FLAG)
            paint1.style = Paint.Style.FILL
            paint1.color = Color.WHITE
            paint1.xfermode = PorterDuffXfermode(PorterDuff.Mode.CLEAR)
            val paint2 = Paint()
            paint2.style = Paint.Style.STROKE
            paint2.color = Color.WHITE
            paint2.strokeWidth = 4f
            val corner = 26
            val cornerSpace = 18
            paint.strokeJoin = Paint.Join.ROUND
            createCornersPath(
                left - cornerSpace,
                top - cornerSpace,
                right + cornerSpace,
                bottom + cornerSpace,
                paint2,
            )
            canvas.drawRoundRect(
                RectF(
                    left.toFloat(),
                    top.toFloat(),
                    right.toFloat(),
                    bottom.toFloat(),
                ),
                corner.toFloat(),
                corner.toFloat(),
                paint1,
            )
            canvas.drawRoundRect(
                RectF(
                    left.toFloat(),
                    top.toFloat(),
                    right.toFloat(),
                    bottom.toFloat(),
                ),
                corner.toFloat(),
                corner.toFloat(),
                paint,
            )
            holder.unlockCanvasAndPost(canvas)
            val params =
                RelativeLayout.LayoutParams(
                    boxWidth,
                    boxHeight,
                )
            params.topMargin = top
            params.addRule(RelativeLayout.CENTER_HORIZONTAL, RelativeLayout.TRUE)
            binding.scan.layoutParams = params
            binding.scan.invalidate()
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    private fun createCornersPath(
        left: Int,
        top: Int,
        right: Int,
        bottom: Int,
        paint: Paint,
    ) {
        val radius = 40f
        val lengthF = 80f
        canvas.drawLine(left + radius, top.toFloat(), left + lengthF, top.toFloat(), paint)
        canvas.drawLine(left.toFloat(), top + radius, left.toFloat(), top + lengthF, paint)
        val rectTL = RectF(left.toFloat(), top.toFloat(), left + radius * 2, top + radius * 2)
        canvas.drawArc(rectTL, 180f, 90f, false, paint)
        canvas.drawLine(right - radius, top.toFloat(), right - lengthF, top.toFloat(), paint)
        canvas.drawLine(right.toFloat(), top + radius, right.toFloat(), top + lengthF, paint)
        val rectTR = RectF(right - radius * 2, top.toFloat(), right.toFloat(), top + radius * 2)
        canvas.drawArc(rectTR, 270f, 90f, false, paint)
        canvas.drawLine(left + radius, bottom.toFloat(), left + lengthF, bottom.toFloat(), paint)
        canvas.drawLine(left.toFloat(), bottom - radius, left.toFloat(), bottom - lengthF, paint)
        val rectBL = RectF(left.toFloat(), bottom - radius * 2, left + radius * 2, bottom.toFloat())
        canvas.drawArc(rectBL, 90f, 90f, false, paint)
        canvas.drawLine(
            right - radius,
            bottom.toFloat(),
            right - lengthF,
            bottom.toFloat(),
            paint,
        )
        canvas.drawLine(
            right.toFloat(),
            bottom - radius,
            right.toFloat(),
            bottom - lengthF,
            paint,
        )
        val rectBR =
            RectF(right - radius * 2, bottom - radius * 2, right.toFloat(), bottom.toFloat())
        canvas.drawArc(rectBR, 0f, 90f, false, paint)
    }

    override fun surfaceChanged(
        holder: SurfaceHolder,
        format: Int,
        width: Int,
        height: Int,
    ) {
    }

    override fun surfaceDestroyed(holder: SurfaceHolder) {}

    @Deprecated("Deprecated in Java")
    override fun onBackPressed() {
        if (binding.customLoader.root.visibility != View.VISIBLE) openBackPressDialog()
    }

    private fun openBackPressDialog() {
        val dialog = Dialog(this)
        dialog.requestWindowFeature(Window.FEATURE_NO_TITLE)
        val binding = DialogEmailPhoneConfirmationBinding.inflate(dialog.layoutInflater)
        dialog.setContentView(binding.root)
        var layoutParams = binding.dialogHeader.layoutParams as LinearLayout.LayoutParams
        var padding = textSizeConverter!!.getPaddingORMarginValue(16)
        layoutParams.setMargins(padding, padding, padding, 0)
        binding.dialogHeader.layoutParams = layoutParams
        val layoutParams2 = binding.cardConfirmation.layoutParams
        layoutParams2.width = textSizeConverter!!.getWidth(280)
        binding.cardConfirmation.layoutParams = layoutParams2
        binding.cardConfirmation.radius = textSizeConverter!!.calculateRadius(14).toFloat()
        binding.dialogHeader.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter!!.getTextSize(18),
        )
        binding.dialogSubtitle.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter!!.getTextSize(14),
        )
        binding.btnNo.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSizeConverter!!.getTextSize(16))
        binding.btnYes.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSizeConverter!!.getTextSize(16))
        layoutParams = binding.dialogSubtitle.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(padding, padding, padding, padding)
        binding.dialogSubtitle.layoutParams = layoutParams
        padding = textSizeConverter!!.getPaddingORMarginValue(12)
        layoutParams = binding.btnNo.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(0, padding, 0, padding)
        binding.btnNo.layoutParams = layoutParams
        layoutParams = binding.btnYes.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(
            textSizeConverter!!.getPaddingORMarginValue(24),
            padding,
            textSizeConverter!!.getPaddingORMarginValue(16),
            padding,
        )
        binding.btnYes.layoutParams = layoutParams
        binding.dialogHeader.typeface =
            ResourcesCompat.getFont(applicationContext, R.font.pingmedium)
        binding.dialogSubtitle.typeface =
            ResourcesCompat.getFont(applicationContext, R.font.regular)
        binding.dialogHeader.text = getString(R.string.back_dialog_header)
        binding.dialogSubtitle.text = getString(R.string.back_dialog_title)
        binding.btnYes.setOnClickListener {
            dialog.dismiss()
            val i = Intent(this@CameraXLivePreviewActivity, StartScanningActivity::class.java)
            i.putExtra(
                resources.getString(R.string.doc_key),
                viewModel!!.cameraXLiveData.getSelectedDoc().value,
            )
            startActivity(i)
            finish()
        }
        binding.btnNo.setOnClickListener { dialog.dismiss() }
        dialog.show()
        dialog.setCancelable(false)
        dialog.window!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
        dialog.window!!.attributes.windowAnimations = R.style.DialogAnimation
    }

    companion object {
        private const val TAG = "CameraXLivePreview"
        private const val DELAY_MILLIS: Long = 15000
        private const val DELAY_INSTRUCTION_MILLIS: Long = 3000
    }
}
