package nashid.verify.sdk.data.repository

import android.content.Context
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import nashid.verify.sdk.data.remote.ApiService
import nashid.verify.sdk.model.DocumentVerification
import nashid.verify.sdk.model.SubmitVerification
import nashid.verify.sdk.model.UploadFileResponse
import nashid.verify.sdk.model.VerificationResponse
import nashid.verify.sdk.model.VerificationResultResponse
import nashid.verify.sdk.utils.NetWorkResult
import nashid.verify.sdk.utils.helpers.createCustomGson
import nashid.verify.sdkNew.R
import okhttp3.MultipartBody
import retrofit2.Response

class ScanDocumentRepository(
    private val apiService: ApiService,
    private val context: Context,
    private val ioDispatcher: CoroutineDispatcher = Dispatchers.IO,
) {
    suspend fun uploadFile(
        token: String,
        appKey: String,
        file: MultipartBody.Part,
    ): NetWorkResult<UploadFileResponse> =
        withContext(ioDispatcher) {
            try {
                val response: Response<UploadFileResponse> = apiService.uploadFile("Bearer $token", appKey, file)
                processAuthenticateUserResponse(response)
            } catch (e: Exception) {
                NetWorkResult.Error(e.localizedMessage ?: context.getString(R.string.error_unknown))
            }
        }

    suspend fun submitVerification(
        token: String,
        appKey: String,
        verification: DocumentVerification,
    ): NetWorkResult<VerificationResponse> =
        withContext(ioDispatcher) {
            try {
                val gson = createCustomGson()
                val json = gson.toJson(verification)
                val response: Response<VerificationResponse> = apiService.submitVerification("Bearer $token", appKey, gson.fromJson(json, SubmitVerification::class.java))
                processVerificationResponse(response)
            } catch (e: Exception) {
                NetWorkResult.Error(e.localizedMessage ?: context.getString(R.string.error_unknown))
            }
        }

    suspend fun getVerification(
        token: String,
        appKey: String,
        verificationId: String,
    ): NetWorkResult<VerificationResultResponse> =
        withContext(ioDispatcher) {
            try {
                val response: Response<VerificationResultResponse> = apiService.getVerification("Bearer $token", appKey, verificationId)
                processVerificationResultResponse(response)
            } catch (e: Exception) {
                NetWorkResult.Error(e.localizedMessage ?: context.getString(R.string.error_unknown))
            }
        }

    private fun processAuthenticateUserResponse(response: Response<UploadFileResponse>): NetWorkResult<UploadFileResponse> =
        if (response.isSuccessful) {
            val body = response.body()
            if (body != null) {
                if (response.code() == 200) {
                    NetWorkResult.Success(body)
                } else {
                    NetWorkResult.Error(body.message ?: context.getString(R.string.error_unknown))
                }
            } else {
                NetWorkResult.Error(context.getString(R.string.error_response_body_null))
            }
        } else {
            NetWorkResult.Error(response.errorBody()?.string() ?: context.getString(R.string.error_unknown))
        }

    private fun processVerificationResponse(response: Response<VerificationResponse>): NetWorkResult<VerificationResponse> =
        if (response.isSuccessful) {
            val body = response.body()
            if (body != null) {
                if (response.code() == 200) {
                    NetWorkResult.Success(body)
                } else {
                    NetWorkResult.Error(body.message ?: context.getString(R.string.error_unknown))
                }
            } else {
                NetWorkResult.Error(context.getString(R.string.error_response_body_null))
            }
        } else {
            NetWorkResult.Error(response.errorBody()?.string() ?: context.getString(R.string.error_unknown))
        }

    private fun processVerificationResultResponse(response: Response<VerificationResultResponse>): NetWorkResult<VerificationResultResponse> =
        if (response.isSuccessful) {
            val body = response.body()
            if (body != null) {
                if (response.code() == 200) {
                    NetWorkResult.Success(body)
                } else {
                    NetWorkResult.Error(body.message)
                }
            } else {
                NetWorkResult.Error(context.getString(R.string.error_response_body_null))
            }
        } else {
            NetWorkResult.Error(response.errorBody()?.string() ?: context.getString(R.string.error_unknown))
        }
}
