package nashid.verify.sdk.di

import nashid.verify.sdk.data.remote.ApiService
import nashid.verify.sdk.utils.Constants.BASE_URL_PRODUCTION
import nashid.verify.sdk.utils.Constants.BASE_URL_STAGING
import nashid.verify.sdkNew.BuildConfig
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import okhttp3.Response
import okhttp3.logging.HttpLoggingInterceptor
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import java.io.IOException
import java.util.concurrent.TimeUnit

fun provideHttpClient(): OkHttpClient {
    val loggingInterceptor =
        HttpLoggingInterceptor().apply {
            level = HttpLoggingInterceptor.Level.BODY
        }
    return OkHttpClient
        .Builder()
        .readTimeout(60, TimeUnit.SECONDS)
        .connectTimeout(60, TimeUnit.SECONDS)
        .addInterceptor(loggingInterceptor)
        .addInterceptor(RetryInterceptor())
        .build()
}

fun provideConverterFactory(): GsonConverterFactory = GsonConverterFactory.create()

fun provideRetrofit(
    okHttpClient: OkHttpClient,
    gsonConverterFactory: GsonConverterFactory,
): Retrofit =
    Retrofit
        .Builder()
        .baseUrl(if (BuildConfig.BUILD_TYPE.equals("release", ignoreCase = true)) BASE_URL_PRODUCTION else BASE_URL_STAGING)
        .client(okHttpClient)
        .addConverterFactory(gsonConverterFactory)
        .build()

fun provideService(retrofit: Retrofit): ApiService = retrofit.create(ApiService::class.java)

class RetryInterceptor(
    private val maxRetries: Int = 3,
) : Interceptor {
    override fun intercept(chain: Interceptor.Chain): Response {
        var attempt = 0
        var response: Response? = null
        var exception: Exception? = null

        do {
            try {
                // Close any previous response
                response?.close()

                response = chain.proceed(chain.request())
                if (response.isSuccessful) return response
            } catch (e: Exception) {
                exception = e
            } finally {
                attempt++
            }
        } while (attempt < maxRetries)

        // Ensure response is closed if retries exhausted
        response?.close()
        throw exception ?: IOException("We’re experiencing an issue on our end, and our team is aware of it. A fix will be available soon.")
    }
}
