package nashid.verify.sdk.viewmodel

import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import nashid.verify.sdk.data.repository.ScanDocumentRepository
import nashid.verify.sdk.model.DocumentVerification
import nashid.verify.sdk.utils.Loggers
import nashid.verify.sdk.utils.NetWorkResult
import nashid.verify.sdk.utils.helpers.ScanDocumentResult
import okhttp3.MultipartBody

class ScanDocumentViewModel(
    private val scanDocumentRepository: ScanDocumentRepository,
) : ViewModel() {
    private val logger = Loggers.withTag("ScanDocumentViewModel")
    private val _result = MutableLiveData<ScanDocumentResult>()
    val result: LiveData<ScanDocumentResult> get() = _result

    fun uploadArtifact(
        token: String,
        appKey: String,
        filePart: MultipartBody.Part,
        artifactType: String,
    ) {
        viewModelScope.launch(Dispatchers.IO) {
            when (val response = scanDocumentRepository.uploadFile(token, appKey, filePart)) {
                is NetWorkResult.Success -> {
                    if (response.data?.data?.fileKey != null) {
                        logger.log("Upload successful for $artifactType: ${response.data.message}")
                        _result.postValue(ScanDocumentResult.FileKeyResult(response.data.data.fileKey, artifactType))
                    } else {
                        logger.log("Upload failed for $artifactType: ${response.message}")
                        _result.postValue(ScanDocumentResult.Error(response.message ?: "Upload failed"))
                    }
                }

                is NetWorkResult.Error -> {
                    logger.log("Upload failed for $artifactType: ${response.message}")
                    _result.postValue(ScanDocumentResult.Error(response.message ?: "Upload failed"))
                }

                else -> {}
            }
        }
    }

    fun submitVerification(
        token: String,
        appKey: String,
        verification: DocumentVerification,
    ) {
        viewModelScope.launch(Dispatchers.IO) {
            when (val response = scanDocumentRepository.submitVerification(token, appKey, verification)) {
                is NetWorkResult.Success -> {
                    if (response.data?.data?.verificationId != null) {
                        val verificationId = response.data.data.verificationId
                        logger.log("SubmitVerification successful for $verificationId: ${response.data.message}")
                        _result.postValue(response.data.message?.let { ScanDocumentResult.ScanDocumentResponse(true, it, response.data.data.verificationId) })
                    } else {
                        logger.log("SubmitVerification failed: ${response.message}")
                        _result.postValue(response.data?.message?.let { ScanDocumentResult.ScanDocumentResponse(false, it, response.data.data.verificationId) })
                    }
                }

                is NetWorkResult.Error -> {
                    logger.log("SubmitVerification failed: ${response.message}")
                    _result.postValue(ScanDocumentResult.ScanDocumentResponse(false, response.message ?: "Unexpected error occurred", ""))
                }

                else -> {}
            }
        }
    }

    fun getVerification(
        token: String,
        appKey: String,
        verificationId: String,
    ) {
        viewModelScope.launch(Dispatchers.IO) {
            when (val response = scanDocumentRepository.getVerification(token, appKey, verificationId)) {
                is NetWorkResult.Success -> {
                    logger.log("GetVerification successful for $verificationId: ${response.data?.message}")
                    _result.postValue(response.data?.let { ScanDocumentResult.GetScanResultResponse(true, it.message, response.data) })
                }

                is NetWorkResult.Error -> {
                    logger.log("GetVerification failed for $verificationId: ${response.message}")
                    _result.postValue(
                        ScanDocumentResult.GetScanResultResponse(
                            false,
                            response.message ?: "Unexpected error occurred",
                            null,
                        ),
                    )
                }

                else -> {}
            }
        }
    }
}
