package nashid.verify.sdk

import android.content.Context
import android.content.Intent
import nashid.verify.sdk.model.AuthCallback
import nashid.verify.sdk.model.DocumentType
import nashid.verify.sdk.model.DocumentVerification
import nashid.verify.sdk.model.ExtraData
import nashid.verify.sdk.model.SDKStatus
import nashid.verify.sdk.request.AuthRequest
import nashid.verify.sdk.ui.MainScreen
import nashid.verify.sdk.utils.KoinEnvironment
import nashid.verify.sdk.utils.Loggers
import nashid.verify.sdk.utils.NetWorkResult
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.helpers.LocaleUtil.Companion.applyLocalizedContext
import nashid.verify.sdk.utils.helpers.ScanDocumentResult
import nashid.verify.sdk.utils.helpers.Storage
import nashid.verify.sdk.utils.helpers.VerificationHandler
import nashid.verify.sdk.viewmodel.AuthViewModel
import nashid.verify.sdk.viewmodel.ScanDocumentViewModel
import nashid.verify.sdk.viewmodel.SkipNfcLiveNessViewModel
import org.koin.core.Koin
import org.koin.core.component.KoinComponent
import org.koin.core.component.inject

class VerifySDKManager : KoinComponent, VerificationHandler {
    override fun getKoin(): Koin {
        if (KoinEnvironment.koin == null) {
            throw IllegalStateException("Koin has not been initialized. Make sure MyEnvironment.initKoin(context) is called before using VerifySDKManager.")
        }
        return KoinEnvironment.koin!!
    }

    private val viewModel: AuthViewModel by inject()
    private val scanDocumentViewModel: ScanDocumentViewModel by inject()
    private val skipNfcLiveNessViewModel: SkipNfcLiveNessViewModel by inject()
    private val context: Context by inject()
    private val logger = Loggers.withTag("VerifySDKManager")

    fun initialize(
        sdkKey: String,
        sdkSecret: String,
        languageType: String = "en",
        extraData: ExtraData? = null,
        callback: AuthCallback,
    ) {
        logger.log("Starting initialization with SDK Keys")
        viewModel.authResult.observeForever { result ->
            when (result) {
                is NetWorkResult.Success -> {
                    logger.log("SDK initialization successful: ${result.data?.toString()}")
                    callback.initializeResponse(true, result.data?.message)
                    result.data?.data?.sdkConfig?.let { sdkConfig ->
                        SdkConfig.updateConfig(
                            sdkKey = sdkConfig.sandboxSdkKey,
                            sdkSecret = sdkConfig.sandboxSdkSecret,
                            languageType = languageType,
                            sdkName = sdkConfig.name,
                            sdkDescription = sdkConfig.description,
                            nfcScanningEnabled = sdkConfig.isNfcScanningEnabled,
                            nfcMultiTriesEnabled = sdkConfig.isNfcMultiTriesEnabled,
                            faceMatchingEnabled = sdkConfig.isFaceMatchingEnabled,
                            ocrEnabled = sdkConfig.isOcrEnabled,
                            mrzEnabled = sdkConfig.isMrzEnabled,
                            protocol = sdkConfig.callbackProtocol,
                            host = sdkConfig.callbackHost,
                            instructionsVisible = sdkConfig.showInstructions,
                            nfcSkippable = sdkConfig.isNfcSkippable,
                            nfcRetries = sdkConfig.nfcMaximumRetries ?: 0,
                            extraData = extraData,
                        )
                    }
                }

                is NetWorkResult.Error -> {
                    logger.log("Initialization failed: ${result.message}")
                    SdkConfig.sdkStatus = SDKStatus.FAILED
                    callback.initializeResponse(false, result.message)
                }

                else -> {}
            }
        }

        // Trigger authentication
        viewModel.authenticateUser(AuthRequest(sdkKey, sdkSecret))

        // Configure the SDK appearance
        Storage(context).setPreferredLocale(languageType)
        applyLocalizedContext(context, languageType)
    }

    fun verify(
        documentType: DocumentType = DocumentType.OMAN_ID,
        callback: (ScanDocumentResult.ScanDocumentResponse) -> Unit,
    ) {
        if (SdkConfig.sdkStatus == SDKStatus.INITIALIZED) {
            logger.log("scanDocument called with DocumentType: ${documentType.displayName}")
            SdkConfig.viewType = documentType
            scanDocumentViewModel.result.observeForever {
                if (it is ScanDocumentResult.ScanDocumentResponse) {
                    callback(it)
                    skipNfcLiveNessViewModel.getLiveNessData().requestFinishActivity()
                    if (!it.result) {
                        skipNfcLiveNessViewModel.getLiveNessData().hideProgressBar()
                    }
                }
            }
            val intent =
                Intent(context, MainScreen::class.java).apply {
                    flags = Intent.FLAG_ACTIVITY_NEW_TASK
                }
            context.startActivity(intent)
        } else {
            logger.log("Failed to initialize SDK")
            callback(ScanDocumentResult.ScanDocumentResponse(false, "Failed to initialize SDK", ""))
        }
    }

    fun getVerificationResult(
        verificationId: String,
        callback: (ScanDocumentResult.GetScanResultResponse) -> Unit,
    ) {
        logger.log("getScanResult called with verificationId: $verificationId")
        if (SdkConfig.sdkStatus == SDKStatus.INITIALIZED) {
            scanDocumentViewModel.getVerification(
                SdkConfig.registerToken!!,
                Utility.getInstance().getAppKey(),
                verificationId,
            )
            scanDocumentViewModel.result.observeForever {
                if (it is ScanDocumentResult.GetScanResultResponse) {
                    callback(it)
                    skipNfcLiveNessViewModel.getLiveNessData().requestFinishActivity()
                }
            }
        } else {
            logger.log("SDK not initialized")
        }
    }

    companion object {
        @Volatile
        private var instance: VerifySDKManager? = null

        @JvmStatic
        fun getInstance(): VerifySDKManager =
            instance ?: synchronized(this) {
                instance ?: VerifySDKManager().also { instance = it }
            }
    }

    override fun submitVerification(verification: DocumentVerification) {
        scanDocumentViewModel.submitVerification(
            SdkConfig.registerToken!!,
            Utility.getInstance().getAppKey(),
            verification,
        )
    }
}
