package nashid.verify.sdk.ui

import android.animation.Animator
import android.animation.AnimatorListenerAdapter
import android.annotation.SuppressLint
import android.app.Activity
import android.app.Dialog
import android.app.PendingIntent
import android.content.Intent
import android.content.res.ColorStateList
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Color
import android.graphics.Matrix
import android.graphics.drawable.ColorDrawable
import android.nfc.NfcAdapter
import android.nfc.Tag
import android.nfc.tech.IsoDep
import android.os.Build
import android.os.Bundle
import android.provider.Settings
import android.util.Log
import android.util.TypedValue
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.view.Window
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.ProgressBar
import android.widget.TextView
import androidx.activity.addCallback
import androidx.activity.result.contract.ActivityResultContracts
import androidx.core.content.ContextCompat
import androidx.core.content.res.ResourcesCompat
import com.airbnb.lottie.LottieDrawable
import nashid.verify.sdk.model.TagProvider
import nashid.verify.sdk.utils.OtherConstant
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.helpers.ErrorUtility
import nashid.verify.sdk.utils.helpers.NfcLayoutAndTextSize
import nashid.verify.sdk.utils.helpers.ScanDocumentResult
import nashid.verify.sdk.utils.helpers.TextSizeConverter
import nashid.verify.sdk.utils.id_card.jmrtd.BACKey
import nashid.verify.sdk.utils.id_card.jmrtd.BACKeySpec
import nashid.verify.sdk.viewmodel.NFCViewModel
import nashid.verify.sdk.viewmodel.ScanDocumentViewModel
import nashid.verify.sdkNew.R
import nashid.verify.sdkNew.databinding.ActivityWaitingForNfcBinding
import nashid.verify.sdkNew.databinding.BottomSheetDialogBinding
import nashid.verify.sdkNew.databinding.DialogEmailPhoneConfirmationBinding
import nashid.verify.sdkNew.databinding.DialogFailureNfcBinding
import nashid.verify.sdkNew.databinding.DialogSuccessBinding
import org.koin.android.ext.android.inject
import java.util.Objects

class NfcActivity : BaseActivity() {
    private var binding: ActivityWaitingForNfcBinding? = null
    private val viewModel: NFCViewModel by inject()
    private val textSizeConverter: TextSizeConverter by inject()
    private val scanDocumentViewModel: ScanDocumentViewModel by inject()

    private var mNfcAdapter: NfcAdapter? = null
    private var pendingIntent: PendingIntent? = null
    private lateinit var nfcLayoutAndTextSize: NfcLayoutAndTextSize

    private var errorDialog: Dialog? = null
    private var progressDialog: Dialog? = null
    private var isActivityRunning = false
    private var progressBar: ProgressBar? = null
    private var progressText: TextView? = null
    private var maxRetryCount = 0

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivityWaitingForNfcBinding.inflate(layoutInflater)
        setContentView(binding!!.root)
        initView()
        backPress()
    }

    private fun initView() {
        isActivityRunning = true
        handleIntentData()
        nfcLayoutAndTextSize =
            NfcLayoutAndTextSize(
                textSizeConverter, binding!!, this, storage, viewModel,
                scanDocumentViewModel,
            )
        nfcLayoutAndTextSize.setLayoutAndTextSize()
        setOtherViews()
        handleInternetConnection()
        setObservable()
    }

    private fun setObservable() {
        viewModel.getNfcLiveData().skipNFC.observe(this) { skipNFC: Boolean -> nfcLayoutAndTextSize.handleSkipNFC(skipNFC) }
        viewModel.getNfcLiveData().isInternetAvailable.observe(this) { isAvailable: Boolean ->
            handleInternetConnectionData(
                isAvailable,
            )
        }
        viewModel.getNfcLiveData().mrtdPercentage.observe(this) { progress: Int ->
            setMRTDProgressBarPercentage(
                progress,
            )
        }
        viewModel.getNfcLiveData().showError.observe(this) { message: String? -> showError(message) }
        viewModel.getNfcLiveData().showResult.observe(this) { isDisplaying: Boolean -> displayResult(isDisplaying) }
        scanDocumentViewModel.result.observeForever { result ->
            if (result is ScanDocumentResult.FileKeyResult) {
                Utility.getInstance().saveFileKeyAndContinue(result.fileKey, result.artifactType)
            }
        }
    }

    private fun handleInternetConnectionData(isAvailable: Boolean) {
        logger.log("handleInternetConnectionData: $isAvailable")
        if (!isAvailable) {
            ErrorUtility.getInstance()
                .showNoInternetDialog(this, !this.isFinishing && !this.isDestroyed)
        }
    }

    private fun handleInternetConnection() {
        logger.log("handleInternetConnection: ")
        viewModel.setInternetConnection(isInternetAvailable)
        if (java.lang.Boolean.TRUE == viewModel.getNfcLiveData().isInternetAvailable.value) {
            viewModel.setIsApiCalled(true)
            initAdapter()
            if (errorDialog != null) {
                errorDialog!!.dismiss()
            }
            if (progressDialog != null) {
                progressDialog!!.dismiss()
            }
            viewModel.validateUser()
        }
    }

    public override fun onNewIntent(intent: Intent) {
        super.onNewIntent(intent)
        logger.log("onNewIntent: ")
        TagProvider.closeTag()
        val tagFromIntent: Tag? =
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                intent.getParcelableExtra(NfcAdapter.EXTRA_TAG, Tag::class.java)
            } else {
                @Suppress("DEPRECATION")
                intent.getParcelableExtra(NfcAdapter.EXTRA_TAG)
            }
        if (tagFromIntent != null) {
            TagProvider.setTag(IsoDep.get(tagFromIntent))
        }
        println("Got new intent!")
        if (errorDialog != null) {
            errorDialog!!.dismiss()
        }
        if (progressDialog != null) {
            progressDialog!!.dismiss()
        }
        readPassport(tagFromIntent)
    }

    private fun readPassport(tagFromIntent: Tag?) {
        if (java.lang.Boolean.TRUE == viewModel.getNfcLiveData().isApiCalled.value) {
            showDialog()
            if (Objects
                    .requireNonNull(viewModel.getNfcLiveData().selectedDoc.value)
                    .equals(getString(R.string.e_passport), ignoreCase = true)
            ) {
                viewModel.readPassportNfc(tagFromIntent)
            } else {
                logger.log("readPassport: ")
                if (viewModel.getNfcLiveData().passportNumber.value!!.isNotEmpty() &&
                    viewModel.getNfcLiveData().dateOfExpiration.value!!.isNotEmpty() &&
                    viewModel.getNfcLiveData().dateOfBirth.value!!.isNotEmpty()
                ) {
                    logger.log("readPassport: tag found")
                    if (TagProvider.getTag() != null) {
                        val bacKey: BACKeySpec =
                            BACKey(
                                viewModel.getNfcLiveData().passportNumber.value,
                                viewModel.getNfcLiveData().dateOfBirth.value,
                                viewModel.getNfcLiveData().dateOfExpiration.value,
                            )
                        logger.log("readPassport: $bacKey")
                        viewModel.cancelNfcRead()
                        viewModel.readData(TagProvider.getTag(), bacKey)
                    }
                }
            }
        }
    }

    private fun showDialog() {
        if (isActivityRunning) {
            progressDialog = Dialog(this)
            progressDialog!!.requestWindowFeature(Window.FEATURE_NO_TITLE)
            val binding =
                BottomSheetDialogBinding.inflate(
                    progressDialog!!.layoutInflater,
                )
            progressDialog!!.setContentView(binding.root)
            if (storage.getPreferredLocale().equals("ar", ignoreCase = true)) {
                binding.progressBar.rotation = 180f
            }
            progressBar = binding.progressBar
            progressText = binding.txtDialogPercentage
            val padding = textSizeConverter.getPaddingOrMarginValue(16)
            binding.mainDialogLyt.setPadding(padding, padding, padding, padding)
            val layoutParams2 = binding.progressBar.layoutParams
            layoutParams2.width = textSizeConverter.getWidth(200)
            layoutParams2.height = ViewGroup.LayoutParams.WRAP_CONTENT
            binding.progressBar.layoutParams = layoutParams2
            var layoutParams = binding.progressBar.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(8), 0, 0)
            binding.progressBar.layoutParams = layoutParams
            binding.txtDialogPercentage.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(16).toFloat(),
            )
            binding.txtScanning.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(20).toFloat(),
            )
            binding.txtDialogDesc.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(16).toFloat(),
            )
            layoutParams = binding.txtScanning.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(24), 0, 0)
            binding.txtScanning.layoutParams = layoutParams
            layoutParams = binding.txtDialogDesc.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(
                0,
                textSizeConverter.getPaddingOrMarginValue(16),
                0,
                textSizeConverter.getPaddingOrMarginValue(28),
            )
            binding.txtDialogDesc.layoutParams = layoutParams
            if (Objects
                    .requireNonNull<String?>(viewModel.getNfcLiveData().selectedDoc.value)
                    .equals(getString(R.string.e_passport), ignoreCase = true)
            ) {
                binding.txtDialogDesc.text = getString(R.string.dialog_dontmove_pass)
            } else {
                binding.txtDialogDesc.text = getString(R.string.dialog_dontmove)
            }
            progressDialog!!.show()
            progressDialog!!.setCancelable(false)
            progressDialog!!.window!!.setLayout(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.WRAP_CONTENT,
            )
            progressDialog!!.window!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
            progressDialog!!.window!!.attributes.windowAnimations = R.style.DialogAnimation
            progressDialog!!.window!!.setGravity(Gravity.BOTTOM)
        }
    }

    private fun showError(message: String?) {
        Log.d("NfcActivity", "showError: $message")
        if (message != null) {
            runOnUiThread {
                Log.d("TAG", "showError: $message")
                showFailureDialog(message)
            }
        }
    }

    private fun displayResult(isDisplaying: Boolean) {
        Log.d("NfcActivity", "displayResult: $isDisplaying")
        if (isDisplaying) showResult()
    }

    private fun showResult() {
        nfcLayoutAndTextSize.storeNFCData()
        if (progressDialog != null) {
            progressDialog!!.dismiss()
        }
        setMRTDProgressBarPercentage(96)
        setMRTDProgressBarPercentage(100)
        if (progressBar != null) {
            progressBar!!.progressTintList =
                ColorStateList.valueOf(
                    ContextCompat.getColor(
                        this,
                        R.color.progress_complete,
                    ),
                )
        }
        showSuccessDialog()
    }

    @SuppressLint("SetTextI18n")
    fun setMRTDProgressBarPercentage(progress: Int) {
        Log.d("NfcActivity", "setMRTDProgressBarPercentage: $progress")
        if (progress != 0) {
            runOnUiThread {
                if (progressBar != null && progressText != null) {
                    progressBar!!.progress = progress
                    progressText!!.text = "$progress%"
                }
            }
        }
    }

    private fun showSuccessDialog() {
        if (isActivityRunning) {
            val dialog = Dialog(this)
            dialog.requestWindowFeature(Window.FEATURE_NO_TITLE)
            val binding = DialogSuccessBinding.inflate(dialog.layoutInflater)
            dialog.setContentView(binding.root)
            val padding = textSizeConverter.getPaddingOrMarginValue(16)
            binding.mainDialogLyt.setPadding(padding, padding, padding, padding)
            val layoutParams2 = binding.imgScanComplete.layoutParams
            layoutParams2.width = textSizeConverter.getWidth(56)
            layoutParams2.height = textSizeConverter.getHeight(56)
            binding.imgScanComplete.layoutParams = layoutParams2
            var layoutParams = binding.imgScanComplete.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(8), 0, 0)
            binding.imgScanComplete.layoutParams = layoutParams
            binding.txtScanCompleted.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(24).toFloat(),
            )
            layoutParams = binding.txtScanCompleted.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(
                0,
                textSizeConverter.getPaddingOrMarginValue(16),
                0,
                textSizeConverter.getPaddingOrMarginValue(28),
            )
            binding.txtScanCompleted.layoutParams = layoutParams
            binding.imgScanComplete.playAnimation()
            binding.imgScanComplete.setAnimation(R.raw.scan_completed)
            binding.imgScanComplete.addAnimatorListener(
                object : AnimatorListenerAdapter() {
                    override fun onAnimationEnd(animation: Animator) {
                        super.onAnimationEnd(animation)
                        dialog.dismiss()
                        TagProvider.closeTag()
                        dismissDialog()
                        viewModel.validateLivensUser()
                    }
                },
            )
            dialog.show()
            dialog.setCancelable(false)
            dialog.window?.setLayout(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.WRAP_CONTENT,
            )
            dialog.window!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
            dialog.window!!.attributes.windowAnimations = R.style.DialogAnimation
            dialog.window!!.setGravity(Gravity.BOTTOM)
        }
    }

    private fun showFailureDialog(message: String) {
        if (isActivityRunning) {
            if (errorDialog != null) {
                errorDialog!!.dismiss()
            }
            if (java.lang.Boolean.TRUE == viewModel.getNfcLiveData().skipNFC.value) {
                if (maxRetryCount >= viewModel.getNfcLiveData().maxTry.value!! && viewModel.getNfcLiveData().maxTry.value != 0) {
                    OtherConstant.getInstance().skipNFC = true
                    dismissDialog()
                    viewModel.validateLivensUser()
                }
            }
            maxRetryCount += 1
            errorDialog = Dialog(this)
            errorDialog!!.requestWindowFeature(Window.FEATURE_NO_TITLE)
            val binding =
                DialogFailureNfcBinding.inflate(
                    errorDialog!!.layoutInflater,
                )
            errorDialog!!.setContentView(binding.root)
            var padding = textSizeConverter.getPaddingOrMarginValue(16)
            binding.mainDialogLyt.setPadding(padding, padding, padding, padding)
            var layoutParams2 = binding.imgScanFailure.layoutParams
            layoutParams2.width = textSizeConverter.getWidth(56)
            layoutParams2.height = textSizeConverter.getHeight(56)
            binding.imgScanFailure.layoutParams = layoutParams2
            layoutParams2 = binding.imgInfo1.layoutParams
            layoutParams2.width = textSizeConverter.getWidth(20)
            layoutParams2.height = textSizeConverter.getHeight(20)
            binding.imgInfo1.layoutParams = layoutParams2
            layoutParams2 = binding.imgNfcFailed.layoutParams
            layoutParams2.width = textSizeConverter.getWidth(145)
            layoutParams2.height = textSizeConverter.getHeight(145)
            binding.imgNfcFailed.layoutParams = layoutParams2
            val layoutParams1 = binding.imgNfcFailed.layoutParams as LinearLayout.LayoutParams
            layoutParams1.setMargins(
                0,
                textSizeConverter.getPaddingOrMarginValue(24),
                0,
                textSizeConverter.getPaddingOrMarginValue(18),
            )
            binding.imgNfcFailed.layoutParams = layoutParams1
            layoutParams2 = binding.imgInfo2.layoutParams
            layoutParams2.width = textSizeConverter.getWidth(20)
            layoutParams2.height = textSizeConverter.getHeight(20)
            binding.imgInfo2.layoutParams = layoutParams2
            var layoutParams = binding.imgScanFailure.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(8), 0, 0)
            binding.imgScanFailure.layoutParams = layoutParams
            binding.txtScanCompleted.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(20).toFloat(),
            )
            layoutParams = binding.txtScanCompleted.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(24), 0, 0)
            binding.txtScanCompleted.layoutParams = layoutParams
            binding.txtScanNotCompleteDesc1.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(16).toFloat(),
            )
            layoutParams = binding.txtScanNotCompleteDesc1.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(
                textSizeConverter.getPaddingOrMarginValue(6),
                textSizeConverter.getPaddingOrMarginValue(16),
                0,
                0,
            )
            binding.txtScanNotCompleteDesc1.layoutParams = layoutParams
            binding.txtScanNotCompleteDesc2.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(16).toFloat(),
            )
            binding.imgNfcFailed.playAnimation()
            binding.imgNfcFailed.repeatCount = LottieDrawable.INFINITE
            binding.imgNfcFailed.setAnimation(R.raw.nfc_read_failed)
            if (message.equals("card", ignoreCase = true)) {
                binding.txtScanCompleted.visibility = View.GONE
                binding.imgNfcFailed.visibility = View.GONE
                binding.txtScanNotCompleteDesc1.text = resources.getString(R.string.mrz_wrong)
            } else {
                if (Objects
                        .requireNonNull<String?>(viewModel.getNfcLiveData().selectedDoc.value)
                        .equals(getString(R.string.e_passport), ignoreCase = true)
                ) {
                    binding.txtScanNotCompleteDesc1.text =
                        resources.getString(R.string.passport_scan_not_complete_desc)
                } else {
                    binding.txtScanNotCompleteDesc1.text =
                        resources.getString(R.string.scan_not_complete_desc2)
                }
            }
            layoutParams = binding.txtScanNotCompleteDesc2.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(
                textSizeConverter.getPaddingOrMarginValue(6),
                textSizeConverter.getPaddingOrMarginValue(16),
                0,
                0,
            )
            binding.txtScanNotCompleteDesc2.layoutParams = layoutParams
            layoutParams = binding.imgInfo1.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(16), 0, 0)
            binding.imgInfo1.layoutParams = layoutParams
            layoutParams = binding.imgInfo2.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(0, textSizeConverter.getPaddingOrMarginValue(16), 0, 0)
            binding.imgInfo2.layoutParams = layoutParams
            layoutParams = binding.btnRetry.layoutParams as LinearLayout.LayoutParams
            layoutParams.setMargins(
                0,
                textSizeConverter.getPaddingOrMarginValue(16),
                0,
                textSizeConverter.getPaddingOrMarginValue(12),
            )
            binding.btnRetry.layoutParams = layoutParams
            padding = textSizeConverter.getPaddingOrMarginValue(12)
            binding.txtBtnRetry.setPadding(0, padding, 0, padding)
            binding.btnRetry.radius = textSizeConverter.calculateRadius(8).toFloat()
            binding.txtBtnRetry.setTextSize(
                TypedValue.COMPLEX_UNIT_PX,
                textSizeConverter.getTextSize(16).toFloat(),
            )
            binding.btnRetry.setOnClickListener {
                errorDialog?.dismiss()
                progressDialog?.dismiss()
                viewModel.cancelNfcRead()
            }
            errorDialog!!.show()
            errorDialog!!.setCancelable(false)
            errorDialog!!.window!!.setLayout(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.WRAP_CONTENT,
            )
            errorDialog!!.window!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
            errorDialog!!.window!!.attributes.windowAnimations = R.style.DialogAnimation
            errorDialog!!.window!!.setGravity(Gravity.BOTTOM)
            if (progressDialog != null) progressDialog!!.dismiss()
        }
    }

    private fun initAdapter() {
        mNfcAdapter = NfcAdapter.getDefaultAdapter(this)
        if (mNfcAdapter != null && mNfcAdapter!!.isEnabled) {
            // Yes NFC available
            pendingIntent =
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                    val intent =
                        Intent(
                            this@NfcActivity,
                            this@NfcActivity.javaClass,
                        ).addFlags(
                            Intent.FLAG_ACTIVITY_SINGLE_TOP,
                        )
                    PendingIntent.getActivity(
                        this@NfcActivity,
                        0,
                        intent,
                        PendingIntent.FLAG_MUTABLE,
                    )
                } else {
                    PendingIntent.getActivity(
                        this,
                        0,
                        Intent(this, javaClass).addFlags(Intent.FLAG_ACTIVITY_SINGLE_TOP),
                        0,
                    )
                }
        } else if (mNfcAdapter != null && !mNfcAdapter!!.isEnabled) {
            showNfcNotEnabledDialog()
        } else {
            showNfcNotSupportedDialog()
        }
    }

    private fun showNfcNotSupportedDialog() {
        binding?.apply {
            nfcDialogTurnon.mainDialogLyt.visibility = View.VISIBLE
            nfcDialogTurnon.lyoutError1.visibility = View.VISIBLE
            nfcDialogTurnon.txtScanNotCompleteDesc1.text = getString(R.string.nfc_not_supported_desc)
            nfcDialogTurnon.txtBtnRetry.text = getString(R.string.nfc_not_enabled_btn_ok)
            nfcDialogTurnon.btnRetry.setOnClickListener {
                nfcDialogTurnon.mainDialogLyt.visibility = View.GONE
            }
        }
    }

    private fun showNfcNotEnabledDialog() {
        binding?.apply {
            nfcDialogTurnon.mainDialogLyt.visibility = View.VISIBLE
            nfcDialogTurnon.btnRetry.setOnClickListener {
                val nfcIntent = Intent(Settings.ACTION_NFC_SETTINGS)
                someActivityResultLauncher.launch(nfcIntent)
                nfcDialogTurnon.mainDialogLyt.visibility = View.GONE
            }
        }
    }

    private val someActivityResultLauncher =
        registerForActivityResult(
            ActivityResultContracts.StartActivityForResult(),
        ) { result ->
            if (result.resultCode == Activity.RESULT_OK) {
                mNfcAdapter = NfcAdapter.getDefaultAdapter(this)
                if (mNfcAdapter != null && mNfcAdapter!!.isEnabled) {

                    // Yes NFC available
                    pendingIntent =
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                            val intent =
                                Intent(
                                    this@NfcActivity,
                                    this@NfcActivity.javaClass,
                                ).addFlags(
                                    Intent.FLAG_ACTIVITY_SINGLE_TOP,
                                )
                            PendingIntent.getActivity(
                                this@NfcActivity,
                                0,
                                intent,
                                PendingIntent.FLAG_MUTABLE,
                            )
                        } else {
                            PendingIntent.getActivity(
                                this,
                                0,
                                Intent(this, javaClass).addFlags(Intent.FLAG_ACTIVITY_SINGLE_TOP),
                                0,
                            )
                        }
                } else if (mNfcAdapter != null && !mNfcAdapter!!.isEnabled) {
                    showNfcNotEnabledDialog()
                } else {
                    showNfcNotSupportedDialog()
                }
            }
        }

    private fun setOtherViews() {
        binding!!.layoutHeader.imgBack.setOnClickListener { onBackPressedDispatcher.onBackPressed() }
        try {
            val matrix = Matrix()
            matrix.postRotate(90f)
            val bmp =
                BitmapFactory.decodeByteArray(
                    Utility.getInstance().scannedImage,
                    0,
                    Utility.getInstance().scannedImage!!.size,
                )
            val scaledBitmap = Bitmap.createScaledBitmap(bmp, 2048, 2048, true)
            val rotatedBitmap =
                Bitmap.createBitmap(
                    scaledBitmap,
                    0,
                    0,
                    scaledBitmap.width,
                    scaledBitmap.height,
                    matrix,
                    true,
                )
            val image = findViewById<ImageView>(R.id.imageview)
            image.setImageBitmap(rotatedBitmap)
        } catch (e: Exception) {
            e.printStackTrace()
        }
        binding!!.txtSkip.setOnClickListener {
            OtherConstant.getInstance().skipNFC = true
            dismissDialog()
            viewModel.validateLivensUser()
        }
    }

    private fun dismissDialog() {
        if (errorDialog != null) {
            errorDialog!!.dismiss()
        }
        if (progressDialog != null) {
            progressDialog!!.dismiss()
        }
    }

    private fun handleIntentData() {
        if (intent.hasExtra(getString(R.string.doc_key))) {
            val selectedDoc = intent.getStringExtra(getString(R.string.doc_key))
            if (selectedDoc != null) {
                viewModel.setSelectedDoc(selectedDoc)
            }
        }
    }

    public override fun onPause() {
        super.onPause()
        if (mNfcAdapter != null) {
            mNfcAdapter!!.disableForegroundDispatch(this)
        }
        if (this.isFinishing) {
            isActivityRunning = false
            viewModel.cancelNfcRead()
            if (progressDialog != null && progressDialog!!.isShowing) {
                progressDialog!!.dismiss()
            }
        }
    }

    public override fun onResume() {
        super.onResume()
        try {
            if (mNfcAdapter != null && pendingIntent != null) {
                mNfcAdapter!!.enableForegroundDispatch(this, pendingIntent, null, null)
            }
        } catch (e: Exception) {
            println("onResume error")
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        ErrorUtility.getInstance().unregisterConnectivityManager(this)
        val disposable = viewModel.getDisposable()
        if (!disposable.isDisposed) {
            disposable.dispose()
            disposable.clear()
        }
        isActivityRunning = false
        viewModel.cancelNfcRead()
        viewModel.getNfcLiveData().resetState()
    }

    override fun onStop() {
        super.onStop()
        isActivityRunning = false
        viewModel.cancelNfcRead()
    }

    private fun backPress() {
        onBackPressedDispatcher.addCallback(this) {
            if (binding!!.customLoader.root.visibility != View.VISIBLE) openBackPressDialog()
        }
    }

    private fun openBackPressDialog() {
        val dialog = Dialog(this)
        dialog.requestWindowFeature(Window.FEATURE_NO_TITLE)
        val binding = DialogEmailPhoneConfirmationBinding.inflate(dialog.layoutInflater)
        dialog.setContentView(binding.root)
        var layoutParams = binding.dialogHeader.layoutParams as LinearLayout.LayoutParams
        var padding = textSizeConverter.getPaddingOrMarginValue(16)
        layoutParams.setMargins(padding, padding, padding, 0)
        binding.dialogHeader.layoutParams = layoutParams
        val layoutParams2 = binding.cardConfirmation.layoutParams
        layoutParams2.width = textSizeConverter.getWidth(280)
        binding.cardConfirmation.layoutParams = layoutParams2
        binding.cardConfirmation.radius = textSizeConverter.calculateRadius(14).toFloat()
        binding.dialogHeader.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(18).toFloat(),
        )
        binding.dialogSubtitle.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(14).toFloat(),
        )
        binding.btnNo.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSizeConverter.getTextSize(16).toFloat())
        binding.btnYes.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSizeConverter.getTextSize(16).toFloat())
        layoutParams = binding.dialogSubtitle.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(padding, padding, padding, padding)
        binding.dialogSubtitle.layoutParams = layoutParams
        padding = textSizeConverter.getPaddingOrMarginValue(12)
        layoutParams = binding.btnNo.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(0, padding, 0, padding)
        binding.btnNo.layoutParams = layoutParams
        layoutParams = binding.btnYes.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(
            textSizeConverter.getPaddingOrMarginValue(24),
            padding,
            textSizeConverter.getPaddingOrMarginValue(16),
            padding,
        )
        binding.btnYes.layoutParams = layoutParams
        binding.dialogHeader.typeface = ResourcesCompat.getFont(applicationContext, R.font.pingmedium)
        binding.dialogSubtitle.typeface = ResourcesCompat.getFont(applicationContext, R.font.regular)
        binding.dialogHeader.text = getString(R.string.back_dialog_header)
        binding.dialogSubtitle.text = getString(R.string.back_dialog_title)
        binding.btnYes.setOnClickListener {
            dialog.dismiss()
            finish()
        }
        binding.btnNo.setOnClickListener { dialog.dismiss() }
        dialog.show()
        dialog.setCancelable(false)
        dialog.window!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
        dialog.window!!.attributes.windowAnimations = R.style.DialogAnimation
    }
}
