package nashid.verify.sdk.viewmodel

import android.app.Application
import android.graphics.Bitmap
import androidx.lifecycle.ViewModel
import nashid.verify.sdk.VerifySDKManager
import nashid.verify.sdk.data.repository.SkipLiveNessRepository
import nashid.verify.sdk.model.DocumentVerification
import nashid.verify.sdk.utils.ArtifactType
import nashid.verify.sdk.utils.OtherConstant
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.helpers.ScanDocumentResult
import nashid.verify.sdkNew.R
import java.util.Objects

class SkipNfcLiveNessViewModel(
    private val skipNfcLiveNessLiveData: SkipNfcLiveNessLiveData,
    private val skipLiveNessRepository: SkipLiveNessRepository,
    private val context: Application,
    private val scanDocumentViewModel: ScanDocumentViewModel,
) : ViewModel() {
    init {
        observeScanDocumentViewModel()
    }

    private fun observeScanDocumentViewModel() {
        scanDocumentViewModel.result.observeForever { result ->
            if (result is ScanDocumentResult.FileKeyResult) {
                Utility.getInstance().saveFileKeyAndContinue(result.fileKey, result.artifactType)
                when (result.artifactType) {
                    ArtifactType.LIVENESS_IMAGE.type -> {
                        skipLiveNessRepository.callToComparison()
                    }
                }
            }
            if (result is ScanDocumentResult.Error) {
                skipLiveNessRepository.callToComparison()
            }
        }
    }

    fun handleInternetConnectionData(isInternetAvailable: Boolean) {
        skipNfcLiveNessLiveData.setInternetConnection(isInternetAvailable)
        if (!isInternetAvailable) {
            skipNfcLiveNessLiveData.setInternetConnection(false)
        } else {
            skipNfcLiveNessLiveData.setIsApiCalled(true)
            if (!SdkConfig.isFaceMatchingEnabled) {
                skipLiveNessRepository.callToComparison()
            } else {
                skipNfcLiveNessLiveData.setIsLiveness(true)
            }
        }
    }

    fun getLiveNessData() = skipNfcLiveNessLiveData

    fun uploadArtifact(
        bitmap: Bitmap?,
        artifactType: ArtifactType,
    ) {
        val file = Utility.getInstance().bitmapToFile(bitmap!!, context)
        val filePart = Utility.getInstance().createMultipartBody(file)
        scanDocumentViewModel.uploadArtifact(
            SdkConfig.registerToken!!,
            Utility.getInstance().getAppKey(),
            filePart,
            artifactType.type,
        )
    }

    private fun submitVerification(verification: DocumentVerification) {
        VerifySDKManager.getInstance().submitVerification(verification)
    }

    fun validateJustCompletion() {
        skipLiveNessRepository.callToComparison()
    }

    fun submitVerification(
        latitude: Double,
        longitude: Double,
        isShowStartScanning: Boolean,
    ) {
        val documentVerification =
            prepareResultData(latitude, longitude, isShowStartScanning)
        submitVerification(documentVerification)
    }

    private fun prepareResultData(
        latitude: Double,
        longitude: Double,
        isShowStartScanning: Boolean,
    ): DocumentVerification {
        var isFlowCompleted = true
        if (isShowStartScanning) {
            skipNfcLiveNessLiveData.setIsLiveness(false)
            isFlowCompleted = false
        }
        val documentType =
            if (Objects.requireNonNull<String?>(skipNfcLiveNessLiveData.getSelectedDoc().value)
                    .equals(context.getString(R.string.e_passport), ignoreCase = true)
            ) {
                2
            } else {
                1
            }
        val isNFCEnabled = SdkConfig.isNfcScanningEnabled && !OtherConstant.getInstance().skipNFC
        val data = skipLiveNessRepository.createData(isNFCEnabled, documentType, skipNfcLiveNessLiveData.getIsLiveness().value)
        val metaData = skipLiveNessRepository.createMetaData(latitude, longitude)
        val artifacts = skipLiveNessRepository.handleArtifacts()
        return DocumentVerification(
            verificationMetadata = metaData,
            type = documentType,
            data = data,
            artifacts = artifacts,
            status = if (isFlowCompleted) 1 else 2,
        )
    }
}
