package nashid.verify.sdk.data.repository

import nashid.verify.sdk.model.Artifacts
import nashid.verify.sdk.model.Liveness
import nashid.verify.sdk.model.Location
import nashid.verify.sdk.model.MainData
import nashid.verify.sdk.model.MetaDataModel
import nashid.verify.sdk.model.NFC
import nashid.verify.sdk.model.Scan
import nashid.verify.sdk.utils.ArtifactType
import nashid.verify.sdk.utils.LivenessData
import nashid.verify.sdk.utils.OtherConstant
import nashid.verify.sdk.utils.ScanDocument
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG10File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG11File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG13File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG1File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG4File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG6File
import nashid.verify.sdk.viewmodel.SkipNfcLiveNessLiveData

class SkipLiveNessRepository(
    private val skipNfcLiveNessLiveData: SkipNfcLiveNessLiveData,
) {
    fun callToComparison() {
        skipNfcLiveNessLiveData.setHandleLocationData(true)
    }

    fun createData(
        isNFCEnabled: Boolean,
        documentType: Int,
        isLiveNessEnabled: Boolean?,
    ): MainData {
        return MainData(
            scan = createScanData(documentType) ?: Scan(),
            nfc = createNfcData(documentType, isNFCEnabled) ?: NFC(),
            liveness = createLiveNess(isLiveNessEnabled ?: false),
        )
    }

    private fun createLiveNess(isLiveNessEnabled: Boolean): Liveness {
        if (isLiveNessEnabled) {
            val matchValue = Utility.getInstance().matchPercentage
            return Liveness(
                passiveLivenessConfirmed = matchValue >= 50,
                score = matchValue.toString(),
            )
        }
        return Liveness()
    }

    private fun createScanData(documentType: Int): Scan? {
        val mrzLines =
            when (documentType) {
                1 ->
                    listOfNotNull(
                        Utility.getInstance().mrzLine1,
                        Utility.getInstance().mrzLine2,
                        Utility.getInstance().mrzLine3,
                    )

                2 ->
                    listOfNotNull(
                        Utility.getInstance().mrzLine1,
                        Utility.getInstance().mrzLine2,
                    )

                else -> return null
            }
        return createScan(mrzLines)
    }

    private fun createScan(mrzLines: List<String>): Scan {
        val utility = Utility.getInstance()
        val scanDocument = ScanDocument.getInstance()

        return Scan(
            mrzText = mrzLines.joinToString("\n"),
            documentNo = utility.passportNumber?.replace("<", "") ?: "",
            documentType = utility.documentType ?: "",
            expiryDate = scanDocument.expiryDate?.let { utility.getFormattedDate(it) } ?: "",
            dateOfBirth = scanDocument.dateOfBirth?.let { utility.getFormattedDate(it) } ?: "",
            fullName = utility.name ?: "",
            nationality = utility.nationality ?: "",
            country = utility.country ?: "",
            gender = if (utility.gender.equals("f", ignoreCase = true)) "Female" else "Male",
        )
    }

    private fun createNfcData(
        documentType: Int,
        isNFCEnabled: Boolean,
    ): NFC? {
        return if (isNFCEnabled) {
            when (documentType) {
                1 -> createIdCardNfcData()
                2 -> createPassportNfcData()
                else -> null
            }
        } else {
            null
        }
    }

    private fun createPassportNfcData(): NFC? {
        return try {
            val eDocument = LivenessData.getInstance().geteDocument()
            val personDetails = eDocument.personDetails
            val additionalPersonDetails = eDocument.additionalPersonDetails
            val additionalDGFiles = eDocument.additionalDGFiles
            val placeOfBirth = additionalPersonDetails.placeOfBirth?.getOrNull(0) ?: ""

            NFC(
                fullNameEnglish = additionalPersonDetails.nameOfHolder!!.replace("null ", ""),
                nationalityEnglish = personDetails.personalNumber,
                genderEnglish = personDetails.gender,
                genderArabic = personDetails.gender,
                companyNameEnglish = additionalPersonDetails?.profession ?: "",
                permitType = additionalPersonDetails?.title ?: "",
                countryOfBirthArabic = personDetails.nationality,
                placeOfIssueArabic = personDetails.issuerAuthority,
                identityNumber = personDetails.serialNumber,
                placeOfIssueEnglish = personDetails.issuerAuthority,
                expiryDate = personDetails.expiryDate,
                visaNumber = additionalDGFiles?.personalizationDeviceSerialNumber ?: "",
                issueDate = additionalDGFiles?.personalizationTime ?: "",
                fullNameArabic = additionalPersonDetails?.nameOfHolder ?: "",
                companyNameArabic = additionalPersonDetails?.profession ?: "",
                countryOfBirthEnglish = placeOfBirth,
                nationalityArabic = personDetails.nationality,
                companyAddressArabic = additionalPersonDetails?.permanentAddress?.get(0) ?: "",
                permitNumber = additionalDGFiles?.personalizationDeviceSerialNumber ?: "",
                dateOfBirth = personDetails.birthDate,
                useByDate = personDetails.expiryDate,
            )
        } catch (e: Exception) {
            e.printStackTrace()
            null
        }
    }

    private fun createIdCardNfcData(): NFC? {
        return try {
            val dg1Parser = OmanCardDG1File(Utility.getInstance().dgFile1)
            val dg4Parser = OmanCardDG4File(Utility.getInstance().dgFile4)
            val dg6Parser = OmanCardDG6File(Utility.getInstance().dgFile6)
            val dg10Parser = OmanCardDG10File(Utility.getInstance().dgFile10)
            val dg13Parser = OmanCardDG13File(Utility.getInstance().dgFile13)
            val dg11Parser = OmanCardDG11File(Utility.getInstance().dgFile11)
            val placeOfIssueArabic =
                dg1Parser.placeOfIssueArabic.let {
                    if (it.startsWith("\\u")) OtherConstant.getInstance().getEncodedString(it) else it
                }
            val fullNameArabic =
                dg4Parser.fullNameArabic.let {
                    if (it.startsWith("\\u")) OtherConstant.getInstance().getEncodedString(it) else it
                }
            val countryOfBirthArabic =
                dg6Parser.countryOfBirthArabic.let {
                    if (it.startsWith("\\u")) OtherConstant.getInstance().getEncodedString(it) else it
                }
            val countryOfBirthEnglish =
                dg6Parser.placeOfBirthEnglish.let {
                    if (it.startsWith("\\u")) OtherConstant.getInstance().getEncodedString(it) else it
                }
            val nationalityArabic =
                dg6Parser.nationalityArabic.let {
                    if (it.startsWith("\\u")) OtherConstant.getInstance().getEncodedString(it) else it
                }
            val genderArabic =
                dg6Parser.genderArabic.let {
                    if (it.startsWith("\\u")) OtherConstant.getInstance().getEncodedString(it) else it
                }
            NFC(
                fullNameEnglish = dg4Parser.fullNameEnglish,
                nationalityEnglish = dg6Parser.nationalityEnglish,
                genderEnglish = dg6Parser.genderEnglish,
                genderArabic = genderArabic,
                companyNameEnglish = dg11Parser.companyNameEnglish ?: "",
                permitType = dg13Parser.sponsorRelationship ?: "",
                countryOfBirthArabic = countryOfBirthArabic,
                placeOfIssueArabic = placeOfIssueArabic,
                identityNumber = dg1Parser.idNumber,
                placeOfIssueEnglish = dg1Parser.placeOfIssueEnglish,
                expiryDate = dg1Parser.expiryDate?.let { Utility.getInstance().getFormattedDate(it) } ?: "",
                visaNumber = dg13Parser.sponsorNumber ?: "",
                issueDate = dg1Parser.issueDate?.let { Utility.getInstance().getFormattedDate(it) } ?: "",
                fullNameArabic = fullNameArabic,
                companyNameArabic = dg11Parser.companyNameArabic ?: "",
                countryOfBirthEnglish = countryOfBirthEnglish,
                nationalityArabic = nationalityArabic,
                companyAddressArabic = dg11Parser.companyAddressArabic ?: "",
                permitNumber = dg10Parser.visaNumber ?: "",
                dateOfBirth = dg6Parser.dateOfBirth?.let { Utility.getInstance().getFormattedDate(it) } ?: "",
                useByDate = dg1Parser.expiryDate?.let { Utility.getInstance().getFormattedDate(it) } ?: "",
            )
        } catch (e: Exception) {
            e.printStackTrace()
            null
        }
    }

    fun createMetaData(
        latitude: Double,
        longitude: Double,
        onMetaDataReady: (MetaDataModel) -> Unit,
    ) {
        Utility.getInstance().fetchPublicIPAddresses { ipv4, ipv6 ->
            val metadata =
                MetaDataModel(
                    timestamp = Utility.getInstance().getCurrentFormattedDateTime() ?: "",
                    location = Location(latitude, longitude),
                    deviceIp4 = ipv4 ?: "Unknown",
                    deviceIp6 = ipv6 ?: "Unknown",
                    systemName = Utility.getInstance().getSystemName(),
                    systemVersion = Utility.getInstance().getSystemVersion(),
                    deviceIdentifier = Utility.getInstance().getDeviceIdentifier(),
                    deviceType = Utility.getInstance().getDeviceType(),
                    gender = if (Utility.getInstance().gender.equals("f", ignoreCase = true)) "Female" else "Male",
                    extraData = SdkConfig.isExtraData,
                )
            onMetaDataReady(metadata)
        }
    }

    fun handleArtifacts(): Artifacts {
        return Artifacts(
            backSideImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.BACKSIDE_IMAGE) ?: "",
            frontSideImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.FRONTSIDE_IMAGE) ?: "",
            ocrFaceImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.OCR_FACE_IMAGE) ?: "",
            nfcFaceImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.NFC_IMAGE) ?: "",
            livenessImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.LIVENESS_IMAGE) ?: "",
            livenessImageWithoutBg = Utility.getInstance().getFileKeyForArtifact(ArtifactType.LIVENESS_IMAGE_WITHOUT_BG) ?: "",
        )
    }
}
