package nashid.verify.sdk.ui

import android.app.Activity
import android.content.Context
import android.content.Intent
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.Card
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.colorResource
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.Font
import androidx.compose.ui.text.font.FontFamily
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.navigation.NavController
import com.airbnb.lottie.compose.LottieAnimation
import com.airbnb.lottie.compose.LottieCompositionSpec
import com.airbnb.lottie.compose.rememberLottieComposition
import nashid.verify.sdk.model.DocumentType
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdkNew.R

@Composable
fun ScanningPreview(onBackClick: () -> Unit) {
    val context = LocalContext.current
    Box(
        modifier =
            Modifier
                .fillMaxSize()
                .background(Color.White),
    ) {
        Column(
            modifier =
                Modifier
                    .fillMaxSize(),
        ) {
            Row(
                modifier =
                    Modifier
                        .fillMaxWidth()
                        .padding(top = 10.dp, end = 25.dp, start = 5.dp),
                verticalAlignment = Alignment.CenterVertically,
            ) {
                IconButton(onClick = onBackClick) {
                    Icon(
                        painter = painterResource(R.drawable.back),
                        contentDescription = "Back Arrow",
                        tint = colorResource(id = R.color.header_color),
                    )
                }
                Text(
                    text = stringResource(id = R.string.header_doc_scan),
                    fontFamily = FontFamily(Font(R.font.pingmedium)),
                    color = colorResource(id = R.color.header_color),
                    fontSize = 18.sp,
                    textAlign = TextAlign.Center,
                    modifier = Modifier.weight(4f),
                )
            }

            Box(
                modifier =
                    Modifier
                        .weight(1f)
                        .padding(20.dp),
            ) {
                LottieAnimationScreen(
                    animationRes =
                        when (SdkConfig.viewType) {
                            DocumentType.OMAN_ID -> R.raw.idscan
                            DocumentType.Passport -> R.raw.passportscan
                            else -> R.raw.failure
                        },
                )
            }
            val descriptionRes =
                when (SdkConfig.viewType) {
                    DocumentType.OMAN_ID -> R.string.doc_scan_desc
                    DocumentType.Passport -> R.string.pass_scan_desc
                    else -> R.string.error_unknown
                }
            ScanDescription(stringResource(id = descriptionRes))
            StartVerification {
                Utility.getInstance().cleanup()
                if (SdkConfig.isOcrEnabled || SdkConfig.isMrzEnabled) {
                    openOcrView(context)
                } else if (SdkConfig.isNfcScanningEnabled) {
                    openNFC(context)
                } else {
                    openLiveNess(context)
                }
            }
        }
    }
}

fun openOcrView(context: Context) {
    val intent = Intent(context, CameraXLiveActivity::class.java)
    when (SdkConfig.viewType) {
        DocumentType.Passport -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.e_passport))
        }

        DocumentType.OMAN_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.id_card))
        }

        else -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.error_unknown))
        }
    }
    intent.flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK
    (context as? Activity)?.let {
        it.finish()
        it.startActivity(intent)
    }
}

private fun openNFC(context: Context) {
    val intent = Intent(context, NfcActivity::class.java)
    when (SdkConfig.viewType) {
        DocumentType.Passport -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.e_passport))
        }

        DocumentType.OMAN_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.id_card))
        }

        else -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.error_unknown))
        }
    }
    intent.setFlags(
        Intent.FLAG_ACTIVITY_CLEAR_TOP or
            Intent.FLAG_ACTIVITY_CLEAR_TASK or
            Intent.FLAG_ACTIVITY_NEW_TASK,
    )
    context.startActivity(intent)
}

private fun openLiveNess(context: Context) {
    val intent = Intent(context, SkipNfcLiveNessActivity::class.java)
    when (SdkConfig.viewType) {
        DocumentType.Passport -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.e_passport))
            Utility.getInstance().selectedDoc = context.getString(R.string.e_passport)
        }

        DocumentType.OMAN_ID -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.id_card))
            Utility.getInstance().selectedDoc = context.getString(R.string.id_card)
        }

        else -> {
            intent.putExtra(context.getString(R.string.doc_key), context.getString(R.string.error_unknown))
        }
    }
    intent.setFlags(
        Intent.FLAG_ACTIVITY_CLEAR_TASK or
            Intent.FLAG_ACTIVITY_NEW_TASK,
    )
    context.startActivity(intent)
}

@Composable
fun ScanDescription(description: String) {
    Text(
        modifier = Modifier.padding(10.dp),
        text = description,
        fontFamily = FontFamily(Font(R.font.regular)),
        color = colorResource(id = R.color.header_color),
        fontSize = 18.sp,
        textAlign = TextAlign.Center,
    )
}

@Composable
fun StartVerification(startVerificationClick: () -> Unit) {
    Card(
        shape = RoundedCornerShape(6.dp),
        onClick = startVerificationClick,
        colors = CardDefaults.cardColors(colorResource(id = R.color.button_color)),
        modifier =
            Modifier
                .fillMaxWidth()
                .padding(16.dp),
    ) {
        Text(
            text = stringResource(id = R.string.btn_start_verificaion),
            fontFamily = FontFamily(Font(R.font.pingmedium)),
            color = colorResource(id = R.color.white),
            modifier =
                Modifier
                    .padding(vertical = 16.dp)
                    .fillMaxWidth(),
            fontSize = 16.sp,
            textAlign = TextAlign.Center,
        )
    }
}

@Composable
fun LottieAnimationScreen(animationRes: Int) {
    val composition by rememberLottieComposition(LottieCompositionSpec.RawRes(animationRes))
    Box(
        modifier =
            Modifier
                .fillMaxSize(),
    ) {
        LottieAnimation(
            composition = composition,
            modifier = Modifier.fillMaxSize(),
            iterations = Int.MAX_VALUE,
        )
    }
}

@Composable
fun StartScanning(navController: NavController) {
    val context = LocalContext.current
    ScanningPreview {
        (context as? Activity)?.finish()
    }
}

@Preview(showBackground = true)
@Composable
fun StartScanning() {
    ScanningPreview(
        onBackClick = {},
    )
}
