package nashid.verify.sdk.viewmodel

import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import nashid.verify.sdk.data.repository.AuthRepository
import nashid.verify.sdk.model.AuthResponse
import nashid.verify.sdk.model.RefreshTokenResponse
import nashid.verify.sdk.request.AuthRequest
import nashid.verify.sdk.request.RefreshTokenRequest
import nashid.verify.sdk.utils.NetWorkResult
import nashid.verify.sdk.utils.SdkConfig

class AuthViewModel(private val authRepository: AuthRepository) : ViewModel() {
    private val _authResult = MutableLiveData<NetWorkResult<AuthResponse>>()
    val authResult: LiveData<NetWorkResult<AuthResponse>> = _authResult

    private val _refreshTokenResult = MutableLiveData<NetWorkResult<RefreshTokenResponse>>()
    val refreshTokenResult: LiveData<NetWorkResult<RefreshTokenResponse>> = _refreshTokenResult

    private var refreshToken: String? = null

    fun authenticateUser(authRequest: AuthRequest) {
        if (authRequest.sdkKey.isBlank() || authRequest.sdkSecret.isBlank()) {
            _authResult.value = NetWorkResult.Error("SDK Key and Secret Key are required")
            return
        }

        _authResult.value = NetWorkResult.Loading(true)

        viewModelScope.launch {
            val result = authRepository.authenticateUser(authRequest)
            _authResult.value = result
            if (result is NetWorkResult.Success) {
                refreshToken = result.data?.data?.refreshToken
                SdkConfig.registerToken = result.data?.data?.accessToken
                scheduleTokenRefresh(result.data?.data?.accessTokenExpireAt?.toLong())
            }
        }
    }

    // Method to refresh token
    private fun refreshToken(refreshTokenRequest: RefreshTokenRequest) {
        viewModelScope.launch {
            val result = authRepository.refreshToken(refreshTokenRequest)
            _refreshTokenResult.value = result
            SdkConfig.registerToken = result.data?.data?.accessToken
            if (result is NetWorkResult.Success) {
                scheduleTokenRefresh(result.data?.data?.accessTokenExpireAt?.toLong())
            }
        }
    }

    private fun scheduleTokenRefresh(accessTokenExpiryTime: Long?) {
        val currentTime = System.currentTimeMillis() / 1000
        if (accessTokenExpiryTime != null) {
            val refreshDelay = accessTokenExpiryTime - currentTime
            if (refreshDelay > 0) {
                viewModelScope.launch(Dispatchers.IO) {
                    // Schedule the refresh after the calculated delay (converted to milliseconds)
                    delay(refreshDelay * 1000) // Convert seconds to milliseconds
                    refreshToken?.let { RefreshTokenRequest(it) }?.let { refreshToken(it) }
                }
            } else {
                println("Token is either expired or too close to expiry for a refresh.")
            }
        } else {
            println("Access token expiry time is null.")
        }
    }
}
