package nashid.verify.sdk

import android.content.Context
import android.content.Intent
import androidx.lifecycle.Observer
import nashid.verify.sdk.model.AuthCallback
import nashid.verify.sdk.model.AuthResponse
import nashid.verify.sdk.model.DocumentType
import nashid.verify.sdk.model.DocumentVerification
import nashid.verify.sdk.model.ExtraData
import nashid.verify.sdk.model.SDKStatus
import nashid.verify.sdk.model.ScanDocumentResult
import nashid.verify.sdk.request.AuthRequest
import nashid.verify.sdk.ui.MainScreen
import nashid.verify.sdk.utils.KoinEnvironment
import nashid.verify.sdk.utils.Loggers
import nashid.verify.sdk.utils.NetWorkResult
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.helpers.LocaleUtil.Companion.applyLocalizedContext
import nashid.verify.sdk.utils.helpers.Storage
import nashid.verify.sdk.viewmodel.AuthViewModel
import nashid.verify.sdk.viewmodel.ScanDocumentViewModel
import nashid.verify.sdk.viewmodel.SkipNfcLiveNessViewModel
import org.koin.core.Koin
import org.koin.core.component.KoinComponent
import org.koin.core.component.inject

internal class VerifySDKManagerImpl : KoinComponent, VerifySDKManager {
    override fun getKoin(): Koin {
        if (KoinEnvironment.koin == null) {
            throw IllegalStateException("Koin has not been initialized. Make sure MyEnvironment.initKoin(context) is called before using VerifySDKManager.")
        }
        return KoinEnvironment.koin!!
    }

    private val viewModel: AuthViewModel by inject()
    private val scanDocumentViewModel: ScanDocumentViewModel by inject()
    private val skipNfcLiveNessViewModel: SkipNfcLiveNessViewModel by inject()
    private val context: Context by inject()
    private val logger = Loggers.withTag("VerifySDKManager")
    private var authResultObserver: Observer<NetWorkResult<AuthResponse>>? = null
    private var scanDocumentObserver: Observer<ScanDocumentResult>? = null

    override fun initialize(
        sdkKey: String,
        sdkSecret: String,
        languageType: String,
        extraData: ExtraData?,
        callback: AuthCallback,
    ) {
        resetInstance()
        logger.log("Starting initialization with SDK Keys")
        authResultObserver =
            Observer { result ->
                when (result) {
                    is NetWorkResult.Success -> {
                        logger.log("SDK initialization successful: ${result.data?.toString()}")
                        callback.initializeResponse(true, result.data?.message)
                        result.data?.data?.sdkConfig?.let { sdkConfig ->
                            SdkConfig.updateConfig(
                                sdkKey = sdkConfig.sandboxSdkKey,
                                sdkSecret = sdkConfig.sandboxSdkSecret,
                                languageType = languageType,
                                sdkName = sdkConfig.name,
                                sdkDescription = sdkConfig.description,
                                nfcScanningEnabled = sdkConfig.isNfcScanningEnabled,
                                nfcMultiTriesEnabled = sdkConfig.isNfcMultiTriesEnabled,
                                faceMatchingEnabled = sdkConfig.isFaceMatchingEnabled,
                                ocrEnabled = sdkConfig.isOcrEnabled,
                                mrzEnabled = sdkConfig.isMrzEnabled,
                                protocol = sdkConfig.callbackProtocol,
                                host = sdkConfig.callbackHost,
                                instructionsVisible = sdkConfig.showInstructions,
                                nfcSkippable = sdkConfig.isNfcSkippable,
                                nfcRetries = sdkConfig.nfcMaximumRetries ?: 0,
                                extraData = extraData,
                            )
                        }
                    }

                    is NetWorkResult.Error -> {
                        logger.log("Initialization failed: ${result.message}")
                        SdkConfig.sdkStatus = SDKStatus.FAILED
                        callback.initializeResponse(false, result.message)
                    }

                    else -> {}
                }
            }

        authResultObserver?.let { viewModel.authResult.observeForever(it) }

        // Trigger authentication
        viewModel.authenticateUser(AuthRequest(sdkKey, sdkSecret))

        // Configure the SDK appearance
        Storage(context).setPreferredLocale(languageType)
        applyLocalizedContext(context, languageType)
    }

    override fun verify(
        documentType: DocumentType,
        callback: (ScanDocumentResult.ScanDocumentResponse) -> Unit,
    ) {
        if (SdkConfig.sdkStatus == SDKStatus.INITIALIZED) {
            logger.log("scanDocument called with DocumentType: ${documentType.displayName}")
            SdkConfig.viewType = documentType
            scanDocumentObserver =
                Observer {
                    if (it is ScanDocumentResult.ScanDocumentResponse) {
                        callback(it)
                        if (!it.result) {
                            skipNfcLiveNessViewModel.getLiveNessData().hideProgressBar()
                        } else {
                            skipNfcLiveNessViewModel.getLiveNessData().requestFinishActivity()
                        }
                    }
                }
            scanDocumentObserver?.let { scanDocumentViewModel.result.observeForever(it) }
            val intent =
                Intent(context, MainScreen::class.java).apply {
                    flags = Intent.FLAG_ACTIVITY_NEW_TASK
                }
            context.startActivity(intent)
        } else {
            logger.log("Failed to initialize SDK")
            callback(ScanDocumentResult.ScanDocumentResponse(false, "Failed to initialize SDK", ""))
        }
    }

    override fun getVerificationResult(
        verificationId: String,
        callback: (ScanDocumentResult.GetScanResultResponse) -> Unit,
    ) {
        logger.log("getScanResult called with verificationId: $verificationId")
        if (SdkConfig.sdkStatus == SDKStatus.INITIALIZED) {
            scanDocumentViewModel.getVerification(
                SdkConfig.registerToken!!,
                Utility.getInstance().getAppKey(),
                verificationId,
            )
            scanDocumentObserver =
                Observer {
                    if (it is ScanDocumentResult.GetScanResultResponse) {
                        callback(it)
                        skipNfcLiveNessViewModel.getLiveNessData().requestFinishActivity()
                    }
                }
            scanDocumentObserver?.let { scanDocumentViewModel.result.observeForever(it) }
        } else {
            logger.log("SDK not initialized")
        }
    }

    companion object {
        @Volatile
        private var instance: VerifySDKManagerImpl? = null

        @JvmStatic
        fun getInstance(): VerifySDKManagerImpl =
            instance ?: synchronized(this) {
                instance ?: VerifySDKManagerImpl().also { instance = it }
            }
    }

    override fun submitVerification(verification: DocumentVerification) {
        scanDocumentViewModel.submitVerification(
            SdkConfig.registerToken!!,
            Utility.getInstance().getAppKey(),
            verification,
        )
    }

    private fun resetInstance() {
        instance = null
        SdkConfig.clear()
        authResultObserver?.let { viewModel.authResult.removeObserver(it) }
        scanDocumentObserver?.let { scanDocumentViewModel.result.removeObserver(it) }
        logger.log("VerifySDKManager instance and observers have been reset.")
    }
}
