package com.kyc.nashidmrz.facematches

import android.content.Context
import android.util.Log
import com.kyc.nashidmrz.BuildConfig
import com.kyc.nashidmrz.mrtd2.interfaceClass.RequestResponse
import okhttp3.Call
import okhttp3.MediaType.Companion.toMediaTypeOrNull
import okhttp3.MultipartBody
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.RequestBody.Companion.asRequestBody
import okhttp3.Response
import okhttp3.logging.HttpLoggingInterceptor
import java.io.File
import java.io.FileOutputStream
import java.io.IOException
import java.util.concurrent.TimeUnit

class FaceMatchApi {
    companion object {
        // Load C++ library
        init {
            System.loadLibrary("native-lib")
        }

        @JvmStatic
        external fun faceMatchAPI(): String

        @JvmStatic
        external fun prodFaceMatchAPI(): String
    }

    private val client: OkHttpClient

    init {
        val httpClientBuilder =
            OkHttpClient.Builder()
                .connectTimeout(30, TimeUnit.SECONDS)
                .writeTimeout(30, TimeUnit.SECONDS)
                .readTimeout(30, TimeUnit.SECONDS)

        val loggingInterceptor = HttpLoggingInterceptor()
        loggingInterceptor.setLevel(HttpLoggingInterceptor.Level.BODY)
        httpClientBuilder.addInterceptor(loggingInterceptor)

        client = httpClientBuilder.build()
    }

    fun matchFaces(
        context: Context,
        data1: ByteArray,
        data2: ByteArray,
        requestResponse: RequestResponse,
    ) {
        val mediaType = "image/jpg".toMediaTypeOrNull()
        val filename1 = File(context.filesDir, "filename1.jpg")
        if (filename1.exists()) {
            filename1.delete()
        }
        try {
            FileOutputStream(filename1).use { fos -> fos.write(data1) }
        } catch (e: IOException) {
            e.printStackTrace()
        }
        val filename2 = File(context.filesDir, "filename2.jpg")
        if (filename2.exists()) {
            filename2.delete()
        }
        try {
            FileOutputStream(filename2).use { fos -> fos.write(data2) }
        } catch (e: IOException) {
            e.printStackTrace()
        }
        val requestBody =
            MultipartBody.Builder().setType(MultipartBody.FORM)
                .addFormDataPart("file1", "filename1.jpg", filename1.asRequestBody(mediaType))
                .addFormDataPart("file2", "filename2.jpg", filename2.asRequestBody(mediaType))
                .build()
        val url =
            if (BuildConfig.API_ENDPOINT_SOURCE == "dev") {
                faceMatchAPI()
            } else {
                prodFaceMatchAPI()
            }
        Log.d(FaceMatchApi::class.java.name, "matchFaces: ${BuildConfig.API_ENDPOINT_SOURCE}")
        Log.d("TAG", "onFailure:facematch start")
        val request =
            Request.Builder()
                .url(url)
                .post(requestBody)
                .build()

        client.newCall(request).enqueue(
            object : okhttp3.Callback {
                override fun onResponse(
                    call: Call,
                    response: Response,
                ) {
                    Log.d("TAG", "onFailure:facematch success")
                    if (response.isSuccessful) {
                        if (filename1.exists()) {
                            filename1.delete()
                        }
                        if (filename2.exists()) {
                            filename2.delete()
                        }
                        val res = response.body?.string()
                        try {
                            val intValue = res?.toInt()
                            if (intValue != null) {
                                requestResponse.successResponse(intValue.toString()) // Handle the integer value
                            } else {
                                requestResponse.failureResponse(
                                    res ?: "Fail",
                                )
                            }
                        } catch (e: NumberFormatException) {
                            e.printStackTrace()
                            requestResponse.failureResponse(res ?: "Fail") // Handle the error message
                        }
                    } else {
                        requestResponse.failureResponse("Fail")
                    }
                }

                override fun onFailure(
                    call: Call,
                    e: IOException,
                ) {
                    e.printStackTrace()
                    requestResponse.failureResponse(e.message ?: "Unknown error")
                }
            },
        )
    }
}
