/*
 * JMRTD - A Java API for accessing machine readable travel documents.
 *
 * Copyright (C) 2006 - 2018  The JMRTD team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * $Id: PACEIMMappingResult.java 1763 2018-02-18 07:41:30Z martijno $
 */

package com.kyc.nashidmrz.id_card.jmrtd.protocol;

import java.security.spec.AlgorithmParameterSpec;
import java.util.Arrays;

/**
 * The result of the PACE nonce mapping step in an Integrated Mapping setting.
 *
 * @author The JMRTD team (info@jmrtd.org)
 *
 * @version $Revision: $
 */
public class PACEIMMappingResult extends PACEMappingResult {

  private static final long serialVersionUID = -6415752866407346050L;

  private final byte[] pcdNonce;

  /**
   * Constructs a result.
   *
   * @param staticParameters the static parameters
   * @param piccNonce the nonce sent by the PICC
   * @param pcdNonce the nonce generated by the PCD
   * @param ephemeralParameters the resulting ephemeral parameters
   */
  public PACEIMMappingResult(AlgorithmParameterSpec staticParameters, byte[] piccNonce, byte[] pcdNonce, AlgorithmParameterSpec ephemeralParameters) {
    super(staticParameters, piccNonce, ephemeralParameters);
    this.pcdNonce = pcdNonce == null ? null : Arrays.copyOf(pcdNonce, pcdNonce.length);
  }

  /**
   * Returns the nonce generated by the PCD.
   *
   * @return the nonce generated by the PCD
   */
  public byte[] getPCDNonce() {
    return pcdNonce == null ? null : Arrays.copyOf(pcdNonce, pcdNonce.length);
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = super.hashCode();
    result = prime * result + Arrays.hashCode(pcdNonce);
    return result;
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (!super.equals(obj)) {
      return false;
    }
    if (getClass() != obj.getClass()) {
      return false;
    }

    PACEIMMappingResult other = (PACEIMMappingResult) obj;
    return Arrays.equals(pcdNonce, other.pcdNonce);
  }
}
