package com.kyc.nashidmrz.locationdata;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.pm.PackageManager;
import android.location.Address;
import android.location.Geocoder;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.os.Bundle;
import android.os.Handler;

import androidx.annotation.NonNull;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;

import java.io.IOException;
import java.util.List;
import java.util.Locale;
import java.util.Objects;

public class LocationHelper implements LocationListener {
    private final Activity activity;
    private final LocationManager locationManager;
    private LocationCallback locationCallback;
    private static final long MIN_TIME_BW_UPDATES = 1000 * 60; // 1 minute
    private static final long MIN_DISTANCE_CHANGE_FOR_UPDATES = 10; // 10 meters
    private static final long LOCATION_TIMEOUT = 10000; // 10 seconds

    public interface LocationCallback {
        void onLocationResult(Location location, String address);
    }

    public LocationHelper(Activity activity) {
        this.activity = activity;
        locationManager = (LocationManager) activity.getSystemService(Context.LOCATION_SERVICE);
    }

    public void getLastLocation(LocationCallback callback) {
        this.locationCallback = callback;

        if (ContextCompat.checkSelfPermission(activity, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
            requestLocationPermissions();
            return;
        }

        Location location = locationManager.getLastKnownLocation(LocationManager.GPS_PROVIDER);
        if (location == null) {
            location = locationManager.getLastKnownLocation(LocationManager.NETWORK_PROVIDER);
        }

        if (location == null) {
            // If no last known location is available, request location updates
            requestLocationUpdatesWithTimeout(callback);
        } else {
            String address = getAddressFromLocation(location);
            if (locationCallback != null) {
                locationCallback.onLocationResult(location, address);
            }
        }
    }

    private void requestLocationUpdatesWithTimeout(LocationCallback callback) {
        this.locationCallback = callback;

        if (ActivityCompat.checkSelfPermission(activity, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
            requestLocationPermissions();
            return;
        }

        locationManager.requestLocationUpdates(
                LocationManager.GPS_PROVIDER,
                MIN_TIME_BW_UPDATES,
                MIN_DISTANCE_CHANGE_FOR_UPDATES,
                this
        );

        // Use a handler to remove location updates after a timeout
        new Handler().postDelayed(() -> {
            locationManager.removeUpdates(this);

            Location lastKnownLocation = locationManager.getLastKnownLocation(LocationManager.GPS_PROVIDER);
            if (lastKnownLocation == null) {
                lastKnownLocation = locationManager.getLastKnownLocation(LocationManager.NETWORK_PROVIDER);
            }

            if (lastKnownLocation != null && locationCallback != null) {
                String address = getAddressFromLocation(lastKnownLocation);
                locationCallback.onLocationResult(lastKnownLocation, address);
            } else {
                // Handle case where location is still null after timeout
                Location defaultLocation = new Location("default");
                defaultLocation.setLatitude(0.0);
                defaultLocation.setLongitude(0.0);
                if (locationCallback != null) {
                    String address = getAddressFromLocation(defaultLocation);
                    locationCallback.onLocationResult(defaultLocation, address);
                }
            }
        }, LOCATION_TIMEOUT);
    }

    @Override
    public void onLocationChanged(@NonNull Location location) {
        if (locationCallback != null) {
            String address = getAddressFromLocation(location);
            locationCallback.onLocationResult(location, address);
            locationManager.removeUpdates(this); // Stop location updates after receiving the first location
        }
    }

    @Override
    public void onStatusChanged(String provider, int status, Bundle extras) {
    }

    @Override
    public void onProviderEnabled(@NonNull String provider) {
    }

    @Override
    public void onProviderDisabled(@NonNull String provider) {
    }

    private void requestLocationPermissions() {
        ActivityCompat.requestPermissions(
                activity,
                new String[]{Manifest.permission.ACCESS_FINE_LOCATION},
                1
        );
    }

    private String getAddressFromLocation(Location location) {
        Geocoder geocoder = new Geocoder(activity, Locale.getDefault());
        try {
            List<Address> addresses = geocoder.getFromLocation(location.getLatitude(), location.getLongitude(), 1);
            if (addresses != null && !addresses.isEmpty()) {
                Address address = addresses.get(0);
                return address.getAddressLine(0);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return "Unknown location";
    }
}

