@file:Suppress("DEPRECATION")

package com.mv.liveness

import android.Manifest
import android.animation.Animator
import android.animation.AnimatorListenerAdapter
import android.annotation.SuppressLint
import android.annotation.TargetApi
import android.app.Dialog
import android.content.Context
import android.content.pm.PackageManager
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Color
import android.graphics.PointF
import android.graphics.Rect
import android.graphics.RectF
import android.graphics.SurfaceTexture
import android.graphics.drawable.ColorDrawable
import android.hardware.camera2.CameraCharacteristics
import android.hardware.camera2.CameraManager
import android.media.Image
import android.os.Build
import android.os.Bundle
import android.os.CountDownTimer
import android.util.Log
import android.util.Size
import android.util.TypedValue
import android.view.Surface
import android.view.View
import android.view.View.GONE
import android.view.View.INVISIBLE
import android.view.View.VISIBLE
import android.view.ViewGroup
import android.view.Window
import android.view.WindowInsets
import android.view.WindowInsetsController
import android.widget.LinearLayout
import android.widget.RelativeLayout
import android.widget.Toast
import androidx.camera.core.CameraSelector
import androidx.camera.core.ImageAnalysis
import androidx.camera.core.ImageCapture
import androidx.camera.core.ImageCaptureException
import androidx.camera.core.ImageProxy
import androidx.camera.core.Preview
import androidx.camera.lifecycle.ProcessCameraProvider
import androidx.core.content.ContextCompat
import androidx.core.content.res.ResourcesCompat
import androidx.core.view.doOnLayout
import com.caverock.androidsvg.SVG
import com.google.mlkit.vision.common.InputImage
import com.kyc.nashidmrz.Constant
import com.kyc.nashidmrz.R
import com.kyc.nashidmrz.Utility
import com.kyc.nashidmrz.databinding.DialogBackBinding
import com.kyc.nashidmrz.databinding.LivenessActivityMainBinding
import com.kyc.nashidmrz.mrtd2.activity.BaseActivity
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.ObsoleteCoroutinesApi
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext
import java.io.ByteArrayOutputStream
import java.nio.ReadOnlyBufferException
import java.util.concurrent.ExecutorService
import java.util.concurrent.Executors
import kotlin.experimental.inv

@ObsoleteCoroutinesApi
internal class LivenessMainActivity :
    BaseActivity() {
    private lateinit var binding: LivenessActivityMainBinding

    private var enginePrepared: Boolean = false
    private lateinit var engineWrapper: EngineWrapper
    private var threshold: Float = DEFAULT_THRESHOLD

    private val previewWidth: Int = 640
    private val previewHeight: Int = 480
    private var imageCount = 0

    private val frameOrientation: Int = 7

    private var screenWidth: Int = 0
    private var screenHeight: Int = 0

    private var factorX: Float = 0F
    private var factorY: Float = 0F
    private var scale: Float = 0F

    @OptIn(DelicateCoroutinesApi::class)
    private val detectionContext = newSingleThreadContext("detection")
    private var working: Boolean = false

    private var cameraProvider: ProcessCameraProvider? = null
    private lateinit var cameraExecutor: ExecutorService

    private var cameraSelector: CameraSelector? = null
    private var startRecording = false

    private var blink: Boolean = false
    private var smile: Boolean = false
    private var isStartTimer: Boolean = false

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = LivenessActivityMainBinding.inflate(layoutInflater)
        setContentView(binding.root)
        if (hasPermissions()) {
            init()
        } else {
            requestPermission()
        }
    }

    private fun hasPermissions(): Boolean {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            for (permission in permissions) {
                if (checkSelfPermission(permission) != PackageManager.PERMISSION_GRANTED) {
                    return false
                }
            }
        }
        return true
    }

    private fun getYuvDataFromImage(image: ImageProxy): ByteArray {
        val width = image.width
        val height = image.height
        val ySize = width * height
        val uvSize = width * height / 4

        val nv21 = ByteArray(ySize + uvSize * 2)

        val yBuffer = image.planes[0].buffer // Y

        val uBuffer = image.planes[1].buffer // U

        val vBuffer = image.planes[2].buffer // V

        var rowStride = image.planes[0].rowStride
        assert(image.planes[0].pixelStride === 1)

        var pos = 0

        if (rowStride == width) { // likely
            yBuffer[nv21, 0, ySize]
            pos += ySize
        } else {
            var yBufferPos = -rowStride.toLong() // not an actual position
            while (pos < ySize) {
                yBufferPos += rowStride.toLong()
                yBuffer.position(yBufferPos.toInt())
                yBuffer[nv21, pos, width]
                pos += width
            }
        }

        rowStride = image.planes[2].rowStride
        val pixelStride = image.planes[2].pixelStride

        assert(rowStride == image.planes[1].rowStride)
        assert(pixelStride == image.planes[1].pixelStride)

        if (pixelStride == 2 && rowStride == width && uBuffer[0] == vBuffer[1]) {
            val savePixel = vBuffer[1]
            try {
                vBuffer.put(1, savePixel.inv())
                if (uBuffer[0] == savePixel.inv()) {
                    vBuffer.put(1, savePixel)
                    vBuffer.position(0)
                    uBuffer.position(0)
                    vBuffer[nv21, ySize, 1]
                    uBuffer[nv21, ySize + 1, uBuffer.remaining()]
                    return nv21 // shortcut
                }
            } catch (ex: ReadOnlyBufferException) {
                ex.printStackTrace()
            }

            vBuffer.put(1, savePixel)
        }

        for (row in 0 until height / 2) {
            for (col in 0 until width / 2) {
                val vuPos = col * pixelStride + row * rowStride
                nv21[pos++] = vBuffer[vuPos]
                nv21[pos++] = uBuffer[vuPos]
            }
        }

        return nv21
    }

    @TargetApi(Build.VERSION_CODES.M)
    private fun requestPermission() = requestPermissions(permissions, PERMISSSION_REQUEST_CODE)

    private lateinit var result: DetectionResult

    @OptIn(DelicateCoroutinesApi::class)
    @SuppressLint("UnsafeOptInUsageError")
    private fun onPreviewFrame(imageProxy: ImageProxy) {
        val byteArray = getYuvDataFromImage(imageProxy)
        if (binding.lyoutScanFail.visibility == GONE) {
            if (!startToCaptureImage) {
                if (enginePrepared) {
                    if (!working) {
                        GlobalScope.launch(detectionContext) {
                            working = true

                            result =
                                engineWrapper.detect(
                                    byteArray,
                                    previewWidth,
                                    previewHeight,
                                    frameOrientation,
                                )
                            result.threshold = threshold
                            runOnUiThread {
                                if (!isStartTimer) {
                                    startTimer()
                                    isStartTimer = true
                                }
                            }

                            factorX = screenWidth / previewHeight.toFloat()
                            factorY = screenHeight / previewWidth.toFloat()
                            val rect =
                                calculateBoxLocationOnScreen(
                                    result.left,
                                    result.top,
                                    result.right,
                                    result.bottom,
                                )
                            detectionResult = result.updateLocation(rect)

                            val isOutside = isRectOutside(result)
                            if (isOutside) {
                                this@LivenessMainActivity.runOnUiThread {
                                    if (startRecording) {
                                        binding.coverView.changeColor(Color.WHITE)
                                        binding.txtTakeSelfie.text =
                                            getString(R.string.face_scanning)
                                    }
                                }
                                if (binding.lyoutScanFail.visibility != VISIBLE) {
                                    imageProxy.close()
                                }
                            } else {
                                this@LivenessMainActivity.runOnUiThread {
                                    if (startRecording) {
                                        if (!Constant.isActiveLivenessEnabled) {
                                            if (isFaceCenteredInRectangle(rect)) {
                                                binding.txtTakeSelfie.text =
                                                    getString(R.string.face_center)
                                            } else {
                                                binding.txtTakeSelfie.text =
                                                    getString(R.string.face_smile)
                                            }
                                        }
                                        binding.coverView.changeColor(
                                            ContextCompat.getColor(
                                                applicationContext,
                                                R.color.face_detection,
                                            ),
                                        )
                                    }
                                }
                                if (result.confidence >= 0.985F) {
                                    Log.d(tag, "onPreviewFrame:real ")
                                    if (startRecording) {
                                        if (Constant.isActiveLivenessEnabled) {
                                            if (imageCount >= 10) {
                                                val mediaImage: Image = imageProxy.image!!
                                                InputImage.fromMediaImage(
                                                    mediaImage,
                                                    imageProxy.imageInfo.rotationDegrees,
                                                )
                                                runOnUiThread(
                                                    kotlinx.coroutines.Runnable {
                                                        if (!blink) {
                                                            binding.txtTakeSelfie.text =
                                                                getString(
                                                                    R.string.liveness_blink,
                                                                )
                                                        } else {
                                                            binding.txtTakeSelfie.text =
                                                                getString(
                                                                    R.string.liveness_smile,
                                                                )
                                                            if (!smile) {
                                                                binding.txtTakeSelfie.text =
                                                                    getString(
                                                                        R.string.liveness_smile,
                                                                    )
                                                            } else {
                                                                binding.txtTakeSelfie.text = ""
                                                            }
                                                        }
                                                    },
                                                )
                                            } else {
                                                if (binding.lyoutScanFail.visibility != VISIBLE) {
                                                    imageProxy.close()
                                                }
                                            }
                                        } else {
                                            if (imageCount == 10) {
                                                takeImage()
                                            } else {
                                                if (binding.lyoutScanFail.visibility != VISIBLE) {
                                                    imageProxy.close()
                                                }
                                            }
                                        }
                                        imageCount++
                                    } else {
                                        if (binding.lyoutScanFail.visibility != VISIBLE) {
                                            imageProxy.close()
                                        }
                                    }
                                } else {
                                    Log.d(tag, "onPreviewFrame:fake ")
                                    if (binding.lyoutScanFail.visibility != VISIBLE) {
                                        imageProxy.close()
                                    }
                                }
                            }
                            working = false
                        }
                    } else {
                        if (binding.lyoutScanFail.visibility != VISIBLE) {
                            imageProxy.close()
                        }
                    }
                } else {
                    if (binding.lyoutScanFail.visibility != VISIBLE) {
                        imageProxy.close()
                    }
                }
            }
        }
    }

    private fun isFaceCenteredInRectangle(rect: Rect): Boolean {
        val faceBounds =
            RectF(
                binding.coverView.getRect().left,
                binding.coverView.getRect().top,
                binding.coverView.getRect().right,
                binding.coverView.getRect().bottom,
            )
        val faceCenterX = (faceBounds.left + faceBounds.right) / 2

        val rectCenterX = (rect.left + rect.right) / 2
        val thresholdX = 0.1 // Adjust as needed
        faceCenterX.toDouble() in (rectCenterX - thresholdX * rect.width())..(rectCenterX + thresholdX * rect.width())
        val isTooFarRight = faceCenterX > rectCenterX + thresholdX * rect.width()
        val isTooFarLeft = faceCenterX < rectCenterX - thresholdX * rect.width()

        return isTooFarLeft || isTooFarRight
    }

    private fun isRectOutside(parentRect: DetectionResult): Boolean {
        val rect =
            RectF(
                parentRect.left.toFloat(),
                parentRect.top.toFloat(),
                parentRect.right.toFloat(),
                parentRect.bottom.toFloat(),
            )
        val ovalRect =
            RectF(
                binding.coverView.getRect().left,
                binding.coverView.getRect().top,
                binding.coverView.getRect().right,
                binding.coverView.getRect().bottom,
            )

        val ovalCenterX = ovalRect.centerX()
        val ovalCenterY = ovalRect.centerY()
        val ovalRadiusX = ovalRect.width() / 2
        val ovalRadiusY = ovalRect.height() / 2

        val rectCorners =
            arrayOf(
                PointF(rect.left, rect.top),
                PointF(rect.right, rect.top),
                PointF(rect.left, rect.bottom),
                PointF(rect.right, rect.bottom),
            )

        for (corner in rectCorners) {
            val dx = corner.x - ovalCenterX
            val dy = corner.y - ovalCenterY

            val distance =
                (dx * dx) / (ovalRadiusX * ovalRadiusX) + (dy * dy) / (ovalRadiusY * ovalRadiusY)
            if (distance > 1) {
                return true
            }
        }

        return false
    }

    private lateinit var detectionResult: DetectionResult

    private fun init() {
        detectionResult = DetectionResult()
        changeTextSize()
        startCamera()

        binding.layoutHeader.imgBack.setOnClickListener { onBackPressed() }
        binding.lytStartRecording.setOnClickListener {
            startRecording = true
            binding.cardPressDesc.visibility = GONE
            binding.cardStartDesc.visibility = GONE

            binding.lytStartRecording.visibility = INVISIBLE
            binding.preview.visibility = VISIBLE
            binding.overlayMainLayout.setBackgroundColor(
                ContextCompat.getColor(
                    applicationContext,
                    R.color.transparent,
                ),
            )
            binding.lytTopView.setBackgroundColor(
                ContextCompat.getColor(
                    applicationContext,
                    R.color.face_camera_back,
                ),
            )
            binding.lytBottomMain.setBackgroundColor(
                ContextCompat.getColor(
                    applicationContext,
                    R.color.face_camera_back,
                ),
            )
        }
    }

    private lateinit var imageProxy: ImageProxy

    @SuppressLint("UnsafeOptInUsageError")
    private fun startCamera() {
        calculateSize()
        cameraExecutor = Executors.newSingleThreadExecutor()
        getBestPreviewSize()

        val preview =
            Preview.Builder()
                .build().also {
                    it.setSurfaceProvider(binding.preview.surfaceProvider)
                }

        val imageAnalysis =
            ImageAnalysis.Builder().build().also {
                it.setAnalyzer(
                    cameraExecutor,
                ) { imageProxy ->
                    if (binding.lyoutScanFail.visibility == GONE) {
                        if (startRecording) {
                            this.imageProxy = imageProxy
                            onPreviewFrame(imageProxy)
                        } else {
                            imageProxy.close()
                        }
                    }
                }
            }
        cameraSelector =
            CameraSelector.Builder()
                .requireLensFacing(CameraSelector.LENS_FACING_FRONT)
                .build()

        val cameraProviderFuture = ProcessCameraProvider.getInstance(this)

        cameraProviderFuture.addListener({
            cameraProvider = cameraProviderFuture.get()

            try {
                cameraProvider?.unbindAll()
                cameraProvider?.bindToLifecycle(this, cameraSelector!!, preview, imageAnalysis)
            } catch (ex: Exception) {
                Log.e(tag, "Failed to bind camera", ex)
            }
        }, ContextCompat.getMainExecutor(this))
    }

    private fun changeTextSize() {
        scale = resources.displayMetrics.scaledDensity

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            window.setDecorFitsSystemWindows(false)
            window.insetsController?.apply {
                hide(WindowInsets.Type.statusBars())
                systemBarsBehavior = WindowInsetsController.BEHAVIOR_SHOW_TRANSIENT_BARS_BY_SWIPE
            }
        } else {
            @Suppress("DEPRECATION")
            window.decorView.systemUiVisibility =
                View.SYSTEM_UI_FLAG_LAYOUT_STABLE or View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
        }
        var layoutParams2 = binding.layoutHeader.imgBack.layoutParams
        layoutParams2.width = getWidth(24)
        layoutParams2.height = getHeight(24)
        binding.layoutHeader.imgBack.layoutParams = layoutParams2
        binding.layoutHeader.txtHelp.text = getString(R.string.header_doc_scan)
        binding.layoutHeader.txtHelp.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            getTextSize(17),
        )

        val back = SVG.getFromResource(resources, R.raw.back_white)
        binding.layoutHeader.imgBack.setSVG(back)
        if (storage.getPreferredLocale().equals("ar", ignoreCase = true)) {
            binding.layoutHeader.imgBack.rotation = 180F
        }
        binding.layoutHeader.txtHelp.setText(R.string.face_header)
        binding.layoutHeader.txtHelp.setTextColor(Color.WHITE)
        val layoutParams = getLinearLayoutParam()
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            layoutParams.setMargins(0, getPaddingORMarginValue(4), 0, 0)
        } else {
            layoutParams.setMargins(0, getPaddingORMarginValue(42), 0, 0)
        }
        binding.layoutHeader.lytHeaderMain.layoutParams = layoutParams

        layoutParams2 = binding.layoutStepMain.lytStep.layoutParams
        layoutParams2.width = LinearLayout.LayoutParams.MATCH_PARENT
        layoutParams2.height = getHeight(26)
        binding.layoutStepMain.lytStep.layoutParams = layoutParams2
        var marginLayoutParam =
            binding.layoutStepMain.lytStep.layoutParams as LinearLayout.LayoutParams
        marginLayoutParam.setMargins(0, getPaddingORMarginValue(16), 0, 0)
        binding.layoutStepMain.lytStep.layoutParams = marginLayoutParam

        var padding = getPaddingORMarginValue(16)
        binding.lytTopView.setPadding(padding, padding, padding, 0)
        binding.lytBottomMain.setPadding(padding, 0, padding, padding)

        val stepIconSize = 22
        var stepLayoutParam = binding.layoutStepMain.txtStep1.layoutParams
        stepLayoutParam.width = getWidth(stepIconSize)
        stepLayoutParam.height = getHeight(stepIconSize)
        binding.layoutStepMain.txtStep1.layoutParams = stepLayoutParam

        stepLayoutParam = binding.layoutStepMain.txtStep2.layoutParams
        stepLayoutParam.width = getWidth(stepIconSize)
        stepLayoutParam.height = getHeight(stepIconSize)
        binding.layoutStepMain.txtStep2.layoutParams = stepLayoutParam

        stepLayoutParam = binding.layoutStepMain.txtStep3.layoutParams
        stepLayoutParam.width = getWidth(stepIconSize)
        stepLayoutParam.height = getHeight(stepIconSize)
        binding.layoutStepMain.txtStep3.layoutParams = stepLayoutParam

        binding.txtTakeSelfie.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            getTextSize(14),
        )

        marginLayoutParam = binding.txtTakeSelfie.layoutParams as LinearLayout.LayoutParams
        marginLayoutParam.setMargins(0, getPaddingORMarginValue(20), 0, 0)
        binding.txtTakeSelfie.layoutParams = marginLayoutParam

        val marginLayoutParam1 = binding.cardStartDesc.layoutParams as RelativeLayout.LayoutParams
        marginLayoutParam1.setMargins(getPaddingORMarginValue(34), 0, getPaddingORMarginValue(34), 0)
        binding.cardStartDesc.layoutParams = marginLayoutParam1

        binding.txtStartRecording.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            getTextSize(16),
        )

        binding.txtStartRecordingDesc2.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            getTextSize(16),
        )
        var svg = SVG.getFromResource(resources, R.raw.start_recording)
        binding.imgStartRecording.setSVG(svg)
        marginLayoutParam = binding.imgStartRecording.layoutParams as LinearLayout.LayoutParams
        marginLayoutParam.setMargins(0, getPaddingORMarginValue(16), 0, getPaddingORMarginValue(16))
        binding.imgStartRecording.layoutParams = marginLayoutParam

        stepLayoutParam = binding.imgStartRecording.layoutParams
        stepLayoutParam.width = getWidth(74)
        stepLayoutParam.height = getHeight(74)
        binding.imgStartRecording.layoutParams = stepLayoutParam

        stepLayoutParam = binding.layoutStepMain.viewStep2Divider.layoutParams
        stepLayoutParam.width = LinearLayout.LayoutParams.MATCH_PARENT
        stepLayoutParam.height = getHeight(2)
        binding.layoutStepMain.viewStep2Divider.layoutParams = stepLayoutParam

        binding.layoutStepMain.txtStep1.setImageResource(R.drawable.complete_step1)
        binding.layoutStepMain.txtStep2.setImageResource(R.drawable.complete_step1)
        binding.layoutStepMain.txtStep3.setImageResource(R.drawable.scan_complete_step3)
        binding.layoutStepMain.viewStep2Divider.setBackgroundResource(R.drawable.gradient_straight_line)
        binding.layoutStepMain.viewStep1Divider.background =
            ContextCompat.getDrawable(
                applicationContext,
                R.drawable.complete_straight_line,
            )

        binding.txtPressDesc.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            getTextSize(16),
        )
        binding.txtPressDesc.setPadding(
            getPaddingORMarginValue(4),
            getPaddingORMarginValue(12),
            getPaddingORMarginValue(4),
            getPaddingORMarginValue(12),
        )
        binding.cardPressDesc.radius = calculateRadius(4).toFloat()
        binding.cardStartDesc.radius = calculateRadius(4).toFloat()

        binding.txtScanCompleted.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            getTextSize(20),
        )
        marginLayoutParam = binding.txtScanCompleted.layoutParams as LinearLayout.LayoutParams
        marginLayoutParam.setMargins(
            0,
            getPaddingORMarginValue(24),
            0,
            0,
        )
        binding.txtScanCompleted.layoutParams = marginLayoutParam
        stepLayoutParam = binding.imgScanComplete.layoutParams
        stepLayoutParam.width = getWidth(56)
        stepLayoutParam.height = getHeight(56)
        binding.imgScanComplete.layoutParams = stepLayoutParam
        binding.imgScanComplete.setAnimation(R.raw.scan_completed)
        binding.overlayMainLayout.setBackgroundColor(
            ContextCompat.getColor(
                applicationContext,
                R.color.transparent,
            ),
        )
        binding.lytTopView.setBackgroundColor(
            ContextCompat.getColor(
                applicationContext,
                R.color.face_camera_back,
            ),
        )
        binding.lytBottomMain.setBackgroundColor(
            ContextCompat.getColor(
                applicationContext,
                R.color.face_camera_back,
            ),
        )

        layoutParams2 = binding.imgScanFail.layoutParams
        layoutParams2.width = getWidth(56)
        layoutParams2.height = getHeight(56)
        binding.imgScanFail.layoutParams = layoutParams2

        marginLayoutParam = binding.txtScanFail.layoutParams as LinearLayout.LayoutParams
        marginLayoutParam.setMargins(0, getPaddingORMarginValue(24), 0, 0)
        binding.txtScanFail.layoutParams = marginLayoutParam
        binding.txtScanFail.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            getTextSize(20),
        )

        marginLayoutParam = binding.btnRescanningDocuments.layoutParams as LinearLayout.LayoutParams
        marginLayoutParam.setMargins(
            getPaddingORMarginValue(16),
            0,
            getPaddingORMarginValue(16),
            getPaddingORMarginValue(44),
        )
        binding.btnRescanningDocuments.layoutParams = marginLayoutParam

        padding = getPaddingORMarginValue(12)
        binding.txtRescanningDocuments.setPadding(0, padding, 0, padding)
        binding.txtRescanningDocuments.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            getTextSize(16),
        )
        binding.btnRescanningDocuments.radius = calculateRadius(8).toFloat()
        binding.txtError3.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            getTextSize(16),
        )
        binding.txtError3.setPadding(0, 0, 0, getPaddingORMarginValue(38))
    }

    private fun getLinearLayoutParam(): LinearLayout.LayoutParams {
        return LinearLayout.LayoutParams(
            LinearLayout.LayoutParams.MATCH_PARENT,
            LinearLayout.LayoutParams.WRAP_CONTENT,
        )
    }

    private fun calculateRadius(radius: Int): Int {
        return (radius * scale + 0.5f).toInt()
    }

    private fun getWidth(width: Int): Int {
        return (width * scale + 0.5f).toInt()
    }

    private fun getHeight(height: Int): Int {
        return (height * scale + 0.5f).toInt()
    }

    private fun getPaddingORMarginValue(padding: Int): Int {
        return (padding * scale + 0.5f).toInt()
    }

    private fun getTextSize(textSize: Int): Float {
        val desiredTextSizeInPixels = (textSize * scale + 0.5f).toInt()
        return desiredTextSizeInPixels.toFloat()
    }

    private fun getBestPreviewSize(): Size? {
        val cameraManager = getSystemService(Context.CAMERA_SERVICE) as CameraManager
        val characteristics = cameraManager.getCameraCharacteristics(getFrontCameraId().toString())
        val streamConfigurationMap =
            characteristics.get(CameraCharacteristics.SCALER_STREAM_CONFIGURATION_MAP)

        val outputSizes = streamConfigurationMap?.getOutputSizes(SurfaceTexture::class.java)

        var bestPreviewSize: Size? = null
        var bestPreviewSizeArea = 0

        outputSizes?.forEach { size ->
            val area = size.width * size.height
            if (size.width >= previewWidth && size.height >= previewHeight && area > bestPreviewSizeArea) {
                bestPreviewSize = size
                bestPreviewSizeArea = area
            }
        }

        return bestPreviewSize
    }

    private fun setRescanningDocumentVisibility() {
        binding.lyoutScanFail.visibility = VISIBLE
        binding.btnRescanningDocuments.setOnClickListener {
            binding.cardStartDesc.visibility = GONE
            binding.lyoutScanFail.visibility = GONE
            imageProxy.close()
            cTimer?.start()
        }
    }

    private var delayMillis: Long = 10000
    private var cTimer: CountDownTimer? = null

    private fun startTimer() {
        cTimer =
            object : CountDownTimer(delayMillis, 1000) {
                override fun onTick(millisUntilFinished: Long) {
                }

                override fun onFinish() {
                    try {
                        if (!result.hasFace) {
                            setRescanningDocumentVisibility()
                            delayMillis = 15000
                        } else {
                            cTimer!!.start()
                        }
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }
                }
            }.start()
    }

    private fun getFrontCameraId(): String? {
        val cameraManager = getSystemService(Context.CAMERA_SERVICE) as CameraManager
        val cameraIds = cameraManager.cameraIdList

        for (id in cameraIds) {
            val characteristics = cameraManager.getCameraCharacteristics(id)
            val facing = characteristics.get(CameraCharacteristics.LENS_FACING)
            if (facing == CameraCharacteristics.LENS_FACING_FRONT) {
                return id
            }
        }

        return null
    }

    private fun calculateSize() {
        binding.coverView.doOnLayout {
            screenWidth = binding.coverView.width
            screenHeight = binding.coverView.height
        }
    }

    private var startToCaptureImage = false

    private fun takeImage() {
        runOnUiThread {
            val imageCapture =
                ImageCapture.Builder()
                    .setTargetRotation(Surface.ROTATION_0)
                    .build()
            startToCaptureImage = true
            cameraProvider?.let {
                try {
                    it.bindToLifecycle(this, cameraSelector!!, imageCapture)
                    imageCapture.takePicture(
                        ContextCompat.getMainExecutor(this),
                        object : ImageCapture.OnImageCapturedCallback() {
                            @SuppressLint("RestrictedApi", "UnsafeOptInUsageError")
                            override fun onCaptureSuccess(image: ImageProxy) {
                                val bitmap = image.image?.toBitmap()

                                if (bitmap != null) {
                                    val stream = ByteArrayOutputStream()
                                    bitmap.compress(Bitmap.CompressFormat.PNG, 100, stream)
                                    val bytes = stream.toByteArray()
                                    Utility.getInstance().liveImage = bytes

                                    binding.lytStartRecording.visibility = INVISIBLE
                                    binding.lyutSucess.visibility = VISIBLE
                                    binding.imgScanComplete.playAnimation()
                                    binding.imgScanComplete.setMaxFrame(80)
                                    binding.imgScanComplete.addAnimatorListener(
                                        object :
                                            AnimatorListenerAdapter() {
                                            override fun onAnimationEnd(animation: Animator) {
                                                super.onAnimationEnd(animation)
                                                binding.txtTakeSelfie.visibility = INVISIBLE
                                                Utility.getInstance().liveImage = bytes

                                                finish()
                                            }
                                        },
                                    )
                                }
                            }

                            override fun onError(exc: ImageCaptureException) {
                            }
                        },
                    )
                } catch (ex: Exception) {
                    Log.e(tag, "Failed to capture image", ex)
                }
            }
        }
    }

    private fun Image.toBitmap(): Bitmap? {
        val buffer = planes[0].buffer
        val bytes = ByteArray(buffer.capacity())
        buffer.get(bytes)
        return BitmapFactory.decodeByteArray(bytes, 0, bytes.size, null)
    }

    private fun calculateBoxLocationOnScreen(
        left: Int,
        top: Int,
        right: Int,
        bottom: Int,
    ): Rect =
        Rect(
            (left * factorX).toInt(),
            (top * factorY).toInt(),
            (right * factorX).toInt(),
            (bottom * factorY).toInt(),
        )

    override fun onRequestPermissionsResult(
        requestCode: Int,
        permissions: Array<out String>,
        grantResults: IntArray,
    ) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults)
        if (requestCode == PERMISSSION_REQUEST_CODE) {
            if (grantResults.isNotEmpty() && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                init()
            } else {
                Toast.makeText(this, "请授权相机权限", Toast.LENGTH_LONG).show()
            }
        }
    }

    override fun onResume() {
        engineWrapper = EngineWrapper(assets)
        enginePrepared = engineWrapper.init()

        if (!enginePrepared) {
            Toast.makeText(this, "Engine init failed.", Toast.LENGTH_LONG).show()
        }

        super.onResume()
    }

    @SuppressLint("RestrictedApi")
    override fun onDestroy() {
        engineWrapper.destroy()
        cameraProvider?.unbindAll()
        cTimer?.cancel()
        super.onDestroy()
    }

    @Deprecated("Deprecated in Java")
    override fun onBackPressed() {
        showLeaveOnboardingDialog()
    }

    @SuppressLint("RestrictedApi")
    private fun showLeaveOnboardingDialog() {
        val dialog = Dialog(this)
        dialog.requestWindowFeature(Window.FEATURE_NO_TITLE)
        val binding: DialogBackBinding =
            DialogBackBinding.inflate(dialog.layoutInflater)
        dialog.setContentView(binding.root)

        var layoutParams = binding.dialogHeader.layoutParams as LinearLayout.LayoutParams
        var padding: Int = getPaddingORMarginValue(16)
        layoutParams.setMargins(getPaddingORMarginValue(16), getPaddingORMarginValue(16), padding, 0)
        binding.dialogHeader.layoutParams = layoutParams

        val layoutParams2: ViewGroup.LayoutParams = binding.cardConfirmation.layoutParams
        layoutParams2.width = getWidth(280)
        binding.cardConfirmation.layoutParams = layoutParams2

        binding.cardConfirmation.radius = calculateRadius(14).toFloat()
        binding.dialogHeader.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            getTextSize(18),
        )
        binding.dialogSubtitle.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            getTextSize(14),
        )
        binding.btnNo.setTextSize(TypedValue.COMPLEX_UNIT_PX, getTextSize(16))
        binding.btnYes.setTextSize(TypedValue.COMPLEX_UNIT_PX, getTextSize(16))

        layoutParams = binding.dialogSubtitle.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(padding, padding, padding, padding)
        binding.dialogSubtitle.layoutParams = layoutParams

        padding = getPaddingORMarginValue(12)
        layoutParams = binding.btnNo.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(0, padding, 0, padding)
        binding.btnNo.layoutParams = layoutParams

        layoutParams = binding.btnYes.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(
            getPaddingORMarginValue(24),
            padding,
            getPaddingORMarginValue(16),
            padding,
        )
        binding.btnYes.layoutParams = layoutParams

        binding.dialogHeader.typeface =
            ResourcesCompat.getFont(
                applicationContext,
                R.font.pingmedium,
            )
        binding.dialogSubtitle.typeface =
            ResourcesCompat.getFont(
                applicationContext,
                R.font.regular,
            )

        binding.dialogHeader.text = getString(R.string.liveness_back_dialog_header)
        binding.dialogSubtitle.text = getString(R.string.liveness_back_dialog_title)
        binding.btnNo.text = getString(R.string.liveness_back_dialog_cotionue)
        binding.btnYes.text = getString(R.string.liveness_back_dialog_leave)

        binding.btnYes.setOnClickListener {
            dialog.dismiss()
            cameraProvider?.unbindAll()
            cameraProvider?.shutdown()
            finish()
        }
        binding.btnNo.setOnClickListener { dialog.dismiss() }

        dialog.show()
        dialog.setCancelable(false)
        dialog.window!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
        dialog.window!!.attributes.windowAnimations = R.style.DialogAnimation
    }

    companion object {
        const val TAG = "MainActivity"
        const val DEFAULT_THRESHOLD = 0.915F

        val permissions: Array<String> = arrayOf(Manifest.permission.CAMERA)
        const val PERMISSSION_REQUEST_CODE = 1
    }
}
