package nashid.verify.sdk.mrtd2

import android.app.Activity
import android.graphics.Bitmap
import android.util.Log
import android.view.View
import com.bumptech.glide.Glide
import com.google.android.material.snackbar.Snackbar
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import nashid.verify.sdk.NashidSDK
import nashid.verify.sdk.R
import nashid.verify.sdk.mrtd2.resultcallback.callbackclass.DocumentImages
import nashid.verify.sdk.networking.APIClient
import nashid.verify.sdk.networking.APIInterface
import okhttp3.MediaType.Companion.toMediaTypeOrNull
import okhttp3.MultipartBody
import okhttp3.RequestBody.Companion.asRequestBody
import okhttp3.RequestBody.Companion.toRequestBody
import okhttp3.ResponseBody
import org.json.JSONObject
import retrofit2.Call
import retrofit2.Callback
import retrofit2.Response
import java.io.File
import java.security.SecureRandom
import java.text.SimpleDateFormat
import java.util.Calendar
import java.util.Locale

internal class CallCompanyScanAPI {
    private lateinit var scanType: String
    private lateinit var mainJSONObject: JSONObject
    private var statusCode: Int = 1
    private lateinit var mainImagesObject: JSONObject
    private lateinit var completeCallingCompanyScanListner: CompleteCallingCompanyScanListner

    fun uploadAllImages(
        activity: Activity,
        jsonObject: JSONObject,
        scanType: String,
        statusCode: Int,
        completeCallingCompanyScanListener: CompleteCallingCompanyScanListner,
    ) {
        completeCallingCompanyScanListner = completeCallingCompanyScanListener
        this.scanType = scanType
        this.statusCode = statusCode
        mainJSONObject = jsonObject
        if (jsonObject.has("images")) {
            mainImagesObject = jsonObject.getJSONObject("images")
            val scope = CoroutineScope(Dispatchers.Main)
            scope.launch {
                handleImages(activity)
                mainJSONObject.put("images", mainImagesObject)
                callCompanyScan(activity, mainJSONObject, scanType)
            }
        } else {
            callCompanyScan(activity, jsonObject, scanType)
        }
    }

    private suspend fun handleImages(activity: Activity) {
        val images =
            mapOf(
                R.string.beforeCropFront to activity.getString(R.string.beforeCropFront),
                R.string.beforeCropBack to activity.getString(R.string.beforeCropBack),
                R.string.afterCropFront to activity.getString(R.string.afterCropFront),
                R.string.afterCropBack to activity.getString(R.string.afterCropBack),
                R.string.nfcImage to activity.getString(R.string.nfcImage),
                R.string.scanImage to activity.getString(R.string.scanImage),
                R.string.signatureImage to activity.getString(R.string.signatureImage),
                R.string.livenessImage to activity.getString(R.string.livenessImage),
            )

        for ((key, value) in images) {
            if (mainImagesObject.has(activity.getString(key))) {
                val image = mainImagesObject[activity.getString(key)]
                Log.d("TAG", "handleImages: $key   $value")
                if (image is Bitmap) {
                    uploadAndReplaceImage(activity, image, value)
                } else if (image is String) {
                    loadImageAndUpload(activity, image, value)
                }
            }
        }
    }

    private suspend fun uploadAndReplaceImage(
        activity: Activity,
        bitmap: Bitmap,
        type: String,
    ) {
        val imageFile = bitmapToFile(activity, bitmap, type)
        Log.d("TAG", "uploadAndReplaceImage: " + bitmap + "   " + type)
        val imageUrl = uploadImage(activity, imageFile)
        mainImagesObject.put(type, imageUrl)
    }

    private suspend fun loadImageAndUpload(
        activity: Activity,
        url: String,
        type: String,
    ) {
        val bitmap = loadImage(activity, url)
        if (bitmap != null) {
            uploadAndReplaceImage(activity, bitmap, type)
        }
    }

    private fun bitmapToFile(
        activity: Activity,
        bitmap: Bitmap,
        filename: String,
    ): File {
        val file = File(activity.filesDir, "$filename.png")
        file.outputStream().use {
            bitmap.compress(Bitmap.CompressFormat.PNG, 100, it)
        }
        return file
    }

    private suspend fun uploadImage(
        activity: Activity,
        imageFile: File,
    ): String =
        withContext(Dispatchers.IO) {
            val fileRequestBody = imageFile.asRequestBody("image/jpeg".toMediaTypeOrNull())
            val filePart =
                MultipartBody.Part.createFormData("image", imageFile.name, fileRequestBody)
            val param1 = imageFile.name.toRequestBody("text/plain".toMediaTypeOrNull())

            val response =
                APIClient
                    .getClient(
                        activity,
                        NashidSDK.getInstance().getBaseURL(),
                        NashidSDK.getInstance().getRegisterToken(),
                    ).create(APIInterface::class.java)
                    .callUploadFile(param1, filePart)
                    .execute()
            Log.d("TAG", "uploadImage: '" + response.body())
            if (response.isSuccessful) {
                val jsonObject = JSONObject(response.body()?.string() ?: "{}")
                val data = jsonObject.getJSONObject("data")
                Log.d("TAG", "handleImages:response " + data.getString("image_url"))
                data.getString("image_url")
            } else {
                activity.currentFocus?.rootView?.let {
                    showSnackbarWithRetry(
                        it,
                        activity,
                        imageFile,
                    )
                }
                ""
            }
        }

    private fun showSnackbarWithRetry(
        view: View,
        activity: Activity,
        imageFile: File,
    ) {
        Snackbar
            .make(view, "An error occurred", Snackbar.LENGTH_INDEFINITE)
            .setAction("Retry") {
                CoroutineScope(Dispatchers.Main).launch {
                    uploadImage(activity, imageFile)
                }
            }.show()
    }

    private suspend fun loadImage(
        activity: Activity,
        url: String,
    ): Bitmap? =
        withContext(Dispatchers.IO) {
            val futureTarget =
                Glide
                    .with(activity.applicationContext)
                    .asBitmap()
                    .load(url)
                    .submit()
            try {
                futureTarget.get()
            } catch (e: Exception) {
                null
            }
        }

    private fun callCompanyScan(
        activity: Activity,
        requestData: JSONObject,
        type: String,
    ) {
        var scanCode =
            generateRandomString(50)
        if (!NashidSDK.getInstance().getQRCodeValue().equals("")) {
            scanCode = NashidSDK.getInstance().getQRCodeValue()
            NashidSDK.getInstance().setQRCodeValue("")
        }
        scanType =
            if (type.equals(activity.getString(R.string.e_passport), ignoreCase = true)) {
                "Passport"
            } else {
                "Card"
            }
        APIClient
            .getClient(
                activity,
                NashidSDK.getInstance().getBaseURL(),
                NashidSDK.getInstance().getRegisterToken(),
            ).create(
                APIInterface::class.java,
            ).callCompanyScans(
                scanType,
                NashidSDK.getInstance().getCompanyUUID(),
                requestData.toString(),
                statusCode,
                scanCode,
                getCurrentFormattedDateTime(),
            ).enqueue(
                object : Callback<ResponseBody> {
                    override fun onResponse(
                        call: Call<ResponseBody>,
                        response: Response<ResponseBody>,
                    ) {
                        try {
                            DocumentImages.instance.cleanup()
                            if (response.isSuccessful) {
                                if (completeCallingCompanyScanListner != null) {
                                    completeCallingCompanyScanListner.onSuccess()
                                }
                            } else {
                                if (completeCallingCompanyScanListner != null) {
                                    completeCallingCompanyScanListner.onFailure()
                                }
                            }
                        } catch (e: java.lang.Exception) {
                            e.printStackTrace()
                            if (completeCallingCompanyScanListner != null) {
                                completeCallingCompanyScanListner.onFailure()
                            }
                        }
                    }

                    override fun onFailure(
                        call: Call<ResponseBody>,
                        t: Throwable,
                    ) {
                        if (completeCallingCompanyScanListner != null) {
                            completeCallingCompanyScanListner.onFailure()
                        }
                    }
                },
            )
    }

    private fun getCurrentFormattedDateTime(): String? {
        val calendar = Calendar.getInstance()
        val date = calendar.time
        val sdf = SimpleDateFormat("yyyy-MM-dd HH:mm:ss", Locale.ENGLISH)
        return sdf.format(date)
    }

    @Suppress("ktlint:standard:property-naming")
    private val CHARACTERS = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"

    fun generateRandomString(length: Int): String {
        val random = SecureRandom()
        val sb = StringBuilder(length)
        for (i in 0 until length) {
            val randomIndex = random.nextInt(CHARACTERS.length)
            val randomChar = CHARACTERS[randomIndex]
            sb.append(randomChar)
        }
        return sb.toString()
    }
}
