package nashid.verify.sdk.mrtd2.activity.passportNFC.util

import android.util.Log
import nashid.verify.sdk.mrtd2.activity.passportNFC.jmrtd.FeatureStatus
import nashid.verify.sdk.mrtd2.activity.passportNFC.jmrtd.JMRTDSecurityProvider
import nashid.verify.sdk.mrtd2.activity.passportNFC.jmrtd.MRTDTrustStore
import nashid.verify.sdk.mrtd2.activity.passportNFC.jmrtd.VerificationStatus
import net.sf.scuba.smartcards.CardServiceException
import org.bouncycastle.asn1.ASN1Encodable
import org.bouncycastle.asn1.ASN1Integer
import org.bouncycastle.asn1.DERSequence
import org.jmrtd.BACKey
import org.jmrtd.PACEKeySpec
import org.jmrtd.PassportService
import org.jmrtd.Util
import org.jmrtd.cert.CardVerifiableCertificate
import org.jmrtd.lds.AbstractTaggedLDSFile
import org.jmrtd.lds.ActiveAuthenticationInfo
import org.jmrtd.lds.CVCAFile
import org.jmrtd.lds.CardAccessFile
import org.jmrtd.lds.ChipAuthenticationInfo
import org.jmrtd.lds.ChipAuthenticationPublicKeyInfo
import org.jmrtd.lds.LDSFileUtil
import org.jmrtd.lds.PACEInfo
import org.jmrtd.lds.SODFile
import org.jmrtd.lds.icao.COMFile
import org.jmrtd.lds.icao.DG11File
import org.jmrtd.lds.icao.DG12File
import org.jmrtd.lds.icao.DG14File
import org.jmrtd.lds.icao.DG15File
import org.jmrtd.lds.icao.DG1File
import org.jmrtd.lds.icao.DG2File
import org.jmrtd.lds.icao.DG3File
import org.jmrtd.lds.icao.DG5File
import org.jmrtd.lds.icao.DG7File
import org.jmrtd.lds.icao.MRZInfo
import org.jmrtd.protocol.BACResult
import org.jmrtd.protocol.EACCAResult
import org.jmrtd.protocol.EACTAResult
import org.jmrtd.protocol.PACEResult
import java.io.IOException
import java.io.InputStream
import java.math.BigInteger
import java.security.GeneralSecurityException
import java.security.KeyStore
import java.security.MessageDigest
import java.security.NoSuchAlgorithmException
import java.security.PrivateKey
import java.security.PublicKey
import java.security.SecureRandom
import java.security.Security
import java.security.Signature
import java.security.cert.Certificate
import java.security.cert.X509Certificate
import java.security.interfaces.ECPublicKey
import java.security.interfaces.RSAPublicKey
import java.security.spec.MGF1ParameterSpec
import java.security.spec.PSSParameterSpec
import java.util.ArrayList
import java.util.Arrays
import java.util.Random
import java.util.TreeMap
import java.util.TreeSet
import javax.crypto.Cipher
import javax.security.auth.x500.X500Principal

class PassportNFC
    @Throws(GeneralSecurityException::class)
    private constructor() {
        private var digest: MessageDigest? = null

        val features: FeatureStatus = FeatureStatus()

        val verificationStatus: VerificationStatus

        @Transient
        private var rsaAASignature: Signature? = null

        @Transient
        private var rsaAADigest: MessageDigest? = null

        @Transient
        private val rsaAACipher: Cipher

        @Transient
        private var ecdsaAASignature: Signature? = null

        @Transient
        private var ecdsaAADigest: MessageDigest? = null

        var trustManager: MRTDTrustStore? = null

        var docSigningPrivateKey: PrivateKey? = null
            set(docSigningPrivateKey) {
                field = docSigningPrivateKey
                updateCOMSODFile(null)
            }

        private var service: PassportService? = null

        private val random: Random
        var rasPublicKey: String? = null
        var ecdsaPublicKey: String? = null

        var comFile: COMFile? = null
            private set
        var sodFile: SODFile? = null
            private set
        var dg1File: DG1File? = null
            private set
        var dg2File: DG2File? = null
            private set
        var dg3File: DG3File? = null
            private set
        var dg5File: DG5File? = null
            private set
        var dg7File: DG7File? = null
            private set
        var dg11File: DG11File? = null
            private set
        var dg12File: DG12File? = null
            private set
        var dg14File: DG14File? = null
            private set
        var dg15File: DG15File? = null
            private set
        var cvcaFile: CVCAFile? = null
            private set

        init {
            this.verificationStatus = VerificationStatus()

            this.random = SecureRandom()

            rsaAADigest =
                MessageDigest.getInstance("SHA1") // NOTE: for output length measurement only. -- MO
            rsaAASignature = Signature.getInstance("SHA1WithRSA/ISO9796-2", BC_PROVIDER)
            rsaAACipher = Cipher.getInstance("RSA/NONE/NoPadding")

            // NOTE: These will be updated in doAA after caller has read ActiveAuthenticationSecurityInfo.
            ecdsaAASignature = Signature.getInstance("SHA256withECDSA", BC_PROVIDER)
            ecdsaAADigest =
                MessageDigest.getInstance("SHA-256") // NOTE: for output length measurement only. -- MO
        }

        @Throws(CardServiceException::class, GeneralSecurityException::class)
        constructor(
            ps: PassportService?,
            trustManager: MRTDTrustStore,
            mrzInfo: MRZInfo,
            maxBlockSize: Int,
        ) : this() {
            if (ps == null) {
                throw IllegalArgumentException("Service cannot be null")
            }
            this.service = ps
            this.trustManager = trustManager

            val hasSAC: Boolean
            var isSACSucceeded = false
            var paceResult: PACEResult? = null
            if (ps.isOpen == false) {
                ps.open()
            }
            try {
                try {
                    Log.i(TAG, "Inspecting card access file")
                    val cardAccessFile =
                        CardAccessFile(
                            (service as PassportService).getInputStream(
                                PassportService.EF_CARD_ACCESS,
                                maxBlockSize,
                            ),
                        )
                    val securityInfos = cardAccessFile.securityInfos
                    for (securityInfo in securityInfos) {
                        if (securityInfo is PACEInfo) {
                            features.setSAC(FeatureStatus.Verdict.PRESENT)
                        }
                    }
                } catch (e: Exception) {
                    // NOTE: No card access file, continue to test for BAC.
                    Log.i(TAG, "DEBUG: failed to get card access file: " + e.message)
                    e.printStackTrace()
                }

                hasSAC = features.hasSAC() == FeatureStatus.Verdict.PRESENT

                if (hasSAC) {
                    try {
                        paceResult = doPACE(ps, mrzInfo, maxBlockSize)
                        isSACSucceeded = true
                    } catch (e: Exception) {
                        e.printStackTrace()
                        Log.i(TAG, "PACE failed, falling back to BAC")
                        isSACSucceeded = false
                    }
                }
                (service as PassportService).sendSelectApplet(isSACSucceeded)
            } catch (cse: CardServiceException) {
                throw cse
            } catch (e: Exception) {
                e.printStackTrace()
                throw CardServiceException("Cannot open document. " + e.message)
            }

            try {

                COMFile(
                    (service as PassportService).getInputStream(
                        PassportService.EF_COM,
                        maxBlockSize,
                    ),
                )

                if (isSACSucceeded) {
                    verificationStatus.setSAC(VerificationStatus.Verdict.SUCCEEDED, "Succeeded")
                    features.setBAC(FeatureStatus.Verdict.UNKNOWN)
                    verificationStatus.setBAC(
                        VerificationStatus.Verdict.NOT_CHECKED,
                        "Using SAC, BAC not checked",
                        EMPTY_TRIED_BAC_ENTRY_LIST,
                    )
                } else {
                    // We failed SAC, and we failed BAC.
                    features.setBAC(FeatureStatus.Verdict.NOT_PRESENT)
                    verificationStatus.setBAC(
                        VerificationStatus.Verdict.NOT_PRESENT,
                        "Non-BAC document",
                        EMPTY_TRIED_BAC_ENTRY_LIST,
                    )
                }
            } catch (e: Exception) {
                Log.i(TAG, "Attempt to read EF.COM before BAC failed with: " + e.message)
                features.setBAC(FeatureStatus.Verdict.PRESENT)
                verificationStatus.setBAC(
                    VerificationStatus.Verdict.NOT_CHECKED,
                    "BAC document",
                    EMPTY_TRIED_BAC_ENTRY_LIST,
                )
            }

            val hasBAC = features.hasBAC() == FeatureStatus.Verdict.PRESENT

            if (hasBAC && !(hasSAC && isSACSucceeded)) {
                val bacKey = BACKey(mrzInfo.documentNumber, mrzInfo.dateOfBirth, mrzInfo.dateOfExpiry)
                val triedBACEntries = ArrayList<BACKey>()
                triedBACEntries.add(bacKey)
                try {
                    doBAC(service as PassportService, mrzInfo)
                    verificationStatus.setBAC(
                        VerificationStatus.Verdict.SUCCEEDED,
                        "BAC succeeded with key $bacKey",
                        triedBACEntries,
                    )
                } catch (e: Exception) {
                    verificationStatus.setBAC(
                        VerificationStatus.Verdict.FAILED,
                        "BAC failed",
                        triedBACEntries,
                    )
                }
            }

            val dgNumbersAlreadyRead = TreeSet<Int>()

            try {
                comFile = getComFile(ps, maxBlockSize)
                sodFile = getSodFile(ps, maxBlockSize)
                dg1File = getDG1File(ps, maxBlockSize)
                dgNumbersAlreadyRead.add(1)
            } catch (ioe: IOException) {
                ioe.printStackTrace()
                Log.w(TAG, "Could not read file")
            }

            try {
                dg14File = getDG14File(ps, maxBlockSize)
            } catch (e: Exception) {
                e.printStackTrace()
            }

            try {
                cvcaFile = getCVCAFile(ps, maxBlockSize)
            } catch (e: Exception) {
                e.printStackTrace()
            }

            val dgNumbers = ArrayList<Int>()
            if (sodFile != null) {
                dgNumbers.addAll(sodFile!!.dataGroupHashes.keys)
            } else if (comFile != null) {
                Log.w(TAG, "Failed to get DG list from EF.SOd. Getting DG list from EF.COM.")
                val tagList = comFile!!.tagList
                dgNumbers.addAll(toDataGroupList(tagList)!!)
            }
            dgNumbers.sort()

            Log.i(TAG, "Found DGs: $dgNumbers")

            var hashResults: MutableMap<Int, VerificationStatus.HashMatchResult>? =
                verificationStatus.hashResults
            if (hashResults == null) {
                hashResults = TreeMap<Int, VerificationStatus.HashMatchResult>()
            }

            if (sodFile != null) {
                val storedHashes = sodFile!!.dataGroupHashes
                for (dgNumber in dgNumbers) {
                    val storedHash = storedHashes[dgNumber]
                    var hashResult: VerificationStatus.HashMatchResult? = hashResults[dgNumber]
                    if (hashResult != null) {
                        continue
                    }
                    if (dgNumbersAlreadyRead.contains(dgNumber)) {
                        hashResult = verifyHash(dgNumber)
                    } else {
                        hashResult = VerificationStatus.HashMatchResult(storedHash!!, null)
                    }
                    hashResults[dgNumber] = hashResult!!
                }
            }
            verificationStatus.setHT(
                VerificationStatus.Verdict.UNKNOWN,
                verificationStatus.htReason,
                hashResults,
            )

            if (dgNumbers.contains(14)) {
                features.setEAC(FeatureStatus.Verdict.PRESENT)
                if (isChipAuthenticationInfoAvailable(ps, mrzInfo, dg14File, sodFile)) {
                    features.setCA(FeatureStatus.Verdict.PRESENT)
                } else {
                    features.setCA(FeatureStatus.Verdict.NOT_PRESENT)
                }
            } else {
                features.setEAC(FeatureStatus.Verdict.NOT_PRESENT)
                features.setCA(FeatureStatus.Verdict.NOT_PRESENT)
            }

            val hasCA = features.hasCA() == FeatureStatus.Verdict.PRESENT
            if (hasCA) {
                try {
                    val eaccaResults = doEACCA(ps, mrzInfo, dg14File, sodFile)
                    verificationStatus.setCA(
                        VerificationStatus.Verdict.SUCCEEDED,
                        "EAC succeeded",
                        eaccaResults[0],
                    )
                } catch (e: Exception) {
                    verificationStatus.setCA(VerificationStatus.Verdict.FAILED, "CA Failed", null)
                }
            }

            val hasEAC = features.hasEAC() == FeatureStatus.Verdict.PRESENT
            val cvcaKeyStores = trustManager.cvcaStores
            if (hasEAC && cvcaKeyStores != null && cvcaKeyStores.size > 0 && verificationStatus.ca == VerificationStatus.Verdict.SUCCEEDED) {
                try {
                    val eactaResults =
                        doEACTA(
                            ps,
                            mrzInfo,
                            cvcaFile,
                            paceResult,
                            verificationStatus.caResult,
                            cvcaKeyStores,
                        )
                    verificationStatus.setEAC(
                        VerificationStatus.Verdict.SUCCEEDED,
                        "EAC succeeded",
                        eactaResults[0],
                    )
                } catch (e: Exception) {
                    e.printStackTrace()
                    verificationStatus.setEAC(VerificationStatus.Verdict.FAILED, "EAC Failed", null)
                }

                dgNumbersAlreadyRead.add(14)
            }
            Log.d(TAG, "alldgnumber: " + dgNumbers)

            if (dgNumbers.contains(15)) {
                features.setAA(FeatureStatus.Verdict.PRESENT)
            } else {
                features.setAA(FeatureStatus.Verdict.NOT_PRESENT)
            }
            val hasAA = features.hasAA() == FeatureStatus.Verdict.PRESENT
            if (hasAA) {
                try {
                    dg15File = getDG15File(ps, maxBlockSize)
                    dgNumbersAlreadyRead.add(15)
                } catch (ioe: IOException) {
                    ioe.printStackTrace()
                    Log.w(TAG, "Could not read file")
                } catch (e: Exception) {
                    verificationStatus.setAA(
                        VerificationStatus.Verdict.NOT_CHECKED,
                        "Failed to read DG15",
                    )
                }
            } else {
                verificationStatus.setAA(VerificationStatus.Verdict.NOT_PRESENT, "AA is not supported")
            }

            try {
                dg2File = getDG2File(ps, maxBlockSize)
            } catch (e: Exception) {
                e.printStackTrace()
            }

            try {
                dg3File = getDG3File(ps, maxBlockSize)
            } catch (e: Exception) {
                e.printStackTrace()
            }

            try {
                dg5File = getDG5File(ps, maxBlockSize)
            } catch (e: Exception) {
                e.printStackTrace()
            }

            try {
                dg7File = getDG7File(ps, maxBlockSize)
            } catch (e: Exception) {
                e.printStackTrace()
            }

            try {
                dg11File = getDG11File(ps, maxBlockSize)
            } catch (e: Exception) {
                Log.d(TAG, "dg11: ")
                e.printStackTrace()
            }

            try {
                dg12File = getDG12File(ps, maxBlockSize)
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }

        fun verifySecurity(): VerificationStatus {
            verifyCS()

            verifyDS()

            verifyHT()

            if (service != null && dg15File != null) {
                verifyAA()
            }

            return verificationStatus
        }

        private fun updateCOMSODFile(newCertificate: X509Certificate?) {
            try {
                val digestAlg = sodFile!!.digestAlgorithm
                val signatureAlg = sodFile!!.digestEncryptionAlgorithm
                val cert = newCertificate ?: sodFile!!.docSigningCertificate
                val signature = sodFile!!.encryptedDigest
                val dgHashes = TreeMap<Int, ByteArray>()

                val dgFids = LDSFileUtil.getDataGroupNumbers(sodFile)
                val digest: MessageDigest
                digest = MessageDigest.getInstance(digestAlg)
                for (fid in dgFids) {
                    if (fid != PassportService.EF_COM.toInt() && fid != PassportService.EF_SOD.toInt() && fid != PassportService.EF_CVCA.toInt()) {
                        val dg = getDG(fid)
                        if (dg == null) {
                            Log.w(TAG, "Could not get input stream for " + Integer.toHexString(fid))
                            continue
                        }
                        val tag = dg.encoded[0]
                        dgHashes[LDSFileUtil.lookupDataGroupNumberByTag(tag.toInt())] =
                            digest.digest(dg.encoded)
                        comFile!!.insertTag(tag.toInt() and 0xFF)
                    }
                }
                sodFile =
                    if (this.docSigningPrivateKey != null) {
                        SODFile(digestAlg, signatureAlg, dgHashes, this.docSigningPrivateKey, cert)
                    } else {
                        SODFile(digestAlg, signatureAlg, dgHashes, signature, cert)
                    }
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }

        private fun verifyAA() {
            if (dg15File == null || service == null) {
                verificationStatus.setAA(VerificationStatus.Verdict.FAILED, "AA failed")
                return
            }

            try {
                val pubKey = dg15File!!.publicKey
                val pubKeyAlgorithm = pubKey.algorithm
                var digestAlgorithm = "SHA1"
                var signatureAlgorithm = "SHA1WithRSA/ISO9796-2"
                if ("EC" == pubKeyAlgorithm || "ECDSA" == pubKeyAlgorithm) {
                    val activeAuthenticationInfoList = ArrayList<ActiveAuthenticationInfo>()
                    val securityInfos = dg14File!!.securityInfos
                    for (securityInfo in securityInfos) {
                        if (securityInfo is ActiveAuthenticationInfo) {
                            activeAuthenticationInfoList.add(securityInfo)
                        }
                    }
                    val activeAuthenticationInfoCount = activeAuthenticationInfoList.size
                    if (activeAuthenticationInfoCount < 1) {
                        verificationStatus.setAA(
                            VerificationStatus.Verdict.FAILED,
                            "Found no active authentication info in EF.DG14",
                        )
                        return
                    } else if (activeAuthenticationInfoCount > 1) {
                        Log.w(TAG, "Found $activeAuthenticationInfoCount in EF.DG14, expected 1.")
                    }
                    val activeAuthenticationInfo = activeAuthenticationInfoList[0]

                    val signatureAlgorithmOID = activeAuthenticationInfo.signatureAlgorithmOID

                    signatureAlgorithm =
                        ActiveAuthenticationInfo.lookupMnemonicByOID(signatureAlgorithmOID)

                    digestAlgorithm =
                        Util.inferDigestAlgorithmFromSignatureAlgorithm(signatureAlgorithm)
                }
                val challengeLength = 8
                val challenge = ByteArray(challengeLength)
                random.nextBytes(challenge)
                val aaResult =
                    (service as PassportService).doAA(
                        dg15File!!.publicKey,
                        sodFile!!.digestAlgorithm,
                        sodFile!!.signerInfoDigestAlgorithm,
                        challenge,
                    )
                if (verifyAA(
                        pubKey,
                        digestAlgorithm,
                        signatureAlgorithm,
                        challenge,
                        aaResult.response,
                    )
                ) {
                    verificationStatus.setAA(VerificationStatus.Verdict.SUCCEEDED, "AA succeeded")
                } else {
                    verificationStatus.setAA(
                        VerificationStatus.Verdict.FAILED,
                        "AA failed due to signature failure",
                    )
                }
            } catch (cse: CardServiceException) {
                cse.printStackTrace()
                verificationStatus.setAA(
                    VerificationStatus.Verdict.FAILED,
                    "AA failed due to exception",
                )
            } catch (e: Exception) {
                e.printStackTrace()
                verificationStatus.setAA(
                    VerificationStatus.Verdict.FAILED,
                    "AA failed due to exception",
                )
            }
        }

        @Throws(CardServiceException::class)
        private fun verifyAA(
            publicKey: PublicKey,
            digestAlgorithm: String?,
            signatureAlgorithm: String?,
            challenge: ByteArray,
            response: ByteArray,
        ): Boolean {
            try {
                val pubKeyAlgorithm = publicKey.algorithm
                if ("RSA" == pubKeyAlgorithm) {
                    // FIXME: check that digestAlgorithm = "SHA1" in this case, check (and re-initialize) rsaAASignature (and rsaAACipher).
                    Log.w(
                        TAG,
                        "Unexpected algorithms for RSA AA: " +
                            "digest algorithm = " + (digestAlgorithm ?: "null") +
                            ", signature algorithm = " + (signatureAlgorithm ?: "null"),
                    )

                    rsaAADigest =
                        digestAlgorithm?.let { MessageDigest.getInstance(it) } // NOTE: for output length measurement only. -- MO
                    rsaAASignature = Signature.getInstance(signatureAlgorithm, BC_PROVIDER)

                    val rsaPublicKey = publicKey as RSAPublicKey
                    rasPublicKey = rsaPublicKey.toString()
                    rsaAACipher.init(Cipher.DECRYPT_MODE, rsaPublicKey)
                    rsaAASignature!!.initVerify(rsaPublicKey)

                    val digestLength = rsaAADigest!!.digestLength // SHA1 should be 20 bytes = 160 bits
                    if (digestLength != 20) throw AssertionError()
                    val plaintext = rsaAACipher.doFinal(response)
                    val m1 = Util.recoverMessage(digestLength, plaintext)
                    rsaAASignature!!.update(m1)
                    rsaAASignature!!.update(challenge)
                    return rsaAASignature!!.verify(response)
                } else if ("EC" == pubKeyAlgorithm || "ECDSA" == pubKeyAlgorithm) {
                    val ecdsaPublicKey = publicKey as ECPublicKey
                    this.ecdsaPublicKey = ecdsaPublicKey.toString()
                    if (ecdsaAASignature == null || signatureAlgorithm != null && signatureAlgorithm != ecdsaAASignature!!.algorithm) {
                        Log.w(
                            TAG,
                            "Re-initializing ecdsaAASignature with signature algorithm " + signatureAlgorithm!!,
                        )
                        ecdsaAASignature = Signature.getInstance(signatureAlgorithm)
                    }
                    if (ecdsaAADigest == null || digestAlgorithm != null && digestAlgorithm != ecdsaAADigest!!.algorithm) {
                        Log.w(
                            TAG,
                            "Re-initializing ecdsaAADigest with digest algorithm " + digestAlgorithm!!,
                        )
                        ecdsaAADigest = MessageDigest.getInstance(digestAlgorithm)
                    }

                    ecdsaAASignature!!.initVerify(ecdsaPublicKey)

                    if (response.size % 2 != 0) {
                        Log.w(TAG, "Active Authentication response is not of even length")
                    }

                    val l = response.size / 2
                    val r = Util.os2i(response, 0, l)
                    val s = Util.os2i(response, l, l)

                    ecdsaAASignature!!.update(challenge)

                    try {
                        val asn1Sequence =
                            DERSequence(arrayOf<ASN1Encodable>(ASN1Integer(r), ASN1Integer(s)))
                        return ecdsaAASignature!!.verify(asn1Sequence.encoded)
                    } catch (ioe: IOException) {
                        ioe.printStackTrace()
                        return false
                    }
                } else {
                    return false
                }
            } catch (iae: IllegalArgumentException) {
                // iae.printStackTrace();
                throw CardServiceException(iae.toString())
            } catch (iae: GeneralSecurityException) {
                throw CardServiceException(iae.toString())
            }
        }

        private fun verifyDS() {
            try {
                verificationStatus.setDS(VerificationStatus.Verdict.UNKNOWN, "Unknown")

                val docSigningCert = sodFile!!.docSigningCertificate
                if (docSigningCert == null) {
                    Log.w(TAG, "Could not get document signer certificate from EF.SOd")
                }
                if (checkDocSignature(docSigningCert)) {
                    verificationStatus.setDS(VerificationStatus.Verdict.SUCCEEDED, "Signature checked")
                } else {
                    verificationStatus.setDS(VerificationStatus.Verdict.FAILED, "Signature incorrect")
                }
            } catch (nsae: NoSuchAlgorithmException) {
                verificationStatus.setDS(
                    VerificationStatus.Verdict.FAILED,
                    "Unsupported signature algorithm",
                )
                return
            } catch (e: Exception) {
                e.printStackTrace()
                verificationStatus.setDS(VerificationStatus.Verdict.FAILED, "Unexpected exception")
                return
            }
        }

        private fun verifyCS() {
            try {
                val chain = ArrayList<Certificate>()

                if (sodFile == null) {
                    verificationStatus.setCS(
                        VerificationStatus.Verdict.FAILED,
                        "Unable to build certificate chain",
                        chain,
                    )
                    return
                }

                var docSigningCertificate: X509Certificate? = null
                var sodIssuer: X500Principal? = null
                var sodSerialNumber: BigInteger? = null
                try {
                    sodIssuer = sodFile!!.issuerX500Principal
                    sodSerialNumber = sodFile!!.serialNumber
                    docSigningCertificate = sodFile!!.docSigningCertificate
                } catch (e: Exception) {
                    Log.w(TAG, "Error getting document signing certificate: " + e.message)
                }

                if (docSigningCertificate != null) {
                    chain.add(docSigningCertificate)
                } else {
                    Log.w(TAG, "Error getting document signing certificate from EF.SOd")
                }

                val cscaStores = trustManager?.cscaStores
                if (cscaStores == null || cscaStores.size <= 0) {
                    Log.w(TAG, "No CSCA certificate stores found.")
                    verificationStatus.setCS(
                        VerificationStatus.Verdict.FAILED,
                        "No CSCA certificate stores found",
                        chain,
                    )
                }
                val cscaTrustAnchors = trustManager?.cscaAnchors
                if (cscaTrustAnchors == null || cscaTrustAnchors.size <= 0) {
                    Log.w(TAG, "No CSCA trust anchors found.")
                    verificationStatus.setCS(
                        VerificationStatus.Verdict.FAILED,
                        "No CSCA trust anchors found",
                        chain,
                    )
                }

                if (docSigningCertificate != null) {
                    val docIssuer = docSigningCertificate.issuerX500Principal
                    val docSerialNumber = docSigningCertificate.serialNumber
                    if (sodSerialNumber != null && sodSerialNumber != docSerialNumber) {
                        Log.w(
                            TAG,
                            "Security object serial number is different from embedded DS certificate serial number!",
                        )
                    }
                }

                val pkixChain =
                    PassportNfcUtils.getCertificateChain(
                        docSigningCertificate,
                        sodIssuer!!,
                        sodSerialNumber!!,
                        cscaStores!!,
                        cscaTrustAnchors!!,
                    )
                if (pkixChain == null) {
                    verificationStatus.setCS(
                        VerificationStatus.Verdict.FAILED,
                        "Could not build chain to trust anchor (pkixChain == null)",
                        chain,
                    )
                    return
                }

                for (certificate in pkixChain) {
                    if (certificate == docSigningCertificate) {
                        continue
                    }
                    chain.add(certificate)
                }

                val chainDepth = chain.size
                if (chainDepth <= 1) {
                    verificationStatus.setCS(
                        VerificationStatus.Verdict.FAILED,
                        "Could not build chain to trust anchor",
                        chain,
                    )
                    return
                }
                if (chainDepth > 1 && verificationStatus.cs == VerificationStatus.Verdict.UNKNOWN) {
                    verificationStatus.setCS(
                        VerificationStatus.Verdict.SUCCEEDED,
                        "Found a chain to a trust anchor",
                        chain,
                    )
                }
            } catch (e: Exception) {
                e.printStackTrace()
                verificationStatus.setCS(
                    VerificationStatus.Verdict.FAILED,
                    "Signature failed",
                    EMPTY_CERTIFICATE_CHAIN,
                )
            }
        }

        /**
         * Checks hashes in the SOd correspond to hashes we compute.
         */
        private fun verifyHT() {
            // Compare stored hashes to computed hashes.
            var hashResults: MutableMap<Int, VerificationStatus.HashMatchResult>? =
                verificationStatus.hashResults
            if (hashResults == null) {
                hashResults = TreeMap<Int, VerificationStatus.HashMatchResult>()
            }

            if (sodFile == null) {
                verificationStatus.setHT(VerificationStatus.Verdict.FAILED, "No SOd", hashResults)
                return
            }

            val storedHashes = sodFile!!.dataGroupHashes
            for (dgNumber in storedHashes.keys) {
                verifyHash(dgNumber, hashResults)
            }
            if (verificationStatus.ht == VerificationStatus.Verdict.UNKNOWN) {
                verificationStatus.setHT(
                    VerificationStatus.Verdict.SUCCEEDED,
                    "All hashes match",
                    hashResults,
                )
            } else {
                // Update storedHashes and computedHashes.
                verificationStatus.setHT(
                    verificationStatus.ht!!,
                    verificationStatus.htReason,
                    hashResults,
                )
            }
        }

        private fun verifyHash(dgNumber: Int): VerificationStatus.HashMatchResult? {
            var hashResults: MutableMap<Int, VerificationStatus.HashMatchResult>? =
                verificationStatus.hashResults
            if (hashResults == null) {
                hashResults = TreeMap<Int, VerificationStatus.HashMatchResult>()
            }
            return verifyHash(dgNumber, hashResults)
        }

        private fun verifyHash(
            dgNumber: Int,
            hashResults: MutableMap<Int, VerificationStatus.HashMatchResult>,
        ): VerificationStatus.HashMatchResult? {
            val fid = LDSFileUtil.lookupFIDByTag(LDSFileUtil.lookupTagByDataGroupNumber(dgNumber))

            // Get the stored hash for the DG.
            var storedHash: ByteArray?
            try {
                val storedHashes = sodFile!!.dataGroupHashes
                storedHash = storedHashes[dgNumber]
            } catch (e: Exception) {
                verificationStatus.setHT(
                    VerificationStatus.Verdict.FAILED,
                    "DG$dgNumber failed, could not get stored hash",
                    hashResults,
                )
                return null
            }

            // Initialize hash.
            val digestAlgorithm = sodFile!!.digestAlgorithm
            try {
                digest = getDigest(digestAlgorithm)
            } catch (nsae: NoSuchAlgorithmException) {
                verificationStatus.setHT(
                    VerificationStatus.Verdict.FAILED,
                    "Unsupported algorithm \"$digestAlgorithm\"",
                    null,
                )
                return null // DEBUG -- MO
            }

            // Read the DG.
            var dgBytes: ByteArray? = null
            try {
            /*InputStream dgIn = null;
            int length = lds.getLength(fid);
            if (length > 0) {
            dgBytes = new byte[length];
            dgIn = lds.getInputStream(fid);
            DataInputStream dgDataIn = new DataInputStream(dgIn);
            dgDataIn.readFully(dgBytes);
            }*/

                val abstractTaggedLDSFile = getDG(fid.toInt())
                if (abstractTaggedLDSFile != null) {
                    dgBytes = abstractTaggedLDSFile.encoded
                }

                if (abstractTaggedLDSFile == null && verificationStatus.eac != VerificationStatus.Verdict.SUCCEEDED && (fid == PassportService.EF_DG3 || fid == PassportService.EF_DG4)) {
                    Log.w(TAG, "Skipping DG$dgNumber during HT verification because EAC failed.")
                    val hashResult = VerificationStatus.HashMatchResult(storedHash!!, null)
                    hashResults[dgNumber] = hashResult
                    return hashResult
                }
                if (abstractTaggedLDSFile == null) {
                    Log.w(
                        TAG,
                        "Skipping DG$dgNumber during HT verification because file could not be read.",
                    )
                    val hashResult = VerificationStatus.HashMatchResult(storedHash!!, null)
                    hashResults[dgNumber] = hashResult
                    return hashResult
                }
            } catch (e: Exception) {
                val hashResult = VerificationStatus.HashMatchResult(storedHash!!, null)
                hashResults[dgNumber] = hashResult
                verificationStatus.setHT(
                    VerificationStatus.Verdict.FAILED,
                    "DG$dgNumber failed due to exception",
                    hashResults,
                )
                return hashResult
            }

            // Compute the hash and compare.
            try {
                val computedHash = digest!!.digest(dgBytes)
                val hashResult = VerificationStatus.HashMatchResult(storedHash!!, computedHash)
                hashResults[dgNumber] = hashResult

                if (!Arrays.equals(storedHash, computedHash)) {
                    verificationStatus.setHT(
                        VerificationStatus.Verdict.FAILED,
                        "Hash mismatch",
                        hashResults,
                    )
                }

                return hashResult
            } catch (ioe: Exception) {
                val hashResult = VerificationStatus.HashMatchResult(storedHash!!, null)
                hashResults[dgNumber] = hashResult
                verificationStatus.setHT(
                    VerificationStatus.Verdict.FAILED,
                    "Hash failed due to exception",
                    hashResults,
                )
                return hashResult
            }
        }

        @Throws(NoSuchAlgorithmException::class)
        private fun getDigest(digestAlgorithm: String): MessageDigest? {
            if (digest != null) {
                digest!!.reset()
                return digest
            }
            Log.i(TAG, "Using hash algorithm $digestAlgorithm")
            digest =
                if (Security.getAlgorithms("MessageDigest").contains(digestAlgorithm)) {
                    MessageDigest.getInstance(digestAlgorithm)
                } else {
                    MessageDigest.getInstance(digestAlgorithm, BC_PROVIDER)
                }
            return digest
        }

        private fun getDG(dg: Int): AbstractTaggedLDSFile? {
            when (dg) {
                1 -> {
                    return dg1File
                }

                2 -> {
                    return dg2File
                }

                3 -> {
                    return dg3File
                }

                5 -> {
                    return dg5File
                }

                7 -> {
                    return dg7File
                }

                11 -> {
                    return dg11File
                }

                12 -> {
                    return dg12File
                }

                14 -> {
                    return dg14File
                }

                15 -> {
                    return dg15File
                }

                else -> {
                    return null
                }
            }
        }

        @Throws(GeneralSecurityException::class)
        private fun checkDocSignature(docSigningCert: Certificate?): Boolean {
            val eContent = sodFile!!.eContent
            val signature = sodFile!!.encryptedDigest

            var digestEncryptionAlgorithm: String? = null
            try {
                digestEncryptionAlgorithm = sodFile!!.digestEncryptionAlgorithm
            } catch (e: Exception) {
                digestEncryptionAlgorithm = null
            }

            if (digestEncryptionAlgorithm == null) {
                val digestAlg = sodFile!!.signerInfoDigestAlgorithm
                var digest: MessageDigest? = null
                try {
                    digest = MessageDigest.getInstance(digestAlg)
                } catch (e: Exception) {
                    digest = MessageDigest.getInstance(digestAlg, BC_PROVIDER)
                }

                digest!!.update(eContent)
                val digestBytes = digest.digest()
                return Arrays.equals(digestBytes, signature)
            }

            if ("SSAwithRSA/PSS" == digestEncryptionAlgorithm) {
                val digestAlg = sodFile!!.signerInfoDigestAlgorithm
                digestEncryptionAlgorithm = digestAlg.replace("-", "") + "withRSA/PSS"
            }

            if ("RSA" == digestEncryptionAlgorithm) {
                val digestJavaString = sodFile!!.signerInfoDigestAlgorithm
                digestEncryptionAlgorithm = digestJavaString.replace("-", "") + "withRSA"
            }

            Log.i(TAG, "digestEncryptionAlgorithm = $digestEncryptionAlgorithm")

            var sig: Signature? = null

            sig = Signature.getInstance(digestEncryptionAlgorithm, BC_PROVIDER)
            if (digestEncryptionAlgorithm.endsWith("withRSA/PSS")) {
                val saltLength =
                    findSaltRsapss(
                        digestEncryptionAlgorithm,
                        docSigningCert,
                        eContent,
                        signature,
                    ) // Unknown salt so we try multiples until we get a success or failure
                val mgf1ParameterSpec = MGF1ParameterSpec("SHA-256")
                val pssParameterSpec =
                    PSSParameterSpec("SHA-256", "MGF1", mgf1ParameterSpec, saltLength, 1)
                sig!!.setParameter(pssParameterSpec)
            }
            sig!!.initVerify(docSigningCert)
            sig.update(eContent)
            return sig.verify(signature)
        }

        private fun findSaltRsapss(
            digestEncryptionAlgorithm: String,
            docSigningCert: Certificate?,
            eContent: ByteArray,
            signature: ByteArray,
        ): Int {
            for (i in 0..512) {
                try {
                    var sig: Signature? = null

                    sig = Signature.getInstance(digestEncryptionAlgorithm, BC_PROVIDER)
                    if (digestEncryptionAlgorithm.endsWith("withRSA/PSS")) {
                        val mgf1ParameterSpec = MGF1ParameterSpec("SHA-256")
                        val pssParameterSpec =
                            PSSParameterSpec("SHA-256", "MGF1", mgf1ParameterSpec, i, 1)
                        sig!!.setParameter(pssParameterSpec)
                    }

                    sig!!.initVerify(docSigningCert)
                    sig.update(eContent)
                    val verify = sig.verify(signature)
                    if (verify) {
                        return i
                    }
                } catch (e: Exception) {
                    e.printStackTrace()
                }
            }
            return 0 // Unable to find it
        }

        @Throws(IOException::class, CardServiceException::class, GeneralSecurityException::class)
        private fun doPACE(
            ps: PassportService,
            mrzInfo: MRZInfo,
            maxBlockSize: Int,
        ): PACEResult {
            var paceResult: PACEResult? = null
            var isCardAccessFile: InputStream? = null
            try {
                val bacKey = BACKey(mrzInfo.documentNumber, mrzInfo.dateOfBirth, mrzInfo.dateOfExpiry)
                val paceKeySpec = PACEKeySpec.createMRZKey(bacKey)
                isCardAccessFile = ps.getInputStream(PassportService.EF_CARD_ACCESS, maxBlockSize)

                val cardAccessFile = CardAccessFile(isCardAccessFile)
                val paceInfos = ArrayList<PACEInfo>()
                for (securityInfo in cardAccessFile.securityInfos) {
                    if (securityInfo is PACEInfo) {
                        paceInfos.add(securityInfo)
                    }
                }

                if (paceInfos.isNotEmpty()) {
                    for (paceInfo in paceInfos) {
                        try {
                            paceResult =
                                ps.doPACE(
                                    paceKeySpec,
                                    paceInfo.objectIdentifier,
                                    PACEInfo.toParameterSpec(paceInfo.parameterId),
                                    paceInfo.parameterId,
                                )
                            break
                        } catch (e: java.lang.Exception) {
                            e.printStackTrace()
                        }
                    }
                }
            } finally {
                if (isCardAccessFile != null) {
                    isCardAccessFile.close()
                    isCardAccessFile = null
                }
            }
            if (paceResult == null) {
                throw java.lang.Exception("PACE authentication failed")
            }
            return paceResult
        }

        @Throws(CardServiceException::class)
        private fun doBAC(
            ps: PassportService,
            mrzInfo: MRZInfo,
        ): BACResult {
            val bacKey = BACKey(mrzInfo.documentNumber, mrzInfo.dateOfBirth, mrzInfo.dateOfExpiry)
            return ps.doBAC(bacKey)
        }

        private fun doEACCA(
            ps: PassportService,
            mrzInfo: MRZInfo,
            dg14File: DG14File?,
            sodFile: SODFile?,
        ): List<EACCAResult> {
            if (dg14File == null) {
                throw NullPointerException("dg14File is null")
            }

            if (sodFile == null) {
                throw NullPointerException("sodFile is null")
            }

            // Chip Authentication
            val eaccaResults = ArrayList<EACCAResult>()

            var chipAuthenticationInfo: ChipAuthenticationInfo? = null

            val chipAuthenticationPublicKeyInfos = ArrayList<ChipAuthenticationPublicKeyInfo>()
            val securityInfos = dg14File.securityInfos
            val securityInfoIterator = securityInfos.iterator()
            while (securityInfoIterator.hasNext()) {
                val securityInfo = securityInfoIterator.next()
                if (securityInfo is ChipAuthenticationInfo) {
                    chipAuthenticationInfo = securityInfo
                } else if (securityInfo is ChipAuthenticationPublicKeyInfo) {
                    chipAuthenticationPublicKeyInfos.add(securityInfo)
                }
            }

            val publicKeyInfoIterator = chipAuthenticationPublicKeyInfos.iterator()
            while (publicKeyInfoIterator.hasNext()) {
                val authenticationPublicKeyInfo = publicKeyInfoIterator.next()
                try {
                    Log.i("EMRTD", "Chip Authentication starting")
                    val doEACCA =
                        ps.doEACCA(
                            chipAuthenticationInfo!!.keyId,
                            chipAuthenticationInfo.objectIdentifier,
                            chipAuthenticationInfo.protocolOIDString,
                            authenticationPublicKeyInfo.subjectPublicKey,
                        )
                    eaccaResults.add(doEACCA)
                } catch (cse: CardServiceException) {
                    cse.printStackTrace()
                }
            }

            return eaccaResults
        }

        fun isChipAuthenticationInfoAvailable(
            ps: PassportService,
            mrzInfo: MRZInfo,
            dg14File: DG14File?,
            sodFile: SODFile?,
        ): Boolean {
            if (dg14File == null) {
                throw NullPointerException("dg14File is null")
            }

            if (sodFile == null) {
                throw NullPointerException("sodFile is null")
            }
            val chipAuthenticationPublicKeyInfos = ArrayList<ChipAuthenticationPublicKeyInfo>()
            val securityInfos = dg14File.securityInfos
            val securityInfoIterator = securityInfos.iterator()
            while (securityInfoIterator.hasNext()) {
                val securityInfo = securityInfoIterator.next()
                if (securityInfo is ChipAuthenticationPublicKeyInfo) {
                    chipAuthenticationPublicKeyInfos.add(securityInfo)
                }
            }
            return chipAuthenticationPublicKeyInfos.isNotEmpty()
        }

        @Throws(
            IOException::class,
            CardServiceException::class,
            GeneralSecurityException::class,
            IllegalArgumentException::class,
            NullPointerException::class,
        )
        private fun doEACTA(
            ps: PassportService,
            mrzInfo: MRZInfo,
            cvcaFile: CVCAFile?,
            paceResult: PACEResult?,
            eaccaResult: EACCAResult?,
            cvcaKeyStores: List<KeyStore>,
        ): List<EACTAResult> {
            if (cvcaFile == null) {
                throw NullPointerException("CVCAFile is null")
            }

            if (eaccaResult == null) {
                throw NullPointerException("EACCAResult is null")
            }

            val eactaResults = ArrayList<EACTAResult>()
            val possibleCVCAReferences = arrayOf(cvcaFile.caReference, cvcaFile.altCAReference)

            // EAC
            for (caReference in possibleCVCAReferences) {
                val eacCredentials =
                    PassportNfcUtils.getEACCredentials(caReference, cvcaKeyStores)
                        ?: continue

                val privateKey = eacCredentials.privateKey
                val chain = eacCredentials.chain
                val terminalCerts = ArrayList<CardVerifiableCertificate>(chain.size)
                for (c in chain) {
                    terminalCerts.add(c as CardVerifiableCertificate)
                }

                try {
                    if (paceResult == null) {
                        val eactaResult =
                            ps.doEACTA(
                                caReference,
                                terminalCerts,
                                privateKey,
                                null,
                                eaccaResult,
                                mrzInfo.documentNumber,
                            )
                        eactaResults.add(eactaResult)
                    } else {
                        val eactaResult =
                            ps.doEACTA(
                                caReference,
                                terminalCerts,
                                privateKey,
                                null,
                                eaccaResult,
                                paceResult,
                            )
                        eactaResults.add(eactaResult)
                    }
                } catch (cse: CardServiceException) {
                    cse.printStackTrace()
                    // NOTE: Failed? Too bad, try next public key.
                    continue
                }

                break
            }

            return eactaResults
        }

        @Throws(CardServiceException::class, IOException::class)
        private fun getComFile(
            ps: PassportService,
            maxBlockSize: Int,
        ): COMFile {
            // COM FILE
            var isComFile: InputStream? = null
            try {
                isComFile = ps.getInputStream(PassportService.EF_COM, maxBlockSize)
                return LDSFileUtil.getLDSFile(PassportService.EF_COM, isComFile) as COMFile
            } finally {
                if (isComFile != null) {
                    isComFile.close()
                    isComFile = null
                }
            }
        }

        @Throws(CardServiceException::class, IOException::class)
        private fun getSodFile(
            ps: PassportService,
            maxBlockSize: Int,
        ): SODFile {
            // SOD FILE
            var isSodFile: InputStream? = null
            try {
                isSodFile = ps.getInputStream(PassportService.EF_SOD, maxBlockSize)
                return LDSFileUtil.getLDSFile(PassportService.EF_SOD, isSodFile) as SODFile
            } finally {
                if (isSodFile != null) {
                    isSodFile.close()
                    isSodFile = null
                }
            }
        }

        @Throws(CardServiceException::class, IOException::class)
        private fun getDG1File(
            ps: PassportService,
            maxBlockSize: Int,
        ): DG1File {
            // Basic data
            var isDG1: InputStream? = null
            try {
                isDG1 = ps.getInputStream(PassportService.EF_DG1, maxBlockSize)
                return LDSFileUtil.getLDSFile(PassportService.EF_DG1, isDG1) as DG1File
            } finally {
                if (isDG1 != null) {
                    isDG1.close()
                    isDG1 = null
                }
            }
        }

        @Throws(CardServiceException::class, IOException::class)
        private fun getDG2File(
            ps: PassportService,
            maxBlockSize: Int,
        ): DG2File {
            // Basic data
            var isDG2: InputStream? = null
            try {
                isDG2 = ps.getInputStream(PassportService.EF_DG2, maxBlockSize)
                return LDSFileUtil.getLDSFile(PassportService.EF_DG2, isDG2) as DG2File
            } finally {
                if (isDG2 != null) {
                    isDG2.close()
                    isDG2 = null
                }
            }
        }

        @Throws(CardServiceException::class, IOException::class)
        private fun getDG3File(
            ps: PassportService,
            maxBlockSize: Int,
        ): DG3File {
            // Basic data
            var isDG3: InputStream? = null
            try {
                isDG3 = ps.getInputStream(PassportService.EF_DG3, maxBlockSize)
                return LDSFileUtil.getLDSFile(PassportService.EF_DG3, isDG3) as DG3File
            } finally {
                if (isDG3 != null) {
                    isDG3.close()
                    isDG3 = null
                }
            }
        }

        @Throws(CardServiceException::class, IOException::class)
        private fun getDG5File(
            ps: PassportService,
            maxBlockSize: Int,
        ): DG5File {
            // Basic data
            var isDG5: InputStream? = null
            try {
                isDG5 = ps.getInputStream(PassportService.EF_DG5, maxBlockSize)
                return LDSFileUtil.getLDSFile(PassportService.EF_DG5, isDG5) as DG5File
            } finally {
                if (isDG5 != null) {
                    isDG5.close()
                    isDG5 = null
                }
            }
        }

        @Throws(CardServiceException::class, IOException::class)
        private fun getDG7File(
            ps: PassportService,
            maxBlockSize: Int,
        ): DG7File {
            // Basic data
            var isDG7: InputStream? = null
            try {
                isDG7 = ps.getInputStream(PassportService.EF_DG7, maxBlockSize)
                return LDSFileUtil.getLDSFile(PassportService.EF_DG7, isDG7) as DG7File
            } finally {
                if (isDG7 != null) {
                    isDG7.close()
                    isDG7 = null
                }
            }
        }

        @Throws(CardServiceException::class, IOException::class)
        private fun getDG11File(
            ps: PassportService,
            maxBlockSize: Int,
        ): DG11File {
            // Basic data
            var isDG11: InputStream? = null
            try {
                isDG11 = ps.getInputStream(PassportService.EF_DG11, maxBlockSize)
                return LDSFileUtil.getLDSFile(PassportService.EF_DG11, isDG11) as DG11File
            } finally {
                if (isDG11 != null) {
                    isDG11.close()
                    isDG11 = null
                }
            }
        }

        @Throws(CardServiceException::class, IOException::class)
        private fun getDG12File(
            ps: PassportService,
            maxBlockSize: Int,
        ): DG12File {
            // Basic data
            var isDG12: InputStream? = null
            try {
                isDG12 = ps.getInputStream(PassportService.EF_DG12, maxBlockSize)
                return LDSFileUtil.getLDSFile(PassportService.EF_DG12, isDG12) as DG12File
            } finally {
                if (isDG12 != null) {
                    isDG12.close()
                    isDG12 = null
                }
            }
        }

        @Throws(CardServiceException::class, IOException::class)
        private fun getDG14File(
            ps: PassportService,
            maxBlockSize: Int,
        ): DG14File {
            // Basic data
            var isDG14: InputStream? = null
            try {
                isDG14 = ps.getInputStream(PassportService.EF_DG14, maxBlockSize)
                return LDSFileUtil.getLDSFile(PassportService.EF_DG14, isDG14) as DG14File
            } finally {
                if (isDG14 != null) {
                    isDG14.close()
                    isDG14 = null
                }
            }
        }

        @Throws(CardServiceException::class, IOException::class)
        private fun getDG15File(
            ps: PassportService,
            maxBlockSize: Int,
        ): DG15File {
            // Basic data
            var isDG15: InputStream? = null
            try {
                isDG15 = ps.getInputStream(PassportService.EF_DG15, maxBlockSize)
                return LDSFileUtil.getLDSFile(PassportService.EF_DG15, isDG15) as DG15File
            } finally {
                if (isDG15 != null) {
                    isDG15.close()
                    isDG15 = null
                }
            }
        }

        @Throws(CardServiceException::class, IOException::class)
        private fun getCVCAFile(
            ps: PassportService,
            maxBlockSize: Int,
        ): CVCAFile {
            // Basic data
            var isEfCvCa: InputStream? = null
            try {
                isEfCvCa = ps.getInputStream(PassportService.EF_CVCA, maxBlockSize)
                return LDSFileUtil.getLDSFile(PassportService.EF_CVCA, isEfCvCa) as CVCAFile
            } finally {
                if (isEfCvCa != null) {
                    isEfCvCa.close()
                    isEfCvCa = null
                }
            }
        }

        private fun toDataGroupList(tagList: IntArray?): List<Int>? {
            if (tagList == null) {
                return null
            }
            val dgNumberList = ArrayList<Int>(tagList.size)
            for (tag in tagList) {
                try {
                    val dgNumber = LDSFileUtil.lookupDataGroupNumberByTag(tag)
                    dgNumberList.add(dgNumber)
                } catch (nfe: NumberFormatException) {
                    Log.w(TAG, "Could not find DG number for tag: " + Integer.toHexString(tag))
                    nfe.printStackTrace()
                }
            }
            return dgNumberList
        }

        companion object {
            private val TAG = PassportNFC::class.java.simpleName

            private val BC_PROVIDER = JMRTDSecurityProvider.spongyCastleProvider

            private val EMPTY_TRIED_BAC_ENTRY_LIST = emptyList<BACKey>()
            private val EMPTY_CERTIFICATE_CHAIN = emptyList<Certificate>()

            val MAX_BLOCK_SIZE: Int = PassportService.DEFAULT_MAX_BLOCKSIZE
            val MAX_TRANSCEIVE_LENGTH_FOR_SECURE_MESSAGING: Int =
                PassportService.NORMAL_MAX_TRANCEIVE_LENGTH
            val MAX_TRANSCEIVE_LENGTH_FOR_PACE: Int = PassportService.NORMAL_MAX_TRANCEIVE_LENGTH
        }
    }
