package nashid.verify.sdk.ui

import android.content.Context
import android.content.pm.PackageManager
import android.os.Bundle
import android.view.View
import androidx.appcompat.app.AppCompatActivity
import nashid.verify.sdk.utils.Loggers
import nashid.verify.sdk.utils.OnInternetAvailabilityListener
import nashid.verify.sdk.utils.helpers.ErrorUtility
import nashid.verify.sdk.utils.helpers.LocaleUtil
import nashid.verify.sdk.utils.helpers.Storage
import org.koin.core.component.KoinComponent

open class BaseActivity :
    AppCompatActivity(),
    KoinComponent,
    OnInternetAvailabilityListener {
    lateinit var storage: Storage
    private lateinit var oldPrefLocaleCode: String
    var isInternetAvailable: Boolean = true
    val tag = "BaseActivity"
    val logger = Loggers.withTag(this.javaClass.simpleName)

    private fun getPreferredLocale(context: Context): String {
        if (!::storage.isInitialized) {
            storage = Storage(context)
        }
        return storage.getPreferredLocale()
    }

    override fun onAvailable() {
        logger.log("onAvailable: ")
        isInternetAvailable = true
    }

    override fun onLost() {
        isInternetAvailable = false
        ErrorUtility.getInstance().showNoInternetDialog(
            this,
            !this.isFinishing && !this.isDestroyed,
        )
    }

    override fun onDestroy() {
        super.onDestroy()
        ErrorUtility.getInstance().unregisterConnectivityManager(this)
    }

    private fun resetTitle() {
        try {
            val label =
                packageManager.getActivityInfo(componentName, PackageManager.GET_META_DATA).labelRes
            if (label != 0) {
                setTitle(label)
            }
        } catch (e: PackageManager.NameNotFoundException) {
            e.printStackTrace()
        }
    }

    override fun attachBaseContext(newBase: Context) {
        getPreferredLocale(newBase)
        oldPrefLocaleCode = Storage(newBase).getPreferredLocale()
        applyOverrideConfiguration(LocaleUtil.getLocalizedConfiguration(oldPrefLocaleCode))
        super.attachBaseContext(newBase)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        isInternetAvailable = ErrorUtility.getInstance().registerConnectivityManager(this, this)
        resetTitle()
        updateLayoutDirection()
    }

    private fun updateLayoutDirection() {
        val config = resources.configuration
        if (config.layoutDirection == View.LAYOUT_DIRECTION_RTL) {
            window.decorView.layoutDirection = View.LAYOUT_DIRECTION_RTL
        } else {
            window.decorView.layoutDirection = View.LAYOUT_DIRECTION_LTR
        }
    }

    override fun onResume() {
        val currentLocaleCode = Storage(this).getPreferredLocale()
        if (oldPrefLocaleCode != currentLocaleCode) {
            recreate()
            oldPrefLocaleCode = currentLocaleCode
        }
        super.onResume()
        isInternetAvailable = ErrorUtility.getInstance().registerConnectivityManager(this, this)
    }
}
