package nashid.verify.sdk.viewmodel

import android.app.Application
import android.content.Intent
import android.content.res.Resources
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.util.Log
import androidx.camera.lifecycle.ProcessCameraProvider
import androidx.core.content.ContextCompat
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import com.google.mlkit.vision.common.InputImage
import com.google.mlkit.vision.text.Text
import com.google.mlkit.vision.text.TextRecognizer
import nashid.verify.sdk.data.repository.IdCardRepository
import nashid.verify.sdk.data.repository.PassportRepository
import nashid.verify.sdk.model.ScanDocumentResult
import nashid.verify.sdk.ui.CameraXLiveActivity
import nashid.verify.sdk.ui.NfcActivity
import nashid.verify.sdk.ui.SkipNfcLiveNessActivity
import nashid.verify.sdk.utils.ArtifactType
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.helpers.DocumentImages
import nashid.verify.sdk.utils.helpers.FaceDetectionUtil
import nashid.verify.sdkNew.R
import java.text.SimpleDateFormat
import java.util.Date
import java.util.Locale
import java.util.Objects

class CameraXLiveViewModel(
    val cameraXLiveData: CameraXLiveData,
    private val textRecognizer: TextRecognizer,
    private val app: Application,
    private val idCardRepository: IdCardRepository,
    private val passportRepository: PassportRepository,
    private val scanDocumentViewModel: ScanDocumentViewModel,
) : ViewModel() {
    private var cameraProviderLiveData: MutableLiveData<ProcessCameraProvider>? = null

    init {
        observeScanDocumentViewModel()
    }

    private fun observeScanDocumentViewModel() {
        scanDocumentViewModel.result.observeForever { result ->
            if (result is ScanDocumentResult.FileKeyResult) {
                Utility.getInstance().saveFileKeyAndContinue(result.fileKey, result.artifactType)
            }
        }
    }

    fun setSelectedDoc(
        selectedDoc: String,
        resources: Resources,
    ) {
        cameraXLiveData.setSelectedDoc(selectedDoc)
        if (selectedDoc.equals(resources.getString(R.string.e_passport), ignoreCase = true)) {
            cameraXLiveData.setStatusText(resources.getString(R.string.scan_passport_top_text))
            cameraXLiveData.setBottomText(resources.getString(R.string.scan_passport_bottom))
        } else {
            cameraXLiveData.setStatusText(resources.getString(R.string.scan_id_top_text))
            cameraXLiveData.setBottomText(resources.getString(R.string.scan_id_bottom))
        }
    }

    fun processTextFromBitmap(bitmap: Bitmap?) {
        textRecognizer
            .process(InputImage.fromBitmap(bitmap!!, 0))
            .addOnSuccessListener { textResult: Text ->
                val blocks = textResult.textBlocks
                passportRepository.readPassportData(blocks)
                val text = textResult.text
                readTextFromString(text)
            }.addOnFailureListener { cameraXLiveData.setIsSuccess(false) }
            .addOnCompleteListener {
                if (cameraXLiveData.getIsSuccess().value == false) {
                    cameraXLiveData.setCloseImageProxy(true)
                }
            }
    }

    private fun readTextFromString(text: String) {
        try {
            val textAll = text.split("\n".toRegex()).dropLastWhile { it.isEmpty() }.toTypedArray()
            if (Objects.requireNonNull(cameraXLiveData.getSelectedDoc().value).equals(
                    app.getString(R.string.e_passport),
                    ignoreCase = true,
                )
            ) {
                passportRepository.readAllDetailOfPassport()
            } else {
                if (cameraXLiveData.getIsFrontView().value == false) {
                    idCardRepository.readIdCardFrontView(text)
                } else {
                    idCardRepository.readIdCardBackView(textAll)
                }
            }
        } catch (e: Exception) {
            e.printStackTrace()
            cameraXLiveData.setIsSuccess(false)
        }
    }

    fun printCurrentTimeInLogs(): String {
        val sdf = SimpleDateFormat("yyyy-MM-dd HH:mm:ss", Locale.getDefault())
        val currentTime = sdf.format(Date())
        return currentTime
    }

    fun handleAnimationCompleted(activity: CameraXLiveActivity) {
        cameraXLiveData.setCTimer(true)
        Utility.getInstance().selectedDoc = cameraXLiveData.getSelectedDoc().value
        val fullBitmap =
            BitmapFactory.decodeByteArray(
                Utility.getInstance().scannedImage,
                0,
                Utility.getInstance().scannedImage?.size ?: 0,
            )
        if (Utility.getInstance().selectedDoc.equals(app.resources.getString(R.string.e_passport), ignoreCase = true)) {
            uploadArtifactForPassport(fullBitmap)
        } else {
            val bitmap =
                BitmapFactory.decodeByteArray(
                    Utility.getInstance().scannedIdFrontView,
                    0,
                    Utility.getInstance().scannedIdFrontView?.size ?: 0,
                )
            uploadArtifactForIdCard(bitmap, ArtifactType.BACKSIDE_IMAGE)
        }
        if (fullBitmap != null) {
            detectAndCropFace(
                fullBitmap,
                object : OnImageProcessedListener {
                    override fun onImageProcessed(croppedBitmap: Bitmap?) {
                        DocumentImages.instance.scanPhoto = croppedBitmap
                        uploadFaceArtifact(croppedBitmap)
                        if (SdkConfig.isNfcScanningEnabled) {
                            val intent = Intent(app, NfcActivity::class.java)
                            intent.putExtra(
                                app.resources.getString(R.string.doc_key),
                                cameraXLiveData.getSelectedDoc().value,
                            )
                            intent.setFlags(
                                Intent.FLAG_ACTIVITY_NEW_TASK,
                            )
                            app.startActivity(intent)
                            activity.finish()
                        } else {
                            openLiveNessScreen(activity)
                        }
                    }
                },
            )
        }
    }

    private fun openLiveNessScreen(activity: CameraXLiveActivity) {
        cameraXLiveData.setUnbindCameraProvider(true)
        Utility.getInstance().selectedDoc = cameraXLiveData.getSelectedDoc().value
        val intent = Intent(app, SkipNfcLiveNessActivity::class.java)
        intent.putExtra(
            app.resources.getString(R.string.doc_key),
            cameraXLiveData.getSelectedDoc().value,
        )
        intent.setFlags(
            Intent.FLAG_ACTIVITY_NEW_TASK,
        )
        app.startActivity(intent)
        activity.finish()
    }

    private fun detectAndCropFace(
        bitmap: Bitmap,
        listener: OnImageProcessedListener?,
    ) {
        val failureBitmap =
            BitmapFactory.decodeResource(app.resources, R.drawable.face_not_found)
        FaceDetectionUtil.cropFaceFromIDCard(bitmap) { croppedBitmap ->
            if (croppedBitmap != null) {
                listener?.onImageProcessed(croppedBitmap)
            } else {
                listener?.onImageProcessed(failureBitmap)
                Log.d("FaceDetector", "No faces detected")
            }
        }
    }

    val processCameraProvider: LiveData<ProcessCameraProvider>
        get() {
            if (cameraProviderLiveData == null) {
                cameraProviderLiveData = MutableLiveData()
                val cameraProviderFuture = ProcessCameraProvider.getInstance(app)
                cameraProviderFuture.addListener(
                    {
                        try {
                            cameraProviderLiveData!!.setValue(cameraProviderFuture.get())
                        } catch (e: InterruptedException) {
                            e.printStackTrace()
                        }
                    },
                    ContextCompat.getMainExecutor(app),
                )
            }
            return cameraProviderLiveData!!
        }

    internal interface OnImageProcessedListener {
        fun onImageProcessed(croppedBitmap: Bitmap?)
    }

    private fun uploadFaceArtifact(bitmap: Bitmap?) {
        bitmap?.let {
            uploadArtifact(it, ArtifactType.OCR_FACE_IMAGE)
        }
    }

    private fun uploadArtifact(
        bitmap: Bitmap?,
        artifactType: ArtifactType,
    ) {
        val file = Utility.getInstance().bitmapToFile(bitmap!!, app)
        val filePart = Utility.getInstance().createMultipartBody(file)
        scanDocumentViewModel.uploadArtifact(
            SdkConfig.registerToken!!,
            Utility.getInstance().getAppKey(),
            filePart,
            artifactType.type,
        )
    }

    private fun uploadArtifactForIdCard(
        bitmap: Bitmap?,
        artifactType: ArtifactType,
    ) {
        bitmap?.let {
            uploadArtifact(it, artifactType)
        }
    }

    private fun uploadArtifactForPassport(bitmap: Bitmap?) {
        bitmap?.let {
            uploadArtifact(it, ArtifactType.FRONTSIDE_IMAGE)
        }
    }

    fun scannedIdFrontViewCompleted(byteArray: ByteArray) {
        val fullBitmap =
            BitmapFactory.decodeByteArray(
                byteArray,
                0,
                byteArray.size,
            )
        uploadArtifactForIdCard(fullBitmap, ArtifactType.FRONTSIDE_IMAGE)
    }

    override fun onCleared() {
        super.onCleared()
        cameraXLiveData.resetState()
    }
}
