package nashid.verify.sdk.viewmodel

import android.app.Application
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import nashid.verify.sdk.data.repository.ScanDocumentRepository
import nashid.verify.sdk.model.Artifacts
import nashid.verify.sdk.model.DocumentType
import nashid.verify.sdk.model.DocumentVerification
import nashid.verify.sdk.model.ScanDocumentResult
import nashid.verify.sdk.utils.ArtifactType
import nashid.verify.sdk.utils.Loggers
import nashid.verify.sdk.utils.NetWorkResult
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdkNew.R
import okhttp3.MultipartBody
import java.util.*

class ScanDocumentViewModel(
    private val scanDocumentRepository: ScanDocumentRepository,
    private val context: Application,
) : ViewModel() {
    private val logger = Loggers.withTag("ScanDocumentViewModel")
    private val _result = MutableLiveData<ScanDocumentResult?>()
    val result: LiveData<ScanDocumentResult?> get() = _result

    companion object {
        private val uploadedArtifacts = Collections.synchronizedSet(mutableSetOf<String>())
        private val requiredArtifacts = Collections.synchronizedSet(mutableSetOf<String>())
    }

    private var isVerificationSubmissionInProgress = false
    private val maxRetries = 10
    private val retryDelayMs = 1000L

    init {
        updateRequiredArtifacts()
    }

    private fun updateRequiredArtifacts() {
        synchronized(requiredArtifacts) {
            requiredArtifacts.clear()
            requiredArtifacts.addAll(
                buildList {
                    if (SdkConfig.isOcrEnabled || SdkConfig.isMrzEnabled) {
                        add(ArtifactType.FRONTSIDE_IMAGE.type)
                        if (SdkConfig.viewType != DocumentType.Passport) {
                            add(ArtifactType.BACKSIDE_IMAGE.type)
                        }
                        add(ArtifactType.OCR_FACE_IMAGE.type)
                    }
                    if (SdkConfig.isNfcScanningEnabled && SdkConfig.isNfcSkipable != true) {
                        add(ArtifactType.NFC_IMAGE.type)
                    }
                    if (SdkConfig.isFaceMatchingEnabled || SdkConfig.isPassiveLiveNessEnabled) {
                        add(ArtifactType.LIVENESS_IMAGE.type)
                    }
                },
            )
        }
        logger.log("Required artifacts updated: $requiredArtifacts")
    }

    fun uploadArtifact(
        token: String,
        appKey: String,
        filePart: MultipartBody.Part,
        artifactType: String,
    ) {
        viewModelScope.launch(Dispatchers.IO) {
            try {
                val docType =
                    when (artifactType) {
                        ArtifactType.FRONTSIDE_IMAGE.type,
                        ArtifactType.BACKSIDE_IMAGE.type,
                        ArtifactType.OCR_FACE_IMAGE.type,
                        -> 1
                        ArtifactType.NFC_IMAGE.type -> 2
                        ArtifactType.LIVENESS_IMAGE.type,
                        -> 3
                        else -> 1
                    }
                when (val response = scanDocumentRepository.uploadFile(token, appKey, filePart)) {
                    is NetWorkResult.Success -> {
                        if (response.data?.data?.fileKey != null) {
                            synchronized(uploadedArtifacts) {
                                uploadedArtifacts.add(artifactType)
                                logger.log("Upload successful for $artifactType. Current uploads: $uploadedArtifacts")
                            }
                            _result.postValue(ScanDocumentResult.FileKeyResult(response.data.data.fileKey, artifactType))
                            logCurrentState()
                        } else {
                            logger.log("Upload failed for $artifactType: ${response.message}")
                            _result.postValue(ScanDocumentResult.Error(response.message ?: "Upload failed"))
                        }
                    }

                    is NetWorkResult.Error -> {
                        logger.log("Upload failed for $artifactType: ${response.message}")
                        _result.postValue(ScanDocumentResult.Error(response.message ?: "Upload failed"))
                    }

                    else -> {}
                }
            } catch (e: Exception) {
                logger.log("Upload failed for $artifactType: ${e.message}")
                _result.postValue(ScanDocumentResult.Error(e.message ?: context.getString(R.string.unexpected_error_occurred)))
            }
        }
    }

    private fun logCurrentState() {
        logger.log("Current state - Required: $requiredArtifacts, Uploaded: $uploadedArtifacts")
    }

    fun submitVerificationWithRetry(
        token: String,
        appKey: String,
        verification: DocumentVerification,
    ) {
        if (isVerificationSubmissionInProgress) {
            logger.log("Verification submission already in progress")
            _result.postValue(ScanDocumentResult.Error("Verification submission already in progress"))
            return
        }

        viewModelScope.launch(Dispatchers.IO) {
            isVerificationSubmissionInProgress = true
            try {
                val currentUploads = synchronized(uploadedArtifacts) { uploadedArtifacts.toSet() }
                val currentRequired = synchronized(requiredArtifacts) { requiredArtifacts.toSet() }
                val missingArtifacts = currentRequired - currentUploads

                if (missingArtifacts.isEmpty()) {
                    logger.log("All required artifacts present. Proceeding with verification submission.")
                    submitVerification(token, appKey, verification)
                    return@launch
                }

                var retryCount = 0
                while (retryCount < maxRetries && isVerificationSubmissionInProgress) {
                    logger.log("Retry attempt ${retryCount + 1} of $maxRetries")

                    val updatedUploads = synchronized(uploadedArtifacts) { uploadedArtifacts.toSet() }
                    val updatedMissing = currentRequired - updatedUploads

                    if (updatedMissing.isEmpty()) {
                        logger.log("All artifacts uploaded. Proceeding with verification.")
                        val updatedVerification =
                            verification.copy(
                                artifacts =
                                    Artifacts(
                                        backSideImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.BACKSIDE_IMAGE) ?: "",
                                        frontSideImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.FRONTSIDE_IMAGE) ?: "",
                                        ocrFaceImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.OCR_FACE_IMAGE) ?: "",
                                        nfcFaceImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.NFC_IMAGE) ?: "",
                                        livenessImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.LIVENESS_IMAGE) ?: "",
                                    ),
                            )
                        submitVerification(token, appKey, updatedVerification)
                        break
                    }
                    logger.log("Still missing artifacts: $updatedMissing")
                    _result.postValue(ScanDocumentResult.Error("Waiting for artifacts: $updatedMissing"))

                    delay(retryDelayMs)
                    retryCount++

                    if (retryCount >= maxRetries) {
                        logger.log("Max retries reached. Missing artifacts: $updatedMissing")
                        _result.postValue(
                            ScanDocumentResult.ScanDocumentResponse(false, "Max retries reached. Missing artifacts: $updatedMissing\"", ""),
                        )
                    }
                }
            } catch (e: Exception) {
                logger.log("Verification submission failed with exception: ${e.message}")
                _result.postValue(ScanDocumentResult.Error("Verification failed: ${e.message}"))
            } finally {
                isVerificationSubmissionInProgress = false
            }
        }
    }

    fun resetArtifacts() {
        synchronized(uploadedArtifacts) {
            uploadedArtifacts.clear()
            logger.log("Cleared uploaded artifacts")
        }
        Utility.getInstance().clearArtifactKeys()
        updateRequiredArtifacts()
        viewModelScope.launch(Dispatchers.Main) {
            _result.value = null
            isVerificationSubmissionInProgress = false
        }
        logger.log("Artifacts reset. Required: $requiredArtifacts, Uploaded: $uploadedArtifacts")
    }

    private suspend fun submitVerification(
        token: String,
        appKey: String,
        verification: DocumentVerification,
    ) {
        try {
            when (val response = scanDocumentRepository.submitVerification(token, appKey, verification)) {
                is NetWorkResult.Success -> {
                    if (response.data?.data?.verificationId != null) {
                        val verificationId = response.data.data.verificationId
                        logger.log("SubmitVerification successful for $verificationId: ${response.data.message}")

                        // Post the success result before resetting artifacts
                        withContext(Dispatchers.Main) {
                            _result.value =
                                ScanDocumentResult.ScanDocumentResponse(
                                    true,
                                    response.data.message ?: "Verification submitted successfully",
                                    verificationId,
                                )
                        }
                        // Small delay to ensure the success result is processed before resetting artifacts
                        delay(100)
                        resetArtifacts()
                    } else {
                        withContext(Dispatchers.Main) {
                            _result.value =
                                ScanDocumentResult.ScanDocumentResponse(
                                    false,
                                    response.data?.message ?: "Verification submission failed",
                                    "",
                                )
                        }
                    }
                }
                is NetWorkResult.Error -> {
                    withContext(Dispatchers.Main) {
                        _result.value =
                            ScanDocumentResult.ScanDocumentResponse(
                                false,
                                response.message ?: context.getString(R.string.unexpected_error_occurred),
                                "",
                            )
                    }
                }
                else -> {}
            }
        } catch (e: Exception) {
            logger.log("SubmitVerification failed with exception: ${e.message}")
            withContext(Dispatchers.Main) {
                _result.value =
                    ScanDocumentResult.ScanDocumentResponse(
                        false,
                        e.message ?: context.getString(R.string.unexpected_error_occurred),
                        "",
                    )
            }
        }
    }

    fun getVerification(
        token: String,
        appKey: String,
        verificationId: String,
    ) {
        viewModelScope.launch(Dispatchers.IO) {
            when (val response = scanDocumentRepository.getVerification(token, appKey, verificationId)) {
                is NetWorkResult.Success -> {
                    logger.log("GetVerification successful for $verificationId: ${response.data?.message}")
                    _result.postValue(response.data?.let { ScanDocumentResult.GetScanResultResponse(true, it.message, response.data) })
                }

                is NetWorkResult.Error -> {
                    logger.log("GetVerification failed for $verificationId: ${response.message}")
                    _result.postValue(
                        ScanDocumentResult.GetScanResultResponse(
                            false,
                            response.message ?: context.getString(R.string.unexpected_error_occurred),
                            null,
                        ),
                    )
                }

                else -> {}
            }
        }
    }

    fun clearScanDocumentData() {
        _result.value = null
    }
}
