package nashid.verify.sdk.ui

import android.Manifest
import android.animation.Animator
import android.animation.AnimatorListenerAdapter
import android.annotation.SuppressLint
import android.annotation.TargetApi
import android.app.Dialog
import android.content.Context
import android.content.pm.PackageManager
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Color
import android.graphics.PointF
import android.graphics.Rect
import android.graphics.RectF
import android.graphics.SurfaceTexture
import android.graphics.drawable.ColorDrawable
import android.hardware.camera2.CameraCharacteristics
import android.hardware.camera2.CameraManager
import android.media.Image
import android.os.Build
import android.os.Bundle
import android.os.CountDownTimer
import android.util.Log
import android.util.Size
import android.util.TypedValue
import android.view.Surface
import android.view.View.GONE
import android.view.View.INVISIBLE
import android.view.View.VISIBLE
import android.view.ViewGroup
import android.view.Window
import android.widget.LinearLayout
import android.widget.Toast
import androidx.activity.addCallback
import androidx.camera.core.CameraSelector
import androidx.camera.core.ImageAnalysis
import androidx.camera.core.ImageCapture
import androidx.camera.core.ImageCaptureException
import androidx.camera.core.ImageProxy
import androidx.camera.core.Preview
import androidx.camera.lifecycle.ProcessCameraProvider
import androidx.core.content.ContextCompat
import androidx.core.content.res.ResourcesCompat
import androidx.core.view.doOnLayout
import com.google.mlkit.vision.common.InputImage
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext
import nashid.mv.liveness.DetectionResults
import nashid.mv.liveness.EngineWrappers
import nashid.verify.sdk.utils.Constants
import nashid.verify.sdk.utils.helpers.LiveNessCameraLayout
import nashid.verify.sdk.utils.helpers.TextSizeConverter
import nashid.verify.sdk.utils.helpers.Utility
import nashid.verify.sdkNew.R
import nashid.verify.sdkNew.databinding.DialogBackBinding
import nashid.verify.sdkNew.databinding.LivenessActivityMainBinding
import org.koin.android.ext.android.inject
import java.io.ByteArrayOutputStream
import java.util.concurrent.ExecutorService
import java.util.concurrent.Executors

@OptIn(ExperimentalCoroutinesApi::class)
internal class LiveNessActivity : BaseActivity() {
    private lateinit var liveNessLayout: LiveNessCameraLayout
    private lateinit var binding: LivenessActivityMainBinding
    private lateinit var engineWrappers: EngineWrappers
    private var enginePrepared: Boolean = false
    private var cameraProvider: ProcessCameraProvider? = null
    private val textSizeConverter: TextSizeConverter by inject()
    private var startToCaptureImage = false
    private var cameraSelector: CameraSelector? = null
    private var imageCount = 0

    @OptIn(DelicateCoroutinesApi::class)
    private val detectionContext = newSingleThreadContext("detection")

    @TargetApi(Build.VERSION_CODES.M)
    private fun requestPermission() = requestPermissions(permissions, PERMISSION_REQUEST_CODE)

    private var working: Boolean = false
    private lateinit var result: DetectionResults
    private lateinit var detectionResults: DetectionResults
    private var delayMillis: Long = 10000
    private var cTimer: CountDownTimer? = null
    private lateinit var imageProxy: ImageProxy
    private val previewWidth: Int = 640
    private val previewHeight: Int = 480
    private lateinit var cameraExecutor: ExecutorService
    private val frameOrientation: Int = 7
    private var threshold: Float = DEFAULT_THRESHOLD
    private var startRecording = false
    private var blink: Boolean = false
    private var smile: Boolean = false
    private var isStartTimer: Boolean = false
    private var factorX: Float = 0F
    private var factorY: Float = 0F
    private var screenWidth: Int = 0
    private var screenHeight: Int = 0

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = LivenessActivityMainBinding.inflate(layoutInflater)
        setContentView(binding.root)
        if (hasPermissions()) {
            init()
        } else {
            requestPermission()
        }
    }

    private fun init() {
        backPress()
        detectionResults = DetectionResults()
        liveNessLayout = LiveNessCameraLayout(this, binding, storage, textSizeConverter)
        liveNessLayout.changeTextSize()
        startCamera()
        binding.layoutHeader.imgBack.setOnClickListener { backPress() }
        binding.lytStartRecording.setOnClickListener {
            startRecording = true
            binding.cardPressDesc.visibility = GONE
            binding.cardStartDesc.visibility = GONE

            binding.lytStartRecording.visibility = INVISIBLE
            binding.preview.visibility = VISIBLE
            binding.overlayMainLayout.setBackgroundColor(
                ContextCompat.getColor(
                    applicationContext,
                    R.color.transparent,
                ),
            )
            binding.lytTopView.setBackgroundColor(
                ContextCompat.getColor(
                    applicationContext,
                    R.color.face_camera_back,
                ),
            )
            binding.lytBottomMain.setBackgroundColor(
                ContextCompat.getColor(
                    applicationContext,
                    R.color.face_camera_back,
                ),
            )
        }
    }

    private fun backPress() {
        onBackPressedDispatcher.addCallback(this) {
            showLeaveOnboardingDialog()
        }
    }

    private fun hasPermissions(): Boolean {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            for (permission in permissions) {
                if (checkSelfPermission(permission) != PackageManager.PERMISSION_GRANTED) {
                    return false
                }
            }
        }
        return true
    }

    private fun takeImage() {
        runOnUiThread {
            val imageCapture =
                ImageCapture
                    .Builder()
                    .setTargetRotation(Surface.ROTATION_0)
                    .build()
            startToCaptureImage = true
            cameraProvider?.let {
                try {
                    it.unbindAll()
                    it.bindToLifecycle(this, cameraSelector!!, imageCapture)
                    imageCapture.takePicture(
                        ContextCompat.getMainExecutor(this),
                        object : ImageCapture.OnImageCapturedCallback() {
                            @SuppressLint("RestrictedApi", "UnsafeOptInUsageError")
                            override fun onCaptureSuccess(image: ImageProxy) {
                                val bitmap = image.image?.toBitmap()

                                if (bitmap != null) {
                                    val stream = ByteArrayOutputStream()
                                    bitmap.compress(Bitmap.CompressFormat.PNG, 100, stream)
                                    val bytes = stream.toByteArray()
                                    Utility.getInstance().liveImage = bytes
                                    cameraProvider?.unbindAll()
                                    binding.lytStartRecording.visibility = INVISIBLE
                                    binding.lyutSucess.visibility = VISIBLE
                                    binding.imgScanComplete.playAnimation()
                                    binding.imgScanComplete.setMaxFrame(80)
                                    binding.imgScanComplete.addAnimatorListener(
                                        object : AnimatorListenerAdapter() {
                                            override fun onAnimationEnd(animation: Animator) {
                                                super.onAnimationEnd(animation)
                                                binding.txtTakeSelfie.visibility = INVISIBLE
                                                Utility.getInstance().liveImage = bytes
                                                finish()
                                            }
                                        },
                                    )
                                }
                                image.close()
                            }

                            override fun onError(exc: ImageCaptureException) {
                            }
                        },
                    )
                } catch (ex: Exception) {
                    Log.e(tag, "Failed to capture image", ex)
                }
            }
        }
    }

    private fun Image.toBitmap(): Bitmap? {
        val buffer = planes[0].buffer
        val bytes = ByteArray(buffer.capacity())
        buffer.get(bytes)
        return BitmapFactory.decodeByteArray(bytes, 0, bytes.size, null)
    }

    private fun startTimer() {
        cTimer =
            object : CountDownTimer(delayMillis, 1000) {
                override fun onTick(millisUntilFinished: Long) {
                }

                override fun onFinish() {
                    try {
                        if (!result.hasFace) {
                            setRescanningDocumentVisibility()
                            delayMillis = 15000
                        } else {
                            cTimer!!.start()
                        }
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }
                }
            }.start()
    }

    private fun setRescanningDocumentVisibility() {
        binding.lyoutScanFail.visibility = VISIBLE
        binding.btnRescanningDocuments.setOnClickListener {
            binding.cardStartDesc.visibility = GONE
            binding.lyoutScanFail.visibility = GONE
            imageProxy.close()
            cTimer?.start()
        }
    }

    @OptIn(DelicateCoroutinesApi::class)
    @SuppressLint("UnsafeOptInUsageError")
    private fun onPreviewFrame(imageProxy: ImageProxy) {
        val byteArray = liveNessLayout.getYuvDataFromImage(imageProxy)
        if (binding.lyoutScanFail.visibility == GONE) {
            if (!startToCaptureImage) {
                if (enginePrepared) {
                    if (!working) {
                        GlobalScope.launch(detectionContext) {
                            working = true

                            result =
                                engineWrappers.detect(
                                    byteArray,
                                    previewWidth,
                                    previewHeight,
                                    frameOrientation,
                                )
                            result.threshold = threshold
                            runOnUiThread {
                                if (!isStartTimer) {
                                    startTimer()
                                    isStartTimer = true
                                }
                            }

                            factorX = screenWidth / previewHeight.toFloat()
                            factorY = screenHeight / previewWidth.toFloat()
                            val rect =
                                calculateBoxLocationOnScreen(
                                    result.left,
                                    result.top,
                                    result.right,
                                    result.bottom,
                                )
                            detectionResults = result.updateLocation(rect)

                            val isOutside = isRectOutside(result)
                            if (isOutside) {
                                this@LiveNessActivity.runOnUiThread {
                                    if (startRecording) {
                                        binding.coverView.changeColor(Color.WHITE)
                                        binding.txtTakeSelfie.text =
                                            getString(R.string.face_scanning)
                                    }
                                }
                                if (binding.lyoutScanFail.visibility != VISIBLE) {
                                    imageProxy.close()
                                }
                            } else {
                                this@LiveNessActivity.runOnUiThread {
                                    if (startRecording) {
                                        if (!Constants.IS_ACTIVE_LIVENESS_ENABLED) {
                                            if (liveNessLayout.isFaceCenteredInRectangle(rect)) {
                                                binding.txtTakeSelfie.text =
                                                    getString(R.string.face_center)
                                            } else {
                                                binding.txtTakeSelfie.text =
                                                    getString(R.string.face_smile)
                                            }
                                        }
                                        binding.coverView.changeColor(
                                            ContextCompat.getColor(
                                                applicationContext,
                                                R.color.face_detection,
                                            ),
                                        )
                                    }
                                }
                                if (result.confidence >= 0.985F) {
                                    Log.d(tag, "onPreviewFrame:real ")
                                    if (startRecording) {
                                        if (Constants.IS_ACTIVE_LIVENESS_ENABLED) {
                                            if (imageCount >= 10) {
                                                val mediaImage: Image = imageProxy.image!!
                                                InputImage.fromMediaImage(
                                                    mediaImage,
                                                    imageProxy.imageInfo.rotationDegrees,
                                                )
                                                runOnUiThread(
                                                    kotlinx.coroutines.Runnable {
                                                        if (!blink) {
                                                            binding.txtTakeSelfie.text =
                                                                getString(
                                                                    R.string.liveness_blink,
                                                                )
                                                        } else {
                                                            binding.txtTakeSelfie.text =
                                                                getString(
                                                                    R.string.liveness_smile,
                                                                )
                                                            if (!smile) {
                                                                binding.txtTakeSelfie.text =
                                                                    getString(
                                                                        R.string.liveness_smile,
                                                                    )
                                                            } else {
                                                                binding.txtTakeSelfie.text = getString(R.string.empty_text)
                                                            }
                                                        }
                                                    },
                                                )
                                            } else {
                                                if (binding.lyoutScanFail.visibility != VISIBLE) {
                                                    imageProxy.close()
                                                }
                                            }
                                        } else {
                                            if (imageCount == 10) {
                                                takeImage()
                                                setLiveNessScore()
                                            } else {
                                                if (binding.lyoutScanFail.visibility != VISIBLE) {
                                                    imageProxy.close()
                                                }
                                            }
                                        }
                                        imageCount++
                                    } else {
                                        if (binding.lyoutScanFail.visibility != VISIBLE) {
                                            imageProxy.close()
                                        }
                                    }
                                } else {
                                    Log.d(tag, "onPreviewFrame:fake ")
                                    if (binding.lyoutScanFail.visibility != VISIBLE) {
                                        imageProxy.close()
                                    }
                                }
                            }
                            working = false
                        }
                    } else {
                        if (binding.lyoutScanFail.visibility != VISIBLE) {
                            imageProxy.close()
                        }
                    }
                } else {
                    if (binding.lyoutScanFail.visibility != VISIBLE) {
                        imageProxy.close()
                    }
                }
            }
        }
    }

    private fun setLiveNessScore() {
        val liveNessScore = result.confidence * 100
        Utility.getInstance().liveNessScore = liveNessScore.toInt()
    }

    private fun isRectOutside(parentRect: DetectionResults): Boolean {
        val rect =
            RectF(
                parentRect.left.toFloat(),
                parentRect.top.toFloat(),
                parentRect.right.toFloat(),
                parentRect.bottom.toFloat(),
            )
        val ovalRect =
            RectF(
                binding.coverView.getRect().left,
                binding.coverView.getRect().top,
                binding.coverView.getRect().right,
                binding.coverView.getRect().bottom,
            )

        val ovalCenterX = ovalRect.centerX()
        val ovalCenterY = ovalRect.centerY()
        val ovalRadiusX = ovalRect.width() / 2
        val ovalRadiusY = ovalRect.height() / 2

        val rectCorners =
            arrayOf(
                PointF(rect.left, rect.top),
                PointF(rect.right, rect.top),
                PointF(rect.left, rect.bottom),
                PointF(rect.right, rect.bottom),
            )

        for (corner in rectCorners) {
            val dx = corner.x - ovalCenterX
            val dy = corner.y - ovalCenterY

            val distance =
                (dx * dx) / (ovalRadiusX * ovalRadiusX) + (dy * dy) / (ovalRadiusY * ovalRadiusY)
            if (distance > 1) {
                return true
            }
        }

        return false
    }

    private fun calculateBoxLocationOnScreen(
        left: Int,
        top: Int,
        right: Int,
        bottom: Int,
    ): Rect =
        Rect(
            (left * factorX).toInt(),
            (top * factorY).toInt(),
            (right * factorX).toInt(),
            (bottom * factorY).toInt(),
        )

    private fun getBestPreviewSize(): Size? {
        val cameraManager = getSystemService(Context.CAMERA_SERVICE) as CameraManager
        val characteristics = cameraManager.getCameraCharacteristics(liveNessLayout.getFrontCameraId().toString())
        val streamConfigurationMap =
            characteristics.get(CameraCharacteristics.SCALER_STREAM_CONFIGURATION_MAP)

        val outputSizes = streamConfigurationMap?.getOutputSizes(SurfaceTexture::class.java)

        var bestPreviewSize: Size? = null
        var bestPreviewSizeArea = 0

        outputSizes?.forEach { size ->
            val area = size.width * size.height
            if (size.width >= previewWidth && size.height >= previewHeight && area > bestPreviewSizeArea) {
                bestPreviewSize = size
                bestPreviewSizeArea = area
            }
        }

        return bestPreviewSize
    }

    @SuppressLint("UnsafeOptInUsageError")
    private fun startCamera() {
        calculateSize()
        cameraExecutor = Executors.newSingleThreadExecutor()
        getBestPreviewSize()

        val preview =
            Preview
                .Builder()
                .build()
                .also {
                    it.surfaceProvider = binding.preview.surfaceProvider
                }

        val imageAnalysis =
            ImageAnalysis.Builder().build().also {
                it.setAnalyzer(
                    cameraExecutor,
                ) { imageProxy ->
                    if (binding.lyoutScanFail.visibility == GONE) {
                        if (startRecording) {
                            this.imageProxy = imageProxy
                            onPreviewFrame(imageProxy)
                        } else {
                            imageProxy.close()
                        }
                    }
                }
            }
        cameraSelector =
            CameraSelector
                .Builder()
                .requireLensFacing(CameraSelector.LENS_FACING_FRONT)
                .build()

        val cameraProviderFuture = ProcessCameraProvider.getInstance(this)

        cameraProviderFuture.addListener({
            cameraProvider = cameraProviderFuture.get()

            try {
                cameraProvider?.unbindAll()
                cameraProvider?.bindToLifecycle(this, cameraSelector!!, preview, imageAnalysis)
            } catch (ex: Exception) {
                Log.e(tag, "Failed to bind camera", ex)
            }
        }, ContextCompat.getMainExecutor(this))
    }

    private fun calculateSize() {
        binding.coverView.doOnLayout {
            screenWidth = binding.coverView.width
            screenHeight = binding.coverView.height
        }
    }

    override fun onResume() {
        engineWrappers = EngineWrappers(assets)
        enginePrepared = engineWrappers.init()
        if (!enginePrepared) {
            Toast.makeText(this, "Engine init failed.", Toast.LENGTH_LONG).show()
        }
        super.onResume()
    }

    @SuppressLint("RestrictedApi")
    override fun onDestroy() {
        engineWrappers.destroy()
        cameraProvider?.unbindAll()
        cTimer?.cancel()
        super.onDestroy()
    }

    @SuppressLint("RestrictedApi", "VisibleForTests")
    private fun showLeaveOnboardingDialog() {
        val dialog = Dialog(this)
        dialog.requestWindowFeature(Window.FEATURE_NO_TITLE)
        val binding: DialogBackBinding =
            DialogBackBinding.inflate(dialog.layoutInflater)
        dialog.setContentView(binding.root)

        var layoutParams = binding.dialogHeader.layoutParams as LinearLayout.LayoutParams
        var padding: Int = textSizeConverter.getPaddingOrMarginValue(16)
        layoutParams.setMargins(textSizeConverter.getPaddingOrMarginValue(16), textSizeConverter.getPaddingOrMarginValue(16), padding, 0)
        binding.dialogHeader.layoutParams = layoutParams

        val layoutParams2: ViewGroup.LayoutParams = binding.cardConfirmation.layoutParams
        layoutParams2.width = textSizeConverter.getWidth(280)
        binding.cardConfirmation.layoutParams = layoutParams2

        binding.cardConfirmation.radius = textSizeConverter.calculateRadius(14).toFloat()
        binding.dialogHeader.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(18).toFloat(),
        )
        binding.dialogSubtitle.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(14).toFloat(),
        )
        binding.btnNo.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSizeConverter.getTextSize(16).toFloat())
        binding.btnYes.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSizeConverter.getTextSize(16).toFloat())

        layoutParams = binding.dialogSubtitle.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(padding, padding, padding, padding)
        binding.dialogSubtitle.layoutParams = layoutParams

        padding = textSizeConverter.getPaddingOrMarginValue(12)
        layoutParams = binding.btnNo.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(0, padding, 0, padding)
        binding.btnNo.layoutParams = layoutParams

        layoutParams = binding.btnYes.layoutParams as LinearLayout.LayoutParams
        layoutParams.setMargins(
            textSizeConverter.getPaddingOrMarginValue(24),
            padding,
            textSizeConverter.getPaddingOrMarginValue(16),
            padding,
        )
        binding.btnYes.layoutParams = layoutParams

        binding.dialogHeader.typeface =
            ResourcesCompat.getFont(
                applicationContext,
                R.font.pingmedium,
            )
        binding.dialogSubtitle.typeface =
            ResourcesCompat.getFont(
                applicationContext,
                R.font.regular,
            )

        binding.dialogHeader.text = getString(R.string.liveness_back_dialog_header)
        binding.dialogSubtitle.text = getString(R.string.liveness_back_dialog_title)
        binding.btnNo.text = getString(R.string.liveness_back_dialog_cotionue)
        binding.btnYes.text = getString(R.string.liveness_back_dialog_leave)

        binding.btnYes.setOnClickListener {
            dialog.dismiss()
            cameraProvider?.unbindAll()
            cameraProvider?.shutdownAsync()
            finish()
        }
        binding.btnNo.setOnClickListener { dialog.dismiss() }

        dialog.show()
        dialog.setCancelable(false)
        dialog.window!!.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
        dialog.window!!.attributes.windowAnimations = R.style.DialogAnimation
    }

    companion object {
        const val TAG = "LiveNessActivity"
        const val DEFAULT_THRESHOLD = 0.915F
        val permissions: Array<String> = arrayOf(Manifest.permission.CAMERA)
        const val PERMISSION_REQUEST_CODE = 1
    }
}
