package nashid.verify.sdk.ui

import android.content.Intent
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Matrix
import android.os.Bundle
import android.util.TypedValue
import android.view.View
import android.view.animation.AnimationUtils
import android.widget.LinearLayout
import android.widget.Toast
import androidx.activity.addCallback
import androidx.activity.result.contract.ActivityResultContracts
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import nashid.verify.sdk.utils.ArtifactType
import nashid.verify.sdk.utils.PermissionAndLocationHelper
import nashid.verify.sdk.utils.helpers.ErrorUtility
import nashid.verify.sdk.utils.helpers.TextSizeConverter
import nashid.verify.sdk.utils.helpers.Utility
import nashid.verify.sdk.viewmodel.SkipNfcLiveNessViewModel
import nashid.verify.sdkNew.R
import nashid.verify.sdkNew.databinding.ActivitySkipNfcBinding
import org.koin.android.ext.android.inject

class SkipNfcLiveNessActivity : BaseActivity() {
    private lateinit var binding: ActivitySkipNfcBinding
    private val textSizeConverter: TextSizeConverter by inject()
    private val viewModel: SkipNfcLiveNessViewModel by inject()
    private val maxRetries: Int = 3
    private var attempt = 0

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivitySkipNfcBinding.inflate(layoutInflater)
        setContentView(binding.root)
        initView()
        backPress()
    }

    private fun handleIntentExtras() {
        if (intent.hasExtra(getString(R.string.doc_key))) {
            val selectedDoc = intent.getStringExtra(getString(R.string.doc_key))
            if (selectedDoc != null) {
                viewModel.getLiveNessData().setSelectedDoc(selectedDoc)
            }
        }
    }

    private fun observeViewModel() {
        viewModel.getLiveNessData().getFinishActivity().observe(this) { handleFinishActivity(it) }
        viewModel.getLiveNessData().getIsLiveness().observe(this) { handleLiveNess(it) }
        viewModel.getLiveNessData().getInternetConnection().observe(this) { handleInternetConnection(it) }
        viewModel.getLiveNessData().getHandleLocationData().observe(this) { handleLocationData(it) }
        viewModel.getLiveNessData().getLivenessBitmap().observe(this) {
            it?.let { it1 ->
                setLiveNessBitmap(
                    it1,
                )
            }
        }
        viewModel.getLiveNessData().getScanBitmap().observe(this) { it?.let { it1 -> setScanBitmap(it1) } }
        viewModel.getLiveNessData().getFaceMatchApiFail().observe(this) {}
        viewModel.getLiveNessData().getHideProgressBar().observe(this) {
            if (it) {
                binding.imgLoader.clearAnimation()
                binding.imgLoader.visibility = View.GONE
                binding.txtValidating.text = getString(R.string.some_thing_went_wrong)
                binding.retryButton.visibility = View.VISIBLE
            }
        }
        binding.retryButton.setOnClickListener {
            if (attempt < maxRetries) {
                attempt++
                binding.retryButton.visibility = View.GONE
                binding.txtValidating.text = getString(R.string.validating)
                binding.imgLoader.visibility = View.VISIBLE
                setLayoutAndTextSize()
                handleLocationData(true)
            } else {
                Toast.makeText(this@SkipNfcLiveNessActivity, "Please try it again later!", Toast.LENGTH_SHORT).show()
                finish()
            }
        }
    }

    private fun handleInternetConnection(isAvailable: Boolean) {
        if (!isAvailable) {
            ErrorUtility.getInstance().showNoInternetDialog(this, !this.isFinishing && !this.isDestroyed)
        }
    }

    private fun setLiveNessBitmap(bitmap: Bitmap) {
        binding.imgTest123.setImageBitmap(bitmap)
    }

    private fun setScanBitmap(bitmap: Bitmap) {
        binding.imgTest12.setImageBitmap(bitmap)
    }

    private fun handleLiveNess(isLiveNess: Boolean) {
        if (isLiveNess) {
            openLiveNessScreen()
        }
    }

    private fun openLiveNessScreen() {
        val intent = Intent(this, LiveNessActivity::class.java)
        someActivityResultLauncher.launch(intent)
    }

    private val someActivityResultLauncher =
        registerForActivityResult(
            ActivityResultContracts.StartActivityForResult(),
        ) { _ ->
            if (Utility.getInstance().liveImage != null) {
                handleLivenessImage()
            } else {
                finish()
            }
        }

    private fun handleLivenessImage() {
        CoroutineScope(Dispatchers.IO).launch {
            try {
                val livenessImage = processLivenessImage().compressImage()
                withContext(Dispatchers.Main) {
                    binding.txtValidating.text = getString(R.string.validating)
                    binding.imgLoader.visibility = View.VISIBLE
                }
                viewModel.uploadArtifact(livenessImage, ArtifactType.LIVENESS_IMAGE)
                withContext(Dispatchers.Main) {
                    viewModel.validateJustCompletion()
                    binding.imgLoader.visibility = View.GONE
                }
            } catch (e: Exception) {
                withContext(Dispatchers.Main) {
                    Toast.makeText(this@SkipNfcLiveNessActivity, "Error processing images", Toast.LENGTH_SHORT).show()
                    binding.imgLoader.visibility = View.GONE
                }
            }
        }
    }

    private suspend fun processLivenessImage(): Bitmap =
        withContext(Dispatchers.IO) {
            val liveImage =
                BitmapFactory.decodeByteArray(
                    Utility.getInstance().liveImage,
                    0,
                    Utility.getInstance().liveImage?.size ?: 0,
                )

            val matrix = Matrix()
            val rotation =
                when (liveImage.width > liveImage.height) {
                    true -> 270f
                    false -> 0f
                }
            matrix.postRotate(rotation)

            if (rotation != 0f) {
                Bitmap.createBitmap(
                    liveImage,
                    0,
                    0,
                    liveImage.width,
                    liveImage.height,
                    matrix,
                    true,
                )
            } else {
                liveImage
            }
        }

    private fun handleFinishActivity(shouldFinish: Boolean) {
        if (shouldFinish) {
            binding.imgLoader.clearAnimation()
            binding.imgLoader.visibility = View.GONE
            binding.txtValidating.visibility = View.GONE
            Utility.getInstance().restartApp(this)
        }
    }

    private fun backPress() {
        onBackPressedDispatcher.addCallback(this) {
            if (viewModel.getLiveNessData().getIsApiCalled().value == false) onBackPressedDispatcher.onBackPressed()
        }
    }

    private fun setLayoutAndTextSize() {
        textSizeConverter.changeStatusBarColor(this)
        val layoutParams2 = binding.imgLoader.layoutParams
        layoutParams2.width = textSizeConverter.getWidth(24)
        layoutParams2.height = textSizeConverter.getHeight(24)
        binding.imgLoader.layoutParams = layoutParams2

        val marginLayoutParam = binding.txtValidating.layoutParams as LinearLayout.LayoutParams
        marginLayoutParam.setMargins(0, textSizeConverter.getPaddingOrMarginValue(32), 0, 0)
        binding.txtValidating.layoutParams = marginLayoutParam
        binding.txtValidating.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(20).toFloat(),
        )

        val rotation = AnimationUtils.loadAnimation(this, R.anim.rotate)
        rotation.fillAfter = true
        binding.imgLoader.startAnimation(rotation)
    }

    private fun initView() {
        setLayoutAndTextSize()
        handleIntentExtras()
        observeViewModel()
        viewModel.handleInternetConnectionData(isInternetAvailable)
    }

    override fun onAvailable() {
        if (viewModel.getLiveNessData().getIsApiCalled().value == false) {
            viewModel.handleInternetConnectionData(
                isInternetAvailable,
            )
        }
    }

    private fun handleLocationData(callLocationData: Boolean) {
        if (callLocationData) {
            PermissionAndLocationHelper.requestPermissionAndLocation(
                this,
                object : PermissionAndLocationHelper.ResultCallback {
                    override fun onResult(
                        latitude: Double,
                        longitude: Double,
                        address: String,
                    ) {
                        viewModel.submitVerification(
                            latitude,
                            longitude,
                            false,
                        )
                    }
                },
            )
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        ErrorUtility.getInstance().unregisterConnectivityManager(this)
        viewModel.getLiveNessData().clear()
    }

    private fun Bitmap.compressImage(
        maxWidth: Int = 1024,
        maxHeight: Int = 1024,
    ): Bitmap {
        if (width <= maxWidth && height <= maxHeight) {
            return this
        }

        val ratioX = maxWidth.toFloat() / width
        val ratioY = maxHeight.toFloat() / height
        val ratio = minOf(ratioX, ratioY)

        val newWidth = (width * ratio).toInt()
        val newHeight = (height * ratio).toInt()

        return Bitmap.createScaledBitmap(this, newWidth, newHeight, true).also {
            if (this != it) {
                this.recycle()
            }
        }
    }
}
