package nashid.verify.sdk.viewmodel

import android.app.Application
import android.content.Intent
import android.nfc.Tag
import android.nfc.tech.IsoDep
import android.util.Log
import androidx.lifecycle.ViewModel
import io.reactivex.disposables.CompositeDisposable
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.cancel
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import nashid.verify.sdk.data.repository.NfcRepository
import nashid.verify.sdk.ui.SkipNfcLiveNessActivity
import nashid.verify.sdk.utils.EDocument
import nashid.verify.sdk.utils.LivenessData
import nashid.verify.sdk.utils.OtherConstant
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.helpers.Utility
import nashid.verify.sdk.utils.id_card.jmrtd.BACKeySpec
import nashid.verify.sdkNew.R
import java.lang.Boolean.FALSE
import java.util.Objects

class NFCViewModel(
    private val nfcLiveData: NfcLiveData,
    private val app: Application,
    private val nfcRepository: NfcRepository,
) : ViewModel() {
    private var coroutineScope = CoroutineScope(Dispatchers.Main + SupervisorJob())
    private val readDataCoroutineScope = CoroutineScope(Dispatchers.Main)

    fun setSelectedDoc(doc: String) {
        nfcLiveData.setSelectedDoc(doc)
    }

    fun getNfcLiveData(): NfcLiveData = nfcLiveData

    fun callNfcStatus() {
        SdkConfig.nfcMaximumRetries?.let { nfcLiveData.setMaxTry(it) }
        SdkConfig.isNfcSkipable?.let { nfcLiveData.setSkipNFC(it) }
    }

    fun setInternetConnection(isAvailable: Boolean) {
        nfcLiveData.setInternetConnection(isAvailable)
    }

    fun setIsApiCalled(isApiCalled: Boolean) {
        nfcLiveData.setIsApiCalled(isApiCalled)
    }

    fun cancelNfcRead() {
        coroutineScope.cancel()
    }

    fun readData(
        isoDep: IsoDep,
        bacKey: BACKeySpec,
    ) {
        readDataCoroutineScope.launch {
            try {
                readIdCardNfcData(isoDep, bacKey)
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
    }

    fun getEDocument(): EDocument = nfcRepository.fetchEDocument()

    fun getDisposable(): CompositeDisposable = nfcRepository.fetchDisposable()

    private suspend fun readIdCardNfcData(
        isoDep: IsoDep,
        bacKey: BACKeySpec,
    ) {
        coroutineScope = CoroutineScope(Dispatchers.Main + SupervisorJob())
        coroutineScope.launch {
            val result =
                withContext(Dispatchers.IO) {
                    nfcRepository.readOmanIdNFC(isoDep, bacKey)
                }
            result?.let {
                Log.d("NFCViewModel", "readIdCardNfcData: " + it.percentage)
                if (it.percentage == 100) {
                    nfcLiveData.setShowResult(true)
                }
                nfcLiveData.setMrtPercentage(it.percentage)
                it.error?.let { error -> nfcLiveData.setShowError(error) }
            }
        }
    }

    fun validateLivensUser() {
        resetData()
        if (Utility.getInstance().hasNfcScanFailed()) {
            val intent =
                Intent(
                    app,
                    SkipNfcLiveNessActivity::class.java,
                )
            intent.putExtra(
                app.getString(R.string.doc_key),
                nfcLiveData.selectedDoc.value,
            )
            intent.setFlags(
                Intent.FLAG_ACTIVITY_NEW_TASK,
            )
            app.startActivity(intent)
        } else if (OtherConstant.getInstance().skipNFC) {
            val intent =
                Intent(
                    app,
                    SkipNfcLiveNessActivity::class.java,
                )
            intent.putExtra(
                app.getString(R.string.doc_key),
                nfcLiveData.selectedDoc.value,
            )
            intent.setFlags(
                Intent.FLAG_ACTIVITY_NEW_TASK,
            )
            app.startActivity(intent)
        } else {
            if (Objects
                    .requireNonNull<String>(nfcLiveData.selectedDoc.value)
                    .equals(
                        app.getString(R.string.e_passport),
                        ignoreCase = true,
                    )
            ) {
                LivenessData
                    .getInstance()
                    .passportNumber =
                    nfcLiveData.passportNumber.value
                val intent =
                    Intent(
                        app,
                        SkipNfcLiveNessActivity::class.java,
                    )
                intent.putExtra(
                    app.getString(R.string.doc_key),
                    nfcLiveData.selectedDoc.value,
                )
                intent.setFlags(
                    Intent.FLAG_ACTIVITY_NEW_TASK,
                )
                app.startActivity(intent)
            } else {
                LivenessData
                    .getInstance()
                    .passportNumber =
                    nfcLiveData.passportNumber.value

                val intent =
                    Intent(
                        app,
                        SkipNfcLiveNessActivity::class.java,
                    )
                intent.putExtra(
                    app.getString(R.string.doc_key),
                    nfcLiveData.selectedDoc.value,
                )
                intent.setFlags(
                    Intent.FLAG_ACTIVITY_NEW_TASK,
                )
                app.startActivity(intent)
            }
        }
    }

    fun readPassportNfc(tagFromIntent: Tag?) {
        nfcRepository.readPassportNfc(tagFromIntent)
    }

    fun resetData() {
        nfcLiveData.setShowResult(FALSE)
        nfcLiveData.setDateOfBirth("")
        nfcLiveData.setExpiryDate("")
        nfcLiveData.setDocumentNumber("")
        nfcLiveData.setShowError(null)
    }
}
