package nashid.verify.sdk.ui

import android.app.Activity
import androidx.compose.foundation.Image
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.Card
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.colorResource
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.Font
import androidx.compose.ui.text.font.FontFamily
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.navigation.NavController
import androidx.navigation.compose.rememberNavController
import nashid.verify.sdk.utils.Screens
import nashid.verify.sdkNew.R

@Composable
fun InstructionPreview(
    steps: List<Step>,
    navController: NavController,
    onBackClick: () -> Unit,
) {
    Box(
        modifier =
            Modifier
                .fillMaxSize()
                .background(Color.White),
    ) {
        Column(
            modifier =
                Modifier
                    .fillMaxSize(),
        ) {
            Row(
                modifier =
                    Modifier
                        .fillMaxWidth()
                        .padding(top = 10.dp, end = 25.dp, start = 5.dp),
                verticalAlignment = Alignment.CenterVertically,
            ) {
                IconButton(onClick = onBackClick) {
                    Icon(
                        painter = painterResource(R.drawable.back),
                        contentDescription = "Back Arrow",
                        tint = colorResource(id = R.color.header_color),
                    )
                }
                Text(
                    text = stringResource(R.string.verification_steps),
                    fontFamily = FontFamily(Font(R.font.pingmedium)),
                    color = colorResource(id = R.color.header_color),
                    fontSize = 18.sp,
                    textAlign = TextAlign.Center,
                    modifier = Modifier.weight(4f),
                )
            }

            Spacer(modifier = Modifier.height(8.dp))
            LazyColumn(
                modifier =
                    Modifier
                        .weight(1f)
                        .padding(20.dp),
            ) {
                items(steps) { step ->
                    StepListItem(step)
                }
            }
            NextButton {
                navController.navigate(Screens.StartScanning.route) {
                    popUpTo(Screens.InstructionScreen.route) { inclusive = true }
                }
            }
        }
    }
}

@Composable
fun StepListItem(step: Step) {
    Row(
        modifier =
            Modifier
                .fillMaxWidth()
                .padding(vertical = 8.dp),
        verticalAlignment = Alignment.CenterVertically,
    ) {
        Card(
            modifier =
                Modifier
                    .size(85.dp)
                    .padding(top = 8.dp),
            colors = CardDefaults.cardColors(colorResource(id = R.color.step_bg_color)),
            shape = RoundedCornerShape(6.dp),
        ) {
            Image(
                painter = painterResource(step.imageRes),
                contentDescription = "Step icon",
                modifier = Modifier.fillMaxSize(),
                contentScale = ContentScale.Inside,
            )
        }

        Spacer(modifier = Modifier.width(8.dp))

        Column(
            verticalArrangement = Arrangement.Center,
            modifier =
                Modifier
                    .fillMaxWidth()
                    .padding(start = 8.dp),
        ) {
            Text(
                text = step.title,
                fontFamily = FontFamily(Font(R.font.pingmedium)),
                color = colorResource(id = R.color.step_header),
            )
            Text(
                text = step.description,
                fontFamily = FontFamily(Font(R.font.pingmedium)),
                color = colorResource(id = R.color.header_color),
            )
        }
    }
}

@Composable
fun NextButton(onNextClick: () -> Unit) {
    Card(
        shape = RoundedCornerShape(6.dp),
        onClick = onNextClick,
        colors = CardDefaults.cardColors(colorResource(id = R.color.button_color)),
        modifier =
            Modifier
                .fillMaxWidth()
                .padding(16.dp),
    ) {
        Text(
            text = stringResource(id = R.string.next),
            fontFamily = FontFamily(Font(R.font.pingmedium)),
            color = colorResource(id = R.color.white),
            modifier =
                Modifier
                    .padding(vertical = 16.dp)
                    .fillMaxWidth(),
            fontSize = 16.sp,
            textAlign = TextAlign.Center,
        )
    }
}

data class Step(val title: String, val description: String, val imageRes: Int)

@Composable
fun InstructionScreen(navController: NavController) {
    val context = LocalContext.current
    val steps =
        listOf(
            Step(stringResource(id = R.string.scan), stringResource(id = R.string.scan_desc), R.drawable.step1),
            Step(stringResource(id = R.string.tap), stringResource(id = R.string.tap_desc), R.drawable.step2),
            Step(stringResource(id = R.string.smile), stringResource(id = R.string.smile_desc), R.drawable.step3),
        )
    InstructionPreview(steps, navController) {
        (context as? Activity)?.finish()
    }
}

@Preview(showBackground = true)
@Composable
fun InstructionScreen() {
    InstructionPreview(
        steps =
            listOf(
                Step(stringResource(id = R.string.scan), stringResource(id = R.string.scan_desc), R.drawable.step1),
                Step(stringResource(id = R.string.tap), stringResource(id = R.string.tap_desc), R.drawable.step2),
                Step(stringResource(id = R.string.smile), stringResource(id = R.string.smile_desc), R.drawable.step3),
            ),
        navController = rememberNavController(),
        onBackClick = {},
    )
}
