package nashid.verify.sdk.utils

import android.os.Parcel
import android.os.Parcelable
import net.sf.scuba.util.Hex
import org.jmrtd.BACKey
import org.jmrtd.protocol.EACCAResult
import org.jmrtd.protocol.EACTAResult
import java.io.IOException
import java.io.ObjectInputStream
import java.io.ObjectOutputStream
import java.io.Serializable
import java.security.cert.Certificate
import java.util.ArrayList
import java.util.Arrays
import java.util.TreeMap

/**
 * A data type for communicating document verification check information.
 *
 * @author The JMRTD team (info@jmrtd.org)
 *
 * @version $Revision: 1559 $
 */
class VerificationStatus : Parcelable {
    var aa: Verdict? = null
        private set

    var bac: Verdict? = null
        private set

    var sac: Verdict? = null
        private set

    var cs: Verdict? = null
        private set

    var ht: Verdict? = null
        private set

    var ds: Verdict? = null
        private set

    var eac: Verdict? = null
        private set

    var ca: Verdict? = null
        private set

    var aaReason: String? = null
        private set

    var bacReason: String? = null
        private set

    var sacReason: String? = null
        private set

    var csReason: String? = null
        private set

    var htReason: String? = null
        private set

    var dsReason: String? = null
        private set

    var eacReason: String? = null
        private set

    var caReason: String? = null
        private set

    // By products of the verification process that may be useful for relying parties to display.
    private var triedBACEntries: List<BACKey>? = null // As a result of BAC testing, this contains all tried BAC entries.
    var hashResults: MutableMap<Int, HashMatchResult>? = null // As a result of HT testing, this contains stored and computed hashes.
    private var certificateChain: List<Certificate>? = null // As a result of CS testing, this contains certificate chain from DSC to CSCA.

    var eacResult: EACTAResult? = null
        private set

    var caResult: EACCAResult? = null
        private set

    enum class Verdict {
        UNKNOWN, // Unknown
        NOT_PRESENT, // Not present
        NOT_CHECKED, // Present, not checked
        FAILED, // Present, checked, and not ok
        SUCCEEDED,
        // Present, checked, and ok
    }

    constructor() {
        setAll(Verdict.UNKNOWN, null)
    }

    fun setAA(
        v: Verdict,
        reason: String?,
    ) {
        this.aa = v
        this.aaReason = reason
    }

    fun getTriedBACEntries(): List<*>? {
        return triedBACEntries
    }

    fun setBAC(
        v: Verdict,
        reason: String?,
        triedBACEntries: List<BACKey>?,
    ) {
        this.bac = v
        this.bacReason = reason
        this.triedBACEntries = triedBACEntries
    }

    fun setSAC(
        v: Verdict,
        reason: String,
    ) {
        this.sac = v
        this.sacReason = reason
    }

    fun getCertificateChain(): List<*>? {
        return certificateChain
    }

    fun setCS(
        v: Verdict,
        reason: String?,
        certificateChain: List<Certificate>?,
    ) {
        this.cs = v
        this.csReason = reason
        this.certificateChain = certificateChain
    }

    fun setDS(
        v: Verdict,
        reason: String?,
    ) {
        this.ds = v
        this.dsReason = reason
    }

    fun setHT(
        v: Verdict,
        reason: String?,
        hashResults: MutableMap<Int, HashMatchResult>?,
    ) {
        this.ht = v
        this.htReason = reason
        this.hashResults = hashResults
    }

    fun setEAC(
        v: Verdict,
        reason: String?,
        eacResult: EACTAResult?,
    ) {
        this.eac = v
        this.eacReason = reason
        this.eacResult = eacResult
    }

    fun setCA(
        v: Verdict,
        reason: String,
        eaccaResult: EACCAResult?,
    ) {
        this.ca = v
        this.caReason = reason
        this.caResult = eaccaResult
    }

    fun setAll(
        verdict: Verdict,
        reason: String?,
    ) {
        setAA(verdict, reason)
        setBAC(verdict, reason, null)
        setCS(verdict, reason, null)
        setDS(verdict, reason)
        setHT(verdict, reason, null)
        setEAC(verdict, reason, null)
    }

    constructor(`in`: Parcel) {
        this.aa = if (`in`.readInt() == 1) Verdict.valueOf(`in`.readString()!!) else null
        this.bac = if (`in`.readInt() == 1) Verdict.valueOf(`in`.readString()!!) else null
        this.sac = if (`in`.readInt() == 1) Verdict.valueOf(`in`.readString()!!) else null
        this.cs = if (`in`.readInt() == 1) Verdict.valueOf(`in`.readString()!!) else null
        this.ht = if (`in`.readInt() == 1) Verdict.valueOf(`in`.readString()!!) else null
        this.ds = if (`in`.readInt() == 1) Verdict.valueOf(`in`.readString()!!) else null
        this.eac = if (`in`.readInt() == 1) Verdict.valueOf(`in`.readString()!!) else null
        this.ca = if (`in`.readInt() == 1) Verdict.valueOf(`in`.readString()!!) else null

        this.aaReason = if (`in`.readInt() == 1) `in`.readString() else null
        this.bacReason = if (`in`.readInt() == 1) `in`.readString() else null
        this.sacReason = if (`in`.readInt() == 1) `in`.readString() else null
        this.csReason = if (`in`.readInt() == 1) `in`.readString() else null
        this.htReason = if (`in`.readInt() == 1) `in`.readString() else null
        this.dsReason = if (`in`.readInt() == 1) `in`.readString() else null
        this.eacReason = if (`in`.readInt() == 1) `in`.readString() else null
        this.caReason = if (`in`.readInt() == 1) `in`.readString() else null

        if (`in`.readInt() == 1) {
            triedBACEntries = ArrayList()
            `in`.readList(triedBACEntries!!, BACKey::class.java.classLoader)
        }

        if (`in`.readInt() == 1) {
            hashResults = TreeMap()
            val size = `in`.readInt()
            for (i in 0 until size) {
                val key = `in`.readInt()
                val value = `in`.readSerializable() as HashMatchResult
                hashResults!![key] = value
            }
        }

        if (`in`.readInt() == 1) {
            certificateChain = ArrayList()
            `in`.readList(certificateChain!!, Certificate::class.java.classLoader)
        }

        if (`in`.readInt() == 1) {
            eacResult = `in`.readSerializable() as EACTAResult
        }

        if (`in`.readInt() == 1) {
            caResult = `in`.readSerializable() as EACCAResult
        }
    }

    override fun describeContents(): Int {
        return 0
    }

    override fun writeToParcel(
        dest: Parcel,
        flags: Int,
    ) {
        dest.writeInt(if (this.aa != null) 1 else 0)
        if (aa != null) {
            dest.writeString(aa?.name)
        }
        dest.writeInt(if (this.bac != null) 1 else 0)
        if (bac != null) {
            dest.writeString(bac?.name)
        }
        dest.writeInt(if (this.sac != null) 1 else 0)
        if (sac != null) {
            dest.writeString(sac?.name)
        }
        dest.writeInt(if (this.cs != null) 1 else 0)
        if (cs != null) {
            dest.writeString(cs?.name)
        }
        dest.writeInt(if (this.ht != null) 1 else 0)
        if (ht != null) {
            dest.writeString(ht?.name)
        }
        dest.writeInt(if (this.ds != null) 1 else 0)
        if (ds != null) {
            dest.writeString(ds?.name)
        }
        dest.writeInt(if (this.eac != null) 1 else 0)
        if (eac != null) {
            dest.writeString(eac?.name)
        }
        dest.writeInt(if (this.ca != null) 1 else 0)
        if (ca != null) {
            dest.writeString(ca?.name)
        }

        dest.writeInt(if (aaReason != null) 1 else 0)
        if (aaReason != null) {
            dest.writeString(aaReason)
        }

        dest.writeInt(if (bacReason != null) 1 else 0)
        if (bacReason != null) {
            dest.writeString(bacReason)
        }

        dest.writeInt(if (sacReason != null) 1 else 0)
        if (sacReason != null) {
            dest.writeString(sacReason)
        }

        dest.writeInt(if (csReason != null) 1 else 0)
        if (csReason != null) {
            dest.writeString(csReason)
        }

        dest.writeInt(if (htReason != null) 1 else 0)
        if (htReason != null) {
            dest.writeString(htReason)
        }

        dest.writeInt(if (dsReason != null) 1 else 0)
        if (dsReason != null) {
            dest.writeString(dsReason)
        }

        dest.writeInt(if (eacReason != null) 1 else 0)
        if (eacReason != null) {
            dest.writeString(eacReason)
        }

        dest.writeInt(if (caReason != null) 1 else 0)
        if (caReason != null) {
            dest.writeString(caReason)
        }

        dest.writeInt(if (triedBACEntries != null) 1 else 0)
        if (triedBACEntries != null) {
            dest.writeList(triedBACEntries)
        }

        dest.writeInt(if (hashResults != null) 1 else 0)
        if (hashResults != null) {
            dest.writeInt(hashResults!!.size)
            for ((key, value) in hashResults!!) {
                dest.writeInt(key)
                dest.writeSerializable(value)
            }
        }

        dest.writeInt(if (certificateChain != null) 1 else 0)
        if (certificateChain != null) {
            dest.writeList(certificateChain)
        }

        dest.writeInt(if (eacResult != null) 1 else 0)
        if (eacResult != null) {
            dest.writeSerializable(eacResult)
        }

        dest.writeInt(if (caResult != null) 1 else 0)
        if (caResult != null) {
            dest.writeSerializable(caResult)
        }
    }

    override fun toString(): String {
        return "VerificationStatus(aa=$aa, bac=$bac, sac=$sac, cs=$cs, ht=$ht, ds=$ds, eac=$eac, ca=$ca, aaReason=$aaReason, bacReason=$bacReason, sacReason=$sacReason, csReason=$csReason, htReason=$htReason, dsReason=$dsReason, eacReason=$eacReason, caReason=$caReason, triedBACEntries=$triedBACEntries, hashResults=$hashResults, certificateChain=$certificateChain, eacResult=$eacResult, caResult=$caResult)"
    }

    class HashMatchResult
    (storedHash: ByteArray, computedHash: ByteArray?) : Serializable {
        var storedHash: ByteArray? = null
            private set

        var computedHash: ByteArray? = null
            private set

        val isMatch: Boolean
            get() = Arrays.equals(storedHash, computedHash)

        init {
            this.storedHash = storedHash
            this.computedHash = computedHash
        }

        override fun toString(): String {
            return "HashResult [" + isMatch + ", stored: " + Hex.bytesToHexString(storedHash) + ", computed: " + Hex.bytesToHexString(computedHash)
        }

        override fun hashCode(): Int {
            return 11 + 3 * Arrays.hashCode(storedHash) + 5 * Arrays.hashCode(computedHash)
        }

        override fun equals(other: Any?): Boolean {
            if (other == null) {
                return false
            }
            if (other === this) {
                return true
            }
            if (other.javaClass != this.javaClass) {
                return false
            }
            val otherHashResult = other as HashMatchResult?
            return Arrays.equals(otherHashResult!!.computedHash, computedHash) && Arrays.equals(otherHashResult.storedHash, storedHash)
        }

        // NOTE: Part of our serializable implementation.
        @Throws(IOException::class, ClassNotFoundException::class)
        private fun readObject(inputStream: ObjectInputStream) {
            // 			inputStream.defaultReadObject();
            storedHash = readBytes(inputStream)
            computedHash = readBytes(inputStream)
        }

        // NOTE: Part of our serializable implementation.
        @Throws(IOException::class)
        private fun writeObject(outputStream: ObjectOutputStream) {
            // 			outputStream.defaultWriteObject();
            writeByteArray(storedHash, outputStream)
            writeByteArray(computedHash, outputStream)
        }

        @Throws(IOException::class)
        private fun readBytes(inputStream: ObjectInputStream): ByteArray? {
            val length = inputStream.readInt()
            if (length < 0) {
                return null
            }
            val bytes = ByteArray(length)
            for (i in 0 until length) {
                val b = inputStream.readInt()
                bytes[i] = b.toByte()
            }
            return bytes
        }

        @Throws(IOException::class)
        private fun writeByteArray(
            bytes: ByteArray?,
            outputStream: ObjectOutputStream,
        ) {
            if (bytes == null) {
                outputStream.writeInt(-1)
            } else {
                outputStream.writeInt(bytes.size)
                for (b in bytes) {
                    outputStream.writeInt(b.toInt())
                }
            }
        }

        companion object {
            private const val serialVersionUID = 263961258911936111L
        }
    }

    companion object {
        @JvmField
        val CREATOR: Parcelable.Creator<*> =
            object : Parcelable.Creator<VerificationStatus> {
                override fun createFromParcel(pc: Parcel): VerificationStatus {
                    return VerificationStatus(pc)
                }

                override fun newArray(size: Int): Array<VerificationStatus?> {
                    return arrayOfNulls(size)
                }
            }
    }
}
