package nashid.verify.sdk.utils.helpers

import android.Manifest
import android.animation.Animator
import android.animation.AnimatorListenerAdapter
import android.annotation.SuppressLint
import android.content.pm.PackageManager
import android.graphics.Bitmap
import android.graphics.Color
import android.graphics.PixelFormat
import android.graphics.drawable.Drawable
import android.os.Build
import android.os.CountDownTimer
import android.util.Size
import android.view.Surface
import android.view.SurfaceHolder
import android.view.View.GONE
import android.view.View.VISIBLE
import android.widget.CompoundButton
import androidx.camera.core.CameraInfoUnavailableException
import androidx.camera.core.CameraSelector
import androidx.camera.core.ImageAnalysis
import androidx.camera.core.ImageCapture
import androidx.camera.core.ImageCapture.OnImageCapturedCallback
import androidx.camera.core.ImageProxy
import androidx.camera.core.Preview
import androidx.camera.lifecycle.ProcessCameraProvider
import androidx.core.content.ContextCompat
import androidx.vectordrawable.graphics.drawable.Animatable2Compat
import com.airbnb.lottie.LottieDrawable
import com.bumptech.glide.Glide
import com.bumptech.glide.load.DataSource
import com.bumptech.glide.load.engine.GlideException
import com.bumptech.glide.load.resource.gif.GifDrawable
import com.bumptech.glide.request.RequestListener
import com.bumptech.glide.request.target.Target
import com.google.mlkit.vision.common.InputImage
import com.google.mlkit.vision.common.internal.ImageConvertUtils
import nashid.verify.sdk.ui.CameraXLiveActivity
import nashid.verify.sdk.utils.Constants
import nashid.verify.sdk.utils.GlareDetector
import nashid.verify.sdk.utils.Loggers
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.viewmodel.CameraXLiveViewModel
import nashid.verify.sdkNew.BuildConfig
import nashid.verify.sdkNew.R
import nashid.verify.sdkNew.databinding.ActivityCameraXliveAcitivityBinding
import java.io.ByteArrayOutputStream
import java.util.Objects
import java.util.concurrent.Executor
import java.util.concurrent.Executors
import kotlin.math.min

class CameraOverlayManager(
    private val activity: CameraXLiveActivity,
    private val binding: ActivityCameraXliveAcitivityBinding,
    private val textSizeConverter: TextSizeConverter,
    private val storage: Storage,
    private val viewModel: CameraXLiveViewModel,
    private val isInternetAvailable: Boolean,
    private val logger: Loggers.LoggerInstance,
) : SurfaceHolder.Callback, CompoundButton.OnCheckedChangeListener {
    private lateinit var holder: SurfaceHolder
    private var cameraProvider: ProcessCameraProvider? = null
    private var analysisUseCase: ImageAnalysis? = null
    private val executor: Executor = Executors.newSingleThreadExecutor()
    private var imageCapture: ImageCapture? = null
    private var bitmapImage: Bitmap? = null
    private var originalBitmapImage: Bitmap? = null
    private var croppedBitmap: Bitmap? = null
    private var mPImage: ImageProxy? = null
    private var cTimer: CountDownTimer? = null
    private var previewUseCase: Preview? = null
    private val delayMillisecond: Long = 15000
    private val screenDivideValue = 1.8
    private var lensFacing = CameraSelector.LENS_FACING_BACK
    private var cameraSelector: CameraSelector? = null

    private var boxWidth = 0
    private var boxHeight = 0
    private var left = 0
    private var right = 0
    private var top = 0
    private var bottom = 0
    private var diameter = 0

    fun setupOverlayAndAnimation() {
        setupPaddingAndOverlay()
        setupHeaderAndAnimation()
        requestPermissions()
    }

    private fun setupPaddingAndOverlay() {
        binding.txtBottom.setPadding(textSizeConverter.getPaddingOrMarginValue(6), 0, 0, 0)
        binding.overlay.setZOrderMediaOverlay(true)
        holder = binding.overlay.holder
        holder.setFormat(PixelFormat.TRANSPARENT)
        holder.addCallback(this)
        cameraSelector = CameraSelector.Builder().requireLensFacing(lensFacing).build()
    }

    private fun setupHeaderAndAnimation() {
        binding.layoutHeader.imgBack.setImageResource(R.drawable.back_white)
        binding.layoutHeader.lytHeaderMain.setBackgroundColor(Color.TRANSPARENT)

        // Rotate the back button if the locale is Arabic
        if (storage.getPreferredLocale().equals("ar", ignoreCase = true)) {
            binding.layoutHeader.imgBack.rotation = 180f
        }

        // Set up the Lottie animation
        binding.lottieAnimationView.setAnimation(R.raw.scan)
        binding.lottieAnimationView.playAnimation()
        binding.lottieAnimationView.repeatCount = LottieDrawable.INFINITE
    }

    private fun requestPermissions() {
        PermissionHelper.requestsPermissions(
            activity,
            object : PermissionHelper.PermissionCallback {
                override fun onResult(
                    allGranted: Boolean,
                    grantedList: List<String>,
                    deniedList: List<String>,
                ) {
                    if (allGranted) {
                        startCameraInit()
                    }
                }
            },
        )
    }

    // Implement required SurfaceHolder.Callback methods
    override fun surfaceCreated(holder: SurfaceHolder) {
        FocusRectDrawer(activity, holder, binding, screenDivideValue).drawFocusRect((ContextCompat.getColor(activity, R.color.overlay_color)))
    }

    override fun surfaceChanged(
        holder: SurfaceHolder,
        format: Int,
        width: Int,
        height: Int,
    ) {
    }

    override fun surfaceDestroyed(holder: SurfaceHolder) {
    }

    private fun startCameraInit() {
        logger.log("start camera..")
        initCameraViewModelProvider()
        initClick()
        startTimer()
    }

    private fun initCameraViewModelProvider() {
        viewModel.processCameraProvider.observe(activity) { provider: ProcessCameraProvider? ->
            cameraProvider = provider
            bindAllCameraUseCases()
            bindAnalysisUseCase()
        }
    }

    private fun initClick() {
        binding.layoutHeader.imgBack.setOnClickListener { activity.onBackPressedDispatcher.onBackPressed() }
        binding.btnRescanningDocuments.setOnClickListener {
            viewModel.cameraXLiveData.setIsOverlayVisible(true)
            viewModel.cameraXLiveData.setIsScanVisible(true)
            viewModel.cameraXLiveData.setIsScanFailVisible(false)
            viewModel.cameraXLiveData.setIsTransparentVisible(false)
            viewModel.cameraXLiveData.setIsScanCompleteVisible(false)
            viewModel.cameraXLiveData.setIsBackCardScanVisible(false)
            cTimer!!.start()
            if (mPImage != null) mPImage!!.close()
        }
    }

    private fun bindAllCameraUseCases() {
        if (cameraProvider != null) {
            cameraProvider!!.unbindAll()
            bindPreviewUseCase()
        }
    }

    @SuppressLint("UnsafeOptInUsageError")
    fun bindAnalysisUseCase() {
        if (ContextCompat.checkSelfPermission(
                activity,
                Manifest.permission.CAMERA,
            ) == PackageManager.PERMISSION_GRANTED
        ) {
            setUpAnalysisUseCase()
        }
    }

    @SuppressLint("RestrictedApi")
    private fun setUpAnalysisUseCase() {
        activity.runOnUiThread { viewModel.cameraXLiveData.setIsScanVisible(true) }
        if (cameraProvider == null) {
            return
        }
        if (analysisUseCase != null) {
            cameraProvider!!.unbind(analysisUseCase)
        }
        analysisUseCase =
            ImageAnalysis
                .Builder()
                .setDefaultResolution(Size(1280, 720))
                .setBackpressureStrategy(ImageAnalysis.STRATEGY_KEEP_ONLY_LATEST)
                .build()
        analysisUseCase!!.setAnalyzer(executor) { pImage: ImageProxy ->
            if (isInternetAvailable) {
                analyzeImage(pImage)
            } else {
                pImage.close()
            }
        }
        cameraProvider!!.bindToLifecycle(activity, cameraSelector!!, imageCapture, analysisUseCase)
    }

    private fun analyzeImage(pImage: ImageProxy) {
        @SuppressLint("UnsafeOptInUsageError")
        val mediaImage = pImage.image
        if (mediaImage == null) {
            pImage.close()
            return
        }
        logger.log("analyze:image ")
        val image = InputImage.fromMediaImage(mediaImage, pImage.imageInfo.rotationDegrees)
        try {
            bitmapImage = ImageConvertUtils.getInstance().convertToUpRightBitmap(image)
            originalBitmapImage = bitmapImage
        } catch (e: Exception) {
            e.printStackTrace()
        }
        cropBitmapByPercentage()
        if (java.lang.Boolean.TRUE == viewModel.cameraXLiveData.getFlag().value) {
            passImageToRead(pImage)
        } else {
            assert(analysisUseCase != null)
            analysisUseCase!!.clearAnalyzer()
        }
    }

    private fun cropBitmapByPercentage() {
        val height = bitmapImage!!.height
        val width = bitmapImage!!.width
        diameter = min(height, width)
        val offset = (Constants.VISION_API_OFFSET * diameter).toInt()
        logger.log("analyzeImage: $width   $diameter")
        diameter -= offset
        val value = 4
        logger.log("analyzeImage:after removing offset $width   $diameter")
        left = width - diameter
        top = (height / screenDivideValue - diameter / value).toInt()
        right = diameter
        bottom = (height / screenDivideValue + diameter / value).toInt()
        boxHeight = bottom - top
        boxWidth = right - left
        val cutPercentage = 16
        val cutAmount = (width * (cutPercentage / 100.0)).toInt()
        left += cutAmount
        right -= cutAmount
        val newBoxWidth = right - left
        if (java.lang.Boolean.FALSE == viewModel.cameraXLiveData.getIsFrontView().value) {
            croppedBitmap = Bitmap.createBitmap(bitmapImage!!, left, top, newBoxWidth, boxHeight)
        } else {
            val cutPercentageTopBottom = 5
            val cutAmountTopBottom = (height * (cutPercentageTopBottom / 100.0)).toInt()
            top += cutAmountTopBottom
            bottom -= cutAmountTopBottom
            val newBoxHeight = bottom - top
            croppedBitmap = Bitmap.createBitmap(bitmapImage!!, left, top, newBoxWidth, newBoxHeight)
        }
        val stream = ByteArrayOutputStream()
        croppedBitmap!!.compress(Bitmap.CompressFormat.PNG, 100, stream)
    }

    @SuppressLint("SetTextI18n")
    private fun passImageToRead(pImage: ImageProxy) {
        mPImage = pImage
        if (BuildConfig.BUILD_TYPE.equals("debug", ignoreCase = true)) {
            activity.runOnUiThread {
                logger.log("run: " + croppedBitmap!!.height + " " + croppedBitmap!!.width)
                binding.testimg1.visibility = VISIBLE
                binding.testimg1.setImageBitmap(croppedBitmap)
            }
        }
        if (binding.lyoutScanFail.visibility == GONE) {
            val imageLight = activity.getString(R.string.image_light_normal)
            if (imageLight.equals(activity.getString(R.string.image_light_normal), ignoreCase = true)) {
                val hasGlare = GlareDetector.isGlareDetected(croppedBitmap!!)
                if (!hasGlare) {
                    activity.runOnUiThread { viewModel.cameraXLiveData.setGlareTextVisibility(false) }
                    filterScannedText()
                } else {
                    activity.runOnUiThread {
                        viewModel.cameraXLiveData.setGlareTextVisibility(true)
                        viewModel.cameraXLiveData.setGlareText(activity.getString(R.string.glare))
                        activity.runOnUiThread { pImage.close() }
                    }
                }
            } else {
                activity.runOnUiThread {
                    viewModel.cameraXLiveData.setGlareTextVisibility(true)
                    viewModel.cameraXLiveData.setGlareText(activity.getString(R.string.image_light) + imageLight)
                    activity.runOnUiThread { pImage.close() }
                }
            }
        }
    }

    private fun filterScannedText() {
        try {
//            this only happen id selected document is idCard
//            The following lines are used to divide the photo into 4 parts. The top left side of the photo is used to read the text from "ROYAL OMAN POLICE D.G.OF CIVIL STATUS." After successfully reading the data, the bitmap is divided horizontally, and the MRZ code is read from the bottom part of the image. This is done to increase the speed and accuracy of reading the data.
            if (!Objects
                    .requireNonNull(viewModel.cameraXLiveData.getSelectedDoc().value)
                    .equals(activity.getString(R.string.e_passport), ignoreCase = true)
            ) {
                if (java.lang.Boolean.FALSE == viewModel.cameraXLiveData.getAlgoHeaderDetect().value && java.lang.Boolean.TRUE == viewModel.cameraXLiveData.getIsFrontView().value) {
                    croppedBitmap =
                        Bitmap.createBitmap(
                            croppedBitmap!!,
                            0,
                            0,
                            croppedBitmap!!.width / 2,
                            croppedBitmap!!.height / 2,
                        )
                } else {
                    if (java.lang.Boolean.TRUE == viewModel.cameraXLiveData.getIsFrontView().value) {
                        croppedBitmap =
                            Bitmap.createBitmap(
                                croppedBitmap!!,
                                0,
                                croppedBitmap!!.height / 2,
                                croppedBitmap!!.width,
                                croppedBitmap!!.height / 2,
                            )
                    }
                }
            }

            if (BuildConfig.BUILD_TYPE.equals("debug", ignoreCase = true)) {
                activity.runOnUiThread {
                    binding.testimg2.visibility = VISIBLE
                    binding.testimg2.setImageBitmap(croppedBitmap)
                }
            }
            viewModel.processTextFromBitmap(croppedBitmap)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    private fun startTimer() {
        cTimer =
            object : CountDownTimer(delayMillisecond, 1000) {
                override fun onTick(millisUntilFinished: Long) {}

                override fun onFinish() {
                    if (isInternetAvailable) {
                        if (java.lang.Boolean.TRUE == viewModel.cameraXLiveData.getIsFrontView().value) {
                            setRescanningDocumentVisibility()
                        } else if (Utility.getInstance().passportNumber == null && Utility.getInstance().dateOfBirth == null && Utility.getInstance().expiryDate == null) {
                            setRescanningDocumentVisibility()
                        } else {
                            cTimer!!.start()
                        }
                    } else {
                        cTimer!!.start()
                    }
                }
            }.start()
    }

    @SuppressLint("UnsafeOptInUsageError")
    private fun bindPreviewUseCase() {
        if (ContextCompat.checkSelfPermission(
                activity,
                Manifest.permission.CAMERA,
            ) == PackageManager.PERMISSION_GRANTED
        ) {
            if (cameraProvider == null) {
                return
            }
            if (previewUseCase != null) {
                cameraProvider!!.unbind(previewUseCase)
            }
            val builder = Preview.Builder()
            previewUseCase = builder.build()
            previewUseCase!!.setSurfaceProvider(binding.previewView.surfaceProvider)
            val rotation =
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
                    activity.display?.rotation ?: Surface.ROTATION_0
                } else {
                    @Suppress("DEPRECATION")
                    activity.windowManager.defaultDisplay.rotation
                }
            imageCapture =
                ImageCapture
                    .Builder()
                    .setTargetRotation(rotation)
                    .build()
            cameraProvider!!.bindToLifecycle(activity, cameraSelector!!, imageCapture, previewUseCase)
        }
    }

    private fun setRescanningDocumentVisibility() {
        viewModel.cameraXLiveData.setIsOverlayVisible(false)
        viewModel.cameraXLiveData.setIsScanVisible(false)
        viewModel.cameraXLiveData.setIsScanFailVisible(true)
        viewModel.cameraXLiveData.setIsTransparentVisible(true)
        viewModel.cameraXLiveData.setIsScanCompleteVisible(false)
    }

    fun handleCloseImageProxy(needToClose: Boolean) {
        if (needToClose) {
            if (mPImage != null) {
                mPImage?.close()
            }
        }
    }

    fun handleCTimer(needToCloseTimer: Boolean) {
        if (cTimer != null) {
            if (needToCloseTimer) {
                cTimer?.cancel()
            } else {
                cTimer?.start()
            }
        }
    }

    fun handleCaptureAnImage(needToCallCapture: Boolean) {
        if (needToCallCapture) {
            captureAnImage()
        }
    }

    fun handleAnalysisUseCase(needToClearAnalyzer: Boolean) {
        if (needToClearAnalyzer) {
            analysisUseCase?.clearAnalyzer()
        }
    }

    fun unBindCameraProvider(isUnbind: Boolean) {
        if (isUnbind) {
            assert(cameraProvider != null)
            cameraProvider?.unbindAll()
        }
    }

    private fun captureAnImage() {
        logger.log("capturing photo")
        imageCapture?.takePicture(
            executor,
            object : OnImageCapturedCallback() {
                override fun onCaptureSuccess(image: ImageProxy) {
                    binding.imgScanComplete.setMaxFrame(80)
                    super.onCaptureSuccess(image)
                    if (Objects
                            .requireNonNull(viewModel.cameraXLiveData.getSelectedDoc().value)
                            .equals(activity.getString(R.string.e_passport), ignoreCase = true)
                    ) {
                        capturePassportImage()
                    } else {
                        val stream = reCropDocBitmap()
                        if (java.lang.Boolean.TRUE == viewModel.cameraXLiveData.getIsFrontView().value && java.lang.Boolean.FALSE == viewModel.cameraXLiveData.getIsFrontViewScanned().value) {
                            captureFrontViewIdCard(stream)
                        } else {
                            captureBackViewIdCard(stream)
                        }
                    }
                }
            },
        )
    }

    private fun capturePassportImage() {
        val stream = ByteArrayOutputStream()
        croppedBitmap?.compress(Bitmap.CompressFormat.PNG, 100, stream)
        val byteArray = stream.toByteArray()
        Utility.getInstance().scannedImage = byteArray
        activity.runOnUiThread {
            viewModel.cameraXLiveData.setStatusText("")
            viewModel.cameraXLiveData.setIsOverlayVisible(false)
            viewModel.cameraXLiveData.setIsScanVisible(false)
            viewModel.cameraXLiveData.setIsTransparentVisible(true)
            viewModel.cameraXLiveData.setIsScanCompleteVisible(true)
            viewModel.cameraXLiveData.setIsLayoutBottomVisible(false)
            binding.imgScanComplete.playAnimation()
            binding.imgScanComplete.addAnimatorListener(
                object : AnimatorListenerAdapter() {
                    override fun onAnimationEnd(animation: Animator) {
                        super.onAnimationEnd(animation)
                        logger.log("passport scan completed")
                        viewModel.handleAnimationCompleted()
                    }
                },
            )
        }
    }

    private fun reCropDocBitmap(): ByteArrayOutputStream {
        val height = originalBitmapImage!!.height
        val width = originalBitmapImage!!.width
        diameter = min(height, width)
        val offset = (Constants.VISION_API_OFFSET * diameter).toInt()
        logger.log("analyzeImage: $width   $diameter")
        diameter -= offset
        val value = 4
        logger.log("analyzeImage:after removing offset $width   $diameter")
        left = width - diameter
        top = (height / screenDivideValue - diameter / value).toInt()
        right = diameter
        bottom = (height / screenDivideValue + diameter / value).toInt()
        boxHeight = bottom - top
        boxWidth = right - left
        croppedBitmap = Bitmap.createBitmap(originalBitmapImage!!, left, top, boxWidth, boxHeight)
        val stream = ByteArrayOutputStream()
        croppedBitmap!!.compress(Bitmap.CompressFormat.PNG, 100, stream)
        return stream
    }

    private fun captureFrontViewIdCard(stream: ByteArrayOutputStream) {
        val byteArray = stream.toByteArray()
        Utility.getInstance().scannedImage = byteArray
        activity.runOnUiThread {
            viewModel.cameraXLiveData.setStatusText("")
            viewModel.cameraXLiveData.setIsOverlayVisible(false)
            viewModel.cameraXLiveData.setIsScanVisible(false)
            viewModel.cameraXLiveData.setIsTransparentVisible(true)
            viewModel.cameraXLiveData.setIsScanCompleteVisible(true)
            binding.imgScanComplete.playAnimation()
            binding.imgScanComplete.addAnimatorListener(
                object : AnimatorListenerAdapter() {
                    override fun onAnimationEnd(animation: Animator) {
                        super.onAnimationEnd(animation)
                        if (viewModel.cameraXLiveData.getIdNo().value == null && viewModel.cameraXLiveData.dOB.value == null && viewModel.cameraXLiveData.getExpiryDate().value == null) {
                            viewModel.cameraXLiveData.setIsScanCompleteVisible(false)
                            viewModel.cameraXLiveData.setIsBackCardScanVisible(true)
                            viewModel.cameraXLiveData.setStatusText("")
                            viewModel.scannedIdFrontViewCompleted(byteArray)
                            logger.log("loading card flip animation")
                            val requestBuilder =
                                Glide
                                    .with(activity)
                                    .asGif()
                                    .load(R.drawable.card_flip)
                                    .listener(
                                        object : RequestListener<GifDrawable> {
                                            override fun onLoadFailed(
                                                e: GlideException?,
                                                model: Any?,
                                                target: Target<GifDrawable>,
                                                isFirstResource: Boolean,
                                            ): Boolean = false

                                            override fun onResourceReady(
                                                resource: GifDrawable,
                                                model: Any,
                                                target: Target<GifDrawable>,
                                                dataSource: DataSource,
                                                isFirstResource: Boolean,
                                            ): Boolean {
                                                binding.txtTurnDoc.visibility = VISIBLE
                                                binding.txtTurnDoc.text = activity.getString(R.string.turn_document)
                                                logger.log("resource ready to load")
                                                resource.setLoopCount(1)
                                                resource.registerAnimationCallback(
                                                    object :
                                                        Animatable2Compat.AnimationCallback() {
                                                        override fun onAnimationEnd(drawable: Drawable) {
                                                            logger.log("animation end")
                                                            binding.txtTurnDoc.visibility = GONE
                                                            viewModel.cameraXLiveData.setStatusText(activity.getString(R.string.scan_id_top_back_text))
                                                            viewModel.cameraXLiveData.setIsOverlayVisible(true)
                                                            viewModel.cameraXLiveData.setIsScanVisible(true)
                                                            viewModel.cameraXLiveData.setIsTransparentVisible(
                                                                false,
                                                            )
                                                            viewModel.cameraXLiveData.setIsScanCompleteVisible(
                                                                false,
                                                            )
                                                            viewModel.cameraXLiveData.setIsBackCardScanVisible(
                                                                false,
                                                            )
                                                            logger.log(
                                                                "onAnimationEnd: start time " + viewModel.printCurrentTimeInLogs(),
                                                            )
                                                            mPImage?.close()
                                                        }
                                                    },
                                                )
                                                return false
                                            }
                                        },
                                    )
                            requestBuilder.into(binding.imgBackcrdScan)
                        }
                    }
                },
            )
        }
        activity.runOnUiThread {
            viewModel.cameraXLiveData.setIdLine1(true)
            viewModel.cameraXLiveData.setIdLine2(true)
            viewModel.cameraXLiveData.setIdLine3(true)
            viewModel.cameraXLiveData.setIsFrontViewScanned(true)
        }
    }

    private fun captureBackViewIdCard(stream: ByteArrayOutputStream) {
        activity.runOnUiThread { viewModel.cameraXLiveData.setStatusText("") }
        val byteArray = stream.toByteArray()
        Utility.getInstance().scannedIdFrontView = byteArray
        activity.runOnUiThread {
            viewModel.cameraXLiveData.setIsOverlayVisible(false)
            viewModel.cameraXLiveData.setIsScanVisible(false)
            viewModel.cameraXLiveData.setIsTransparentVisible(true)
            viewModel.cameraXLiveData.setIsBackCardScanVisible(false)
            viewModel.cameraXLiveData.setIsScanCompleteVisible(true)
            binding.imgScanComplete.playAnimation()
            logger.log("onAnimationEnd: end time " + viewModel.printCurrentTimeInLogs())
            binding.imgScanComplete.addAnimatorListener(
                object : AnimatorListenerAdapter() {
                    override fun onAnimationEnd(animation: Animator) {
                        super.onAnimationEnd(animation)
                        viewModel.handleAnimationCompleted()
                    }
                },
            )
        }
    }

    override fun onCheckedChanged(
        buttonView: CompoundButton,
        isChecked: Boolean,
    ) {
        if (ContextCompat.checkSelfPermission(
                activity,
                Manifest.permission.CAMERA,
            ) == PackageManager.PERMISSION_GRANTED
        ) {
            if (cameraProvider == null) {
                return
            }
            val newLensFacing =
                if (lensFacing == CameraSelector.LENS_FACING_FRONT) CameraSelector.LENS_FACING_BACK else CameraSelector.LENS_FACING_FRONT
            val newCameraSelector =
                CameraSelector.Builder().requireLensFacing(newLensFacing).build()
            try {
                if (cameraProvider!!.hasCamera(newCameraSelector)) {
                    lensFacing = newLensFacing
                    cameraSelector = newCameraSelector
                    bindAllCameraUseCases()
                }
            } catch (e: CameraInfoUnavailableException) {
                e.printStackTrace()
            }
        }
    }
}
