package nashid.verify.sdk.utils.helpers

import android.Manifest
import android.app.Activity
import android.content.Context
import android.content.pm.PackageManager
import android.location.Geocoder
import android.location.Location
import android.location.LocationListener
import android.location.LocationManager
import android.os.Bundle
import android.os.Handler
import android.os.Looper
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import java.io.IOException
import java.util.Locale

class LocationHelper(private val activity: Activity) : LocationListener {
    private val locationManager = activity.getSystemService(Context.LOCATION_SERVICE) as LocationManager
    private var locationCallback: LocationCallback? = null

    companion object {
        private const val MIN_TIME_BW_UPDATES: Long = 1000 * 60 // 1 minute
        private const val MIN_DISTANCE_CHANGE_FOR_UPDATES = 10f // 10 meters
        private const val LOCATION_TIMEOUT: Long = 10000 // 10 seconds
    }

    interface LocationCallback {
        fun onLocationResult(
            location: Location,
            address: String,
        )
    }

    fun getLastLocation(callback: LocationCallback) {
        locationCallback = callback

        if (ContextCompat.checkSelfPermission(activity, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
            requestLocationPermissions()
            return
        }

        var location = locationManager.getLastKnownLocation(LocationManager.GPS_PROVIDER)
        if (location == null) {
            location = locationManager.getLastKnownLocation(LocationManager.NETWORK_PROVIDER)
        }

        if (location == null) {
            requestLocationUpdatesWithTimeout(callback)
        } else {
            val address = getAddressFromLocation(location)
            locationCallback?.onLocationResult(location, address)
        }
    }

    private fun requestLocationUpdatesWithTimeout(callback: LocationCallback) {
        locationCallback = callback

        if (ActivityCompat.checkSelfPermission(activity, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
            requestLocationPermissions()
            return
        }

        locationManager.requestLocationUpdates(
            LocationManager.GPS_PROVIDER,
            MIN_TIME_BW_UPDATES,
            MIN_DISTANCE_CHANGE_FOR_UPDATES,
            this,
        )

        Handler(Looper.getMainLooper()).postDelayed({
            locationManager.removeUpdates(this)

            var lastKnownLocation = locationManager.getLastKnownLocation(LocationManager.GPS_PROVIDER)
            if (lastKnownLocation == null) {
                lastKnownLocation = locationManager.getLastKnownLocation(LocationManager.NETWORK_PROVIDER)
            }

            if (lastKnownLocation != null && locationCallback != null) {
                val address = getAddressFromLocation(lastKnownLocation)
                locationCallback?.onLocationResult(lastKnownLocation, address)
            } else {
                val defaultLocation =
                    Location("default").apply {
                        latitude = 0.0
                        longitude = 0.0
                    }
                val address = getAddressFromLocation(defaultLocation)
                locationCallback?.onLocationResult(defaultLocation, address)
            }
        }, LOCATION_TIMEOUT)
    }

    override fun onLocationChanged(location: Location) {
        locationCallback?.let {
            val address = getAddressFromLocation(location)
            it.onLocationResult(location, address)
            locationManager.removeUpdates(this)
        }
    }

    override fun onStatusChanged(
        provider: String?,
        status: Int,
        extras: Bundle?,
    ) {
        // Optional override, can leave empty
    }

    override fun onProviderEnabled(provider: String) {
        // Optional override, can leave empty
    }

    override fun onProviderDisabled(provider: String) {
        // Optional override, can leave empty
    }

    private fun requestLocationPermissions() {
        ActivityCompat.requestPermissions(
            activity,
            arrayOf(Manifest.permission.ACCESS_FINE_LOCATION),
            1,
        )
    }

    private fun getAddressFromLocation(location: Location): String {
        val geocoder = Geocoder(activity, Locale.getDefault())
        return try {
            val addresses = geocoder.getFromLocation(location.latitude, location.longitude, 1)
            if (!addresses.isNullOrEmpty()) {
                addresses[0].getAddressLine(0) ?: "Unknown location"
            } else {
                "Unknown location"
            }
        } catch (e: IOException) {
            e.printStackTrace()
            "Unknown location"
        }
    }
}
