package nashid.verify.sdk.data.repository

import android.content.Context
import android.widget.Toast
import nashid.verify.sdk.model.Artifacts
import nashid.verify.sdk.model.Liveness
import nashid.verify.sdk.model.Location
import nashid.verify.sdk.model.MainData
import nashid.verify.sdk.model.MetaDataModel
import nashid.verify.sdk.model.NFC
import nashid.verify.sdk.model.Scan
import nashid.verify.sdk.utils.ArtifactType
import nashid.verify.sdk.utils.LivenessData
import nashid.verify.sdk.utils.OtherConstant
import nashid.verify.sdk.utils.ScanDocument
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG10File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG11File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG13File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG1File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG4File
import nashid.verify.sdk.utils.id_card.omn.OmanCardDG6File
import nashid.verify.sdk.viewmodel.SkipNfcLiveNessLiveData

class SkipLiveNessRepository(
    private val skipNfcLiveNessLiveData: SkipNfcLiveNessLiveData,
    private val context: Context,
) {
    fun callToComparison() {
        skipNfcLiveNessLiveData.setHandleLocationData(true)
    }

    fun createData(
        isNFCEnabled: Boolean,
        documentType: Int,
        isLiveNessEnabled: Boolean?,
    ): MainData {
        return MainData(
            scan =
                if (SdkConfig.isOcrEnabled || SdkConfig.isMrzEnabled) {
                    createScanData(documentType) ?: Scan()
                } else {
                    Scan()
                },
            nfc =
                if (isNFCEnabled && SdkConfig.isNfcScanningEnabled) {
                    createNfcData(documentType, isNFCEnabled) ?: NFC()
                } else {
                    NFC()
                },
            liveness =
                if (SdkConfig.isFaceMatchingEnabled || SdkConfig.isPassiveLiveNessEnabled) {
                    createLiveNess(isLiveNessEnabled ?: false)
                } else {
                    Liveness()
                },
        )
    }

    private fun createLiveNess(isLiveNessEnabled: Boolean): Liveness {
        if (isLiveNessEnabled) {
            return when {
                SdkConfig.isActiveLiveNessEnabled -> {
                    Liveness(
                        activeLivenessConfirmed = true,
                        passiveLivenessConfirmed = false,
                        score = Utility.getInstance().getLivenessScore().toString(),
                    )
                }
                SdkConfig.isPassiveLiveNessEnabled -> {
                    Liveness(
                        activeLivenessConfirmed = false,
                        passiveLivenessConfirmed = true,
                        score = Utility.getInstance().getLivenessScore().toString(),
                    )
                }
                else -> Liveness()
            }
        }
        return Liveness()
    }

    private fun createScanData(documentType: Int): Scan? {
        if (!SdkConfig.isOcrEnabled && !SdkConfig.isMrzEnabled && !Utility.getInstance().isBarCodeCard) {
            return null
        }

        if (Utility.getInstance().isBarCodeCard) {
            return createScan(emptyList())
        }

        val mrzLines =
            when (documentType) {
                1 ->
                    listOfNotNull(
                        Utility.getInstance().mrzLine1,
                        Utility.getInstance().mrzLine2,
                        Utility.getInstance().mrzLine3,
                    ).takeIf { it.isNotEmpty() }

                2 ->
                    listOfNotNull(
                        Utility.getInstance().mrzLine1,
                        Utility.getInstance().mrzLine2,
                    ).takeIf { it.isNotEmpty() }

                else -> null
            } ?: return null

        return createScan(mrzLines)
    }

    private fun createScan(mrzLines: List<String>): Scan {
        val utility = Utility.getInstance()
        val scanDocument = ScanDocument.getInstance()

        return Scan(
            mrzText = mrzLines.joinToString("\n"),
            documentNo = utility.passportNumber?.replace("<", "") ?: "",
            documentType = utility.documentType ?: "",
            expiryDate = scanDocument.expiryDate?.let { utility.getFormattedDate(it) } ?: "",
            dateOfBirth = scanDocument.dateOfBirth?.let { utility.getFormattedDate(it) } ?: "",
            fullName = utility.name ?: "",
            nationality = utility.nationality ?: "",
            country = utility.country ?: "",
            gender =
                when (utility.gender?.uppercase()) {
                    "F" -> "Female"
                    "M" -> "Male"
                    else -> ""
                },
        )
    }

    private fun createNfcData(
        documentType: Int,
        isNFCEnabled: Boolean,
    ): NFC? {
        return if (isNFCEnabled) {
            when (documentType) {
                1 -> createIdCardNfcData()
                2 -> createPassportNfcData()
                else -> null
            }
        } else {
            null
        }
    }

    private fun createPassportNfcData(): NFC? {
        return try {
            val eDocument = LivenessData.getInstance().geteDocument()
            val personDetails = eDocument.personDetails
            val additionalPersonDetails = eDocument.additionalPersonDetails
            val additionalDGFiles = eDocument.additionalDGFiles
            val placeOfBirth = additionalPersonDetails.placeOfBirth?.getOrNull(0) ?: ""

            NFC(
                passportDocType = eDocument.docType.name,
                passportNationality = personDetails.nationality,
                passDocNumber = personDetails.serialNumber,
                passportIssuingCountry = personDetails.issuerAuthority,
                passportDateOfBirth = personDetails.birthDate,
                passportGender = personDetails.gender,
                passportDocExpiryDate = personDetails.expiryDate,
                passportName = personDetails.name,
                passportPlaceOfBirth = additionalPersonDetails.placeOfBirth?.get(0) ?: "",
                passportAddress = additionalPersonDetails.permanentAddress?.get(0) ?: "",
                passportProfession = additionalPersonDetails.profession ?: "",
                passportTitle = additionalPersonDetails.title ?: "",
                passportPersonalSummary = additionalPersonDetails.personalSummary ?: "",
                passportTDNumber = additionalPersonDetails.otherValidTDNumbers?.get(0) ?: "",
                passportPhoneNumber = additionalPersonDetails.telephone ?: "",
                passportNameOfHolder = additionalPersonDetails.nameOfHolder ?: "",
                passportIssuingAuthority = additionalDGFiles.issueingAuthority,
            )
        } catch (e: Exception) {
            e.printStackTrace()
            null
        }
    }

    private fun createIdCardNfcData(): NFC? {
        return try {
            val dg1Parser = Utility.getInstance().dgFile1?.let { OmanCardDG1File(it) }
            val dg4Parser = Utility.getInstance().dgFile4?.let { OmanCardDG4File(it) }
            val dg6Parser = Utility.getInstance().dgFile6?.let { OmanCardDG6File(it) }
            val dg10Parser = Utility.getInstance().dgFile10?.let { OmanCardDG10File(it) }
            val dg13Parser = Utility.getInstance().dgFile13?.let { OmanCardDG13File(it) }
            val dg11Parser = Utility.getInstance().dgFile11?.let { OmanCardDG11File(it) }

            val placeOfIssueArabic =
                dg1Parser?.placeOfIssueArabic?.let {
                    if (it.startsWith("\\u")) OtherConstant.getInstance().getEncodedString(it) else it
                } ?: ""

            val fullNameArabic =
                dg4Parser?.fullNameArabic?.let {
                    if (it.startsWith("\\u")) OtherConstant.getInstance().getEncodedString(it) else it
                } ?: ""

            val countryOfBirthArabic =
                dg6Parser?.countryOfBirthArabic?.let {
                    if (it.startsWith("\\u")) OtherConstant.getInstance().getEncodedString(it) else it
                } ?: ""

            val countryOfBirthEnglish =
                dg6Parser?.placeOfBirthEnglish?.let {
                    if (it.startsWith("\\u")) {
                        OtherConstant.getInstance().getEncodedString(it)
                    } else if (Utility.getInstance().containsArabic(it)) {
                        ""
                    } else {
                        it
                    }
                } ?: ""

            val nationalityArabic =
                dg6Parser?.nationalityArabic?.let {
                    if (it.startsWith("\\u")) OtherConstant.getInstance().getEncodedString(it) else it
                } ?: ""

            val genderArabic =
                dg6Parser?.genderArabic?.let {
                    if (it.startsWith("\\u")) OtherConstant.getInstance().getEncodedString(it) else it
                } ?: ""

            NFC(
                fullNameEnglish = dg4Parser?.fullNameEnglish ?: "",
                nationalityEnglish = dg6Parser?.nationalityEnglish ?: "",
                genderEnglish = dg6Parser?.genderEnglish ?: "",
                genderArabic = genderArabic,
                companyNameEnglish = dg11Parser?.companyNameEnglish ?: "",
                countryOfBirthArabic = countryOfBirthArabic,
                placeOfIssueArabic = placeOfIssueArabic,
                identityNumber = dg1Parser?.idNumber ?: "",
                placeOfIssueEnglish = dg1Parser?.placeOfIssueEnglish ?: "",
                expiryDate = dg1Parser?.expiryDate?.let { Utility.getInstance().getNfcFormattedDate(it) } ?: "",
                issueDate = dg1Parser?.issueDate?.let { Utility.getInstance().getNfcFormattedDate(it) } ?: "",
                fullNameArabic = fullNameArabic,
                companyNameArabic = dg11Parser?.companyNameArabic ?: "",
                countryOfBirthEnglish = countryOfBirthEnglish,
                nationalityArabic = nationalityArabic,
                companyAddressArabic = dg11Parser?.companyAddressArabic ?: "",
                dateOfBirth = dg6Parser?.dateOfBirth?.let { Utility.getInstance().getNfcFormattedDate(it) } ?: "",
                visaNumber = dg10Parser?.visaNumber ?: "",
                permitNumber = dg13Parser?.sponsorNumber ?: "",
                permitType = dg13Parser?.sponsorRelationship ?: "",
                useByDate = dg10Parser?.visaExpiryDate?.let { Utility.getInstance().getNfcFormattedDate(it) } ?: "",
            )
        } catch (e: Exception) {
            e.printStackTrace()
            Toast.makeText(context, "Error creating NFC data: ${e.message}", Toast.LENGTH_LONG).show()
            null
        }
    }

    fun createMetaData(
        latitude: Double,
        longitude: Double,
        onMetaDataReady: (MetaDataModel) -> Unit,
    ) {
        Utility.getInstance().fetchPublicIPAddresses { ipv4, ipv6 ->
            val metadata =
                MetaDataModel(
                    timestamp = Utility.getInstance().getCurrentFormattedDateTime() ?: "",
                    location = Location(latitude, longitude),
                    deviceIp4 = ipv4,
                    deviceIp6 = ipv6,
                    systemName = Utility.getInstance().getSystemName(),
                    systemVersion = Utility.getInstance().getSystemVersion(),
                    appVersion = Utility.getInstance().getAppVersion(context) ?: "",
                    deviceIdentifier = Utility.getInstance().getDeviceIdentifier(),
                    deviceType = Utility.getInstance().getDeviceType(),
                    gender =
                        when (Utility.getInstance().gender?.uppercase()) {
                            "F" -> "Female"
                            "M" -> "Male"
                            else -> ""
                        },
                    extraData = SdkConfig.isExtraData,
                )
            onMetaDataReady(metadata)
        }
    }

    fun handleArtifacts(): Artifacts {
        return Artifacts(
            backSideImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.BACKSIDE_IMAGE) ?: "",
            frontSideImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.FRONTSIDE_IMAGE) ?: "",
            ocrFaceImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.OCR_FACE_IMAGE) ?: "",
            nfcFaceImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.NFC_IMAGE) ?: "",
            livenessImage = Utility.getInstance().getFileKeyForArtifact(ArtifactType.LIVENESS_IMAGE) ?: "",
        )
    }
}
