package nashid.verify.sdk.utils.helpers

import com.google.gson.Gson
import com.google.gson.GsonBuilder
import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.google.gson.JsonSerializationContext
import com.google.gson.JsonSerializer
import nashid.verify.sdk.model.Artifacts
import nashid.verify.sdk.model.DocumentVerification
import nashid.verify.sdk.model.Liveness
import nashid.verify.sdk.model.MainData
import nashid.verify.sdk.model.NFC
import nashid.verify.sdk.model.Scan
import java.lang.reflect.Type

class DocumentVerificationSerializer : JsonSerializer<DocumentVerification> {
    override fun serialize(
        src: DocumentVerification,
        typeOfSrc: Type,
        context: JsonSerializationContext,
    ): JsonElement {
        val jsonObject = JsonObject()
        src.metaDataModel?.let {
            jsonObject.add("metadata", context.serialize(it))
        }
        if (src.type != 0) {
            jsonObject.addProperty("type", src.type)
        }
        src.data?.let {
            jsonObject.add("data", context.serialize(it))
        }
        src.artifacts?.let {
            jsonObject.add("artifacts", context.serialize(it))
        }
        if (src.status != 0) {
            jsonObject.addProperty("status", src.status)
        }

        return jsonObject
    }
}

class MainDataSerializer : JsonSerializer<MainData> {
    override fun serialize(
        src: MainData,
        typeOfSrc: Type,
        context: JsonSerializationContext,
    ): JsonElement {
        val jsonObject = JsonObject()

        if (src.scan != null && !isScanEmpty(src.scan)) {
            jsonObject.add("scan", context.serialize(src.scan))
        }

        if (src.nfc != null) {
            val nfcObject = JsonObject()
            with(src.nfc) {
                if (fullNameEnglish.isNotEmpty()) nfcObject.addProperty("full_name_english", fullNameEnglish)
                if (nationalityEnglish.isNotEmpty()) nfcObject.addProperty("nationality_english", nationalityEnglish)
                if (genderEnglish.isNotEmpty()) nfcObject.addProperty("gender_english", genderEnglish)
                if (genderArabic.isNotEmpty()) nfcObject.addProperty("gender_arabic", genderArabic)
                if (companyNameEnglish.isNotEmpty()) nfcObject.addProperty("company_name_english", companyNameEnglish)
                if (permitType.isNotEmpty()) nfcObject.addProperty("permit_type", permitType)
                if (countryOfBirthArabic.isNotEmpty()) nfcObject.addProperty("country_of_birth_arabic", countryOfBirthArabic)
                if (placeOfIssueArabic.isNotEmpty()) nfcObject.addProperty("place_of_issue_arabic", placeOfIssueArabic)
                if (identityNumber.isNotEmpty()) nfcObject.addProperty("identity_number", identityNumber)
                if (placeOfIssueEnglish.isNotEmpty()) nfcObject.addProperty("place_of_issue_english", placeOfIssueEnglish)
                if (expiryDate.isNotEmpty()) nfcObject.addProperty("expiry_date", expiryDate)
                if (visaNumber.isNotEmpty()) nfcObject.addProperty("visa_number", visaNumber)
                if (issueDate.isNotEmpty()) nfcObject.addProperty("issue_date", issueDate)
                if (fullNameArabic.isNotEmpty()) nfcObject.addProperty("full_name_arabic", fullNameArabic)
                if (companyNameArabic.isNotEmpty()) nfcObject.addProperty("company_name_arabic", companyNameArabic)
                if (countryOfBirthEnglish.isNotEmpty()) nfcObject.addProperty("country_of_birth_english", countryOfBirthEnglish)
                if (nationalityArabic.isNotEmpty()) nfcObject.addProperty("nationality_arabic", nationalityArabic)
                if (companyAddressArabic.isNotEmpty()) nfcObject.addProperty("company_address_arabic", companyAddressArabic)
                if (permitNumber.isNotEmpty()) nfcObject.addProperty("permit_number", permitNumber)
                if (dateOfBirth.isNotEmpty()) nfcObject.addProperty("date_of_birth", dateOfBirth)
                if (useByDate.isNotEmpty()) nfcObject.addProperty("use_by_date", useByDate)

                if (passportDocType.isNotEmpty()) nfcObject.addProperty("document_type", passportDocType)
                if (passportNationality.isNotEmpty()) nfcObject.addProperty("nationality", passportNationality)
                if (passDocNumber.isNotEmpty()) nfcObject.addProperty("document_number", passDocNumber)
                if (passportIssuingCountry.isNotEmpty()) nfcObject.addProperty("issuing_country", passportIssuingCountry)
                if (passportDateOfBirth.isNotEmpty()) nfcObject.addProperty("passport_date_of_birth", passportDateOfBirth)
                if (passportGender.isNotEmpty()) nfcObject.addProperty("gender", passportGender)
                if (passportDocExpiryDate.isNotEmpty()) nfcObject.addProperty("document_expiry_date", passportDocExpiryDate)
                if (passportName.isNotEmpty()) nfcObject.addProperty("name", passportName)
                if (passportPlaceOfBirth.isNotEmpty()) nfcObject.addProperty("place_of_birth", passportPlaceOfBirth)
                if (passportAddress.isNotEmpty()) nfcObject.addProperty("address", passportAddress)
                if (passportProfession.isNotEmpty()) nfcObject.addProperty("profession", passportProfession)
                if (passportTitle.isNotEmpty()) nfcObject.addProperty("title", passportTitle)
                if (passportPersonalSummary.isNotEmpty()) nfcObject.addProperty("personal_summary", passportPersonalSummary)
                if (passportTDNumber.isNotEmpty()) nfcObject.addProperty("td_number", passportTDNumber)
                if (passportPhoneNumber.isNotEmpty()) nfcObject.addProperty("phone_number", passportPhoneNumber)
                if (passportNameOfHolder.isNotEmpty()) nfcObject.addProperty("name_of_holder", passportNameOfHolder)
                if (passportIssuingAuthority.isNotEmpty()) nfcObject.addProperty("issuing_authority", passportIssuingAuthority)
            }
            if (nfcObject.size() > 0) {
                jsonObject.add("NFC", nfcObject)
            }
        }

        if (src.liveness != null && !isLivenessEmpty(src.liveness)) {
            jsonObject.add("liveness", context.serialize(src.liveness))
        }

        return jsonObject
    }

    private fun isScanEmpty(scan: Scan): Boolean {
        return scan == Scan()
    }

    private fun isNfcEmpty(nfc: NFC): Boolean {
        return nfc == NFC()
    }

    private fun isLivenessEmpty(liveness: Liveness): Boolean {
        return liveness == Liveness()
    }
}

class ArtifactsSerializer : JsonSerializer<Artifacts> {
    override fun serialize(
        src: Artifacts,
        typeOfSrc: Type,
        context: JsonSerializationContext,
    ): JsonElement {
        val jsonObject = JsonObject()

        if (src.backSideImage.isNotEmpty()) {
            jsonObject.addProperty("back_side_image", src.backSideImage)
        }
        if (src.frontSideImage.isNotEmpty()) {
            jsonObject.addProperty("front_side_image", src.frontSideImage)
        }
        if (src.ocrFaceImage.isNotEmpty()) {
            jsonObject.addProperty("ocr_face_image", src.ocrFaceImage)
        }
        if (src.nfcFaceImage.isNotEmpty()) {
            jsonObject.addProperty("nfc_face_image", src.nfcFaceImage)
        }
        if (src.livenessImage.isNotEmpty()) {
            jsonObject.addProperty("liveness_image", src.livenessImage)
        }
        if (src.livenessImageWithoutBg.isNotEmpty()) {
            jsonObject.addProperty("livness_image_without_bg", src.livenessImageWithoutBg)
        }

        return jsonObject
    }
}

fun createCustomGson(): Gson {
    return GsonBuilder()
        .registerTypeAdapter(DocumentVerification::class.java, DocumentVerificationSerializer())
        .registerTypeAdapter(MainData::class.java, MainDataSerializer())
        .registerTypeAdapter(Artifacts::class.java, ArtifactsSerializer())
        .create()
}
