/*
 * JMRTD - A Java API for accessing machine readable travel documents.
 *
 * Copyright (C) 2006 - 2018  The JMRTD team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * $Id: PACEInfo.java 1805 2018-11-26 21:39:46Z martijno $
 */

package nashid.verify.sdk.utils.id_card.jmrtd.lds;

import nashid.verify.sdk.utils.id_card.jmrtd.Util;
import nashid.verify.sdk.utils.id_card.jmrtd.lds.SecurityInfo;

import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.ASN1Integer;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.DLSequence;
import org.bouncycastle.crypto.params.DHParameters;
import org.bouncycastle.jce.ECNamedCurveTable;
import org.bouncycastle.util.encoders.Hex;

import java.math.BigInteger;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.ECParameterSpec;
import java.util.Arrays;
import java.util.Set;
import java.util.TreeSet;

import javax.crypto.spec.DHParameterSpec;

/**
 * PACE Info object as per SAC TR 1.01, November 11, 2010.
 *
 * @author The JMRTD team (info@jmrtd.org)
 *
 * @version $Revision: 1805 $
 *
 * @since 0.5.0
 */
public class PACEInfo extends SecurityInfo {

  private static final long serialVersionUID = 7960925013249578359L;

  /**
   * A DH parameter specification which also keeps track of
   * the prime order of the subgroup generated by the generator.
   *
   * @author The JMRTD team (info@jmrtd.org)
   *
   * @version $Revision: 1805 $
   */
  public static class DHCParameterSpec extends DHParameterSpec {

    private final BigInteger q;

    /**
     * Creates a parameter specification.
     *
     * @param p the prime
     * @param g the generator
     * @param q the prime order of subgroup generated by {@code g}
     */
    public DHCParameterSpec(BigInteger p, BigInteger g, BigInteger q) {
      super(p, g);
      this.q = q;
    }

    /**
     * Returns the prime order of subgroup generated by {@code g}.
     *
     * @return the prime order of subgroup generated by {@code g}
     */
    public BigInteger getQ() {
      return q;
    }
  }

  /** Generic mapping and Integrated mapping and CAM mapping. */
  public enum MappingType {
    /** Generic Mapping. */
    GM,

    /** Integrated Mapping. */
    IM,

    /** Chip Authentication Mapping. */
    CAM
  }

  /** Standardized domain parameters. Based on Table 6. */
  public static final int PARAM_ID_GFP_1024_160 = 0;

  /** Standardized domain parameters. Based on Table 6. */
  public static final int PARAM_ID_GFP_2048_224 = 1;

  /** Standardized domain parameters. Based on Table 6. */
  public static final int PARAM_ID_GFP_2048_256 = 2;

  /* RFU 3 - 7 */

  /** Standardized domain parameters. Based on Table 6. */
  public static final int PARAM_ID_ECP_NIST_P192_R1 = 8;

  /** Standardized domain parameters. Based on Table 6. */
  public static final int PARAM_ID_ECP_BRAINPOOL_P192_R1 = 9;

  /** Standardized domain parameters. Based on Table 6. */
  public static final int PARAM_ID_ECP_NIST_P224_R1 = 10;

  /** Standardized domain parameters. Based on Table 6. */
  public static final int PARAM_ID_ECP_BRAINPOOL_P224_R1 = 11;

  /** Standardized domain parameters. Based on Table 6. */
  public static final int PARAM_ID_ECP_NIST_P256_R1 = 12;

  /** Standardized domain parameters. Based on Table 6. */
  public static final int  PARAM_ID_ECP_BRAINPOOL_P256_R1 = 13;

  /** Standardized domain parameters. Based on Table 6. */
  public static final int  PARAM_ID_ECP_BRAINPOOL_P320_R1 = 14;

  /** Standardized domain parameters. Based on Table 6. */
  public static final int   PARAM_ID_ECP_NIST_P384_R1 = 15;

  /** Standardized domain parameters. Based on Table 6. */
  public static final int   PARAM_ID_ECP_BRAINPOOL_P384_R1 = 16;

  /** Standardized domain parameters. Based on Table 6. */
  public static final int   PARAM_ID_ECP_BRAINPOOL_P512_R1 = 17;

  /** Standardized domain parameters. Based on Table 6. */  public static final int PARAM_ID_ECP_NIST_P521_R1 = 18;

  /* RFU 19-31 */
  private static DHParameters fromPGQ(String hexP, String hexG, String hexQ)
  {
    return new DHParameters(fromHex(hexP), fromHex(hexG), fromHex(hexQ));
  }
  private static BigInteger fromHex(String hex)
  {
    return new BigInteger(1, Hex.decode(hex));
  }
  private static final String rfc5114_1024_160_p = "B10B8F96A080E01DDE92DE5EAE5D54EC52C99FBCFB06A3C6"
          + "9A6A9DCA52D23B616073E28675A23D189838EF1E2EE652C0" + "13ECB4AEA906112324975C3CD49B83BFACCBDD7D90C4BD70"
          + "98488E9C219A73724EFFD6FAE5644738FAA31A4FF55BCCC0" + "A151AF5F0DC8B4BD45BF37DF365C1A65E68CFDA76D4DA708"
          + "DF1FB2BC2E4A4371";
  private static final String rfc5114_1024_160_g = "A4D1CBD5C3FD34126765A442EFB99905F8104DD258AC507F"
          + "D6406CFF14266D31266FEA1E5C41564B777E690F5504F213" + "160217B4B01B886A5E91547F9E2749F4D7FBD7D3B9A92EE1"
          + "909D0D2263F80A76A6A24C087A091F531DBF0A0169B6A28A" + "D662A4D18E73AFA32D779D5918D08BC8858F4DCEF97C2A24"
          + "855E6EEB22B3B2E5";
  private static final String rfc5114_1024_160_q = "F518AA8781A8DF278ABA4E7D64B7CB9D49462353";
  private static final DHParameterSpec PARAMS_GFP_1024_160 = Util.toExplicitDHParameterSpec( fromPGQ(rfc5114_1024_160_p, rfc5114_1024_160_g,
          rfc5114_1024_160_q));
  private static final String rfc5114_2048_224_p = "AD107E1E9123A9D0D660FAA79559C51FA20D64E5683B9FD1"
          + "B54B1597B61D0A75E6FA141DF95A56DBAF9A3C407BA1DF15" + "EB3D688A309C180E1DE6B85A1274A0A66D3F8152AD6AC212"
          + "9037C9EDEFDA4DF8D91E8FEF55B7394B7AD5B7D0B6C12207" + "C9F98D11ED34DBF6C6BA0B2C8BBC27BE6A00E0A0B9C49708"
          + "B3BF8A317091883681286130BC8985DB1602E714415D9330" + "278273C7DE31EFDC7310F7121FD5A07415987D9ADC0A486D"
          + "CDF93ACC44328387315D75E198C641A480CD86A1B9E587E8" + "BE60E69CC928B2B9C52172E413042E9B23F10B0E16E79763"
          + "C9B53DCF4BA80A29E3FB73C16B8E75B97EF363E2FFA31F71" + "CF9DE5384E71B81C0AC4DFFE0C10E64F";
  private static final String rfc5114_2048_224_g = "AC4032EF4F2D9AE39DF30B5C8FFDAC506CDEBE7B89998CAF"
          + "74866A08CFE4FFE3A6824A4E10B9A6F0DD921F01A70C4AFA" + "AB739D7700C29F52C57DB17C620A8652BE5E9001A8D66AD7"
          + "C17669101999024AF4D027275AC1348BB8A762D0521BC98A" + "E247150422EA1ED409939D54DA7460CDB5F6C6B250717CBE"
          + "F180EB34118E98D119529A45D6F834566E3025E316A330EF" + "BB77A86F0C1AB15B051AE3D428C8F8ACB70A8137150B8EEB"
          + "10E183EDD19963DDD9E263E4770589EF6AA21E7F5F2FF381" + "B539CCE3409D13CD566AFBB48D6C019181E1BCFE94B30269"
          + "EDFE72FE9B6AA4BD7B5A0F1C71CFFF4C19C418E1F6EC0179" + "81BC087F2A7065B384B890D3191F2BFA";
  private static final String rfc5114_2048_256_p = "87A8E61DB4B6663CFFBBD19C651959998CEEF608660DD0F2"
          + "5D2CEED4435E3B00E00DF8F1D61957D4FAF7DF4561B2AA30" + "16C3D91134096FAA3BF4296D830E9A7C209E0C6497517ABD"
          + "5A8A9D306BCF67ED91F9E6725B4758C022E0B1EF4275BF7B" + "6C5BFC11D45F9088B941F54EB1E59BB8BC39A0BF12307F5C"
          + "4FDB70C581B23F76B63ACAE1CAA6B7902D52526735488A0E" + "F13C6D9A51BFA4AB3AD8347796524D8EF6A167B5A41825D9"
          + "67E144E5140564251CCACB83E6B486F6B3CA3F7971506026" + "C0B857F689962856DED4010ABD0BE621C3A3960A54E710C3"
          + "75F26375D7014103A4B54330C198AF126116D2276E11715F" + "693877FAD7EF09CADB094AE91E1A1597";
  private static final String rfc5114_2048_256_g = "3FB32C9B73134D0B2E77506660EDBD484CA7B18F21EF2054"
          + "07F4793A1A0BA12510DBC15077BE463FFF4FED4AAC0BB555" + "BE3A6C1B0C6B47B1BC3773BF7E8C6F62901228F8C28CBB18"
          + "A55AE31341000A650196F931C77A57F2DDF463E5E9EC144B" + "777DE62AAAB8A8628AC376D282D6ED3864E67982428EBC83"
          + "1D14348F6F2F9193B5045AF2767164E1DFC967C1FB3F2E55" + "A4BD1BFFE83B9C80D052B985D182EA0ADB2A3B7313D3FE14"
          + "C8484B1E052588B9B7D2BBD2DF016199ECD06E1557CD0915" + "B3353BBB64E0EC377FD028370DF92B52C7891428CDC67EB6"
          + "184B523D1DB246C32F63078490F00EF8D647D148D4795451" + "5E2327CFEF98C582664B4C0F6CC41659";
  private static final String rfc5114_2048_256_q = "8CF83642A709A097B447997640129DA299B1A47D1EB3750B"
          + "A308B0FE64F5FBD3";
  private static final String rfc5114_2048_224_q = "801C0D34C58D93FE997177101F80535A4738CEBCBF389A99B36371EB";

  private static final DHParameterSpec PARAMS_GFP_2048_224 = Util.toExplicitDHParameterSpec(fromPGQ(rfc5114_2048_224_p, rfc5114_2048_224_g,
          rfc5114_2048_224_q));
  private static final DHParameterSpec PARAMS_GFP_2048_256 = Util.toExplicitDHParameterSpec( fromPGQ(rfc5114_2048_256_p, rfc5114_2048_256_g,
          rfc5114_2048_256_q));

  private static final ECParameterSpec PARAMS_ECP_NIST_P192_R1 = Util.toExplicitECParameterSpec(ECNamedCurveTable.getParameterSpec("secp192r1"));
  private static final ECParameterSpec PARAMS_ECP_NIST_P224_R1 = Util.toExplicitECParameterSpec(ECNamedCurveTable.getParameterSpec("secp224r1"));
  private static final ECParameterSpec PARAMS_ECP_NIST_P256_R1 = Util.toExplicitECParameterSpec(ECNamedCurveTable.getParameterSpec("secp256r1"));
  private static final ECParameterSpec PARAMS_ECP_NIST_P384_R1 = Util.toExplicitECParameterSpec(ECNamedCurveTable.getParameterSpec("secp384r1"));
  private static final ECParameterSpec PARAMS_ECP_NIST_P521_R1 = Util.toExplicitECParameterSpec(ECNamedCurveTable.getParameterSpec("secp521r1"));
  private static final ECParameterSpec PARAMS_ECP_BRAINPOOL_P192_R1 = Util.toExplicitECParameterSpec(ECNamedCurveTable.getParameterSpec("brainpoolp192r1"));
  private static final ECParameterSpec PARAMS_ECP_BRAINPOOL_P224_R1 = Util.toExplicitECParameterSpec(ECNamedCurveTable.getParameterSpec("brainpoolp224r1"));
  private static final ECParameterSpec PARAMS_ECP_BRAINPOOL_P256_R1 = Util.toExplicitECParameterSpec(ECNamedCurveTable.getParameterSpec("brainpoolp256r1"));
  private static final ECParameterSpec PARAMS_ECP_BRAINPOOL_P320_R1 = Util.toExplicitECParameterSpec(ECNamedCurveTable.getParameterSpec("brainpoolp320r1"));
  private static final ECParameterSpec PARAMS_ECP_BRAINPOOL_P384_R1 = Util.toExplicitECParameterSpec(ECNamedCurveTable.getParameterSpec("brainpoolp384r1"));
  private static final ECParameterSpec PARAMS_ECP_BRAINPOOL_P512_R1 = Util.toExplicitECParameterSpec(ECNamedCurveTable.getParameterSpec("brainpoolp512r1"));

  private static final Set<String> ALLOWED_REQUIRED_IDENTIFIERS = new TreeSet<String>(Arrays.asList(ID_PACE_DH_GM_3DES_CBC_CBC,
          ID_PACE_DH_GM_AES_CBC_CMAC_128,
          ID_PACE_DH_GM_AES_CBC_CMAC_192,
          ID_PACE_DH_GM_AES_CBC_CMAC_256,
          ID_PACE_DH_IM_3DES_CBC_CBC,
          ID_PACE_DH_IM_AES_CBC_CMAC_128,
          ID_PACE_DH_IM_AES_CBC_CMAC_192,
          ID_PACE_DH_IM_AES_CBC_CMAC_256,
          ID_PACE_ECDH_GM_3DES_CBC_CBC,
          ID_PACE_ECDH_GM_AES_CBC_CMAC_128,
          ID_PACE_ECDH_GM_AES_CBC_CMAC_192,
          ID_PACE_ECDH_GM_AES_CBC_CMAC_256,
          ID_PACE_ECDH_IM_3DES_CBC_CBC,
          ID_PACE_ECDH_IM_AES_CBC_CMAC_128,
          ID_PACE_ECDH_IM_AES_CBC_CMAC_192,
          ID_PACE_ECDH_IM_AES_CBC_CMAC_256,
          ID_PACE_ECDH_CAM_AES_CBC_CMAC_128,
          ID_PACE_ECDH_CAM_AES_CBC_CMAC_192,
          ID_PACE_ECDH_CAM_AES_CBC_CMAC_256));

  private final String protocolOID;
  private final int version;
  private final BigInteger parameterId;

  /**
   * Constructs a PACE info object.
   *
   * @param oid the object identifier, indicating what PACE variant
   *        is to be used (agreement protocol, mapping type, and secure channel properties)
   * @param version a version number, which should be 2
   * @param parameterId either a standardized domain parameter id from table 6 or a proprietary domain parameter
   */
  public PACEInfo(String oid, int version, int parameterId) {
    this(oid, version, BigInteger.valueOf(parameterId));
  }

  /**
   * Creates a PACE info object.
   *
   * @param oid the object identifier, indicating what PACE variant
   *        is to be used (agreement protocol, mapping type, and secure channel properties)
   * @param version a version number, which should be 2
   * @param parameterId either a standardized domain parameter id from table 6 or a proprietary domain parameter
   */
  public PACEInfo(String oid, int version, BigInteger parameterId) {
    if (!checkRequiredIdentifier(oid)) {
      throw new IllegalArgumentException("Invalid OID");
    }
    if (version != 2) {
      throw new IllegalArgumentException("Invalid version, must be 2");
    }
    this.protocolOID = oid;
    this.version = version;
    this.parameterId = parameterId;
  }

  /**
   * Creates a PACE info from an encoding.
   *
   * @param paceInfoBytes the encoded bytes
   *
   * @return a PACE info object
   */
  public static PACEInfo createPACEInfo(byte[] paceInfoBytes) {
    /*
     * FIXME: Should add a constructor to PACEInfo that takes byte[] or InputStream, or
     * align this with SecurityInfo.getInstance().
     */
    ASN1Sequence sequence = ASN1Sequence.getInstance(paceInfoBytes);
    String oid = ((ASN1ObjectIdentifier)sequence.getObjectAt(0)).getId();
    ASN1Primitive requiredData = sequence.getObjectAt(1).toASN1Primitive();
    ASN1Primitive optionalData = null;
    if (sequence.size() == 3) {
      optionalData = sequence.getObjectAt(2).toASN1Primitive();
    }

    int version = ((ASN1Integer)requiredData).getValue().intValue();
    BigInteger parameterId = null;
    if (optionalData != null) {
      parameterId = ((ASN1Integer)optionalData).getValue();
    }

    return new PACEInfo(oid, version, parameterId);
  }

  /**
   * Returns the PACE protocol object identifier.
   *
   * @return the PACE protocol object identifier
   */
  @Override
  public String getObjectIdentifier() {
    return protocolOID;
  }

  /**
   * Returns the protocol object identifier as a human readable string.
   *
   * @return a string describing the PACE protocol object identifier
   */
  @Override
  public String getProtocolOIDString() {
    return toProtocolOIDString(protocolOID);
  }

  /**
   * Returns the version.
   *
   * @return the version
   */
  public int getVersion() {
    return version;
  }

  /**
   * Returns the parameter identifier.
   *
   * @return the parameter identifier
   */
  public BigInteger getParameterId() {
    return parameterId;
  }

  /**
   * Returns a DER object with this SecurityInfo data (DER sequence).
   *
   * @return a DER object with this SecurityInfo data
   *
   * @deprecated this method will be removed from visible interface (because of dependency on BC API)
   */
  @Deprecated
  @Override
  public ASN1Primitive getDERObject() {
    ASN1EncodableVector vector = new ASN1EncodableVector();

    /* Protocol */
    vector.add(new ASN1ObjectIdentifier(protocolOID));

    /* Required data */
    vector.add(new ASN1Integer(version));

    /* Optional data */
    if (parameterId != null) {
      vector.add(new ASN1Integer(parameterId));
    }
    return new DLSequence(vector);
  }

  @Override
  public String toString() {
    StringBuilder result = new StringBuilder();
    result.append("PACEInfo [");
    result.append("protocol: ").append(toProtocolOIDString(protocolOID));
    result.append(", version: ").append(version);
    if (parameterId != null) {
      result.append(", parameterId: ").append(toStandardizedParamIdString(parameterId));
    }
    result.append("]");
    return result.toString();
  }

  @Override
  public int hashCode() {
    return 1234567891
        + 7 * protocolOID.hashCode()
        + 5 * version
        + 3 * (parameterId == null ? 1991 : parameterId.hashCode());
  }

  @Override
  public boolean equals(Object other) {
    if (other == null) {
      return false;
    }
    if (other == this) {
      return true;
    }
    if (!PACEInfo.class.equals(other.getClass())) {
      return false;
    }

    PACEInfo otherPACEInfo = (PACEInfo) other;
    return getDERObject().equals(otherPACEInfo.getDERObject());
  }

  /**
   * Checks whether the object identifier is valid for describing a PACE protocol.
   *
   * @param oid a PACE object identifier
   *
   * @return a boolean indicating whether the object identifier describes a known PACE protocol
   */
  public static boolean checkRequiredIdentifier(String oid) {
    return ALLOWED_REQUIRED_IDENTIFIERS.contains(oid);
  }

  /*
   * FIXME: perhaps we should introduce an enum for PACE identifiers (with a String toOID() method),
   * so that we can get rid of static methods below. -- MO
   */

  /**
   * Returns the mapping type for a given PACE protocol object identifier.
   *
   * @param oid a PACE protocol object identifier
   *
   * @return the mapping type
   */
  public static MappingType toMappingType(String oid) {
    if (ID_PACE_DH_GM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_DH_GM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_DH_GM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_DH_GM_AES_CBC_CMAC_256.equals(oid)
        || ID_PACE_ECDH_GM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_ECDH_GM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_ECDH_GM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_ECDH_GM_AES_CBC_CMAC_256.equals(oid)) {
      return MappingType.GM;
    } else if (ID_PACE_DH_IM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_DH_IM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_DH_IM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_DH_IM_AES_CBC_CMAC_256.equals(oid)
        || ID_PACE_ECDH_IM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_ECDH_IM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_ECDH_IM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_ECDH_IM_AES_CBC_CMAC_256.equals(oid)) {
      return MappingType.IM;
    } else if (ID_PACE_ECDH_CAM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_ECDH_CAM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_ECDH_CAM_AES_CBC_CMAC_256.equals(oid)) {
      return MappingType.CAM;
    }

    throw new NumberFormatException("Unknown OID: \"" + oid + "\"");
  }

  /**
   * Returns the key agreement algorithm ({@code "DH"} or {@code "ECDH"})
   * for a PACE protocol object identifier.
   *
   * @param oid a PACE protocol object identifier
   *
   * @return a key agreement algorithm as JCE mnemonic string
   */
  public static String toKeyAgreementAlgorithm(String oid) {
    if (ID_PACE_DH_GM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_DH_GM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_DH_GM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_DH_GM_AES_CBC_CMAC_256.equals(oid)
        || ID_PACE_DH_IM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_DH_IM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_DH_IM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_DH_IM_AES_CBC_CMAC_256.equals(oid)) {
      return "DH";
    } else if (ID_PACE_ECDH_GM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_ECDH_GM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_ECDH_GM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_ECDH_GM_AES_CBC_CMAC_256.equals(oid)
        || ID_PACE_ECDH_IM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_ECDH_IM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_ECDH_IM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_ECDH_IM_AES_CBC_CMAC_256.equals(oid)
        || ID_PACE_ECDH_CAM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_ECDH_CAM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_ECDH_CAM_AES_CBC_CMAC_256.equals(oid)) {
      return "ECDH";
    }

    throw new NumberFormatException("Unknown OID: \"" + oid + "\"");
  }

  /**
   * Returns the encryption algorithm described in the PACE protocol object identifier.
   *
   * @param oid the PACE protocol object identifier
   *
   * @return a encryption algorithm as JCE mnemonic string
   */
  public static String toCipherAlgorithm(String oid) {
    if (ID_PACE_DH_GM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_DH_IM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_ECDH_GM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_ECDH_IM_3DES_CBC_CBC.equals(oid)
        ) {
      return "DESede";
    } else if (ID_PACE_DH_GM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_DH_GM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_DH_GM_AES_CBC_CMAC_256.equals(oid)
        || ID_PACE_DH_IM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_DH_IM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_DH_IM_AES_CBC_CMAC_256.equals(oid)
        || ID_PACE_ECDH_GM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_ECDH_GM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_ECDH_GM_AES_CBC_CMAC_256.equals(oid)
        || ID_PACE_ECDH_IM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_ECDH_IM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_ECDH_IM_AES_CBC_CMAC_256.equals(oid)
        || ID_PACE_ECDH_CAM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_ECDH_CAM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_ECDH_CAM_AES_CBC_CMAC_256.equals(oid)) {
      return "AES";
    }

    throw new NumberFormatException("Unknown OID: \"" + oid + "\"");
  }

  /**
   * Returns the digest algorithm described in the PACE protocol object identifier.
   *
   * @param oid the PACE protocol object identifier
   *
   * @return a digest algorithm as JCE mnemonic string
   */
  public static String toDigestAlgorithm(String oid) {
    if (ID_PACE_DH_GM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_DH_IM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_ECDH_GM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_ECDH_IM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_DH_GM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_DH_IM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_ECDH_GM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_ECDH_IM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_ECDH_CAM_AES_CBC_CMAC_128.equals(oid)) {
      return "SHA-1";
    } else if (ID_PACE_DH_GM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_DH_IM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_ECDH_GM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_ECDH_IM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_ECDH_CAM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_DH_GM_AES_CBC_CMAC_256.equals(oid)
        || ID_PACE_DH_IM_AES_CBC_CMAC_256.equals(oid)
        || ID_PACE_ECDH_GM_AES_CBC_CMAC_256.equals(oid)
        || ID_PACE_ECDH_IM_AES_CBC_CMAC_256.equals(oid)
        || ID_PACE_ECDH_CAM_AES_CBC_CMAC_256.equals(oid)) {
      return "SHA-256";
    }

    throw new NumberFormatException("Unknown OID: \"" + oid + "\"");
  }

  /**
   * Returns the key length (128, 192, or 256) described in the given PACE protocol object identifier.
   *
   * @param oid a PACE protocol object identifier
   *
   * @return the key length in bits
   */
  public static int toKeyLength(String oid) {
    if (ID_PACE_DH_GM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_DH_IM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_ECDH_GM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_ECDH_IM_3DES_CBC_CBC.equals(oid)
        || ID_PACE_DH_GM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_DH_IM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_ECDH_GM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_ECDH_IM_AES_CBC_CMAC_128.equals(oid)
        || ID_PACE_ECDH_CAM_AES_CBC_CMAC_128.equals(oid)) {
      return 128;
    } else if (ID_PACE_DH_GM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_ECDH_GM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_DH_IM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_ECDH_IM_AES_CBC_CMAC_192.equals(oid)
        || ID_PACE_ECDH_CAM_AES_CBC_CMAC_192.equals(oid)) {
      return 192;
    } else if (ID_PACE_DH_GM_AES_CBC_CMAC_256.equals(oid)
        || ID_PACE_DH_IM_AES_CBC_CMAC_256.equals(oid)
        || ID_PACE_ECDH_GM_AES_CBC_CMAC_256.equals(oid)
        || ID_PACE_ECDH_IM_AES_CBC_CMAC_256.equals(oid)
        || ID_PACE_ECDH_CAM_AES_CBC_CMAC_256.equals(oid)) {
      return 256;
    }

    throw new NumberFormatException("Unknown OID: \"" + oid + "\"");
  }

  /**
   * Derives a JCE algorithm parameter specification from a PACE standard domain parameter integer.
   *
   * @param stdDomainParam the standard domain parameter
   *
   * @return a JCE algorithm parameter specification
   */
  public static AlgorithmParameterSpec toParameterSpec(BigInteger stdDomainParam) {
    return toParameterSpec(stdDomainParam.intValue());
  }

  /**
   * Derives a JCE algorithm parameter specification from a PACE standard domain parameter integer.
   *
   * @param stdDomainParam the standard domain parameter
   *
   * @return a JCE algorithm parameter specification
   */
  public static AlgorithmParameterSpec toParameterSpec(int stdDomainParam) {
    switch (stdDomainParam) {
      case PARAM_ID_GFP_1024_160:
        return PARAMS_GFP_1024_160;
      case PARAM_ID_GFP_2048_224:
        return PARAMS_GFP_2048_224;
      case PARAM_ID_GFP_2048_256:
        return PARAMS_GFP_2048_256;
      case PARAM_ID_ECP_NIST_P192_R1:
        return PARAMS_ECP_NIST_P192_R1;
      case PARAM_ID_ECP_BRAINPOOL_P192_R1:
        return PARAMS_ECP_BRAINPOOL_P192_R1;
      case PARAM_ID_ECP_NIST_P224_R1:
        return PARAMS_ECP_NIST_P224_R1;
      case PARAM_ID_ECP_BRAINPOOL_P224_R1:
        return PARAMS_ECP_BRAINPOOL_P224_R1;
      case PARAM_ID_ECP_NIST_P256_R1:
        return PARAMS_ECP_NIST_P256_R1;
      case PARAM_ID_ECP_BRAINPOOL_P256_R1:
        return PARAMS_ECP_BRAINPOOL_P256_R1;
      case PARAM_ID_ECP_BRAINPOOL_P320_R1:
        return PARAMS_ECP_BRAINPOOL_P320_R1;
      case PARAM_ID_ECP_NIST_P384_R1:
        return PARAMS_ECP_NIST_P384_R1;
      case PARAM_ID_ECP_BRAINPOOL_P384_R1:
        return PARAMS_ECP_BRAINPOOL_P384_R1;
      case PARAM_ID_ECP_BRAINPOOL_P512_R1:
        return PARAMS_ECP_BRAINPOOL_P512_R1;
      case PARAM_ID_ECP_NIST_P521_R1:
        return PARAMS_ECP_NIST_P521_R1;
      default:
        throw new NumberFormatException("Unknown standardized domain parameters " + stdDomainParam);
    }
  }

  /**
   * Derives a human readable algorithm description from a PACE standard domain parameter integer.
   *
   * @param stdDomainParam the standard domain parameter
   *
   * @return a human readable algorithm description
   */
  public static String toStandardizedParamIdString(BigInteger stdDomainParam) {
    if (stdDomainParam == null) {
      return "null";
    }

    switch (stdDomainParam.intValue()) {
      case PARAM_ID_GFP_1024_160: /* 0 */
        return "1024-bit MODP Group with 160-bit Prime Order Subgroup";
      case PARAM_ID_GFP_2048_224: /* 1 */
        return "2048-bit MODP Group with 224-bit Prime Order Subgroup";
      case PARAM_ID_GFP_2048_256: /* 2 */
        return "2048-bit MODP Group with 256-bit Prime Order Subgroup";
        /* 3 - 7 RFU */
      case PARAM_ID_ECP_NIST_P192_R1: /* 8 */
        return "NIST P-192 (secp192r1)";
      case PARAM_ID_ECP_BRAINPOOL_P192_R1:/* 9 */
        return "BrainpoolP192r1";
      case PARAM_ID_ECP_NIST_P224_R1: /* 10 */
        return "NIST P-224 (secp224r1)";
      case PARAM_ID_ECP_BRAINPOOL_P224_R1: /* 11 */
        return "BrainpoolP224r1";
      case PARAM_ID_ECP_NIST_P256_R1: /* 12 */
        return "NIST P-256 (secp256r1)";
      case PARAM_ID_ECP_BRAINPOOL_P256_R1: /* 13 */
        return "BrainpoolP256r1";
      case PARAM_ID_ECP_BRAINPOOL_P320_R1: /* 14 */
        return "BrainpoolP320r1";
      case PARAM_ID_ECP_NIST_P384_R1: /* 15 */
        return "NIST P-384 (secp384r1)";
      case PARAM_ID_ECP_BRAINPOOL_P384_R1: /* 16 */
        return "BrainpoolP384r1";
      case PARAM_ID_ECP_BRAINPOOL_P512_R1: /* 17 */
        return "BrainpoolP512r1";
      case PARAM_ID_ECP_NIST_P521_R1: /* 18 */
        return "NIST P-521 (secp521r1)";
        /* 19-31 RFU */
      default:
        return stdDomainParam.toString();
    }
  }

  /**
   * Returns an ASN1 name for the given object identifier.
   *
   * @param oid a PACE protocol object identifier
   *
   * @return an ASN1 name if known, or the original object identifier if not
   */
  private String toProtocolOIDString(String oid) {
    if (ID_PACE_DH_GM_3DES_CBC_CBC.equals(oid)) {
      return "id-PACE-DH-GM-3DES-CBC-CBC";
    }
    if (ID_PACE_DH_GM_AES_CBC_CMAC_128.equals(oid)) {
      return "id-PACE-DH-GM-AES-CBC-CMAC-128";
    }
    if (ID_PACE_DH_GM_AES_CBC_CMAC_192.equals(oid)) {
      return "id-PACE-DH-GM-AES-CBC-CMAC-192";
    }
    if (ID_PACE_DH_GM_AES_CBC_CMAC_256.equals(oid)) {
      return "id-PACE-DH-GM-AES-CBC-CMAC-256";
    }
    if (ID_PACE_DH_IM_3DES_CBC_CBC.equals(oid)) {
      return "id-PACE-DH-IM-3DES-CBC-CBC";
    }
    if (ID_PACE_DH_IM_AES_CBC_CMAC_128.equals(oid)) {
      return "id-PACE-DH-IM-AES-CBC-CMAC-128";
    }
    if (ID_PACE_DH_IM_AES_CBC_CMAC_192.equals(oid)) {
      return "id-PACE-DH-IM-AES-CBC-CMAC-192";
    }
    if (ID_PACE_DH_IM_AES_CBC_CMAC_256.equals(oid)) {
      return "id-PACE_DH-IM-AES-CBC-CMAC-256";
    }
    if (ID_PACE_ECDH_GM_3DES_CBC_CBC.equals(oid)) {
      return "id-PACE-ECDH-GM-3DES-CBC-CBC";
    }
    if (ID_PACE_ECDH_GM_AES_CBC_CMAC_128.equals(oid)) {
      return "id-PACE-ECDH-GM-AES-CBC-CMAC-128";
    }
    if (ID_PACE_ECDH_GM_AES_CBC_CMAC_192.equals(oid)) {
      return "id-PACE-ECDH-GM-AES-CBC-CMAC-192";
    }
    if (ID_PACE_ECDH_GM_AES_CBC_CMAC_256.equals(oid)) {
      return "id-PACE-ECDH-GM-AES-CBC-CMAC-256";
    }
    if (ID_PACE_ECDH_IM_3DES_CBC_CBC.equals(oid)) {
      return "id-PACE-ECDH-IM_3DES-CBC-CBC";
    }
    if (ID_PACE_ECDH_IM_AES_CBC_CMAC_128.equals(oid)) {
      return "id-PACE-ECDH-IM-AES-CBC-CMAC-128";
    }
    if (ID_PACE_ECDH_IM_AES_CBC_CMAC_192.equals(oid)) {
      return "id-PACE-ECDH-IM-AES-CBC-CMAC-192";
    }
    if (ID_PACE_ECDH_IM_AES_CBC_CMAC_256.equals(oid)) {
      return "id-PACE-ECDH-IM-AES-CBC-CMAC-256";
    }
    if (ID_PACE_ECDH_CAM_AES_CBC_CMAC_128.equals(oid)) {
      return "id-PACE-ECDH-CAM-AES-CBC-CMAC-128";
    }
    if (ID_PACE_ECDH_CAM_AES_CBC_CMAC_192.equals(oid)) {
      return "id-PACE-ECDH-CAM-AES-CBC-CMAC-192";
    }
    if (ID_PACE_ECDH_CAM_AES_CBC_CMAC_256.equals(oid)) {
      return "id-PACE-ECDH-CAM-AES-CBC-CMAC-256";
    }

    return oid;
  }
}
