package nashid.verify.sdk.data.repository

import nashid.verify.sdk.model.DocumentType
import nashid.verify.sdk.utils.Constants.ID_CARD_REGEX1
import nashid.verify.sdk.utils.Constants.ID_CARD_REGEX2
import nashid.verify.sdk.utils.Constants.ID_CARD_REGEX3
import nashid.verify.sdk.utils.Loggers
import nashid.verify.sdk.utils.ScanDocument
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.helpers.CardReaderHelper.Companion.cleanString
import nashid.verify.sdk.utils.helpers.CardReaderHelper.Companion.isBahCard
import nashid.verify.sdk.utils.helpers.CardReaderHelper.Companion.isKuwCard
import nashid.verify.sdk.utils.helpers.CardReaderHelper.Companion.isOmanCard
import nashid.verify.sdk.utils.helpers.CardReaderHelper.Companion.isUaeCard
import nashid.verify.sdk.utils.helpers.CardReaderHelper.Companion.replaceAlfaWithNumber
import nashid.verify.sdk.utils.helpers.CardReaderHelper.Companion.replaceNumberWithAlfa
import nashid.verify.sdk.viewmodel.CameraXLiveData
import java.util.Locale
import java.util.Objects
import java.util.regex.Pattern

class IdCardRepository(
    private val cameraXLiveData: CameraXLiveData,
    private val scanDocument: ScanDocument,
) {
    private val logger = Loggers.withTag("IdCardRepository")

    fun readIdCardBackView(textAll: Array<String>) {
        var gender = ""
        val pattern1 = Pattern.compile(ID_CARD_REGEX1)
        val pattern2 = Pattern.compile(ID_CARD_REGEX2)
        val pattern3 = Pattern.compile(ID_CARD_REGEX3)
        if (cameraXLiveData.getAlgoHeaderDetect().value == false) {
            if (Utility.getInstance().isUAECard == true) {
                if (isUaeCard(java.lang.String.join(" ", *textAll))) {
                    cameraXLiveData.setAlgoHeaderDetect(true)
                }
            } else if (Utility.getInstance().isBahCard == true) {
                if (isBahCard(java.lang.String.join(" ", *textAll))) {
                    cameraXLiveData.setAlgoHeaderDetect(true)
                }
            } else if (Utility.getInstance().isKuwCard == true) {
                if (isKuwCard(java.lang.String.join(" ", *textAll))) {
                    cameraXLiveData.setAlgoHeaderDetect(true)
                }
            } else if (Utility.getInstance().isOmanCard == true) {
                if (isOmanCard(java.lang.String.join(" ", *textAll))) {
                    cameraXLiveData.setAlgoHeaderDetect(true)
                }
            } else {
                if (isOmanCard(java.lang.String.join(" ", *textAll))) {
                    cameraXLiveData.setAlgoHeaderDetect(true)
                }
            }
        }
        if (cameraXLiveData.getAlgoHeaderDetect().value == true) {
            if (cameraXLiveData.getIdLine3().value == true) {
                var validLine3Found = false
                for (data in textAll) {
                    var updatedString = StringBuilder(data.replace(" ", ""))
                    val removeKString = removeKValueFromString(updatedString.toString().uppercase())
                    updatedString = StringBuilder(removeKString.replace("«", "<"))
                    val matcher3 = pattern3.matcher(updatedString.toString())
                    if (matcher3.matches()) {
                        val originalName = updatedString.toString().replace("<", " ").replace("«", " ").trim()
                        val nameParts = originalName.split(" ")
                        val formattedName = StringBuilder()
                        formattedName.append(nameParts[0])
                        formattedName.append("<")
                        for (i in 1 until nameParts.size) {
                            formattedName.append(nameParts[i])
                            if (i < nameParts.size - 1) {
                                formattedName.append("<")
                            }
                        }
                        while (formattedName.length < 30) {
                            formattedName.append("<")
                        }
                        if (formattedName.length == 30) {
                            Utility.getInstance().name = originalName
                            Utility.getInstance().mrzLine3 = formattedName.toString()
                            cameraXLiveData.setIdLine3(false)
                            validLine3Found = true
                            break
                        }
                    }
                }
                if (!validLine3Found) {
                    cameraXLiveData.setIsSuccess(false)
                    return
                }
            }
            if (cameraXLiveData.getIdLine3().value == false) {
                for (data in textAll) {
                    var updatedString = StringBuilder(data.replace(" ", ""))
                    val removeKString = removeKValueFromString(updatedString.toString().uppercase())
                    updatedString = StringBuilder(removeKString.replace("«", "<"))
                    val matcher3 = pattern3.matcher(updatedString.toString())
                    logger.log("Attempting to match name pattern with string: $updatedString")

                    // reading person name MRZ
                    if (matcher3.matches() && cameraXLiveData.getIdLine3().value == true) {
                        logger.log("MRZ3 pattern matched for name")

                        // Store the original name without extra formatting
                        Utility.getInstance().name = updatedString.toString().replace("<", " ").replace("«", " ").trim()
                        val rawName = updatedString.toString()
                        logger.log("Raw name from MRZ: $rawName")

                        Utility.getInstance().name =
                            rawName
                                .replace("<", " ")
                                .replace("«", " ")
                                .trim()
                        logger.log("Processed name: ${Utility.getInstance().name}")

                        // Format name for MRZ line 3 with proper padding
                        val nameParts = Utility.getInstance().name?.split(" ") ?: listOf()
                        logger.log("Name parts: ${nameParts.joinToString(", ")}")

                        val formattedName = StringBuilder()

                        // Add first name
                        formattedName.append(nameParts.getOrNull(0) ?: "")
                        formattedName.append("<")
                        logger.log("Added first name: $formattedName")

                        // Add remaining name parts
                        for (i in 1 until (nameParts.size)) {
                            formattedName.append(nameParts[i])
                            if (i < nameParts.size - 1) {
                                formattedName.append("<")
                            }
                            logger.log("Added name part $i: ${nameParts[i]}")
                        }

                        // Pad with '<' to reach 30 characters
                        while (formattedName.length < 30) {
                            formattedName.append("<")
                        }
                        logger.log("Final formatted MRZ3 name: $formattedName")

                        Utility.getInstance().mrzLine3 = formattedName.toString()
                        logger.log("Stored MRZ3 line in Utility: ${Utility.getInstance().mrzLine3}")
                        cameraXLiveData.setIdLine3(false)
                        logger.log("Name processing completed successfully")
                    } else {
                        logger.log("Name pattern did not match or IdLine3 was false")
                        logger.log("Matcher result: ${matcher3.matches()}")
                        logger.log("IdLine3 value: ${cameraXLiveData.getIdLine3().value}")
                    }
                    val index = updatedString.indexOf("MN")
                    var replacedString = updatedString.toString()
                    // Replace the 0 with O if MRZ contain MN(OMN) string
                    if (index > 1 && updatedString[index - 1] == '0') {
                        val firstPart =
                            updatedString.substring(
                                0,
                                index - 1,
                            ) // Exclude the character before "MN"
                        var secondPart = updatedString.substring(index - 1) // Include "MN" and the rest of the string
                        if (secondPart.startsWith("0")) {
                            secondPart = secondPart.replace("0", "O") // Replace "O0" with "00"
                        }
                        replacedString = firstPart + secondPart // Concatenate the two parts back together
                    }
                    // replace the text at last position of string if last letter is char instead of digit
                    val lastChar = replacedString[replacedString.length - 1]
                    if (Character.isLetter(lastChar)) {
                        if (lastChar == 'O' || lastChar == 'o') {
                            replacedString = replacedString.substring(0, replacedString.length - 1) + '0'
                        }
                    }
                    if (replacedString.length >= 25) {
                        replacedString = addCharactersToReachLength(replacedString, 30)
                    }
                    val matcher2 = pattern2.matcher(cleanString(replacedString))
                    var matcherString: String? = updatedString.toString()
                    if (updatedString.length > 15) {
                        matcherString = updatedString.substring(0, 15)
                    }

                    // Reading the MRZ1 line
                    val matcher1 = pattern1.matcher(matcherString.toString())
                    if (matcher1.matches() && cameraXLiveData.getIdLine1().value == true) {
                        val length = 30 - updatedString.length
                        for (i in 0 until length) {
                            updatedString.append("<")
                        }
                        Utility.getInstance().mrzLine1 = removeKValueFromString(updatedString.toString().uppercase())
                        cameraXLiveData.setIdNo(
                            replaceAlfaWithNumber(
                                Objects.requireNonNull(
                                    matcher1.group(3),
                                ),
                            ),
                        )
                        Utility.getInstance().mrzLine1 = updatedString.toString()

                        if (Utility.getInstance().isKuwCard == true) {
                            // Extract the document number from the end portion for Kuwait cards
                            val fullMrz = updatedString.toString()
                            val kuwaitDocMatch = ".*?(\\d{12})[<]*$".toRegex().find(fullMrz)
                            if (kuwaitDocMatch != null) {
                                val docNumber = kuwaitDocMatch.groupValues[1] // Get the captured group
                                logger.log("Full MRZ: $fullMrz")
                                logger.log("Kuwait ID document number extracted: $docNumber")
                                cameraXLiveData.setIdNo(docNumber)
                            } else {
                                val lastTwelveDigits = "\\d{12}".toRegex().findAll(fullMrz).lastOrNull()?.value
                                if (lastTwelveDigits != null) {
                                    logger.log("Kuwait ID document number (last 12 digits): $lastTwelveDigits")
                                    cameraXLiveData.setIdNo(lastTwelveDigits)
                                } else {
                                    logger.log("Kuwait ID pattern not found, falling back to default")
                                    cameraXLiveData.setIdNo(
                                        replaceAlfaWithNumber(
                                            Objects.requireNonNull(
                                                matcher1.group(3),
                                            ),
                                        ),
                                    )
                                }
                            }
                        } else {
                            // Original behavior for other cards
                            cameraXLiveData.setIdNo(
                                replaceAlfaWithNumber(
                                    Objects.requireNonNull(
                                        matcher1.group(3),
                                    ),
                                ),
                            )
                        }

                        val country = replaceNumberWithAlfa(Objects.requireNonNull(matcher1.group(2)))
                        Utility.getInstance().country = country
                        Utility.getInstance().documentType = matcher1.group(1)
                        cameraXLiveData.setIdLine1(false)
                    }
                    // Reading MRZLine 2
                    if (replacedString.length == 30 && matcher2.matches() && cameraXLiveData.getIdLine2().value == true) {
                        updatedString = StringBuilder(cleanString(replacedString))
                        Utility.getInstance().mrzLine2 = updatedString.toString()
                        cameraXLiveData.setDOB(matcher2.group(1))
                        cameraXLiveData.setExpiryDate(matcher2.group(4))
                        val rawGender = matcher2.group(3)?.toString()?.trim()?.uppercase()
                        gender =
                            when (rawGender) {
                                "F", "E" -> "F" // Sometimes OCR might read F as E
                                "M" -> "M"
                                "X" -> "X"
                                else -> {
                                    ""
                                }
                            }

                        Utility.getInstance().gender = gender
                        val nationality = matcher2.group(6)
                        Utility.getInstance().nationality = nationality

                        // Special handling for Kuwait ID address number
                        if (Utility.getInstance().isKuwCard == true) {
                            // Extract address number after EGY in MRZ line 2
                            val addressMatch = "EGY(\\d{8})".toRegex().find(updatedString.toString())
                            if (addressMatch != null) {
                                val addressNumber = addressMatch.groupValues[1]
                                logger.log("Kuwait ID address number extracted: $addressNumber")
                                Utility.getInstance().addressNumber = addressNumber
                            }
                        }

                        cameraXLiveData.setIdLine2(false)
                    }
                    if (cameraXLiveData.getIdLine1().value == false && cameraXLiveData.getIdLine2().value == false && cameraXLiveData.getIdLine3().value == false) {
                        break
                    }
                }
            }
        }
        //        will call this line when successfully read all MRZLines
        if (Utility.getInstance().mrzLine1 != null && Utility.getInstance().mrzLine2 != null && Utility.getInstance().mrzLine3 != null) {
            logger.log("All MRZ lines found. Proceeding with validation")

            if (cameraXLiveData.getIsFrontViewScanned().value == false) {
                cameraXLiveData.setIdNo(null)
                cameraXLiveData.setDOB(null)
                cameraXLiveData.setExpiryDate(null)
                cameraXLiveData.setIdLine1(true)
                cameraXLiveData.setIdLine2(true)
                cameraXLiveData.setIdLine3(true)
                cameraXLiveData.setIsSuccess(false)
                //                isSuccess = false;
            } else if (cameraXLiveData.getExpiryDate().value != null && cameraXLiveData.getExpiryDate().value!!.isNotEmpty()) {
                if (cameraXLiveData.getIsFrontViewScanned().value == true &&
                    Objects.requireNonNull<String?>(
                        cameraXLiveData.getIdNo().value,
                    ).isNotEmpty() &&
                    Objects.requireNonNull<String?>(
                        cameraXLiveData.dOB.value,
                    ).isNotEmpty()
                ) {
                    val documentNumber = cameraXLiveData.getIdNo().value
                    val dateOfBirthDay = cameraXLiveData.dOB.value
                    val expiryDate = cameraXLiveData.getExpiryDate().value
                    try {
                        if (expiryDate!!.isNotEmpty()) {
                            scanDocument.documentNo = documentNumber
                            scanDocument.dateOfBirth = dateOfBirthDay
                            scanDocument.expiryDate = expiryDate
                            Utility.getInstance().passportNumber = documentNumber
                            Utility.getInstance().dateOfBirth = dateOfBirthDay
                            Utility.getInstance().expiryDate = expiryDate
                            Utility.getInstance().gender = gender.uppercase()
                            cameraXLiveData.setFlag(false)
                            cameraXLiveData.setCloseAnalysisUseCase(true)
                            cameraXLiveData.setIsSuccess(true)
                            cameraXLiveData.setCTimer(true)
                            cameraXLiveData.setCaptureAnImage(true)
                        } else {
                            cameraXLiveData.setIsSuccess(false)
                        }
                    } catch (e: Exception) {
                        e.printStackTrace()
                        cameraXLiveData.setIsSuccess(false)
                    }
                } else {
                    cameraXLiveData.setIsSuccess(false)
                }
            } else {
                cameraXLiveData.setIsSuccess(false)
            }
        } else {
            logger.log("Missing MRZ lines - MRZ1: ${Utility.getInstance().mrzLine1 != null}, MRZ2: ${Utility.getInstance().mrzLine2 != null}, MRZ3: ${Utility.getInstance().mrzLine3 != null}")
            cameraXLiveData.setIsSuccess(false)
        }
    }

    fun readIdCardFrontView(originalText: String) {
        var text = originalText
        text = text.replace(" ", "").uppercase(Locale.getDefault())
        text = text.replace("\n", " ").uppercase(Locale.getDefault())
        logger.log("Front view scanning : $text")

        when (SdkConfig.viewType) {
            DocumentType.OMAN_ID -> {
                if (text.contains("SULTANATEOFOMAN") && (text.contains("IDENTITY CARD") || text.contains("RESIDENT CARD"))) {
                    handleOmanFrontView()
                } else {
                    cameraXLiveData.setIsSuccess(false)
                }
            }
            DocumentType.KUWAITI_ID -> {
                if (text.contains("STATEOFKUWAITCIVILIDCARD") || text.contains("CivilID")) {
                    handleKuwFrontView()
                } else {
                    cameraXLiveData.setIsSuccess(false)
                }
            }
            DocumentType.EMIRATI_ID -> {
                if (text.contains("UNITEDARABEMIRATES") && text.contains("IDENTITYCARD") || text.contains("UNITEDARABEMIRATES")) {
                    handleUAEFrontView()
                } else {
                    cameraXLiveData.setIsSuccess(false)
                }
            }
            DocumentType.BAHRAINI_ID -> {
                if (text.contains("KINGDOMOFBAHRAIN") && text.contains("IDENTITYCARD") || text.contains("KINGDOMOFBAHRAIN")) {
                    handleBahCard()
                } else {
                    cameraXLiveData.setIsSuccess(false)
                }
            }
            DocumentType.QATARI_ID -> {
                if (text.contains("STATEOFQATAR") || text.contains("QATARID")) {
                    handleQatarView()
                } else {
                    cameraXLiveData.setIsSuccess(false)
                }
            }
            DocumentType.SAUDI_ID -> {
                if (text.contains("KINGDOMOFSAUDIARABIA") || text.contains("MINISTRYOFINTERIOR") ||
                    text.contains("بطاقةالهويةالوطنية") || text.contains("وزارةالداخلية") ||
                    text.contains("المملكةالعربيةالسعودية") || text.contains("نسخة")
                ) {
                    handleKSAView()
                } else {
                    cameraXLiveData.setIsSuccess(false)
                }
            }
            else -> {
                cameraXLiveData.setIsSuccess(false)
            }
        }
        cameraXLiveData.setCTimer(false)
    }

    private fun handleQatarView() {
        cameraXLiveData.apply {
            setIdNo(null)
            setDOB(null)
            setExpiryDate(null)
            setIsFrontView(true)
            setIdLine1(false)
            setIdLine2(false)
            setIdLine3(false)
            setCaptureAnImage(true)
            setIsSuccess(true)
        }
        Utility.getInstance().apply {
            isBarCodeCard = true
            isQatarCard = true
            isUAECard = false
            isBahCard = false
            isOmanCard = false
        }
        cameraXLiveData.setCTimer(false)
    }

    private fun handleKSAView() {
        cameraXLiveData.apply {
            setIdNo(null)
            setDOB(null)
            setExpiryDate(null)
            setIsFrontView(true)
            setIdLine1(false)
            setIdLine2(false)
            setIdLine3(false)
            setCaptureAnImage(true)
            setIsSuccess(true)
        }
        Utility.getInstance().apply {
            isBarCodeCard = true
            isKSACard = true
            isUAECard = false
            isBahCard = false
            isOmanCard = false
        }
        cameraXLiveData.setCTimer(false)
    }

    private fun handleOmanFrontView() {
        cameraXLiveData.setIdNo(null)
        cameraXLiveData.setDOB(null)
        cameraXLiveData.setExpiryDate(null)
        cameraXLiveData.setIsFrontView(true)
        cameraXLiveData.setIdLine1(true)
        cameraXLiveData.setIdLine2(true)
        cameraXLiveData.setCaptureAnImage(true)
        cameraXLiveData.setIsSuccess(true)
        Utility.getInstance().isOmanCard = true
    }

    private fun handleUAEFrontView() {
        cameraXLiveData.setIdNo(null)
        cameraXLiveData.setDOB(null)
        cameraXLiveData.setExpiryDate(null)
        cameraXLiveData.setIsFrontView(true)
        cameraXLiveData.setIdLine1(true)
        cameraXLiveData.setIdLine2(true)
        cameraXLiveData.setCaptureAnImage(true)
        cameraXLiveData.setIsSuccess(true)
        Utility.getInstance().isUAECard = true
    }

    private fun handleKuwFrontView() {
        cameraXLiveData.setIdNo(null)
        cameraXLiveData.setDOB(null)
        cameraXLiveData.setExpiryDate(null)
        cameraXLiveData.setIsFrontView(true)
        cameraXLiveData.setIdLine1(true)
        cameraXLiveData.setIdLine2(true)
        cameraXLiveData.setCaptureAnImage(true)
        cameraXLiveData.setIsSuccess(true)
        Utility.getInstance().isKuwCard = true
    }

    private fun handleBahCard() {
        cameraXLiveData.setIdNo(null)
        cameraXLiveData.setDOB(null)
        cameraXLiveData.setExpiryDate(null)
        cameraXLiveData.setIsFrontView(true)
        cameraXLiveData.setIdLine1(true)
        cameraXLiveData.setIdLine2(true)
        cameraXLiveData.setCaptureAnImage(true)
        cameraXLiveData.setIsSuccess(true)
        Utility.getInstance().isBahCard = true
    }

    private fun removeKValueFromString(modifiedMRZText: String): String {
        val latest = StringBuilder()
        for (i in modifiedMRZText.indices) {
            if (modifiedMRZText[i] == 'K' || modifiedMRZText[i] == 'C') {
                val prevChar = if (i > 0) modifiedMRZText[i - 1] else null
                val nextChar = if (i < modifiedMRZText.length - 1) modifiedMRZText[i + 1] else null
                if ((prevChar == '<' || prevChar == 'K' || prevChar == 'C') && (nextChar == '<' || nextChar == 'K' || nextChar == 'C')) {
                    latest.append('<')
                } else {
                    latest.append('K')
                }
            } else {
                latest.append(modifiedMRZText[i])
            }
        }
        return latest.toString()
    }

    private fun addCharactersToReachLength(
        input: String,
        desiredLength: Int,
    ): String {
        if (input.length >= desiredLength) {
            return input
        }
        val lastChar = input[input.length - 1]
        if (Character.isDigit(lastChar)) {
            val additionalChars = desiredLength - input.length
            val sb = StringBuilder(input)
            for (i in 0 until additionalChars) {
                sb.insert(sb.length - 1, '<')
            }
            return sb.toString()
        }
        return input
    }
}
