package nashid.verify.sdk.data.repository

import android.content.Context
import com.google.mlkit.vision.text.Text.TextBlock
import nashid.verify.sdk.model.DocumentType
import nashid.verify.sdk.utils.Constants.PASSPORT_TD_3_LINE_1_REGEX
import nashid.verify.sdk.utils.Constants.PASSPORT_TD_3_LINE_2_REGEX
import nashid.verify.sdk.utils.ScanDocument
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.helpers.CardReaderHelper.Companion.replaceAlfaWithNumber
import nashid.verify.sdk.utils.helpers.CardReaderHelper.Companion.replaceNumberWithAlfa
import nashid.verify.sdk.viewmodel.CameraXLiveData
import nashid.verify.sdkNew.R
import java.util.Objects
import java.util.regex.Pattern

class PassportRepository(
    private val cameraXLiveData: CameraXLiveData,
    private val scanDocument: ScanDocument,
    private val context: Context,
) {
    private var passportMRZ = ""
    private var name = ""

    fun replaceSecondChar(input: String): String {
        if (input.length < 2) return input // Ensure the string has at least 2 characters

        return if (input[1] in listOf('K', 'k', 'C', 'c')) {
            input.substring(0, 1) + "<" + input.substring(2)
        } else {
            input
        }
    }

    fun readPassportData(blocks: List<TextBlock>) {
        if (Objects.requireNonNull<String?>(cameraXLiveData.getSelectedDoc().value).equals(
                context.getString(R.string.e_passport),
                ignoreCase = true,
            )
        ) {
            for (i in blocks.indices) {
                val lines = blocks[i].lines
                for (j in lines.indices) {
                    val elements = lines[j].elements
                    for (k in elements.indices) {
                        if (elements[k].text.length > 30) {
                            try {
                                val countrySubstring =
                                    elements[k].text.substring(
                                        10,
                                        13,
                                    ) // Adjust the indices based on the actual MRZ format
                                val newCountryCode =
                                    countrySubstring.replace(
                                        "0",
                                        "O",
                                    ) // Replace with the desired new country code
                                var modifiedMRZText =
                                    elements[k].text.replace(countrySubstring, newCountryCode)
                                modifiedMRZText = modifiedMRZText.replace("«", "<")
                                modifiedMRZText = replaceSecondChar(modifiedMRZText)

                                val patternPassportTD3Line2 =
                                    Pattern.compile(PASSPORT_TD_3_LINE_2_REGEX)
                                val patternPassportTD3Line1 =
                                    Pattern.compile(PASSPORT_TD_3_LINE_1_REGEX)
                                val matcherPassportTD3Line2 =
                                    patternPassportTD3Line2.matcher(modifiedMRZText)
                                val matcherPassportTD3Line1 =
                                    patternPassportTD3Line1.matcher(modifiedMRZText.uppercase())
                                if (matcherPassportTD3Line2.matches() && passportMRZ.isEmpty()) {
                                    val latest = removeKValueFromString(modifiedMRZText.uppercase())
                                    passportMRZ = latest
                                    Utility.getInstance().mrzLine2 = passportMRZ.replace("«", "<")
                                }

                                if (matcherPassportTD3Line1.matches() && name.isEmpty()) {
                                    val latest = removeKValueFromString(modifiedMRZText.uppercase())
                                    Utility.getInstance().mrzLine1 = latest.uppercase()
                                    var newString = latest.substring(5, latest.length - 1).replace("<", " ")
                                    val cleanedString = cleanExcessSpaces(newString).lowercase()
                                    name = correctOcrMisreads(cleanedString).uppercase()
                                    Utility.getInstance().name = name.trim { it <= ' ' }
                                    Utility.getInstance().country = latest.substring(2, 5)
                                    Utility.getInstance().documentType = DocumentType.INTERNATIONAL_PASSPORT.displayName
                                }
                            } catch (e: Exception) {
                                e.printStackTrace()
                            }
                        }
                    }
                }
            }
        }
    }

    fun cleanExcessSpaces(input: String): String {
        val words = input.trim().split(Regex("\\s+")) // Split by any spaces
        val result = StringBuilder()
        var spaceCount = 0

        for (i in words.indices) {
            if (i > 0) {
                val spacesBetween = input.indexOf(words[i]) - input.indexOf(words[i - 1]) - words[i - 1].length
                if (spacesBetween > 3) {
                    break // Ignore words after excessive spaces
                }
                result.append(" ")
            }
            result.append(words[i])
        }
        return result.toString().trim()
    }

    private fun correctOcrMisreads(input: String): String {
        val replacements =
            mapOf(
                '1' to 'I',
                '0' to 'O',
                '5' to 'S',
                '8' to 'B',
                '6' to 'G',
            )

        return input.map { char -> replacements[char] ?: char }.joinToString("")
    }

    fun readAllDetailOfPassport() {
        if (Utility.getInstance().mrzLine1.isNullOrEmpty()) {
            cameraXLiveData.setIsSuccess(false)
            return
        }
        if (passportMRZ.isNotEmpty() && name.isNotEmpty()) {
            val line = passportMRZ
            var documentNumber = line.substring(0, 9)
            var nationality = line.substring(10, 13)
            var dateOfBirthDay = line.substring(13, 19)
            var gender = line.substring(20, 21)
            cameraXLiveData.setExpiryDate(line.substring(21, 27))
            nationality = replaceNumberWithAlfa(nationality)
            dateOfBirthDay = replaceAlfaWithNumber(dateOfBirthDay)
            gender = replaceNumberWithAlfa(gender)
            cameraXLiveData.setExpiryDate(
                replaceAlfaWithNumber(
                    cameraXLiveData.getExpiryDate().value.toString(),
                ),
            )
            documentNumber = documentNumber.replace("O", "0")
            if (documentNumber.isNotEmpty() &&
                dateOfBirthDay.isNotEmpty() &&
                cameraXLiveData.getExpiryDate().value!!.isNotEmpty() &&
                nationality.isNotEmpty() &&
                gender.isNotEmpty()
            ) {
                scanDocument.documentNo = documentNumber
                scanDocument.dateOfBirth = dateOfBirthDay
                scanDocument.expiryDate = cameraXLiveData.getExpiryDate().value
                Utility.getInstance().passportNumber = documentNumber
                Utility.getInstance().dateOfBirth = dateOfBirthDay
                Utility.getInstance().expiryDate = cameraXLiveData.getExpiryDate().value
                Utility.getInstance().nationality = nationality
                Utility.getInstance().gender = gender
                cameraXLiveData.setFlag(false)
                cameraXLiveData.setCloseAnalysisUseCase(true)
                cameraXLiveData.setIsSuccess(true)
                cameraXLiveData.setCaptureAnImage(true)
                passportMRZ = ""
                name = ""
            } else {
                cameraXLiveData.setIsSuccess(false)
            }
        } else {
            cameraXLiveData.setIsSuccess(false)
        }
    }

    private fun removeKValueFromString(modifiedMRZText: String): String {
        val latest = StringBuilder()
        for (i in modifiedMRZText.indices) {
            if (modifiedMRZText[i] == 'K' || modifiedMRZText[i] == 'C') {
                val prevChar = if (i > 0) modifiedMRZText[i - 1] else null
                val nextChar = if (i < modifiedMRZText.length - 1) modifiedMRZText[i + 1] else null
                if ((prevChar == '<' || prevChar == 'K' || prevChar == 'C') && (nextChar == '<' || nextChar == 'K' || nextChar == 'C')) {
                    latest.append('<')
                } else {
                    latest.append('K')
                }
            } else {
                latest.append(modifiedMRZText[i])
            }
        }
        return latest.toString()
    }
}
