package nashid.verify.sdk.viewmodel

import android.app.Application
import android.graphics.Bitmap
import androidx.lifecycle.ViewModel
import nashid.verify.sdk.VerifySDKManager
import nashid.verify.sdk.data.repository.SkipLiveNessRepository
import nashid.verify.sdk.model.DocumentVerification
import nashid.verify.sdk.model.ScanDocumentResult
import nashid.verify.sdk.utils.ArtifactType
import nashid.verify.sdk.utils.OtherConstant
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility

class SkipNfcLiveNessViewModel(
    private val skipNfcLiveNessLiveData: SkipNfcLiveNessLiveData,
    private val skipLiveNessRepository: SkipLiveNessRepository,
    private val context: Application,
    private val scanDocumentViewModel: ScanDocumentViewModel,
) : ViewModel() {
    init {
        observeScanDocumentViewModel()
    }

    private fun observeScanDocumentViewModel() {
        scanDocumentViewModel.result.observeForever { result ->
            if (result is ScanDocumentResult.FileKeyResult) {
                Utility.getInstance().saveFileKeyAndContinue(result.fileKey, result.artifactType)
            }
            if (result is ScanDocumentResult.Error) {
                skipLiveNessRepository.callToComparison()
            }
        }
    }

    fun handleInternetConnectionData(isInternetAvailable: Boolean) {
        skipNfcLiveNessLiveData.setInternetConnection(isInternetAvailable)
        if (!isInternetAvailable) {
            skipNfcLiveNessLiveData.setInternetConnection(false)
        } else {
            skipNfcLiveNessLiveData.setIsApiCalled(true)
            if (Utility.getInstance().ocrScanningFailed()) {
                skipLiveNessRepository.callToComparison()
            } else if (!SdkConfig.isFaceMatchingEnabled && !SdkConfig.isPassiveLiveNessEnabled) {
                skipLiveNessRepository.callToComparison()
            } else {
                skipNfcLiveNessLiveData.setIsLiveness(true)
            }
        }
    }

    fun getLiveNessData() = skipNfcLiveNessLiveData

    fun uploadArtifact(
        bitmap: Bitmap?,
        artifactType: ArtifactType,
    ) {
        val file = Utility.getInstance().bitmapToFile(bitmap!!, context)
        val filePart = file?.let { Utility.getInstance().createMultipartBody(it) }
        if (filePart != null) {
            scanDocumentViewModel.uploadArtifact(
                SdkConfig.registerToken!!,
                Utility.getInstance().getAppKey(),
                filePart,
                artifactType.type,
            )
        }
    }

    fun updateArtifacts() {
        scanDocumentViewModel.updateRequiredArtifacts()
    }

    private fun submitVerification(verification: DocumentVerification) {
        VerifySDKManager.getInstance().submitVerification(verification)
    }

    fun validateJustCompletion() {
        skipLiveNessRepository.callToComparison()
    }

    fun submitVerification(
        latitude: Double,
        longitude: Double,
        isShowStartScanning: Boolean,
    ) {
        prepareResultData(
            latitude,
            longitude,
            isShowStartScanning,
            callback =
                object : DocumentVerificationCallback {
                    override fun onResult(result: DocumentVerification) {
                        submitVerification(result)
                    }
                },
        )
    }

    private fun prepareResultData(
        latitude: Double,
        longitude: Double,
        isShowStartScanning: Boolean,
        callback: DocumentVerificationCallback,
    ) {
        var isFlowCompleted = true
        if (isShowStartScanning) {
            skipNfcLiveNessLiveData.setIsLiveness(false)
            isFlowCompleted = false
        }
        val documentType = SdkConfig.viewType.value
        skipLiveNessRepository.createMetaData(latitude, longitude) { metaData ->
            val isNFCEnabled = SdkConfig.isNfcScanningEnabled && !OtherConstant.getInstance().skipNFC
            val data = skipLiveNessRepository.createData(isNFCEnabled, documentType, skipNfcLiveNessLiveData.getIsLiveness().value)
            val artifacts = skipLiveNessRepository.handleArtifacts()
            val result =
                DocumentVerification(
                    metaDataModel = metaData,
                    type = documentType,
                    data = data,
                    artifacts = artifacts,
                    status =
                        if (Utility.getInstance().ocrScanningFailed() || Utility.getInstance().missingArtifacts) {
                            2
                        } else if (isFlowCompleted) {
                            1
                        } else {
                            2
                        },
                )
            callback.onResult(result)
        }
    }

    override fun onCleared() {
        super.onCleared()
        skipNfcLiveNessLiveData.clear()
    }

    interface DocumentVerificationCallback {
        fun onResult(result: DocumentVerification)
    }
}
