package nashid.verify.sdk.ui

import android.content.Intent
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Matrix
import android.os.Bundle
import android.util.TypedValue
import android.view.View
import android.view.animation.AnimationUtils
import android.widget.LinearLayout
import android.widget.Toast
import androidx.activity.addCallback
import androidx.activity.result.contract.ActivityResultContracts
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import nashid.verify.sdk.utils.ArtifactType
import nashid.verify.sdk.utils.PermissionAndLocationHelper
import nashid.verify.sdk.utils.SdkConfig
import nashid.verify.sdk.utils.Utility
import nashid.verify.sdk.utils.helpers.ErrorUtility
import nashid.verify.sdk.utils.helpers.TextSizeConverter
import nashid.verify.sdk.viewmodel.SkipNfcLiveNessViewModel
import nashid.verify.sdkNew.R
import nashid.verify.sdkNew.databinding.ActivitySkipNfcBinding
import org.koin.android.ext.android.inject

class SkipNfcLiveNessActivity : BaseActivity() {
    private lateinit var binding: ActivitySkipNfcBinding
    private val textSizeConverter: TextSizeConverter by inject()
    private val viewModel: SkipNfcLiveNessViewModel by inject()
    private val maxRetries: Int = 3
    private var attempt = 0

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivitySkipNfcBinding.inflate(layoutInflater)
        setContentView(binding.root)
        initView()
        backPress()
    }

    private fun handleIntentExtras() {
        if (intent.hasExtra(getString(R.string.doc_key))) {
            val selectedDoc = intent.getStringExtra(getString(R.string.doc_key))
            if (selectedDoc != null) {
                viewModel.getLiveNessData().setSelectedDoc(selectedDoc)
            }
        }
    }

    private fun observeViewModel() {
        viewModel.getLiveNessData().getFinishActivity().observe(this) { handleFinishActivity(it) }
        viewModel.getLiveNessData().getIsLiveness().observe(this) { handleLiveNess(it) }
        viewModel.getLiveNessData().getInternetConnection().observe(this) { handleInternetConnection(it) }
        viewModel.getLiveNessData().getHandleLocationData().observe(this) { handleLocationData(it) }
        viewModel.getLiveNessData().getLivenessBitmap().observe(this) {
            it?.let { it1 ->
                setLiveNessBitmap(
                    it1,
                )
            }
        }
        viewModel.getLiveNessData().getScanBitmap().observe(this) { it?.let { it1 -> setScanBitmap(it1) } }
        viewModel.getLiveNessData().getFaceMatchApiFail().observe(this) {}
        viewModel.getLiveNessData().getHideProgressBar().observe(this) {
            if (it) {
                if (attempt < maxRetries) {
                    attempt++
                    binding.txtValidating.text = getString(R.string.validating)
                    binding.imgLoader.visibility = View.VISIBLE
                    setLayoutAndTextSize()
                    handleLocationData(true)
                } else {
                    Toast.makeText(this@SkipNfcLiveNessActivity, getString(R.string.some_thing_went_wrong), Toast.LENGTH_SHORT).show()
                    finish()
                }
            }
        }
    }

    private fun handleInternetConnection(isAvailable: Boolean) {
        if (!isAvailable) {
            ErrorUtility.getInstance().showNoInternetDialog(this, !this.isFinishing && !this.isDestroyed)
        }
    }

    private fun setLiveNessBitmap(bitmap: Bitmap) {
        binding.imgTest123.setImageBitmap(bitmap)
    }

    private fun setScanBitmap(bitmap: Bitmap) {
        binding.imgTest12.setImageBitmap(bitmap)
    }

    private fun handleLiveNess(isLiveNess: Boolean) {
        if (isLiveNess) {
            openLiveNessScreen()
        }
    }

    private fun openLiveNessScreen() {
        val intent = Intent(this, LiveNessActivity::class.java)
        someActivityResultLauncher.launch(intent)
    }

    private val someActivityResultLauncher =
        registerForActivityResult(
            ActivityResultContracts.StartActivityForResult(),
        ) { result ->
            if (result.resultCode == RESULT_OK) {
                if (Utility.getInstance().liveImage != null) {
                    handleLiveNessImage()
                } else {
                    Utility.getInstance().setLiveNessScanFailed(true)
                    viewModel.updateArtifacts()
                    viewModel.validateJustCompletion()
                }
            } else if (result.resultCode == RESULT_CANCELED) {
                Utility.getInstance().setLiveNessScanFailed(true)
                viewModel.updateArtifacts()
                viewModel.validateJustCompletion()
            }
        }

    private fun handleLiveNessImage() {
        CoroutineScope(Dispatchers.IO).launch {
            try {
                // Process and upload active liveNess image if enabled
                if (SdkConfig.isActiveLiveNessEnabled) {
                    logger.log("Processing active liveNess image")
                    val activeImageBytes = Utility.getInstance().getActiveLivenessImage()
                    if (activeImageBytes == null) {
                        logger.log("Active liveNess image not found in Utility")
                    } else {
                        logger.log("Active liveNess image size: ${activeImageBytes.size}")
                        val activeLiveNessImage = processActiveLiveNessImage().compressImage()
                        withContext(Dispatchers.Main) {
                            binding.txtValidating.text = getString(R.string.validating)
                            binding.imgLoader.visibility = View.VISIBLE
                        }
                        viewModel.uploadArtifact(activeLiveNessImage, ArtifactType.ACTIVE_LIVENESS_IMAGE)
                        logger.log("Active liveNess image processed and uploaded")
                    }
                }

                // Process and upload passive liveNess image if enabled
                if (SdkConfig.isPassiveLiveNessEnabled) {
                    logger.log("Processing passive liveNess image")
                    val passiveLiveNessImage = processPassiveLiveNessImage().compressImage()
                    withContext(Dispatchers.Main) {
                        binding.txtValidating.text = getString(R.string.validating)
                        binding.imgLoader.visibility = View.VISIBLE
                    }
                    viewModel.uploadArtifact(passiveLiveNessImage, ArtifactType.PASSIVE_LIVENESS_IMAGE)
                    logger.log("Passive liveNess image processed and uploaded")
                }

                withContext(Dispatchers.Main) {
                    viewModel.validateJustCompletion()
                    binding.imgLoader.visibility = View.GONE
                }
            } catch (e: Exception) {
                logger.log("Error in handleLivenNessImage : $e")
                withContext(Dispatchers.Main) {
                    binding.imgLoader.clearAnimation()
                    binding.imgLoader.visibility = View.GONE
                    binding.txtValidating.text = getString(R.string.some_thing_went_wrong)
                    binding.retryButton.visibility = View.VISIBLE

                    // Mark the liveNess scan as failed
                    Utility.getInstance().setLiveNessScanFailed(true)
                    viewModel.updateArtifacts()
                }
            }
        }
    }

    private suspend fun processActiveLiveNessImage(): Bitmap =
        withContext(Dispatchers.IO) {
            val activeImageBytes = Utility.getInstance().getActiveLivenessImage()
            if (activeImageBytes == null) {
                logger.log("Active liveNess image bytes are null")
                throw IllegalStateException("Active liveNess image not found")
            }

            try {
                val liveImage =
                    BitmapFactory.decodeByteArray(
                        activeImageBytes,
                        0,
                        activeImageBytes.size,
                    ) ?: throw IllegalStateException("Failed to decode active liveNess image")

                val matrix = Matrix()
                val rotation =
                    when (liveImage.width > liveImage.height) {
                        true -> 270f
                        false -> 0f
                    }
                matrix.postRotate(rotation)

                if (rotation != 0f) {
                    Bitmap.createBitmap(
                        liveImage,
                        0,
                        0,
                        liveImage.width,
                        liveImage.height,
                        matrix,
                        true,
                    ).also {
                        if (it != liveImage) {
                            liveImage.recycle()
                        }
                    }
                } else {
                    liveImage
                }
            } catch (e: Exception) {
                logger.log("Error processing active liveNess image : $e")
                throw e
            }
        }

    private suspend fun processPassiveLiveNessImage(): Bitmap =
        withContext(Dispatchers.IO) {
            val passiveImageBytes = Utility.getInstance().getPassiveLiveNessImage()
            if (passiveImageBytes == null) {
                logger.log("Passive liveNess image bytes are null")
                throw IllegalStateException("Passive liveNess image not found")
            }

            try {
                val liveImage =
                    BitmapFactory.decodeByteArray(
                        passiveImageBytes,
                        0,
                        passiveImageBytes.size,
                    ) ?: throw IllegalStateException("Failed to decode passive liveNess image")

                val matrix = Matrix()
                val rotation =
                    when (liveImage.width > liveImage.height) {
                        true -> 270f
                        false -> 0f
                    }
                matrix.postRotate(rotation)

                if (rotation != 0f) {
                    Bitmap.createBitmap(
                        liveImage,
                        0,
                        0,
                        liveImage.width,
                        liveImage.height,
                        matrix,
                        true,
                    ).also {
                        if (it != liveImage) {
                            liveImage.recycle()
                        }
                    }
                } else {
                    liveImage
                }
            } catch (e: Exception) {
                logger.log("Error processing passive liveNess image : $e")
                throw e
            }
        }

    private fun handleFinishActivity(shouldFinish: Boolean) {
        if (shouldFinish) {
            binding.imgLoader.clearAnimation()
            binding.imgLoader.visibility = View.GONE
            binding.txtValidating.visibility = View.GONE
            Utility.getInstance().restartApp(this)
        }
    }

    private fun backPress() {
        onBackPressedDispatcher.addCallback(this) {
            if (viewModel.getLiveNessData().getIsApiCalled().value == false) onBackPressedDispatcher.onBackPressed()
        }
    }

    private fun setLayoutAndTextSize() {
        textSizeConverter.changeStatusBarColor(this)
        val layoutParams2 = binding.imgLoader.layoutParams
        layoutParams2.width = textSizeConverter.getWidth(24)
        layoutParams2.height = textSizeConverter.getHeight(24)
        binding.imgLoader.layoutParams = layoutParams2

        val marginLayoutParam = binding.txtValidating.layoutParams as LinearLayout.LayoutParams
        marginLayoutParam.setMargins(0, textSizeConverter.getPaddingOrMarginValue(32), 0, 0)
        binding.txtValidating.layoutParams = marginLayoutParam
        binding.txtValidating.setTextSize(
            TypedValue.COMPLEX_UNIT_PX,
            textSizeConverter.getTextSize(20).toFloat(),
        )

        val rotation = AnimationUtils.loadAnimation(this, R.anim.rotate)
        rotation.fillAfter = true
        binding.imgLoader.startAnimation(rotation)
    }

    private fun initView() {
        setLayoutAndTextSize()
        handleIntentExtras()
        observeViewModel()
        viewModel.handleInternetConnectionData(isInternetAvailable)
    }

    override fun onAvailable() {
        if (viewModel.getLiveNessData().getIsApiCalled().value == false) {
            viewModel.handleInternetConnectionData(
                isInternetAvailable,
            )
        }
    }

    private fun handleLocationData(callLocationData: Boolean) {
        if (callLocationData) {
            PermissionAndLocationHelper.requestPermissionAndLocation(
                this,
                object : PermissionAndLocationHelper.ResultCallback {
                    override fun onResult(
                        latitude: Double,
                        longitude: Double,
                        address: String,
                    ) {
                        viewModel.submitVerification(
                            latitude,
                            longitude,
                            false,
                        )
                    }
                },
            )
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        ErrorUtility.getInstance().unregisterConnectivityManager(this)
        viewModel.getLiveNessData().clear()
    }

    private fun Bitmap.compressImage(
        maxWidth: Int = 1024,
        maxHeight: Int = 1024,
    ): Bitmap {
        if (width <= maxWidth && height <= maxHeight) {
            return this
        }

        val ratioX = maxWidth.toFloat() / width
        val ratioY = maxHeight.toFloat() / height
        val ratio = minOf(ratioX, ratioY)

        val newWidth = (width * ratio).toInt()
        val newHeight = (height * ratio).toInt()

        return Bitmap.createScaledBitmap(this, newWidth, newHeight, true).also {
            if (this != it) {
                this.recycle()
            }
        }
    }
}
